import { useTranslation } from "react-i18next";
import {
  Ellipsis,
  Trash2,
  Download,
} from "lucide-react"
import { useEcho } from '@laravel/echo-react';
import { useQueryClient } from "@tanstack/react-query";
import { Button } from "@/components/ui/button";
import {
  useCreateBackup,
  useGetBackups,
  useDeleteBackup,
  useDownloadBackup
} from "@/components/administration/queries";
import { BaseBackup } from "@/components/administration/types";
import { DataTable } from "@/components/ui/DataTable/DataTable";
import DataTableSkeleton from "@/components/ui/DataTable/DataTableSkeleton";
import { ColumnDef } from "@tanstack/react-table";
import {
    PageLayout,
    PageHeader,
    PageTitle,
    PageDescription,
    PageSeparator,
} from "@/components/Page";
import {
  DropdownMenu,
  DropdownMenuTrigger,
  DropdownMenuContent,
  DropdownMenuItem,
} from "@/components/ui/dropdown-menu";
import { Badge, badgeVariants } from "@/components/ui/badge";
import type { VariantProps } from "class-variance-authority"
import { toast } from "sonner";
type BadgeVariant = VariantProps<typeof badgeVariants>["variant"]

export default function Backup() {
  const { t } = useTranslation();
  const { data, isLoading } = useGetBackups();
  const createBackup = useCreateBackup();
  const deleteBackup = useDeleteBackup();
  const downloadBackup = useDownloadBackup();
  const queryClient = useQueryClient();

  useEcho('NNA.Admins', 'BackupUpdate', (event: { status: string }) => {
    queryClient.invalidateQueries({ queryKey: ["backups"] });

    // Show toast notifications based on backup status
    if (event.status === 'Success' || event.status === 'completed') {
      toast.success(t('Backup completed successfully'));
    } else if (event.status === 'Failed') {
      toast.error(t('Backup failed. Check /var/www/html/nagiosna/storage/logs/laravel.log for details.'));
    } else if (event.status === 'In Progress') {
      toast.info(t('Backup is now running...'));
    }
  });

  const columns: ColumnDef<BaseBackup>[] = [
    {
      accessorKey: 'file_name',
      id: "File Name",
      header: t('File Name'),
    },
    {
      accessorKey: 'date',
      header: t('Date'),
    },
    {
      accessorKey: 'status',
      header: t('Status'),
      cell: ({ row }) => {
        const status = row.getValue("status")
        let variant: BadgeVariant = "default"
    
        if (status === "completed") {
          variant = "green"
        } else if (status === "running" || status === "Pending") {
          variant = "blue"
        } else if (status === "Failed" || status === "Stopped") {
          variant = "red"
        } 
        return <Badge variant={variant}>{String(status)}</Badge>
      },
    },
    {
      accessorKey: 'size',
      header: t('Size'),
    },
    {
      id: 'actions',
      header: '',
      cell: ({ row }) => (
        <DropdownMenu>
        <DropdownMenuTrigger asChild>
          <Button variant="ghost">
            <Ellipsis />
          </Button>
        </DropdownMenuTrigger>
        <DropdownMenuContent>
          <DropdownMenuItem onClick={() => downloadBackup.mutate(row.original.file_name)}>
            <Download />
            {t('Download')}
          </DropdownMenuItem>
          <DropdownMenuItem onClick={() => deleteBackup.mutate(row.original.file_name)}>
            <Trash2 />
            {t('Delete')}
          </DropdownMenuItem>
        </DropdownMenuContent>
      </DropdownMenu>
      ),
    },
  ];

  return (
      <PageLayout>
        <PageHeader>
          <PageTitle>{t('System Backup')}</PageTitle>
          <PageDescription>{t('Manage your Nagios Network Analyzer backups')}</PageDescription>
          <PageSeparator />
          <Button className="mb-4"
            onClick={() => {
              createBackup.mutate();
            }}
          >
            {t('Create Backup')}
          </Button>
      </PageHeader>
      {isLoading && (
          <DataTableSkeleton></DataTableSkeleton>
      )}
      {!isLoading && (
          <DataTable
            columns={columns}
            data={data || []}
          />
                  )}
    </PageLayout>
  );
}
