import { Button } from "@/components/ui/button";
import {
  Card,
  CardContent,
} from "@/components/ui/card";
import { PageLayout, PageHeader, PageTitle, PageDescription, PageSeparator } from "@/components/Page";
import { useTranslation } from "react-i18next";
import { LoaderCircle, Send } from 'lucide-react';
import { SubmitHandler, useForm } from "react-hook-form";
import { NotificationOptions } from "@/components/administration/types";
import { Form, FormLabel, FormControl, FormField, FormItem, FormDescription } from '@/components/ui/form';
import { PasswordInput } from "@/components/ui/passwordInput";
import { Input } from "@/components/ui/input";
import {
  RadioGroup,
  RadioGroupItem,
} from "@/components/ui/radio-group";
import { useGetNotificationOptions, useEditNotificationOptionsMutation, useTestNotificationOptions } from "@/components/administration/notifications/queries";
import { Switch } from "@/components/ui/switch";

const channels: { id: 'mail' | 'slack' | 'broadcast', label: string }[] = [
  {
    id: 'mail',
    label: 'Mail',
  },
  // {
  //   id: 'slack',
  //   label: 'Slack',
  // },
  // {
  //   id: 'broadcast',
  //   label: 'Broadcast',
  // },
];

function NotificationChannelForm({ notificationChannel }: { notificationChannel: Array<'mail' | 'slack' | 'broadcast'> }) {
  const { t } = useTranslation();
  const editNotificationOptionsMutation = useEditNotificationOptionsMutation();

  const form = useForm<NotificationOptions>({
    values: {
      notification_channel: notificationChannel ?? [],
    }
  });

  const onSubmit: SubmitHandler<NotificationOptions> = (data) => {
    editNotificationOptionsMutation.mutate(data);
  };

  return (
    <Card>
      <CardContent>
        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)} className="grid gap-4">
            <FormField
              control={form.control}
              name="notification_channel"
              render={() => (
                <FormItem className="flex justify-between flex-wrap gap-4">
                  <div className="space-y-1">
                    <FormLabel>{t("Notification Channel")}</FormLabel>
                    <FormDescription>
                      {t("Select the channels via which you would like to send notifications")}
                    </FormDescription>
                  </div>
                  <div className="flex items-center gap-4">
                    {channels.map((channel) => (
                      <FormField
                        key={channel.id}
                        control={form.control}
                        name="notification_channel"
                        render={({ field }) => {
                          return (
                            <FormItem
                              key={channel.id}
                              className="flex flex-row items-center gap-2"
                            >
                              <FormLabel className="text-sm font-normal">
                                {channel.label}
                              </FormLabel>
                              <FormControl>
                                <Switch
                                  checked={field.value?.includes(channel.id)}
                                  onCheckedChange={(checked) => {
                                    const updated = checked
                                      ? [...field.value, channel.id]
                                      : field.value?.filter((value) => value !== channel.id);
                                    field.onChange(updated);
                                    form.handleSubmit(onSubmit)();
                                  }}
                                />
                              </FormControl>
                            </FormItem>
                          )
                        }}
                      />
                    ))}
                  </div>
                </FormItem>
              )}
            />
          </form>
        </Form>
      </CardContent>
    </Card>
  )
}

function NotificationOptionsForm({ notificationOptions }: { notificationOptions: NotificationOptions }) {
  const { t } = useTranslation();
  const editNotificationOptionsMutation = useEditNotificationOptionsMutation();

  const form = useForm<NotificationOptions>({
    values: {
      notification_channel: notificationOptions?.notification_channel ?? [],
      mail_config: {
        send_from: notificationOptions?.mail_config?.send_from ?? "",
        mailer: notificationOptions?.mail_config?.mailer ?? "sendmail",
        smtp_config: {
          host: notificationOptions?.mail_config?.smtp_config?.host ?? "",
          port: notificationOptions?.mail_config?.smtp_config?.port ?? 587,
          username: notificationOptions?.mail_config?.smtp_config?.username ?? "",
          password: notificationOptions?.mail_config?.smtp_config?.password ?? "",
          encryption: notificationOptions?.mail_config?.smtp_config?.encryption ?? "none",
        }
      }
    }
  });

  const { isDirty } = form.formState;

  const onSubmit: SubmitHandler<NotificationOptions> = (data) => {
    editNotificationOptionsMutation.mutate(data);
  };

  return (
    <Card>
      <CardContent>
        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)}>
            <div className="flex flex-row items-center justify-between">
              <h1 className="leading-9 text-md font-large font-semibold">
                {t("Mail Settings")}
              </h1>
              <Button
                disabled={!isDirty}
                variant={isDirty ? "default" : "outline"}
                className="border"
                type="submit"
              >
                {t("Update")}
              </Button>
            </div>
            <PageSeparator />
            <FormField
              control={form.control}
              name="mail_config.send_from"
              render={({ field }) => (
                <FormItem className="flex items-center justify-between flex-wrap">
                  <FormLabel>{t("Send From")}</FormLabel>
                  <FormControl className="w-md">
                    <Input placeholder="example@doman.com" {...field} />
                  </FormControl>
                </FormItem>
              )}
            />
            <PageSeparator />
            <FormField
              control={form.control}
              name="mail_config.mailer"
              render={({ field }) => (
                <FormItem className="flex items-center justify-between flex-wrap">
                  <div>
                    <FormLabel>{t("Mailer")}</FormLabel>
                  </div>
                  <FormControl>
                    <RadioGroup
                      onValueChange={field.onChange}
                      defaultValue={field.value}
                      className="w-md gap-4"
                    >
                      {/* <FormItem className="flex items-center gap-3">
                        <FormControl>
                          <RadioGroupItem value="sendmail" />
                        </FormControl>
                        <FormLabel className="font-normal">
                          Sendmail
                        </FormLabel>
                      </FormItem> */}
                      <FormItem className="flex items-center gap-3">
                        <FormControl>
                          <RadioGroupItem value="smtp" />
                        </FormControl>
                        <FormLabel className="font-normal">
                          SMTP
                        </FormLabel>
                      </FormItem>
                    </RadioGroup>
                  </FormControl>
                </FormItem>
              )}
            />
            <PageSeparator />
            <FormField
              control={form.control}
              name="mail_config.smtp_config.host"
              render={({ field }) => (
                <FormItem className="flex items-center justify-between flex-wrap">
                  <FormLabel>{t("Host")}</FormLabel>
                  <FormControl className="w-md">
                    <Input placeholder="smtp.example.com" {...field} />
                  </FormControl>
                </FormItem>
              )}
            />
            <PageSeparator />
            <FormField
              control={form.control}
              name="mail_config.smtp_config.port"
              render={({ field }) => (
                <FormItem className="flex items-center justify-between flex-wrap">
                  <FormLabel>{t("Port")}</FormLabel>
                  <FormControl className="w-md">
                    <Input {...field} />
                  </FormControl>
                </FormItem>
              )}
            />
            <PageSeparator />
            <FormField
              control={form.control}
              name="mail_config.smtp_config.username"
              render={({ field }) => (
                <FormItem className="flex items-center justify-between flex-wrap">
                  <FormLabel>{t("Username")}</FormLabel>
                  <FormControl className="w-md">
                    <Input placeholder="johndoe" {...field} />
                  </FormControl>
                </FormItem>
              )}
            />
            <PageSeparator />
            <FormField
              control={form.control}
              name="mail_config.smtp_config.password"
              render={({ field }) => (
                <FormItem className="flex items-center justify-between flex-wrap">
                  <FormLabel>{t("Password")}</FormLabel>
                  <FormControl>
                    <div className="w-md">
                      <PasswordInput {...field} />
                    </div>
                  </FormControl>
                </FormItem>
              )}
            />
            <PageSeparator />
            <FormField
              control={form.control}
              name="mail_config.smtp_config.encryption"
              render={({ field }) => (
                <FormItem className="flex items-center justify-between flex-wrap">
                  <div>
                    <FormLabel>{t("Encryption")}</FormLabel>
                  </div>
                  <FormControl>
                    <RadioGroup
                      onValueChange={field.onChange}
                      defaultValue={field.value}
                      className="w-md gap-4"
                    >
                      <FormItem className="flex items-center gap-3">
                        <FormControl>
                          <RadioGroupItem value="none" />
                        </FormControl>
                        <FormLabel className="font-normal">
                          {t("None")}
                        </FormLabel>
                      </FormItem>
                      <FormItem className="flex items-center gap-3">
                        <FormControl>
                          <RadioGroupItem value="tls" />
                        </FormControl>
                        <FormLabel className="font-normal">
                          STARTTLS
                        </FormLabel>
                      </FormItem>
                      <FormItem className="flex items-center gap-3">
                        <FormControl>
                          <RadioGroupItem value="ssl" />
                        </FormControl>
                        <FormLabel className="font-normal">
                          SSL/TLS
                        </FormLabel>
                      </FormItem>
                    </RadioGroup>
                  </FormControl>
                </FormItem>
              )}
            />
          </form>
        </Form>
      </CardContent>
    </Card>
  )
}

export default function Notifications() {
  const { t } = useTranslation();
  const { data } = useGetNotificationOptions();
  const mailMutation = useTestNotificationOptions();
  if (!data) return;

  return (
    <PageLayout>
      <div className="flex justify-between">
        <PageHeader>
          <PageTitle>{t("Notification Settings")}</PageTitle>
          <PageDescription>{t("Configure and test your notification settings")}</PageDescription>
        </PageHeader>
        <Button
          disabled={mailMutation.isPending}
          onClick={() => mailMutation.mutate()}
          className="mt-4"
        >
          {mailMutation.isPending
            ? (
              <LoaderCircle className="animate-spin" />
            ) : (
              <Send />
            )
          }
          <div>{t("Send Test Notification")}</div>
        </Button>
      </div>
      <PageSeparator />
      <NotificationChannelForm notificationChannel={data.notification_channel} />
      <div className="my-4" />
      <NotificationOptionsForm notificationOptions={data} />
    </PageLayout>
  );
}