import { Button } from "@/components/ui/button";
import { Ellipsis, Plus, Pencil, Trash2, Shield, User as UserIcon, Settings } from "lucide-react";
import { useTranslation } from "react-i18next";
import {
  PageLayout,
  PageHeader,
  PageTitle,
  PageSeparator,
  PageDescription,
} from "@/components/Page";
import { DataTable } from "@/components/ui/DataTable/DataTable";
import DataTableSkeleton from "@/components/ui/DataTable/DataTableSkeleton";
import { ColumnDef } from "@tanstack/react-table";
import { BaseUser, User } from "@/components/user/types";
import { useGetUsers } from "@/components/user/queries";
import { useDeleteUser } from "@/components/user/queries";
import {
  DropdownMenu,
  DropdownMenuTrigger,
  DropdownMenuContent,
  DropdownMenuItem,
} from "@/components/ui/dropdown-menu";
import ImportUserDialog from "@/components/administration/ImportUserDialog";
import AddEditUserDialog from "@/components/user/AddEditUserDialog";
import { useQueryClient } from "@tanstack/react-query";
import { useState } from "react";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar"

export default function Users() {
  const { t } = useTranslation();
  const { data, isLoading } = useGetUsers();
  const queryClient = useQueryClient();
  const deleteUserMutation = useDeleteUser();

  const [editDialogOpen, setEditDialogOpen] = useState(false);
  const [editingUser, setEditingUser] = useState<User | undefined>(undefined);

  // Define columns for the DataTable
  const columns: ColumnDef<BaseUser>[] = [
    {
      accessorKey: 'username',
      header: t('User'),
      cell: ({ row }) => (
        <div className="flex items-center gap-2">
          <Avatar>
            <AvatarImage src={`/storage/avatars/${row.original.username}.svg`} className="scale-95" />
            <AvatarFallback>
              {row.original.username.slice(0, 2).toUpperCase()}
            </AvatarFallback>
          </Avatar>
          <span>{row.original.username}</span>
        </div>
      )
    },
    {
      accessorKey: 'email',
      header: t('Email'),
    },
    {
      id: 'role_id',
      header: t('Role'),
      cell: ({ row }) => (
        <span className="flex items-center gap-2">
          <div className="bg-secondary [&>svg]:size-4 size-8 flex items-center justify-center rounded-full border">
            {row.original.role.type === 'admin' ? <Shield /> : row.original.role.type === 'user' ? <UserIcon /> : <Settings />}
          </div>
          {row.original.role.name}
        </span>
      ),
    },
    {
      accessorKey: 'type',
      header: t('Account Type'),
    },
    {
      accessorKey: 'apiaccess',
      header: t('API Access'),
      cell: ({ row }) => (
        <span>{row.original.apiaccess ? t('Yes') : t('No')}</span>
      ),
    },
    {
      id: 'actions',
      header: '',
      cell: ({ row }) => (
        <DropdownMenu>
          <DropdownMenuTrigger asChild>
            <Button variant="ghost">
              <Ellipsis />
            </Button>
          </DropdownMenuTrigger>
          <DropdownMenuContent>
            <DropdownMenuItem 
              onClick={() => {
                setEditingUser(row.original as User);
                setEditDialogOpen(true);
              }}
            >
              <Pencil />
              {t('Edit')}
            </DropdownMenuItem>
            <DropdownMenuItem onClick={() => deleteUserMutation.mutate(row.original.id)}>
              <Trash2 />
              {t('Delete')}
            </DropdownMenuItem>
          </DropdownMenuContent>
        </DropdownMenu>
      ),
    },
  ];

  return (
    <PageLayout>
      <PageHeader>
        <PageTitle>{t('User Management')}</PageTitle>
        <PageDescription>{t("Create, import, edit, and delete users.")}</PageDescription>
      </PageHeader>
      <PageSeparator />
        <div className="space-x-4 pb-4">
          <AddEditUserDialog>
            <Button onClick={() => setEditingUser(undefined)}>
              <Plus />
              {t('Create User')}
            </Button>
          </AddEditUserDialog>
          
          <ImportUserDialog onSuccess={() => queryClient.invalidateQueries({ queryKey: ['Users'] })}>
            <Button> <Plus />{t('Add Users from LDAP/AD')}</Button>
          </ImportUserDialog>
        </div>
      
      {isLoading && (
        <DataTableSkeleton></DataTableSkeleton>
      )}
      {!isLoading && (
        <DataTable columns={columns} data={data || []} />
      )}

      {/* Edit User Dialog - Rendered outside the dropdown */}
      {editingUser && (
        <AddEditUserDialog 
          user={editingUser}
          open={editDialogOpen}
          onOpenChange={(open) => {
            setEditDialogOpen(open);
            if (!open) {
              setEditingUser(undefined);
            }
          }}
        />
      )}
    </PageLayout>
  );
}