import { useTranslation } from "react-i18next";
import { Button } from "@/components/ui/button";
import { DataTable } from "@/components/ui/DataTable/DataTable";
import { ColumnDef, Row } from "@tanstack/react-table";
import { DropdownMenu, DropdownMenuItem, DropdownMenuContent, DropdownMenuTrigger } from "@/components/ui/dropdown-menu";
import {
  ClipboardCheckIcon,
  Loader2Icon,
  MoreHorizontalIcon,
  PlayIcon,
  PowerIcon,
  PowerOffIcon,
  Trash2Icon,
} from "lucide-react";
import {
  useDeleteCheck,
  useForceCheck,
  useGetChecks,
  useToggleActiveCheck,
} from "@/components/alerting/queries";
import { Badge } from "@/components/ui/badge";
import { Check, SystemCheckMetric } from "@/components/alerting/types";
import { useState } from "react";
import { DataTableColumnHeader } from "@/components/ui/DataTable/DataTableColumnHeader";
import { Switch } from "@/components/ui/switch";
import EditSystemCheckModal from "@/components/alerting/modals/system-checks/EditSystemCheck";
import { useEcho } from '@laravel/echo-react';
import { useQueryClient } from '@tanstack/react-query';

export default function AlertingSystem() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  const { data: checks, isLoading: checksLoading } = useGetChecks({ check_type: 'system' });
  const { mutate: deleteCheck } = useDeleteCheck();
  const { mutate: runCheck } = useForceCheck();
  const { mutate: toggleActiveCheck } = useToggleActiveCheck();

  useEcho(`NNA.Users`, 'CheckJobCompleted', () => {
    queryClient.invalidateQueries({ queryKey: ['checks', 'system'] });
  });

  const alertingChecksColumns: ColumnDef<Check>[] = [
    {
      accessorKey: 'active',
      header: ({ column }) => {
        return <DataTableColumnHeader column={column} title="Active" />
      },
      cell: ({ row }: { row: Row<any> }) => {
        const value = row.getValue("active");

        return (
          <div className="flex px-2.5">
            <Switch
              checked={value as boolean}
              onCheckedChange={() => toggleActiveCheck(row.original.id)}
            />
          </div>
        )
      }
    },
    {
      accessorKey: 'name',
      header: ({ column }) => {
        return <DataTableColumnHeader column={column} title="Name" />
      },
    },
    {
      accessorKey: 'metric',
      header: t('Check Metric'),
      cell: ({ row }: { row: Row<any> }) => {

        const metricLabels: Record<SystemCheckMetric, string> = {
          cpu_usage: t("CPU Usage (%)"),
          used_percent: t("Memory Used (%)"),
          swap_percent: t("Memory Swap (%)"),
          available: t("Storage Available (bytes)"),
          percent: t("Storage Used (%)"),
        };

        return (
          <div>
            {metricLabels[row.original.metric as SystemCheckMetric]}
          </div>
        )
      }
    },
    {
      accessorKey: 'last_code',
      header: () => {
        return (
          <div className="w-min">
            {t('Last Status')}
          </div>
        )
      },
      cell: ({ row }: { row: Row<any> }) => {
        return <ReturnCodeStatus status_code={row.original.last_code} />
      }
    },
    {
      accessorKey: 'last_run',
      header: t('Last Run'),
      cell: ({ row }: { row: Row<any> }) => {
        if (!row.original.last_run) {
          return (
            <div>{`-`}</div>
          )
        }

        const datetime_formatted = new Date(`${row.original.last_run}`).toLocaleString()
        const datetime_array = datetime_formatted.split(",");
        return (
          <div className="flex flex-col gap-0 h-min">
            <span>{`${datetime_array[0]},`}</span>
            <span>{`${datetime_array[1]}`}</span>
          </div>
        )
      }
    },
    {
      accessorKey: 'last_stdout',
      header: t('Last Stdout'),
      cell: ({ row }: { row: Row<any> }) => {
        if (!row.original.last_stdout) {
          return (
            <div>{`-`}</div>
          )
        }

        return (
          <div className="flex text-wrap">
            {row.original.last_stdout}
          </div>
        )
      }
    },
    {
      accessorKey: 'actions',
      enableSorting: false,
      header: () => (
        <div className="hidden" />
      ),
      cell: ({ row }: { row: Row<any> }) => {
        const [dropdownOpen, setDrodownOpen] = useState(false);

        return (
          <div className="flex justify-end">
            <DropdownMenu open={dropdownOpen} onOpenChange={setDrodownOpen}>
              <DropdownMenuTrigger asChild>
                <Button size="sm" variant="ghost" className="gap-2 text-xs">
                  <MoreHorizontalIcon />
                </Button>
              </DropdownMenuTrigger>
              <DropdownMenuContent align="end">
                {
                  checks && (
                    <DropdownMenuItem onClick={() => runCheck(row.original.id)}>
                      <PlayIcon />
                      <span className="hidden lg:block">{t(`Run Check`)}</span>
                    </DropdownMenuItem>
                  )
                }
                <EditSystemCheckModal check={checks.filter((check: Check) => check.id == row.original.id)[0]} onModalClose={() => setDrodownOpen(false)} />
                <DropdownMenuItem
                  onClick={() => toggleActiveCheck(row.original.id)}
                >
                  {row.original.active ? <PowerOffIcon /> : <PowerIcon />}
                  <span className="hidden lg:block">{row.original.active ? t("Deactivate") : t("Activate")}</span>
                </DropdownMenuItem>
                <DropdownMenuItem onClick={() => deleteCheck(row.original.id)}>
                  <Trash2Icon />
                  <span className="hidden lg:block">{t(`Delete`)}</span>
                </DropdownMenuItem>
              </DropdownMenuContent>
            </DropdownMenu>
          </div>
        )
      },
    }
  ];

  return (
    <div className="flex flex-col gap-4">
      <h1 className="text-xl font-medium flex items-center gap-1.5">
        <ClipboardCheckIcon size={20} />
        {t(`System Checks`)}
      </h1>
      {
        checksLoading ? (
          <div className="flex justify-center items-center h-48">
            <Loader2Icon className="animate-spin" />
          </div>
        ) : (
          checks && <DataTable columns={alertingChecksColumns} data={checks || []} />
        )
      }
    </div>
  );
}

const ReturnCodeStatus = ({ status_code }: { status_code: number }) => {
  const status_code_dict: {
    [key: number]: {
      label: string,
      css: string,
      variant: "green" | "yellow" | "red" | "secondary"
    }
  } = {
    0: {
      label: "OK",
      css: "bg-success",
      variant: "green"
    },
    1: {
      label: "WARNING",
      css: "bg-warning",
      variant: "yellow"
    },
    2: {
      label: "CRITICAL",
      css: "bg-error",
      variant: "red"
    },
    3: {
      label: "UNKNOWN",
      css: "bg-secondary",
      variant: "secondary"
    }
  }

  if (status_code == undefined) {
    return (
      <Badge variant={"secondary"}>{"Pending"}</Badge>
    )
  }

  return (
    <Badge className="w-4/5 min-w-fit" variant={status_code_dict[status_code].variant}>{status_code_dict[status_code].label}</Badge>
  )
}