import { ColumnDef } from "@tanstack/react-table";
import {
    Tooltip,
    TooltipContent,
    TooltipTrigger,
    TooltipProvider,
} from "@/components/ui/tooltip";
import { DataTableColumnHeader } from "@/components/ui/DataTable/DataTableColumnHeader";
import { useEffect, useRef, useState } from "react";

export type ExampleData = {
    name: string;
    id: string;
    ip: string;
    number: number;
    port: number;
    status: string;
    description: string;
};

const createColumn = (
    accessorKey: keyof ExampleData,
    title: string
): ColumnDef<ExampleData> => ({
    accessorKey,
    header: ({ column }) => (
        <DataTableColumnHeader
            className="text-muted-foreground"
            column={column}
            title={title}
        />
    ),
    cell: ({ row }) => {
        const value = String(row.getValue(accessorKey as string));
        const textRef = useRef<HTMLDivElement>(null);
        const [isOverflowing, setIsOverflowing] = useState(false);

        useEffect(() => {
            const checkOverflow = () => {
                const el = textRef.current;
                if (el) {
                    const isOver = el.scrollWidth > el.clientWidth;
                    setIsOverflowing(isOver);
                }
            };

            let resizeTimeout: NodeJS.Timeout;
            const handleResize = () => {
                clearTimeout(resizeTimeout);
                resizeTimeout = setTimeout(() => {
                    checkOverflow();
                }, 100);
            };

            checkOverflow();
            window.addEventListener("resize", handleResize);

            return () => {
                window.removeEventListener("resize", handleResize);
                clearTimeout(resizeTimeout);
            };
        }, [value]);

        const TextContent = (
            <div
                ref={textRef}
                className="pl-1 truncate whitespace-nowrap overflow-hidden text-ellipsis w-full max-w-[500px]"
            >
                {value}
            </div>
        );

        return isOverflowing ? (
            <TooltipProvider>
                <Tooltip>
                    <TooltipTrigger asChild>{TextContent}</TooltipTrigger>
                    <TooltipContent>
                        <p>{value}</p>
                    </TooltipContent>
                </Tooltip>
            </TooltipProvider>
        ) : (
            TextContent
        );
    },
});

export const exampleColumns: ColumnDef<ExampleData>[] = [
    createColumn("name", "Name"),
    createColumn("id", "Id"),
    createColumn("ip", "Ip"),
    createColumn("number", "Number"),
    createColumn("port", "Port"),
    createColumn("status", "Status"),
    createColumn("description", "Description"),
];
