import { useTranslation } from 'react-i18next';
import { RadioGroup } from '@radix-ui/react-radio-group';
import { RadioGroupItem } from '@/components/ui/radio-group';
import { Label } from '@/components/ui/label';
import { Button } from "@/components/ui/button"
import { ChevronRight, CircleUser, Info, KeyRound } from 'lucide-react';
import { Input } from '@/components/ui/input';
import { TimeZoneDropdown } from '@/components/administration/global/TimeZoneDropdown';
import LanuageDropdown from '@/components/ui/LanguageDropdown';
import ThemesDropdown from '@/components/ui/ThemesDropdown';
import { useForm } from 'react-hook-form'
import { PasswordInput } from '@/components/ui/passwordInput';
import { useMutation } from '@tanstack/react-query';
import { useNavigate } from 'react-router-dom';
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form"
import {
  PageLayout,
  PageHeader,
  PageTitle,
  PageDescription,
  PageSeparator,
} from "@/components/Page";
import {
  Card,
  CardContent,
} from "@/components/ui/card";
import axios, { AxiosError } from '@/lib/axios';
import { toast } from 'sonner';
import { useQueryClient } from "@tanstack/react-query";
import { z } from 'zod';
import { zodResolver } from '@hookform/resolvers/zod';
import { useState, useEffect } from 'react';
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from '@/components/ui/tooltip';
import { useTranslateError } from '@/components/administration/license/utils';
import { useGetSystemTime } from '@/components/queries';

type StringObject = {
[key: string]: string;
};

export default function Install() {
  const { t } = useTranslation();
  const navigate = useNavigate();
  const translateError = useTranslateError();
  const { data: systemTime } = useGetSystemTime();
  const [errorMessage, setErrorMessage] = useState('');
  const [errorTitle, setErrorTitle] = useState('');

  const FormSchema = z.object({
    licenseOption: z.enum(["trial", "key"]),
    licenseKey: z.string().optional(),
    clientId: z.string().optional(),
    userName: z.string().nonempty(t("Please input a user name.")),
    password: z.string().min(4, t("Password must be at least 6 characters long")),
    passwordConfirm: z.string().min(4, t("Password confirm must be at least 6 characters long")),
    email: z.string().email(t("Invalid email address")),
    language: z.string().nonempty(t("Language is required")),
    timezone: z.string().nonempty(t("Timezone is required")),
    theme: z.string().nonempty(t("Theme is required"))
  }).superRefine((data, ctx) => {
    if (data.licenseOption === "key") {
      if (!data.licenseKey) {
        ctx.addIssue({
          code: "custom",
          message: t("Enter a valid license key."),
          path: ["licenseKey"],
        });
      }

      if (!data.clientId) {
        ctx.addIssue({
          code: "custom",
          message: t("Enter your Client ID or Token."),
          path: ["clientId"],
        });
      }
    }
  });

  const form = useForm({
    resolver: zodResolver(FormSchema),
    values: {
      licenseOption: "trial",
      licenseKey: "",
      clientId: "",
      userName: "nagiosadmin",
      password: "",
      passwordConfirm: "",
      email: "",
      language: "en_US",
      timezone: systemTime?.timezone ?? "",
      theme: "default",
    }
  });

  const licenseOption = form.watch("licenseOption");
  const queryClient = useQueryClient();

  function handleLicenseOptionChange(newValue: 'trial' | 'key') {
    form.setValue('licenseOption', newValue);
    if (newValue === 'trial') {
      form.clearErrors(['licenseKey', 'clientId']);
    }
  }

  const addUser = (data: StringObject) => {
    return axios.post('/api/v1/install', data);
  }
  
  const addInitialUser = useMutation({
    mutationFn: addUser,
    onSuccess: (response) => {
      queryClient.invalidateQueries({ queryKey: ['installed'] });
      queryClient.invalidateQueries({ queryKey: ['license'] });
      toast(t(response.data.message));
      navigate(response.data.redirect);
    },
    onError: (error: AxiosError<{ message: string; title: string }>) => {
      const message = error?.response?.data?.message || error?.message;
      const title = error?.response?.data?.title || '';

      if (message && title) {
        const { title: translatedTitle, message: translatedMessage } = translateError(title, message);
        window.scrollTo({ top: 0, behavior: 'smooth' });
        form.setError('licenseKey', { type: 'manual' });
        setErrorMessage(translatedMessage);
        setErrorTitle(translatedTitle);
      } else {
        toast(t('Failed to create user.'));
      }
    },
  });

  function onSubmit(values: StringObject) {
    addInitialUser.mutate(values);
    setErrorMessage('');
    setErrorTitle('');
  }

  return (
    <div className="max-w-4xl mx-auto my-10 space-y-10">
      <PageHeader>
          <PageTitle>{t("Final Installation Steps")}</PageTitle>
          <PageDescription>
              {t("Almost done! Update the license and configure the admin account.")}
          </PageDescription>
      </PageHeader>
      <Form {...form}>
        <form onSubmit={form.handleSubmit((onSubmit))} className="space-y-10">
          <Card>
            <CardContent>
              <div>
                <div className="flex items-center gap-2 text-lg font-semibold mb-6"><KeyRound />{t("License Setup")}</div>
                <PageDescription>{t('Choose a trial license, enter your key, or ')}
                  <a className="text-primary" href="https://www.nagios.com/products/nagios-network-analyzer/" target="_blank">{t('get a license now')}</a>
                  .
                </PageDescription>
                {errorMessage && (
                  <div className="bg-destructive/10 border border-destructive text-destructive px-4 py-3 rounded relative mt-4">
                    {errorTitle && (
                      <p className="block font-semibold text-lg mb-1">{errorTitle}</p>
                    )}
                    <p className="block">{errorMessage}</p>
                  </div>
                )}
                <PageSeparator/>
                <div className="flex flex-col gap-4">
                  <FormField 
                    control={form.control}
                    name="licenseOption"
                    render={({ field }) => (
                      <FormItem className="grid grid-cols-3 items-center gap-4 w-full">
                        <FormControl>
                          <RadioGroup
                            defaultValue="trial"
                            className="gap-[.5rem] grid"
                            onValueChange={handleLicenseOptionChange}
                            value={field.value}
                          >
                            <div className="flex items-center space-x-2">
                              <RadioGroupItem value="trial" id="trial" />
                              <Label htmlFor="trial">{t('Free 30 day trial')}</Label>
                            </div>
                            <div className="flex items-center space-x-2">
                              <RadioGroupItem value="key" id="key" />
                              <Label htmlFor="key">{t('I already have a key')}</Label>
                            </div>
                          </RadioGroup>
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <FormField 
                    control={form.control}
                    name="licenseKey"
                    render={({ field }) => (
                      <FormItem className="grid grid-cols-3 items-center gap-4 w-full">
                        <FormLabel className="text-sm font-medium">{t('License Key')}</FormLabel>
                        <FormControl className="col-span-2">
                          <Input 
                            {...field}
                            className={licenseOption === "trial" ? "cursor-not-allowed" : ""}
                            disabled={licenseOption === "trial"}
                          />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <FormField 
                    control={form.control}
                    name="clientId"
                    render={({ field }) => (
                      <FormItem className="grid grid-cols-3 items-center gap-4 w-full">
                        <div className="flex items-center gap-1">
                          <FormLabel className="text-sm font-medium">{t("Client ID / Token")}</FormLabel>
                          <TooltipProvider>
                            <Tooltip>
                              <TooltipTrigger asChild>
                                <span tabIndex={0} className="cursor-pointer">
                                  <Info size={16} />
                                </span>
                              </TooltipTrigger>
                              <TooltipContent className="max-w-80 text-wrap">
                                {t(
                                  "Your client ID or token should be in the license email. Contact sales@nagios.com if you do not know yours."
                                )}
                              </TooltipContent>
                            </Tooltip>
                          </TooltipProvider>
                        </div>
                        <FormControl className="col-span-2">
                          <Input 
                            {...field}
                            className={licenseOption === "trial" ? "cursor-not-allowed" : ""}
                            disabled={licenseOption === "trial"}
                          />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </div>
              </div>
            </CardContent>
          </Card>
          <Card>
            <CardContent>
              <div className="pb-8">
                <div className="flex items-center gap-2 text-lg font-semibold mb-6"><CircleUser />{t("Admin Account Setup")}</div>
                <PageDescription>{t('Enter the admin account settings. The default username is nagiosadmin.')}</PageDescription>
                <PageSeparator/>
                <div className="flex flex-col gap-4">
                  <FormField 
                    control={form.control}
                    name="userName"
                    render={({ field }) => (
                      <FormItem className="grid grid-cols-3 items-center gap-4 w-full">
                        <FormLabel className="text-sm font-medium">{t('Username')}</FormLabel>
                        <FormControl className="col-span-2">
                          <Input {...field} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <FormField 
                    control={form.control}
                    name="password"
                    render={({ field }) => (
                      <FormItem className="grid grid-cols-3 items-center gap-4 w-full">
                        <FormLabel className="text-sm font-medium">{t('Password')}</FormLabel>
                        <div className="col-span-2">
                          <FormControl className="col-span-2">
                            <PasswordInput {...field} />
                          </FormControl>
                        </div>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <FormField 
                    control={form.control}
                    name="passwordConfirm"
                    render={({ field }) => (
                      <FormItem className="grid grid-cols-3 items-center gap-4 w-full">
                        <FormLabel className="text-sm font-medium">{t('Confirm Password')}</FormLabel>
                        <div className="col-span-2">
                          <FormControl className="col-span-2">
                            <PasswordInput {...field} />
                          </FormControl>
                        </div>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <FormField
                    control={form.control}
                    name="email"
                    render={({ field }) => (
                      <FormItem className="grid grid-cols-3 items-center gap-4 w-full">
                        <FormLabel className="text-sm font-medium">{t('Email Address')}</FormLabel>
                        <FormControl className="col-span-2">
                          <Input {...field} placeholder="example@email.com" />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />    
                  <FormField
                    control={form.control}
                    name="language"
                    render={({ field }) => (
                      <FormItem className="grid grid-cols-3 items-center gap-4 w-full">
                        <FormLabel className="text-sm font-medium">{t('Language')}</FormLabel>
                        <FormControl className="col-span-2">
                          <LanuageDropdown field={field} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <FormField
                    control={form.control}
                    name="timezone"
                    render={({ field }) => (
                      <FormItem className="grid grid-cols-3 items-center gap-4 w-full">
                        <FormLabel className="text-sm font-medium">{t('System Timezone')}</FormLabel>
                        <FormControl className="col-span-2">
                          <div className="col-span-2 w-full">
                            <TimeZoneDropdown {...field} />
                          </div>
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <FormField
                    control={form.control}
                    name="theme"
                    render={({ field }) => (
                      <FormItem className="grid grid-cols-3 items-center gap-4 w-full">
                        <FormLabel className="text-sm font-medium">{t('User Theme')}</FormLabel>
                        <FormControl className="col-span-2">
                          <ThemesDropdown field={field} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </div>
              </div>
              </CardContent>
            </Card>
          <div className="flex justify-end mt-8">
              <Button className="bg-primary" type="submit">
                {t('Finish Installation')}
                <ChevronRight />
              </Button>
            </div>
        </form>
      </Form>
    </div>
  );
}