import {
  PageLayout,
  PageHeader,
  PageTitle,
} from "@/components/Page"
import { useTranslation } from 'react-i18next';
import { useGetNdiff } from "@/components/nmap/ndiffs/queries";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
} from "@/components/ui/card"
import {
  Tabs,
  TabsContent,
  TabsList,
  TabsTrigger,
} from "@/components/ui/tabs"
import { Skeleton } from "@/components/ui/skeleton"
import {
  Diff,
} from "lucide-react";
import { Button } from "@/components/ui/button";
import { Link } from "react-router-dom";
import { ndiff } from "@/components/nmap/ndiffs/types";
import CopyButton from "@/components/ui/CopyButton";

export default function Ndiff() {
  const { t } = useTranslation();
  const id = window.location.pathname.split('/').pop() ?? '';
  const { data, isPending, isError } = useGetNdiff(id);

  return (
    <>
      <PageLayout>
        <PageHeader>
          <PageTitle>
            <span className="flex items-center gap-2">
              <Diff className="w-7 h-7" />
              {isPending ? <Skeleton className="h-8 w-80" /> : (data?.title ? `${data.title}` : "")}
            </span>
          </PageTitle>
          {data && (
            <div className="flex gap-4 mt-2">
              {data.scan1_id && (
                <div className="flex items-center gap-2">
                  <span className="font-semibold">{t('Base Scan')}:</span>
                  <Button asChild>
                    <Link to={`/nmap/scan/${data.scan1_id}`}>
                      {data.scan1?.title ? data.scan1.title : `#${data.scan1_id}`}
                    </Link>
                  </Button>
                </div>
              )}
              {data.scan2_id && (
                <div className="flex items-center gap-2">
                  <span className="font-semibold">{t('Comparison Scan')}:</span>
                  <Button asChild>
                    <Link to={`/nmap/scan/${data.scan2_id}`}>
                      {data.scan2?.title ? data.scan2.title : `#${data.scan2_id}`}
                    </Link>
                  </Button>
                </div>
              )}
            </div>
          )}
        </PageHeader>

        <Tabs defaultValue="diff_summary" className="pt-6">
          <TabsList>
            <TabsTrigger value="diff_summary">{t('Diff Summary')}</TabsTrigger>
          </TabsList>

          <TabsContent value="diff_summary">
            <Card>
              <CardHeader>
                <CardDescription>
                  {t("Lines starting with '-' were removed and lines starting with '+' were added. If a line starting with '-' is followed by a line starting with '+', it means the line was changed between scans.")}
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-2">
                {isPending && <div>{t('Loading...')}</div>}
                {isError && <div>{t('Failed to load Ndiff.')}</div>}
                {data && data.ndiff_results ? (
                  <div className="space-y-2">
                    {data.ndiff_results ? (
                      <DiffSummary data={data} />
                    ) : (
                      <p className="text-lg text-muted-foreground">
                        {t("No Ndiff output available.")}
                      </p>
                    )}
                  </div>
                ) : (
                  <div>{t('No standard output available.')}</div>
                )}
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </PageLayout>
    </>
  )
}

const DiffSummary = ({ data }: { data: ndiff }) => {
  const { t } = useTranslation();

  return (
    <div>
      {data && data.ndiff_results ? (
        <div className="font-mono text-sm flex flex-col">
          {data.ndiff_results.split('\n').map((line: string, idx: number) => {
            
            if (line.trim() === "") {
              return <></>;
            }

            let bgClass = "";
            if (line.startsWith('-')) {
              bgClass = "bg-red-200 dark:bg-red-950";
            } else if (line.startsWith('+')) {
              bgClass = "bg-green-200 dark:bg-green-950";
            }

            const linePrefix = line.startsWith('-') ? '-' : line.startsWith('+') ? '+' : '';
            const lineWithoutPrefix = line.replace(/^[-+]/, '').trim();

            return (
              <div key={idx} className={bgClass + " px-2 py-1 flex items-center gap-2 justify-between group"}>
                <span className="text-muted-foreground min-w-[1.5em] text-center font-bold text-lg">
                  {linePrefix}
                </span>
                <span className="flex-1">{lineWithoutPrefix}</span>
                <div className="opacity-0 group-hover:opacity-100 transition-opacity">
                  {lineWithoutPrefix && <CopyButton textToCopy={lineWithoutPrefix} />}
                </div>
              </div>
            );
          })}
        </div>
      ) : (
        <div>{t("No summary diff available.")}</div>
      )}
    </div>
  )
}