import { Language } from "@/components/user/types";
import { useUpdatePersonalInformation, useUpdateLanguage, useUpdateAPIKey, useUpdateTheme } from "@/components/user/profile/queries";
import { ChangePasswordDialog } from "@/components/user/profile/ChangePasswordDialog";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import CopyButton from "@/components/ui/CopyButton";
import { cn } from "@/lib/utils";
import { useAuth } from "@/contexts/AuthContext";
import { useTranslation } from "react-i18next";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { Link } from "react-router-dom";
import { Check, ChevronsUpDown, RefreshCcw } from "lucide-react";
import {
    Form,
    FormControl,
    FormDescription,
    FormField,
    FormItem,
    FormLabel,
    FormMessage,
} from "@/components/ui/form";
import {
    PageLayout,
    PageHeader,
    PageTitle,
    PageDescription,
    PageSeparator,
} from "@/components/Page";
import {
    Command,
    CommandEmpty,
    CommandGroup,
    CommandInput,
    CommandItem,
    CommandList,
} from "@/components/ui/command";
import {
    Popover,
    PopoverContent,
    PopoverTrigger,
} from "@/components/ui/popover";
import { Card, CardContent } from "@/components/ui/card";
import { PersonalInformation } from "@/components/user/profile/types";
import { useState } from "react";
import { Lock } from "lucide-react";
import { isAdmin } from "@/components/role/utils";

function PersonalInfo({ data }: { data: PersonalInformation }) {
    const { t } = useTranslation();
    const updatePersonalInformation = useUpdatePersonalInformation();

    const personalInfoSchema = z.object({
        username: z.string().min(1, t("Username is required")),
        first_name: z.string()
            .max(255, t("First name must be less than 255 characters"))
            .optional(),
        last_name: z.string()
            .max(255, t("Last name must be less than 255 characters"))
            .optional(),
        email: z.email({ error: t("Invalid email address") })
            .max(255, t("Email must be less than 255 characters")),
        company: z.string()
            .max(255, t("Company name must be less than 255 characters"))
            .optional(),
        phone: z.e164({ error: t("Invalid phone number") }).optional().or(z.literal("")),
    }) satisfies z.ZodType<PersonalInformation>;

    const form = useForm<PersonalInformation>({
        resolver: zodResolver(personalInfoSchema),
        values: data,
        mode: "onChange",
    });

    const { isDirty } = form.formState;

    function onSubmit(data: PersonalInformation) {
        updatePersonalInformation.mutate({ data: data });
    }

    return (
        <Card>
            <CardContent>
                <Form {...form}>
                    <form onSubmit={form.handleSubmit(onSubmit)}>
                        <div className="flex flex-row items-center justify-between">
                            <h1 className="leading-9 text-md font-large font-semibold">
                                {t("Personal Information")}
                            </h1>
                            <Button
                                disabled={!isDirty}
                                variant={isDirty ? "default" : "outline"}
                                className="border"
                                type="submit"
                            >
                                {t("Update")}
                            </Button>
                        </div>
                        <PageSeparator />
                        {/* Username */}
                        <FormField
                            control={form.control}
                            name="username"
                            render={({ field }) => (
                                <FormItem className="flex items-center justify-between flex-wrap gap-4">
                                    <div className="space-y-1">
                                        <FormLabel>{t("Username")}</FormLabel>
                                        <FormDescription>{t("Your unique identifier in Network Analyzer")}</FormDescription>
                                    </div>
                                    <FormControl className="w-md">
                                        <Input
                                            disabled
                                            placeholder="nagiosadmin"
                                            {...field}
                                            value={field.value || ""}
                                        />
                                    </FormControl>
                                    <FormMessage />
                                </FormItem>
                            )}
                        />
                        <PageSeparator />
                        {/* First & Last Name*/}
                        <FormItem className="flex items-center justify-between flex-wrap gap-4">
                            <div className="space-y-1">
                                <FormLabel>{t("Full Name")}</FormLabel>
                                <FormDescription>
                                    {t("Name displayed in interface")}
                                </FormDescription>
                            </div>
                            <div className="flex gap-2 w-md">
                                <FormField
                                    control={form.control}
                                    name="first_name"
                                    render={({ field }) => (
                                        <FormItem className="w-full">
                                            <FormControl>
                                                <Input placeholder={t("First Name")} {...field} value={field.value} />
                                            </FormControl>
                                            <FormMessage />
                                        </FormItem>
                                    )}
                                />
                                <FormField
                                    control={form.control}
                                    name="last_name"
                                    render={({ field }) => (
                                        <FormItem className="w-full">
                                            <FormControl>
                                                <Input placeholder={t("Last Name")} {...field} value={field.value} />
                                            </FormControl>
                                            <FormMessage />
                                        </FormItem>
                                    )}
                                />
                            </div>
                        </FormItem>
                        <PageSeparator />

                        {/* Email */}
                        <FormField
                            control={form.control}
                            name="email"
                            render={({ field }) => (
                                <FormItem className="flex items-center justify-between flex-wrap gap-4">
                                    <div className="space-y-1">
                                        <FormLabel>{t("Email")}</FormLabel>
                                        <FormDescription>
                                            {t("Your email address for account access")}
                                        </FormDescription>
                                    </div>
                                    <div className="w-md">
                                        <FormControl>
                                            <Input placeholder="example@domain.com" {...field} value={field.value} />
                                        </FormControl>
                                        <FormMessage />
                                    </div>
                                </FormItem>
                            )}
                        />
                        <PageSeparator />

                        {/* Company */}
                        <FormField
                            control={form.control}
                            name="company"
                            render={({ field }) => (
                                <FormItem className="flex items-center justify-between flex-wrap gap-4">
                                    <div className="space-y-1">
                                        <FormLabel>{t("Company")}</FormLabel>
                                        <FormDescription>
                                            {t("Name displayed in Nagios Applications")}
                                        </FormDescription>
                                    </div>
                                    <div className="w-md">
                                        <FormControl>
                                            <Input placeholder="Nagios Enterprises" {...field} value={field.value} />
                                        </FormControl>
                                        <FormMessage />
                                    </div>
                                </FormItem>
                            )}
                        />
                        <PageSeparator />

                        {/* Phone */}
                        <FormField
                            control={form.control}
                            name="phone"
                            render={({ field }) => (
                                <FormItem className="flex items-center justify-between flex-wrap gap-4">
                                    <div className="space-y-1">
                                        <FormLabel>{t("Phone")}</FormLabel>
                                        <FormDescription>{t("Contact phone number")}</FormDescription>
                                    </div>
                                    <div className="w-md">
                                        <FormControl>
                                            <Input placeholder="+11234567890" {...field} value={field.value || ""} />
                                        </FormControl>
                                        <FormMessage />
                                    </div>
                                </FormItem>
                            )}
                        />
                    </form>
                </Form>
            </CardContent>
        </Card>
    );
}

function AccountSettings({ data }: { data: { lang: Language, password: string, theme: string } }) {

    const { t } = useTranslation();
    const updateLanguage = useUpdateLanguage();
    const updateTheme = useUpdateTheme();
    const [openLang, setOpenLang] = useState(false);
    const [openTheme, setOpenTheme] = useState(false);

    const langForm = useForm<{ lang: Language; }>({
        values: { lang: data.lang },
    });

    const themeForm = useForm<{ theme: string }>({
        values: { theme: data.theme },
    });

    const languages: { label: string; value: Language }[] = [
        { label: t("Default"), value: "default" },
        { label: t("Bulgarian"), value: "bg_BG" },
        { label: t("Czech"), value: "cs_CZ" },
        { label: t("German"), value: "de_DE" },
        { label: t("English"), value: "en_US" },
        { label: t("Spanish"), value: "es_ES" },
        { label: t("French"), value: "fr_FR" },
        { label: t("Italian"), value: "it_IT" },
        { label: t("Japanese"), value: "ja_JP" },
        { label: t("Korean"), value: "ko_KR" },
        { label: t("Polish"), value: "pl_PL" },
        { label: t("Portugese"), value: "pt_PT" },
        { label: t("Russian"), value: "ru_RU" },
        { label: t("Chinese (China)"), value: "zh_CN" },
        { label: t("Chinese (Taiwan)"), value: "zh_TW" },
    ];

    const themes = [
        { label: t("Default"), value: "default" },
        { label: t("Light"), value: "light" },
        { label: t("Dark"), value: "dark" },
    ];

    function onSubmit(lang: Language) {
        updateLanguage.mutate(lang);
        setOpenLang(false);
    }

    function onSubmitTheme(theme: string) {
        updateTheme.mutate(theme);
        setOpenTheme(false);
    }

    return (
        <Card>
            <CardContent>
                <Form {...langForm}>
                    <form>
                        <h1 className="leading-9 text-md font-large font-semibold">
                            {t("Account Settings")}
                        </h1>
                        <PageSeparator />
                        <FormField
                            control={langForm.control}
                            name="lang"
                            render={({ field }) => (
                                <FormItem className="flex items-center justify-between flex-wrap gap-4">
                                    <div className="space-y-1">
                                        <FormLabel>{t("Language")}</FormLabel>
                                        <FormDescription>
                                            {t("Change the display language")}
                                        </FormDescription>
                                    </div>
                                    <Popover open={openLang} onOpenChange={setOpenLang}>
                                        <PopoverTrigger asChild>
                                            <FormControl>
                                                <Button
                                                    variant="outline"
                                                    role="combobox"
                                                    className={cn(
                                                        "w-[200px] justify-between",
                                                        !field.value &&
                                                        "text-muted-foreground"
                                                    )}
                                                >
                                                    {field.value
                                                        ? languages.find(
                                                            (lang) =>
                                                                lang.value ===
                                                                field.value
                                                        )?.label
                                                        : "Select language"}
                                                    <ChevronsUpDown className="opacity-50" />
                                                </Button>
                                            </FormControl>
                                        </PopoverTrigger>
                                        <PopoverContent className="w-[200px] p-0">
                                            <Command>
                                                <CommandInput placeholder="Search language..." />
                                                <CommandList>
                                                    <CommandEmpty>
                                                        {t("No language found")}
                                                    </CommandEmpty>
                                                    <CommandGroup>
                                                        {languages.map(
                                                            (lang) => (
                                                                <CommandItem
                                                                    value={lang.label}
                                                                    key={lang.value}
                                                                    onSelect={() => {
                                                                        langForm.handleSubmit(() => onSubmit(lang.value))();
                                                                    }}
                                                                >
                                                                    {lang.label}
                                                                    <Check
                                                                        className={cn(
                                                                            "ml-auto",
                                                                            lang.value ===
                                                                                field.value
                                                                                ? "opacity-100"
                                                                                : "opacity-0"
                                                                        )}
                                                                    />
                                                                </CommandItem>
                                                            )
                                                        )}
                                                    </CommandGroup>
                                                </CommandList>
                                            </Command>
                                        </PopoverContent>
                                    </Popover>
                                    <FormMessage />
                                </FormItem>
                            )}
                        />
                    </form>
                </Form>

                <PageSeparator />

                {/* Theme Selection */}
                <Form {...themeForm}>
                    <form>
                        <FormField
                            control={themeForm.control}
                            name="theme"
                            render={({ field }) => (
                                <FormItem className="flex items-center justify-between flex-wrap gap-4">
                                    <div className="space-y-1">
                                        <FormLabel>{t("Theme")}</FormLabel>
                                        <FormDescription>
                                            {t("Choose the theme for your account")}
                                        </FormDescription>
                                    </div>
                                    <Popover open={openTheme} onOpenChange={setOpenTheme}>
                                        <PopoverTrigger asChild>
                                            <FormControl>
                                                <Button
                                                    variant="outline"
                                                    role="combobox"
                                                    className={cn(
                                                        "w-[200px] justify-between",
                                                        !field.value && "text-muted-foreground"
                                                    )}
                                                >
                                                    {field.value
                                                        ? themes.find(th => th.value === field.value)?.label
                                                        : t("Select theme")}
                                                    <ChevronsUpDown className="opacity-50" />
                                                </Button>
                                            </FormControl>
                                        </PopoverTrigger>
                                        <PopoverContent className="w-[200px] p-0">
                                            <Command>
                                                <CommandInput placeholder={t("Search theme...")} />
                                                <CommandList>
                                                    <CommandEmpty>
                                                        {t("No theme found")}
                                                    </CommandEmpty>
                                                    <CommandGroup>
                                                        {themes.map(theme => (
                                                            <CommandItem
                                                                key={theme.value}
                                                                value={theme.label}
                                                                onSelect={() => {
                                                                    themeForm.handleSubmit(() => onSubmitTheme(theme.value))();
                                                                }}
                                                            >
                                                                {theme.label}
                                                                <Check
                                                                    className={cn(
                                                                        "ml-auto",
                                                                        theme.value === field.value
                                                                            ? "opacity-100"
                                                                            : "opacity-0"
                                                                    )}
                                                                />
                                                            </CommandItem>
                                                        ))}
                                                    </CommandGroup>
                                                </CommandList>
                                            </Command>
                                        </PopoverContent>
                                    </Popover>
                                </FormItem>
                            )}
                        />
                    </form>
                </Form>
                <PageSeparator />
                <div className="flex flex-col xl:flex-row justify-between items-start gap-4">
                    <div className="space-y-1">
                        <label className="flex items-center gap-2 text-sm leading-none font-medium select-none">{t("Security")}</label>
                        <p className="text-muted-foreground text-sm">
                            {t("Manage account credentials")}
                        </p>
                    </div>
                    <ChangePasswordDialog>
                        <Button variant="outline">
                            <Lock />
                            {t("Change Password")}
                        </Button>
                    </ChangePasswordDialog>
                </div>
            </CardContent>
        </Card>
    );
}

function APIKeySettings({ data }: { data: { apikey: string, isAdmin: boolean, apiaccess: string, apikey_id: number } }) {
    const { t } = useTranslation();
    const updateAPIKey = useUpdateAPIKey();

    function handleRegenerateKey(apikey_id: number) {
        updateAPIKey.mutate(apikey_id);
    }

    function getAccessBadge() {
        if (data.apiaccess && data.isAdmin) {
            return <Badge variant="green">{t("Admin - Full Access")}</Badge>;
        } else if (data.apiaccess && !data.isAdmin) {
            return <Badge variant="blue">{t("Integraton Access")}</Badge>;
        } else {
            return <Badge variant="red">{t("No Access")}</Badge>;
        }
    }

    function getDescription() {
        if (data.apiaccess) {
            return t("Your API key for external access.");
        } else {
            return t("API Access is disabled for your account.");
        }
    }

    return (
        <Card>
            <CardContent>
                <div className="flex flex-row items-center justify-between">
                    <h1 className="leading-9 text-md font-large font-semibold">
                        {t("API Key")}
                    </h1>
                    {data.apiaccess && data.apikey_id && (
                        <Button
                            onClick={() => handleRegenerateKey(data.apikey_id)}
                            disabled={updateAPIKey.isPending}
                        >
                            <RefreshCcw
                                className={`w-3 h-3 ${updateAPIKey.isPending
                                        ? "animate-spin"
                                        : ""
                                    }`}
                            />
                            {t("Generate New Key")}
                        </Button>
                    )}
                </div>
                <PageSeparator />

                <div className="flex justify-between items-center flex-wrap gap-4">
                    <div className="space-y-1">
                        <div className="flex gap-2 items-center">
                            <label className="text-sm font-medium leading-none peer-disabled:cursor-not-allowed peer-disabled:opacity-70">
                                {t("Current API Key")}
                            </label>
                            {getAccessBadge()}
                        </div>
                        <p className="text-sm text-muted-foreground">
                            {getDescription()}
                        </p>
                    </div>
                    <div className="w-md">
                        {data.apiaccess ? (
                            <div className="relative">
                                <Input
                                    value={data.apikey || ""}
                                    disabled
                                    className="relative border-dashed border-popover-foreground/60"
                                    placeholder={t("No API key available")}
                                />
                                <div className="absolute flex items-center justify-center top-0 right-0 h-9 min-w-9">
                                    <CopyButton textToCopy={data.apikey || ""} />
                                </div>
                            </div>
                        ) : null}
                    </div>
                </div>
            </CardContent>
        </Card>
    );
}

export default function Profile() {
    const { t } = useTranslation();

    const { user } = useAuth();
    if (!user) return;

    const PersonalInformation: PersonalInformation = {
        username: user.username ?? "",
        first_name: user.first_name ?? "",
        last_name: user.last_name ?? "",
        email: user.email ?? "",
        company: user.company ?? "",
        phone: user.phone ?? undefined,
    }

    const accountSettings = { lang: user.lang, password: user.password, theme: user.theme };
    const apikeySettings = { apikey: user.apikey ?? "", isAdmin: isAdmin(user?.role) , apiaccess: user.apiaccess, apikey_id: user.apikey_id };

    return (
        <PageLayout>
            <PageHeader>
                <PageTitle>{t("My Profile")}</PageTitle>
                <PageDescription>
                    {t("Manage your account settings, personal information, and API key")}
                </PageDescription>
                <PageSeparator />
            </PageHeader>
            <div className="space-y-6">
                <PersonalInfo data={PersonalInformation} />
                <AccountSettings data={accountSettings} />
                <APIKeySettings data={apikeySettings} />
            </div>
        </PageLayout>
    );
}
