#!/bin/bash
#
# Bash script for creating a backup of Nagios Network Analyzer
# Copyright 2014-2025 Nagios Enterprises, LLC. All Rights reserved.
#
# Creates a full backup of Nagios Network Analyzer
#
BASEDIR="$(dirname $0)"
ts=`date +%s`
###############################
# USAGE / HELP
###############################
usage () {
    echo ""
    echo "Use this script to backup Nagios Network Analyzer."
    echo ""
    echo " -n | --name              Set the name of the backup minus the .tar.gz"
    echo " -p | --prepend           Prepend a string to the .tar.gz name"
    echo " -a | --append            Append a string to the .tar.gz name"
    echo " -d | --directory         Change the directory to store the compressed backup"
    echo ""
}
###############################
# ADDING LOGIC FOR NEW BACKUPS
###############################
while [ -n "$1" ]; do
    case "$1" in
        -h | --help)
            usage
            exit 0
            ;;
        -n | --name)
            fullname=$2
            ;;
        -p | --prepend)
            prepend=$2"."
            ;;
        -a | --append)
            append="."$2
            ;;
        -d | --directory)
            rootdir=$2
            ;;
    esac
    shift
done
if [ -z "$rootdir" ]; then
    rootdir="/store/backups/nagiosna"
fi

##############################
# CHECK DISK SPACE
##############################
echo "Checking disk space requirements..."

set -a
source <(grep -E '^(DB_HOST|DB_USERNAME|DB_PASSWORD|DB_DATABASE)=' /var/www/html/nagiosna/.env)
set +a
export MYSQL_PWD="$DB_PASSWORD"

root_avail="$(df -P / | awk 'NR==2 {print $4}')"
echo "Available space on /: $(numfmt --to=iec-i --suffix=B $((root_avail * 1024)))"

total_size=0

echo "Checking flow directories from database..."
mapfile -t flow_directories < <(mysql -h "$DB_HOST" -u "$DB_USERNAME" \
    --batch --raw -N -e "SELECT DISTINCT directory FROM sources WHERE directory IS NOT NULL AND directory != '';" "$DB_DATABASE" 2>/dev/null)

if [ ${#flow_directories[@]} -gt 0 ]; then
    for flow_dir in "${flow_directories[@]}"; do
        if [ -d "$flow_dir" ]; then
            flow_dir_size="$(du -sk "$flow_dir" 2>/dev/null | awk '{print $1}')"
            echo "Size of $flow_dir: $(numfmt --to=iec-i --suffix=B $((flow_dir_size * 1024)))"
            total_size="$((total_size + flow_dir_size))"
        else
            echo "Warning: Flow directory not found: $flow_dir"
        fi
    done
else
    echo "No flow directories found in database, skipping..."
fi

dirs_to_check=(
    "/usr/local/wireshark"
    "/usr/local/nmap"
    "/usr/local/etc/suricata"
    "/usr/local/var/log/suricata"
    "/usr/local/var/lib/suricata"
)

# Loop through each directory
for dir in "${dirs_to_check[@]}"; do
    if [ -d "$dir" ]; then
        dir_size="$(du -sk "$dir" 2>/dev/null | awk '{print $1}')"
        echo "Size of $dir: $(numfmt --to=iec-i --suffix=B $((dir_size * 1024)))"
        total_size="$((total_size + dir_size))"
    else
        echo "Directory $dir not found"
    fi
done

safety_margin="$((total_size * 15 / 100))"
required_space="$((total_size + safety_margin))"

echo ""
echo "Total size of data to backup: $(numfmt --to=iec-i --suffix=B $((total_size * 1024)))"
echo "Estimated space needed (with 15% safety margin): $(numfmt --to=iec-i --suffix=B $((required_space * 1024)))"
echo ""

if [ $root_avail -lt $required_space ]; then
    echo "ERROR: Insufficient disk space!"
    echo "Required: $(numfmt --to=iec-i --suffix=B $((required_space * 1024)))"
    echo "Available: $(numfmt --to=iec-i --suffix=B $((root_avail * 1024)))"
    echo "Please free up space before running this backup."
    exit 1
fi

echo "Disk space check passed. Proceeding with backup..."
echo ""

cd "$rootdir"
#############################
# SET THE NAME & TIME
#############################
name="$fullname"
if [ -z "$fullname" ]; then
    name="$prepend$ts$append"
fi

# Get current Unix timestamp as name
if [ -z "$name" ]; then
    name="$ts"
fi

# My working directory
mydir="$rootdir/$name"
# Make directory for this specific backup
if [ ! -d "$mydir" ]; then
    mkdir -p "$mydir"
fi

##############################
# BACKUP DIRS
##############################
echo "Backing up flow directories..."
if [ ! -d "$mydir/var" ]; then
    mkdir -p "$mydir/var" || { echo "Failed to create "$mydir"/var"; exit 1; }
fi

# Get flow directories from database (reuse credentials from disk check)
mapfile -t flow_directories < <(mysql -h "$DB_HOST" -u "$DB_USERNAME" \
    --batch --raw -N -e "SELECT DISTINCT directory FROM sources WHERE directory IS NOT NULL AND directory != '';" "$DB_DATABASE" 2>/dev/null)

if [ ${#flow_directories[@]} -gt 0 ]; then
    echo "Backing up flow directories from database..."
    for flow_dir in "${flow_directories[@]}"; do
        if [ -d "$flow_dir" ]; then
            echo "Copying flow directory: $flow_dir"
            # Get the basename of the directory to use as subdirectory name
            dir_basename="$(basename "$flow_dir")"
            cp -rp "$flow_dir/." "$mydir/var/$dir_basename" || { echo "Failed to copy flow directory $flow_dir"; exit 1; }
        else
            echo "Warning: Flow directory not found, skipping: $flow_dir"
        fi
    done
else
    echo "No flow directories in database, backing up default location..."
    # Fallback to default location
    if [ -d /usr/local/nagiosna/var ]; then
        cp -rp /usr/local/nagiosna/var/. "$mydir/var" || { echo "Failed to copy Flow directories"; exit 1; }
    fi
fi

# Remove non-dirs from var root
find "$mydir/var" -maxdepth 1 -type f -delete
# Compress flows
cd "$mydir"
tar czfp nagiosna-flows.tar.gz -C var .
tar_res=$?

if [ "$tar_res" -eq 2 ]; then
    echo "FATAL ERROR: tarring flow sources failed. Exiting..."
    exit 1
fi

if [ -d "$mydir/var" ]; then
    rm -rf "$mydir/var"
fi

# Bring version file along for the ride
if [ -f /var/www/html/nagiosna/naversion ]; then
    cp -p /var/www/html/nagiosna/naversion "$mydir" || { echo "Failed to copy nna version file"; exit 1; }
fi

##############################
# BACKUP env file
##############################
if [ -f /var/www/html/nagiosna/.env ]; then
    cp -p /var/www/html/nagiosna/.env "$mydir" || { echo "Failed to copy env file"; exit 1; }
else
    echo "Failed to find env file, if the data base is being backed up this will fail."
fi

##############################
# BACKUP LDAP
##############################
if [ -d /etc/openldap ]; then
    echo "Backing up LDAP directory..."
    if [ ! -d "$mydir/ldap" ]; then
        mkdir -p "$mydir/ldap" || { echo "Failed to create $mydir/ldap"; exit 1; }
    fi
    cp -rp /etc/openldap/* "$mydir/ldap/" || { echo "Failed to copy ldap files"; exit 1; }
elif [ -d /etc/ldap ]; then
    echo "Backing up LDAP directory..."
    if [ ! -d "$mydir/ldap" ]; then
        mkdir -p "$mydir/ldap" || { echo "Failed to create $mydir/ldap"; exit 1; }
    fi
    cp -rp /etc/ldap/* "$mydir/ldap/" || { echo "Failed to copy ldap files"; exit 1; }
    if [ ! -d "$mydir/ldap/cacerts" ]; then
        mkdir -p "$mydir/ldap/cacerts" || { echo "Failed to create $mydir/ldap/cacerts"; exit 1; }
    fi
    cp -rp /etc/ssl/certs/* "$mydir/ldap/cacerts/" || { echo "Failed to copy ldap certs"; exit 1; }
else
    echo "No LDAP directory found, skipping backup."
fi

##############################
# BACKUP DATABASES
##############################
echo "Backing up MySQL databases..."
if [ ! -d "$mydir/mysql" ]; then
    mkdir -p "$mydir/mysql" || { echo "Failed to create $mydir/mysql"; exit 1; }
fi

set -a
source <(grep -E '^(DB_HOST|DB_USERNAME|DB_PASSWORD|DB_DATABASE)=' /var/www/html/nagiosna/.env)
set +a
export MYSQL_PWD="$DB_PASSWORD"
mysqldump --no-tablespaces --skip-tz-utc -h "$DB_HOST" -u "$DB_USERNAME" --add-drop-database "$DB_DATABASE" > "$mydir/mysql/nagiosna.sql"
res=$?
if [ "$res" != 0 ]; then
    echo "Error backing up MySQL database "$DB_DATABASE" - check the password in this script!"
    exit "$res";
fi

##############################
# BACKUP APACHE CONFIG FILES
##############################
echo "Backing up Apache config files..."
apache="$mydir/httpd"
if [ ! -d "$apache" ]; then
    mkdir -p "$apache" || { echo "Failed to create $apache"; exit 1; }
fi

# RHEL system config
if [ -f "/etc/httpd/conf.d/nna.conf" ]; then
    echo "Found RHEL-style Laravel Apache config — backing up"
    cp -p /etc/httpd/conf.d/nna.conf "$apache/" || { echo "Failed to copy httpd conf files"; exit 1; }
fi

# Debian/Ubuntu system config
if [ -f "/etc/apache2/sites-available/nna.conf" ]; then
    echo "Found Debian/Ubuntu-style Laravel Apache config — backing up"
    cp -p /etc/apache2/sites-available/nna.conf "$apache/" || { echo "Failed to copy apache conf files"; exit 1; }
fi

##############################
# BACKUP Wireshark pcap files
##############################
if [ -d "/usr/local/wireshark" ]; then
    echo "Backing up Wireshark files..."
    tar czfp "$mydir/nagiosna-wireshark.tar.gz" -C /usr/local/wireshark .
    tar_res=$?

    if [ "$tar_res" -eq 2 ]; then
        echo "FATAL ERROR: tarring Wireshark files failed. Exiting..."
        exit 1
    fi
else
    echo "Wireshark directory not found, skipping backup."
fi

##############################
# BACKUP NMAP files
##############################
if [ -d "/usr/local/nmap" ]; then
    echo "Backing up Nmap files..."
    tar czfp "$mydir/nagiosna-nmap.tar.gz" -C /usr/local/nmap .
    tar_res=$?

    if [ "$tar_res" -eq 2 ]; then
        echo "FATAL ERROR: tarring Nmap files failed. Exiting..."
        exit 1
    fi
else
    echo "Nmap directory not found, skipping backup."
fi

##############################
# BACKUP Suricata files
##############################
if [ -d "/usr/local/etc/suricata" ] || [ -d "/usr/local/var/log/suricata" ] || [ -d "/usr/local/var/lib/suricata" ]; then
    echo "Backing up Suricata files..."
    # Suricata config
    if [ -d "/usr/local/etc/suricata" ]; then
        tar czfp "$mydir/nagiosna-suricata-config.tar.gz" -C /usr/local/etc/suricata .
        tar_res=$?

        if [ "$tar_res" -eq 2 ]; then
            echo "FATAL ERROR: tarring Suricata config options failed. Exiting..."
            exit 1
        fi
    fi
    # Suricata logs
    if [ -d "/usr/local/var/log/suricata" ]; then
        tar czfp "$mydir/nagiosna-suricata-data.tar.gz" -C /usr/local/var/log/suricata .
        tar_res=$?

        if [ "$tar_res" -eq 2 ]; then
            echo "FATAL ERROR: tarring Suricata logs failed. Exiting..."
            exit 1
        fi
    fi
    # Suricata rules
    if [ -d "/usr/local/var/lib/suricata" ]; then
        tar czfp "$mydir/nagiosna-suricata-rules.tar.gz" -C /usr/local/var/lib/suricata .
        tar_res=$?

        if [ "$tar_res" -eq 2 ]; then
            echo "FATAL ERROR: tarring Suricata rules failed. Exiting..."
            exit 1
        fi
    fi
else
    echo "No Suricata directories found, skipping backup."
fi

##############################
# COMPRESS BACKUP
##############################
echo "Compressing backup..."
# Move back to backup directory
cd "$rootdir"
tar czfp "$name.tar.gz" "$name"
if [ -d "$name" ]; then
    rm -rf "$name"
fi

chown nna:nnacmd "$name.tar.gz"
chmod 0664 "$name.tar.gz"

if [ -s "$name.tar.gz" ];then
    echo " "
    echo "==============="
    echo "BACKUP COMPLETE"
    echo "==============="
    echo "Backup stored in $rootdir/$name.tar.gz"
    exit 0;
else
    echo " "
    echo "==============="
    echo "BACKUP FAILED"
    echo "==============="
    echo "File was not created at $rootdir/$name.tar.gz"
    exit 1;
fi