#!/usr/bin/env python3
import unittest
import re
import MySQLdb as mdb
from pathlib import Path

ENV_PATH = Path("/var/www/html/nagiosna/.env")

def load_env(path=ENV_PATH):
    """Load key=value pairs from a .env file into a dictionary."""
    env_vars = {}
    if not path.exists():
        raise FileNotFoundError(f".env file not found: {path}")
    with path.open("r") as f:
        for line in f:
            line = line.strip()
            # Skip comments or blank lines
            if not line or line.startswith("#"):
                continue
            if "=" not in line:
                continue
            key, value = line.split("=", 1)
            value = value.strip().strip('"').strip("'")
            env_vars[key.strip()] = value
    return env_vars

def get_db_info(key, default=None):
    """Get DB connection info mapped from .env."""
    env = load_env()
    mapping = {
        "hostname": env.get("APP_BACKEND_URL", "localhost"),
        "username": env.get("DB_USERNAME"),
        "password": env.get("DB_PASSWORD"),
        "database": env.get("DB_DATABASE"),
    }
    return mapping.get(key, default)

def query_database(query, commit=False):
    global HOSTNAME
    global USERNAME
    global PASSWORD
    global DATABASE
    try:
        connection = mdb.connect(HOSTNAME, USERNAME, PASSWORD, DATABASE)
        cur = connection.cursor()
        cur.execute(query)
        if commit == True:
            connection.commit()
        else:
            result = cur.fetchall()
            cur.close()
            return result
        connection.close()
    except Exception as e:
        raise e

def safe_filename(filename):
    return re.sub('[^A-Za-z_0-9]', '', filename)

HOSTNAME = get_db_info('hostname')
USERNAME = get_db_info('username')
PASSWORD = get_db_info('password')
DATABASE = get_db_info('database')

class Config_Tests(unittest.TestCase):
    
    def test_get_db_info_username(self):
        username = get_db_info('username')
        self.assertTrue(isinstance(username, str))
        self.assertEqual(username, 'nagiosna')
    
    def test_get_db_info_password(self):
        password = get_db_info('password')
        self.assertTrue(isinstance(password, str))
        self.assertEqual(password, 'nagiosna')
    
    def test_get_db_info_database(self):
        database = get_db_info('database')
        self.assertTrue(isinstance(database, str))
        self.assertEqual(database, 'nagiosna')
    
    def test_get_db_info_hostname(self):
        hostname = get_db_info('hostname')
        self.assertTrue(isinstance(hostname, str))
        self.assertEqual(hostname, 'localhost')

if __name__ == "__main__":
    unittest.main()
