#!/usr/bin/env python3

import optparse
import os
import sys

version = '.'.join([str(x) for x in sys.version_info[:2]])
sys.path.append('/usr/local/lib/python%s/site-packages' % version)
sys.path.append('/usr/local/lib64/python%s/site-packages' % version)

from syslog import syslog, openlog, LOG_ERR, LOG_INFO
import subprocess
import logging
import rrdtool
import capd
import operator
import functools

logger = logging.getLogger('nnabackend')

if os.geteuid() == 0:
    # In testing, we found that this script would produce different results
    # when running as the actual nna user and when running as root with setuid().
    # We're not sure why, but we do need to actually have the nna user run the script.
    # If it runs as root-with-setuid(), the bandwidth graph will stop updating.
    #    -- swolf 2023-12-06
    print("Error: please run this script as the user 'nna'")
    sys.exit(1)

def parse_args():
    usage = '%prog --directory <directory> --port <port>'
    parser = optparse.OptionParser(usage=usage)
    
    parser.add_option('--directory', help='Directory the flow records will be recorded to.')
    parser.add_option('--port', type='int', help='Port that will be listened on for the flow records.')
    
    options, args = parser.parse_args()
    
    if not options.directory:
        parser.error("--directory is a required flag.")
    if not options.port:
        parser.error("--port is a required flag.")
    
    return options

# todo Replace this for influxdb stuff
def create_bandwidth_db(directory, name='bandwidth.rrd'):
    '''Creates the bandwidth DB for the source'''
    rrdtool.create( os.path.join(directory, name),
                    '--no-overwrite',
                    '--step', '300',
                    'DS:flows:GAUGE:600:0:U',
                    'DS:bytes:GAUGE:600:0:U',
                    'DS:packets:GAUGE:600:0:U',
                    'DS:bps:GAUGE:600:0:U',
                    'DS:pps:GAUGE:600:0:U',
                    'DS:bpp:GAUGE:600:0:U',
                    'RRA:AVERAGE:0.5:1:17531',
                    'RRA:AVERAGE:0.5:12:4380',
                    'RRA:AVERAGE:0.5:288:365',
                    'RRA:HWPREDICT:1440:0.1:0.0035:288' )

def create_source_resources(directory):
    '''Creates the folder for the flows to be written to.'''
    global logger

    try:
        os.makedirs(directory)
        subprocess.Popen(['chmod', '5775', directory])
    except OSError as e:
        logger.error(str(e))
    
    flowdir = os.path.join(directory, 'flows')
    try:
        os.mkdir(flowdir)
    except OSError as e:
        logger.error(str(e))
    
    try:
        create_bandwidth_db(directory)
    except Exception as e:
        logger.exception(e)

def no_existing_infrastructure_for(directory):
    '''Checks to see if there is existing created infrastructure for
    a given directory.'''
    flows = os.path.join(directory, 'flows')
    bandwidth = os.path.join(directory, 'bandwidth.rrd')
    bools = [not os.path.exists(x) for x in [directory, flows, bandwidth]]
    if not functools.reduce(operator.or_, bools):
        return False
    else:
        return True

def open_firewall_port(port):
    cmd = ['sudo', '/usr/local/nagiosna/scripts/manage_firewall.sh', '-t', 'udp', '-p', str(port), '--add']
    apply_rule = subprocess.Popen(cmd)
    apply_rule.wait()

def main():
    options = parse_args()
    
    #~ Do we need to create the directory structure for this source?
    if no_existing_infrastructure_for(options.directory):
        create_source_resources(options.directory)
    
    # Try to open firewall port (will not add rule if it already exists)
    open_firewall_port(options.port)

if __name__ == '__main__':
    try:
        main()
    except Exception as e:
        logger.error(str(e))
