#!/bin/bash
#
# Bash script that opens/closes firewall ports
# Copyright 2017-2025 Nagios Enterprises, LLC. All Rights reserved.
#
# Because different systems use different firewalls, we are going to guess
# what the firewall that is being used is (and if it's even running)
#
# Currently implemented:
#   IPT = iptables (CentOS/RHEL 5 & 6, sometimes CentOS/RHEL 7)
#   FWD = firewalld (CentOS/RHEL 7 standard)
#
# Needs implementation:
#   SUS = SuSEfirewall2 (most versions of SuSE/SLES)
#   UFW = uncomplicated firewall (Debian/Ubuntu)
#

FIREWALL="IPT"
FIREWALLSTATUS=0
PATH=$PATH:/sbin:/usr/sbin
BASEDIR=$(dirname $(readlink -f $0))

# Usage information
usage() {
    echo ""
    echo "Script that manages the configuration of the systems firewall, open and close ports,"
    echo "check the list of currently open ports/configurations."
    echo "Copyright 2017-2025 Nagios Enterprises, LLC. All Rights reserved."
    echo ""
    echo "-p | --port       The port to open"
    echo "-t | --type       Either 'tcp', 'udp', or 'both'"
    echo "--add             Add the port"
    echo "--rem             Remove the port"
    echo "--list            List the firewall rules"
    echo ""
    echo "-h | --help       Help information and usage documentation for this script"
    echo ""
}

# Get options passed
while [ -n "$1" ]; do
    case "$1" in
        --h | --help)
        usage
        exit 0
        ;;
        -p | --port)
        PORT=$2
        ;;
        -t | --type)
        TYPE=$2
        ;;
        --add)
        ACTION="add"
        ;;
        --rem)
        if [ "$ACTION" == "add" ]; then
            echo "You can only select either --add or --rem and not both"
            exit 0
        fi
        ACTION="rem"
        ;;
        --list)
        if [ -z "$ACTION" ]; then
            echo "You cannot use --add or --rem at the same time as --list"
            exit 0
        fi
        ACTION="list"
        ;;
    esac
    shift
done

# Type must be tcp, udp, both
if [ "$TYPE" != "tcp" ] && [ "$TYPE" != "udp" ] && [ "$TYPE" != "both" ]; then
    echo "You must specify tcp, udp, or both"
    exit 0
fi

# Port must be present
if [ "x$PORT" == "x" ]; then
    echo "You must specify a port with -p | --port"
    exit 0
fi

# Check which firewall we are using
if [ `command -v firewall-cmd` ]; then
    FIREWALL="FWD"
elif [ `command -v iptables` ]; then
    FIREWALL="IPT"
fi

# Do iptables port adding and deleting
iptables_func() {
    iptables -C INPUT -p $2 --dport $PORT --jump ACCEPT > /dev/null 2>&1
    if [ $? -ne 0 ]; then
        if [ "$1" == "I" ]; then
            iptables -$1 INPUT -p $2 -j ACCEPT --dport $PORT
        fi
    else
        if [ "$1" == "D" ]; then
            iptables -$1 INPUT -p $2 -j ACCEPT --dport $PORT
        fi
    fi
}

# Do firewalld port adding and deleting
firewalld_func() {
    firewall-cmd --zone=public --$1=$PORT/$2 --permanent
}

# Check what action we are taking
if [ "$ACTION" == "add" ]; then

    # Add firewall port
    case $FIREWALL in
        IPT)
            if [ "$TYPE" == "both" ]; then
                iptables_func "I" "udp"
                iptables_func "I" "tcp"
            else
                iptables_func "I" "$TYPE"
            fi
            service iptables save > /dev/null
        ;;
        FWD)
            if [ "$TYPE" == "both" ]; then
                firewalld_func "add-port" "udp"
                firewalld_func "add-port" "tcp"
            else
                firewalld_func "add-port" $TYPE
            fi
            firewall-cmd --reload
        ;;
    esac

elif [ "$ACTION" == "rem" ]; then

    # Remove firewall port
    case $FIREWALL in
        IPT)
            if [ "$TYPE" == "both" ]; then
                iptables_func "D" "udp"
                iptables_func "D" "tcp"
            else
                iptables_func "D" "$TYPE"
            fi
            service iptables save > /dev/null
        ;;
        FWD)
            if [ "$TYPE" == "both" ]; then
                firewalld_func "remove-port" "udp"
                firewalld_func "remove-port" "tcp"
            else
                firewalld_func "remove-port" $TYPE
            fi
            firewall-cmd --reload
        ;;
    esac

elif [ "$ACTION" == "list" ]; then

    # List the ports that are open and what they 
    case $FIREWALL in
        IPT)
            iptables -L
        ;;
        FWD)
            firewall-cmd --list-ports
        ;;
    esac

fi

echo "Firewall updated"
exit 0
