#!/bin/bash
#
# Bash script for restoring Nagios Network Analyzer
# Copyright 2014-2025 Nagios Enterprises, LLC. All Rights reserved.
#
# Restores a backup of Nagios Network Analyzer
#

# Make sure we have the backup file
if [ $# != 1 ]; then
	echo "Usage: $0 <backupfile>"
	echo "This script restores your Nagios Network Analyzer system using a previously made Nagios NA backup file."
	echo "Copyright 2014-2025 Nagios Enterprises, LLC. All Rights reserved."
	exit 1
fi

# Get file name without extention
backupfilefull=$1
backupfile="${backupfilefull%%.*}"
ROOTDIR="/store/backups/nagiosna"
BASEDIR="$(dirname $0)"

# Must be root
if [ "$(id -u)" -ne 0 ]; then
    echo "Error: You must be root to run this script." >&2
	exit 1
fi

##############################
# MAKE SURE BACKUP FILE EXIST
##############################
if [ ! -f "$backupfilefull" ]; then
	echo "Unable to find backup file $backupfilefull!"
	exit 1
fi

##############################
# MAKE TEMP RESTORE DIRECTORY
##############################
ts=`date +%s`
echo "TS=$ts"
mydir="${ROOTDIR}/${ts}-restore"
if [ ! -d "$mydir" ]; then
    mkdir -p "$mydir"
fi
cp -f "$backupfilefull" "$mydir"
if [ ! -d "$mydir" ]; then
	echo "Unable to create restore directory $mydir!"
	exit 1
fi

##############################
# UNZIP BACKUP
##############################
echo "Extracting backup to $mydir..."
cd "$mydir"
tar xzfps "$backupfilefull"

# Change to subdirectory
cd "$(find . -mindepth 1 -maxdepth 1 -type d | head -1)"

# Make sure we have at least the flows tarball in here...
backupdir="$(pwd)"
echo "In $backupdir..."
if [ ! -f "$backupdir/nagiosna-flows.tar.gz" ]; then
	echo "Unable to find flow files to restore in $backupdir"
	exit 1
fi

echo "Backup files look okay.  Preparing to restore..."


##############################
# SHUTDOWN SERVICES
##############################
echo "Shutting down services..."
systemctl stop nagiosna

##############################
# RESTORE DATABASES 
##############################
echo "Restoring MySQL databases..."

set -a
source <(grep -E '^(DB_HOST|DB_USERNAME|DB_PASSWORD|DB_DATABASE)=' /var/www/html/nagiosna/.env)
set +a
export MYSQL_PWD="$DB_PASSWORD"
mysql -h "$DB_HOST" -u "$DB_USERNAME" "$DB_DATABASE" < "$backupdir/mysql/nagiosna.sql"
res=$?
if [ $res != 0 ]; then
	echo "Error restoring MySQL database '$DB_DATABASE' - check database credentials!"
	exit 1
fi

echo "Database restored successfully."

##############################
# RESTORE FLOW DIRS TO CORRECT LOCATIONS
##############################
echo "Restoring flow files to their configured directories..."

# Extract flows to temporary location first
temp_flows_dir="$mydir/flows_temp"
mkdir -p "$temp_flows_dir"
tar xzfps "$backupdir/nagiosna-flows.tar.gz" -C "$temp_flows_dir"
tar_res=$?
if [ "$tar_res" -eq 2 ]; then
    echo "Error extracting flow files - tar returned $tar_res"
    exit 1
fi

# Get flow directories from database
mapfile -t flow_dirs < <(mysql -h "$DB_HOST" -u "$DB_USERNAME" \
    --batch --raw -N -e "SELECT DISTINCT directory FROM sources WHERE directory IS NOT NULL AND directory != '';" "$DB_DATABASE" 2>/dev/null)

if [ ${#flow_dirs[@]} -gt 0 ]; then
    echo "Found ${#flow_dirs[@]} flow directory location(s) in database"
    
    # Restore each directory to its configured location
    for flow_dir in "${flow_dirs[@]}"; do
        dir_basename="$(basename "$flow_dir")"
        
        # Check if we have this directory in the backup
        if [ -d "$temp_flows_dir/$dir_basename" ]; then
            echo "Restoring flows to: $flow_dir"
            
            # Create parent directory if it doesn't exist
            parent_dir="$(dirname "$flow_dir")"
            if [ ! -d "$parent_dir" ]; then
                mkdir -p "$parent_dir"
            fi
            
            # Create the flow directory if it doesn't exist
            if [ ! -d "$flow_dir" ]; then
                mkdir -p "$flow_dir"
            fi
            
            # Copy the flow files
            cp -rp "$temp_flows_dir/$dir_basename/." "$flow_dir/" || {
                echo "Warning: Failed to restore flows to $flow_dir"
            }
        else
            echo "Warning: No backup data found for directory: $flow_dir"
        fi
    done
else
    # Fallback: restore to default location if no directories in database
    echo "No flow directories found in database, using default location..."
    default_flowdir="/usr/local/nagiosna/var/"
    
    if [ ! -d "$default_flowdir" ]; then
        mkdir -p "$default_flowdir"
    fi
    
    cp -rp "$temp_flows_dir/." "$default_flowdir/" || {
        echo "Warning: Failed to restore flows to default location"
    }
fi

# Clean up temporary extraction directory
rm -rf "$temp_flows_dir"
echo "Flow restoration complete."

##############################
# RESTORE LDAP CONFIG AND CERTS
##############################

echo "Restoring LDAP configs and certs..."

# Detect distro type
if grep -qiE "red hat|centos|oracle" /etc/os-release 2>/dev/null; then
    # RHEL/CentOS
    ldap_conf_dir="/etc/openldap"
    ldap_cert_dir="/etc/openldap/certs"
    ldap_cacert_dir="/etc/openldap/cacerts"
elif grep -qiE "debian|ubuntu" /etc/os-release 2>/dev/null; then
    # Debian/Ubuntu
    ldap_conf_dir="/etc/ldap"
    ldap_cert_dir="/etc/ldap/certs"
    ldap_cacert_dir="/etc/ssl/certs"
else
    echo "Unknown distro, skipping LDAP restore"
    ldap_conf_dir=""
fi

if [ -d "$backupdir/ldap" ] && [ -n "$ldap_conf_dir" ]; then
    # Restore ldap.conf if exists
    if [ -f "$backupdir/ldap/ldap.conf" ]; then
        echo "Restoring ldap.conf..."
        if [ ! -d "$ldap_conf_dir" ]; then
            mkdir -p "$ldap_conf_dir"
        fi
        cp -rp "$backupdir/ldap/ldap.conf" "$ldap_conf_dir/"
    fi

    # Restore LDAP certs
    if [ -d "$backupdir/ldap/certs" ]; then
        echo "Restoring LDAP certs..."
        if [ ! -d "$ldap_cert_dir" ]; then
            mkdir -p "$ldap_cert_dir"
        fi
        cp -rp "$backupdir/ldap/certs/"* "$ldap_cert_dir/"
        chmod 644 "$ldap_cert_dir/"*
        chown root:root "$ldap_cert_dir/"*
    fi

    # Restore CA certs if present
    if [ -d "$backupdir/ldap/cacerts" ]; then
        echo "Restoring CA certs..."
        if [ ! -d "$ldap_cacert_dir" ]; then
            mkdir -p "$ldap_cacert_dir"
        fi
        cp -rp "$backupdir/ldap/cacerts/"* "$ldap_cacert_dir/"
        chmod 644 "$ldap_cacert_dir/"*
        chown root:root "$ldap_cacert_dir/"*
    fi

else
    echo "No LDAP backup directory found, skipping LDAP restore."
fi

##############################
# RESTORE Wirshark files
##############################
if [ -f "$backupdir/nagiosna-wireshark.tar.gz" ]; then
    echo "Restoring Wireshark files..."
    if [ ! -d "/usr/local/wireshark" ]; then
        mkdir -p /usr/local/wireshark
    fi

    tar xzfps "$backupdir/nagiosna-wireshark.tar.gz" -C /usr/local/wireshark
    tar_res=$?
    if [ "$tar_res" -eq 2 ]; then
        echo "Error restoring Wireshark files - tar returned $tar_res"
        exit 1
    fi
else
    echo "Wireshark directory not found, skipping Wire restore."
fi

##############################
# Restore NMAP files
##############################
if [ -f "$backupdir/nagiosna-nmap.tar.gz" ]; then
    echo "Restoring Nmap files..."
    if [ ! -d "/usr/local/nmap" ]; then
        mkdir -p /usr/local/nmap
    fi

    tar xzfps "$backupdir/nagiosna-nmap.tar.gz" -C /usr/local/nmap
    tar_res=$?
    if [ "$tar_res" -eq 2 ]; then
        echo "Error restoring Nmap files - tar returned $tar_res"
        exit 1
    fi
else
    echo "Nmap directory not found, skipping Nmap restore."
fi

##############################
# BACKUP Suricata files
##############################
if [ -f "$backupdir/nagiosna-suricata-config.tar.gz" ]; then
    echo "Restoring Suricata files..."
    if [ ! -d /usr/local/etc/suricata ]; then
        mkdir -p /usr/local/etc/suricata
    fi

    if [ ! -d /usr/local/var/log/suricata ]; then
        mkdir -p /usr/local/var/log/suricata
    fi

    if [ ! -d /usr/local/var/lib/suricata ]; then
        mkdir -p /usr/local/var/lib/suricata
    fi

    tar xzfps "$backupdir/nagiosna-suricata-config.tar.gz" -C /usr/local/etc/suricata
    tar_res=$?
    if [ "$tar_res" -eq 2 ]; then
        echo "Error restoring Suricata config - tar returned $tar_res"
        exit 1
    fi

    tar xzfps "$backupdir/nagiosna-suricata-data.tar.gz" -C /usr/local/var/log/suricata
    tar_res=$?
    if [ "$tar_res" -eq 2 ]; then
        echo "Error restoring Suricata data - tar returned $tar_res"
        exit 1
    fi

    tar xzfps "$backupdir/nagiosna-suricata-rules.tar.gz" -C /usr/local/var/lib/suricata
    tar_res=$?
    if [ "$tar_res" -eq 2 ]; then
        echo "Error restoring Suricata rules - tar returned $tar_res"
        exit 1
    fi
else
    echo "No Suricata directories found, skipping backup."
fi

##############################
# RESTORE env file
##############################
echo "Restoring env file..."
if [ -f "$backupdir/.env" ]; then
    # Save current database credentials before overwriting
    current_db_host="$(grep '^DB_HOST=' /var/www/html/nagiosna/.env | cut -d '=' -f2-)"
    current_db_username="$(grep '^DB_USERNAME=' /var/www/html/nagiosna/.env | cut -d '=' -f2-)"
    current_db_password="$(grep '^DB_PASSWORD=' /var/www/html/nagiosna/.env | cut -d '=' -f2-)"
    current_db_database="$(grep '^DB_DATABASE=' /var/www/html/nagiosna/.env | cut -d '=' -f2-)"
    
    # Restore the env file
    cp -p "$backupdir/.env" /var/www/html/nagiosna/.env
    
    # Restore current system's database credentials
    sed -i "s|^DB_HOST=.*|DB_HOST=$current_db_host|" /var/www/html/nagiosna/.env
    sed -i "s|^DB_USERNAME=.*|DB_USERNAME=$current_db_username|" /var/www/html/nagiosna/.env
    sed -i "s|^DB_PASSWORD=.*|DB_PASSWORD=$current_db_password|" /var/www/html/nagiosna/.env
    sed -i "s|^DB_DATABASE=.*|DB_DATABASE=$current_db_database|" /var/www/html/nagiosna/.env
else
    echo "No env file found, skipping restore. If mysql db was restored this will cause issues if the env file is not restored."
fi

IP="$(hostname -I | awk '{print $1}')"
sed -i "s|^APP_URL=.*|APP_URL=http://$IP|" /var/www/html/nagiosna/.env

echo "Restarting database servers..."
service mysqld restart

##############################
# RESTORE APACHE CONFIG FILES
##############################
echo "Restoring Apache config files..."
if [ -f "/etc/httpd/conf.d/nna.conf" ]; then
    cp -rp "$backupdir/httpd/*.conf" /etc/httpd/conf.d/
fi

if [ -f "/etc/apache2/sites-available/nna.conf" ]; then
    cp -rp "$backupdir/httpd/*.conf" /etc/apache2/sites-available/
fi


##############################
# RESTART SERVICES
##############################
systemctl start nagiosna

##############################
# DELETE TEMP RESTORE DIRECTORY
##############################

rm -rf "$mydir"

echo " "
echo "==============="
echo "RESTORE COMPLETE"
echo "==============="

exit 0;