<?php 

/**
 * html_attr() - helper function to print html attributes
 * @param string $attr the attribute you'd like to set val as
 * @param ref string $val the value to set to the attribute you've specified
 * @return string containing formatted text
 */
function html_attr($attr, &$val) {

	if (empty($val) && $val !== 0)
		return '';

	return " {$attr}=\"{$val}\" ";
}


/**
 * html_raw() - helper function to pad any potential html to prevent running into other elements/attributes
 * @param ref string $val
 * @return string containing formatted text
 */
function html_raw(&$val) {

	if (empty($val) && $val !== 0)
		return '';

	return " {$val} ";
}



/**
 * html_set_attribute() - useful for setting default values of arrays if no other value exists. can also merge existing or overwrite entirely
 * @param string $attr - the specific attribute we're looking for
 * @param string $val - the val we'd like to set to attr
 * @param array [ref] $attributes - the array of attributes to check and set
 * @param bool $merge - default false - if true, will merge with any existing val
 * @param bool $overwrite - default false - if true, will overwrite any existing val
 * @param string $merge_char - default is a single space. this is the character that will be prepended to $val before merging
 */
function html_set_attribute($attr, $val, &$attributes, $merge = false, $overwrite = false, $merge_char = ' ') {
	
	if ($overwrite || empty($attributes[$attr])) {

		$attributes[$attr] = $val;

	} else if ($merge && !empty($attributes[$attr])) {

		$found = false;

		// special case for class attr
		if ($attr == 'class') {
			$classes = explode(' ', $attributes[$attr]);
			foreach ($classes as $class) {
				if ($class == $val) {
					$found = true;
				}
			}
		} else {
			$found = strpos($attributes[$attr], $val);
		}
		if ($found === false)
			$attributes[$attr] .= $merge_char . $val;
	}
}

/**
 * html_parse_attributes() - parse an array (or a string converted to an array) of attributes for use in html
 * @param mixed $attributes - the attributes to attempt to parse
 * @return string - a parsed string to be inserted into an element's tag [definition]
 */
function html_parse_attributes($attributes = array()) {

	$attributes = html_get_attributes_array($attributes);

	$parsed = '';
	$self_closing = '';
	foreach ($attributes as $key => $val) {

		if ($val === 'self_closing' || ($key === 'self_closing' && $val === true)) {
			$self_closing = ' /';
			continue;
		}

		if (is_numeric($key)) {
			$parsed .= html_raw($val);
		} else {
			$parsed .= html_attr($key, $val);
		}
	}
	return $parsed . $self_closing;
}


/**
 * html_get_attributes_array() - convert an empty or string val argument to an appropriately formatted attributes style array
 * @param mixed $attributes - if empty you get a blank array, if string it will attempt to convert it to what it thinks is a viable array, if array then you get that array
 * @return array - the appropriate attributes array
 */
function html_get_attributes_array($attributes) {

	if (empty($attributes))
		return array();

	if (!is_array($attributes)) {

		$tmp_array = explode(',', $attributes);
		$attributes_array = array();

		foreach ($tmp_array as $attr) {

			$attr = trim($attr, ' ');

			$tmp_split = explode(' = ', $attr);
			if (count($tmp_split) == 2) {

				$key = trim($tmp_split[0], ' ');
				$val = trim($tmp_split[1], ' ');
				$opts_array[$key] = $val;

			} else {
				$opts_array[] = $attr;
			}
		}
		return $attributes_array;
	}

	return $attributes;
}

function encode_html($rawval) {
	
    return htmlentities($rawval, ENT_COMPAT, 'UTF-8');
}

// this exists for bikeshedding only
function encode_form_val($rawval) {

	//flash_message(_('encode_form_val() is deprecated. use encode_html() instead.'), 'warning');
	return encode_html($rawval);
}

/**
 * Useful to prevent XSS in html but with minimal characters.
 * Shortcut for: htmlspecialchars($rawval, ENT_QUOTES | ENT_SUBSTITUTE | ENT_HTML5, 'UTF-8')
 */
function encode_form_val_minimal($rawval)
{
    return htmlspecialchars($rawval, ENT_QUOTES | ENT_SUBSTITUTE | ENT_HTML5, 'UTF-8');
}

function get_options_from_array($options_array, $selected_value) {

	if (!is_array($options_array))
		return '';

	$output = '';
	foreach($options_array as $value => $text) {
		$selected = '';
		if ($selected_value == $value)
			$selected = ' selected="selected"';

		$output .= "<option value=\"{$value}\"{$selected}>{$text}</option>\n";
	}

	return $output;
}

function get_users_options_from_array($options_array, $selected_value) {

	if (!is_array($options_array))
		return '';

	$output = '';
	
	if (!empty($options_array)) {
		// grab and store the key/value for the "No Mapping" option
		$no_mapping_option = $arrayName = array("none" => $options_array['none']);
		// remove the "No Mapping" option from the array of users
		unset($options_array['none']);
		
		// sort the all of the servers users alphabtically
		asort($options_array, SORT_STRING | SORT_FLAG_CASE);
		
		// append the temporary "No Mapping" option array to the top of the sorted users array
		$options_array = $no_mapping_option + $options_array;
	}
	
	foreach($options_array as $value => $text) {
		$selected = '';
		if ($selected_value === $value)
			$selected = ' selected="selected"';

		$output .= "<option value=\"{$value}\"{$selected}>{$text}</option>\n";
	}

	return $output;
}

function tt_bind($content, $icon = 'fa-question-circle') {
	return '<i class="fa fa-lg ' . $icon . ' tt-bind" title="' . $content . '"></i>';
}
function e_tt_bind($content, $icon = 'fa-question-circle') {
	echo tt_bind($content, $icon);
}

function pop($title, $content, $icon = 'fa-question-circle') {
	return '<i class="fa fa-lg ' . $icon . ' pop" title="' . $title . '" data-content="' . $content . '" data-trigger="hover"></i>';
}
function e_pop($title, $content, $icon = 'fa-question-circle') {
	echo pop($title, $content, $icon);
}

function well_top_page_header($h1, $additional = '') {
	return
	'<div class="well top">
        <h1 class="fl">' . $h1 . '</h1>
        ' . $additional . '
        <div class="clear"></div>
    </div>';
}