<?php

// record a given message into the database
function log_message_to_db($server_id, $user_id, $level, $type, $message) {

	global $db;

	// sometimes a log function might end up before the db is initialized
	// this stops from the application from erroring out
	if (!isset($db) || empty($db))
		return false;

	if (!in_array($level, array(LOG_LEVEL_ERROR, LOG_LEVEL_WARN, LOG_LEVEL_INFO, LOG_LEVEL_DEBUG, LOG_LEVEL_TRACE)))
		return false;

	if (!in_array($type, array(LOG_TYPE_SECURITY, LOG_TYPE_SYSTEM, LOG_TYPE_AUDIT, LOG_TYPE_APPLICATION)))
		return false;

	if (!defined('LOG_LEVEL')) {

		$log_level = get_option('log_level', LOG_LEVEL_ERROR);

		if (!in_array($log_level, array(LOG_LEVEL_ERROR, LOG_LEVEL_WARN, LOG_LEVEL_INFO, LOG_LEVEL_DEBUG, LOG_LEVEL_TRACE)))
			$log_level = LOG_LEVEL_ERROR;

		// stop any more database calls looking for log level
		define('LOG_LEVEL', intval($log_level));
	}

	// if we are type subsystem, or running from cli then echo for output
	if ((defined('SUBSYSTEM') && constant('SUBSYSTEM')) || substr(php_sapi_name(), 0, 3) == 'cli') {
		if ($level <= LOG_LEVEL) {
			echo date('Y-m-d H:i:s') . "[s: {$server_id}, u: {$user_id}] {$message}\n";
		}
	}

	if ($level > LOG_LEVEL)
		return false;

	$bind_array = array(
		':server_id' => $server_id,
		':user_id' => $user_id,
		':level' => $level,
		':type' => $type,
		':message' => $message,
		);

	return $db->exec_query('INSERT INTO log (server_id, user_id, level, type, message) VALUES (:server_id, :user_id, :level, :type, :message)', $bind_array);
}

/* log wrapper functions
   these all redirect to log_message_to_db() */
function log_error($type, $message, $server_id = 0, $user_id = 0) {
	log_message_to_db($server_id, $user_id, LOG_LEVEL_ERROR, $type, $message);
}
function log_warn($type, $message, $server_id = 0, $user_id = 0) {
	log_message_to_db($server_id, $user_id, LOG_LEVEL_WARN, $type, $message);
}
function log_info($type, $message, $server_id = 0, $user_id = 0) {
	log_message_to_db($server_id, $user_id, LOG_LEVEL_INFO, $type, $message);
}
function log_debug($type, $message, $server_id = 0, $user_id = 0) {
	log_message_to_db($server_id, $user_id, LOG_LEVEL_DEBUG, $type, $message);
}
function log_trace($type, $message, $server_id = 0, $user_id = 0) {
	log_message_to_db($server_id, $user_id, LOG_LEVEL_TRACE, $type, $message);
}

/* open the log file handles for appending
   used by logging subsys */
function open_log_file_handles() {

	global $write_log;
	$write_log = get_option('write_log', false);

	global $write_debug; 
	$write_debug = get_option('write_debug', false);

	global $debug_in_db;
	$debug_in_db = get_option('debug_in_db', false);

	global $log_handle;
	$log_handle = false;

	if ($write_log) {
		$log_file = get_option('log_file', DEFAULT_LOG_FILE);
		$log_handle = fopen($log_file, "a");
	}

	global $debug_handle;
	$debug_handle = false;

	if ($write_debug) {
		$debug_file = get_option('debug_file', DEFAULT_DEBUG_FILE);
		$debug_handle = fopen($debug_file, "a");
	}
}

/* close the log file handles
   used by logging subsys */
function close_log_file_handles() {

	global $log_handle;
	global $debug_handle;

	if ($log_handle !== false)
		@fclose($log_handle);

	if ($debug_handle !== false)
		@fclose($debug_handle);
}

// convert log level to human readable
function log_level_to_human_readable($log_level) {
	
	switch ($log_level) {
		case LOG_LEVEL_DISABLED:	return _('Disabled');
		case LOG_LEVEL_ERROR:		return _('Error');
		case LOG_LEVEL_WARN:		return _('Warn');
		case LOG_LEVEL_INFO:		return _('Info');
		case LOG_LEVEL_DEBUG:		return _('Debug');
		case LOG_LEVEL_TRACE:		return _('Trace');
	}

	return _('Unknown');
}

// convert log type to human readable
function log_type_to_human_readable($log_type) {

	switch ($log_type) {
		case LOG_TYPE_APPLICATION:	return _('Application');
		case LOG_TYPE_SECURITY:		return _('Security');
		case LOG_TYPE_SYSTEM:		return _('System');
		case LOG_TYPE_AUDIT:		return _('Audit');
	}

	return _('Unknown');
}

// quick debug
// if var_dump is true, then var_dump $var
// else try and output in a sane manner
function quick_debug($var, $var_dump = false) {

	echo '<pre>';

	if ($var_dump) {
		var_dump($var);
	} else {

		if ($var === true)
			echo '((true))';
		else if ($var === false)
			echo '((false))';
		else if ($var === null)
			echo '((null))';
		else if (is_array($var))
			print_r($var);
		else
			echo $var;
	}

	echo '</pre>';
}

// alias for quick_debug
function qd($var, $var_dump = false) {
	quick_debug($var, $var_dump);
}

// quick debugging into a file for when screen printing is unavailable
function quick_debug_file($var, $file = '/tmp/test') {

	$ouput = '';

	if ($var === true)
		$output = '((true))';
	else if ($var === false)
		$output = '((false))';
	else if ($var === null)
		$output = '((null))';
	else if (is_array($var))
		$output = print_r($var, true);
	else
		$output = $var;

	$output .= "\n";
	file_put_contents($file, $output, FILE_APPEND);
}

// alias for quick_debug_file
function qdf($var, $file = '/tmp/test') {
	quick_debug_file($var, $file);
}
