#!/bin/bash

BASEDIR=$(dirname $(readlink -f $0))

# Import all CFG variables
. $BASEDIR/../var/fusion-sys.cfg
eval $(php $BASEDIR/import_fusionconfig.php)

BACKUPLOG="$proddir/var/log/backup.log"
ts=`date +%s`


###############################
# USAGE / HELP
###############################
usage () {
    echo ""
    echo "Use this script to backup Nagios Fusion."
    echo ""
    echo " -n | --name        Set the full name of the backup minus the .tar.gz"
    echo " -p | --prepend     Prepend a string to the .tar.gz name (ignored when using -n)"
    echo " -a | --append      Append a string to the .tar.gz name (ignored when using -n)"
    echo " -d | --directory   Change the directory to store the compressed backup"
    echo ""
}


###############################
# ADDING LOGIC FOR NEW BACKUPS
###############################
while [ -n "$1" ]; do
    case "$1" in
        -h | --help)
            usage
            exit 0
            ;;
        -n | --name)
            fullname=$2
            ;;
        -p | --prepen)
            prepend=$2"."
            ;;
        -a | --append)
            append="."$2
            ;;
        -d | --directory)
            rootdir=$2
            ;;
    esac
    shift
done

if [ -z $rootdir ]; then
    rootdir="/store/backups/nagiosfusion"
fi

# Move to root dir to store backups
cd $rootdir


#############################
# SET THE NAME & TIME
#############################
name=$fullname

if [ -z $fullname ]; then
    name=$prepend$ts$append
fi

# Get current Unix timestamp as name
if [ -z $name ]; then
    name=$ts
fi

# My working directory
mydir=$rootdir/$name

# Make directory for this specific backup
mkdir -p $mydir


##############################
# BACKUP DIRS
##############################
echo "Backing up Nagios Fusion directory..."
tar czfp $mydir/nagiosfusion.tar.gz /usr/local/nagiosfusion


##############################
# BACKUP DATABASES
##############################
mkdir -p $mydir/mysql
echo "Backing up MySQL databases..."
mysqldump -h $cfg__database__fusion__host -u $cfg__database__fusion__user --password="$cfg__database__fusion__pass" --add-drop-database -B $cfg__database__fusion__dbname > $mydir/mysql/fusion.sql
res=$?
if [ $res != 0 ]; then
    echo "Error backing up MySQL database 'fusion' - check the password in this script!" | tee -a $BACKUPLOG
    rm -r $mydir
    exit $res;
fi


##############################
# BACKUP CRONJOB ENTRIES
##############################
mkdir -p $mydir/cron
echo "Backing up nagiosfusion cronjob..."
cp /etc/cron.d/nagiosfusion $mydir/cron/nagiosfusion
echo "Backing up cronjobs for Apache..."
if [ -f /var/spool/cron/apache ]; then
    cp /var/spool/cron/apache $mydir/cron/apache
fi


##############################
# BACKUP SUDOERS
##############################
sudoers=/etc/sudoers.d/nagiosfusion
if [ -f "$sudoers" ]; then
    mkdir -p $mydir/sudoers
    echo "Backing up nagiosfusion sudoers.d file..."
    cp $sudoers $mydir/sudoers
fi


##############################
# BACKUP LOGROTATE
##############################
echo "Backing up logrotate config files..."
mkdir -p $mydir/logrotate
cp -rp /etc/logrotate.d/nagiosfusion $mydir/logrotate


##############################
# BACKUP APACHE CONFIG FILES
##############################
echo "Backing up Apache config files..."
mkdir -p $mydir/httpd
cp -rp $httpdconfdir/nagiosfusion.conf $mydir/httpd


##############################
# COMPRESS BACKUP
##############################
echo "Compressing backup..."
tar czfp $name.tar.gz $name
rm -rf $name

# Change ownership
chown $nagiosuser:$nagiosgroup $name.tar.gz

if [ -s $name.tar.gz ];then

    echo " "
    echo "==============="
    echo "BACKUP COMPLETE"
    echo "==============="
    echo "Backup stored in $rootdir/$name.tar.gz"

    exit 0;
else
    echo " "
    echo "==============="
    echo "BACKUP FAILED"
    echo "==============="
    echo "File was not created at $rootdir/$name.tar.gz"
    rm -r $mydir
    exit 1;
fi
