#!/bin/bash

# Make sure we have the backup file
if [ $# != 1 ]; then
    echo "Usage: $0 <backupfile>"
    echo "This script restores your Fusion system using a previously made Nagios Fusion backup file."
    exit 1
fi
backupfile=$1


# See if the user wants to continue
echo "WARNING: You are about to attempt a restore"
echo "If something goes wrong, you may be left with an unstable system"
read -r -p "Are you sure you want to continue? [y/N] " ok
case "$ok" in
    Y | y ) : ;;
    * )     exit 1
esac


BASEDIR=$(dirname $(readlink -f $0))

# Import all CFG variables
. $BASEDIR/../var/fusion-sys.cfg
eval $(php $BASEDIR/import_fusionconfig.php)


# MySQL root password
mysqlpass="nagiosfusion"


# Must be root
me=`whoami`
if [ $me != "root" ]; then
    echo "You must be root to run this script."
    exit 1
fi

rootdir=/store/backups/nagiosfusion

##############################
# MAKE SURE BACKUP FILE EXIST
##############################
if [ ! -f $backupfile ]; then
    echo "Unable to find backup file $backupfile!"
    exit 1
fi

backupver=$(eval $(echo $(tar -xzOf $backupfile $(basename $backupfile .tar.gz)/nagiosfusion.tar.gz | tar -xzOf - usr/local/nagiosfusion/var/fusion-sys.cfg |cat|grep ^ver\=));echo $ver)
if [ "$ver" != "$backupver" ]; then
    echo "WARNING: you are trying to restore a OS $backupver backup on a OS $ver system"
    echo ""
    echo "         httpd configurations will NOT be restored."
    echo
    read -r -p "Are you sure you want to continue? [y/N] " ok

    case "$ok" in
        Y | y ) : ;;
        * )     exit 1
    esac
fi


##############################
# MAKE TEMP RESTORE DIRECTORY
##############################
#ts=`echo $backupfile | cut -d . -f 1`
ts=`date +%s`
echo "TS=$ts"
mydir=${rootdir}/${ts}-restore
mkdir -p $mydir
if [ ! -d $mydir ]; then
    echo "Unable to create restore directory $mydir!"
    exit 1
fi


##############################
# UNTAR BACKUP
##############################
echo "Extracting backup to $mydir..."
cd $mydir
tar xzfps $backupfile

# Change to subdirectory
cd `ls`

# Make sure we have some directories here...
backupdir=`pwd`
echo "In $backupdir..."
if [ ! -f nagiosfusion.tar.gz ]; then
    echo "Unable to find files to restore in $backupdir"
    exit 1
fi

echo "Backup files look okay.  Preparing to restore..."



##############################
# RESTORE DIRS
##############################
rootdir=/
echo "Restoring directories to ${rootdir}..."

# Nagios XI
echo "Restoring Nagios Fusion..."
if [ "$ver" == "$backupver" ]; then
    rm -rf /usr/local/nagiosfusion
    cd $rootdir && tar xzfps $backupdir/nagiosfusion.tar.gz 
else
    mv $BASEDIR/../var/fusion-sys.cfg /tmp/fusion-sys.cfg
    rm -rf /usr/local/nagiosfusion
    cd $rootdir && tar xzfps $backupdir/nagiosfusion.tar.gz 
    cp -r /tmp/fusion-sys.cfg $BASEDIR/../var/fusion-sys.cfg
    rm -f /tmp/fusion-sys.cfg
fi


# RE-IMPORT ALL FUSION CFG VARS
. $BASEDIR/../var/fusion-sys.cfg
php $BASEDIR/import_fusionconfig.php > $BASEDIR/config.dat
. $BASEDIR/config.dat
rm $BASEDIR/config.dat


##############################
# RESTORE DATABASES
##############################
echo "Restoring MySQL Fusion database..."
mysql -h $cfg__database__fusion__host -u root --password="$mysqlpass" < $backupdir/mysql/fusion.sql
res=$?
if [ $res != 0 ]; then
    echo "Error restoring MySQL database 'fusion' - check the password in this script!"
    exit;
fi


##############################
# REMOVE ANY BAD DATA
##############################
echo "Truncating polling locks..."
mysql -h $cfg__database__fusion__host -u root --password="$mysqlpass" $cfg__database__fusion__dbname -e "TRUNCATE TABLE polling_lock;"


##############################
# RESTORE CRONJOB ENTRIES
##############################
echo "Restoring nagiosfusion cronjob..."
cp -rp $backupdir/cron/nagiosfusion /etc/cron.d
echo "Restoring Apache cronjobs..."
if [ -f $backupdir/cron/apache ]; then
    cp -rp $backupdir/cron/apache /var/spool/cron/apache
fi


##############################
# RESTORE SUDOERS
##############################
if [ -f $backupdir/sudoers ] && [ "$ver" == "$backupver" ]; then
    echo "Restoring sudoers file..."
    cp -rp $backupdir/sudoers /etc/sudoers.d/nagiosfusion
fi


##############################
# RESTORE LOGROTATE
##############################
echo "Restoring logrotate config files..."
cp -rp $backupdir/logrotate/nagiosfusion /etc/logrotate.d


##############################
# RESTORE APACHE CONFIG FILES
##############################
if [ "$ver" == "$backupver" ]; then
    echo "Restoring Apache config files..."
    cp -rp $backupdir/httpd/*.conf /etc/httpd/conf/
else
    echo "Skipping Apache config files restoration"
fi


##############################
# RESTART SERVICES
##############################
echo "Restarting services..."
$BASEDIR/manage_services.sh restart mysqld
$BASEDIR/manage_services.sh restart httpd


##############################
# DELETE TEMP RESTORE DIRECTORY
##############################
rm -rf $mydir


##############################
# DELETE forceinstall FILE
##############################
rm -f /tmp/nagiosfusion.forceinstall


echo " "
echo "==============="
echo "RESTORE COMPLETE"
echo "==============="

exit 0;
