#!/bin/bash -e

. ./fusion-sys.cfg
. ./functions.sh

# Check if user already configured repos
if [ -f installed.repos ]; then
    echo "Repos already configured - exiting."
    exit 0
fi
echo "Configuring Repos..."

# Check distro is CentOS / RHEL
case "$dist" in
    el8 | el9 )
        if  [ "$dist" == "el8" ] && [ "$distro" == "CentOS" ]; then
            echo "Centos Stream 8 is no longer supported due to end of life."
            echo "Please use CentOS Stream 9 instead."
            exit 1
        fi
        ;;
    ubuntu22 | ubuntu24 )
        :
        ;;
    debian11 | debian12 )
        :
        ;;
    ubuntu20 )
        echo "Ubuntu 20 is no longer supported due to end of life."
        echo "Please use Ubuntu 22, or 24 instead."
        exit 1
        ;;
    ubuntu18 )
        echo "Ubuntu 18 is no longer supported due to end of life."
        echo "Please use Ubuntu 20, 22, or 24 instead."
        exit 1
        ;;
    ubuntu16 )
        echo "Ubuntu 16 is no longer supported due to end of life."
        echo "Please use Ubuntu 20, 22, or 24 instead."
        exit 1
        ;;
    debian9 | debian10 )
        echo "Debian 9/10 is no longer supported due to end of life."
        echo "Please use Debian 11 instead."
        exit 1
        ;;
    el7 )
        echo "CentOS/RHEL/Oracle 7 are no longer supported due to end of life."
        echo "Please use CentOS Stream 9 or RHEL/Oracle 8 or 9 instead."
        exit 1
        ;;
    el6 )
        echo "CentOS / RHEL 6 is no longer supported due to end of life."
        echo "Please use CentOS Stream 9 or RHEL/Oracle 8 or 9 instead."
        exit 1
        ;;
    el5 )
        echo "CentOS / RHEL 5 is no longer supported due to end of life."
        echo "Please use CentOS Stream 9 or RHEL/Oracle 8 or 9 instead."
        exit 1
        ;;
    *)
        echo "$dist is not currently supported. Please use one of the following distros:" >&2
        echo "  CentOS Stream 9 " >&2
        echo "  RHEL, Oracle 8 or 9 " >&2
        echo "  Ubuntu 20, 22, or 24 LTS" >&2
        echo "  Debian 11" >&2
        exit 1
esac

# Check that the Red Hat installation is valid
if [ "$distro" = "RedHatEnterpriseServer" ] && [ -x /usr/sbin/rhn_check ] && ! /usr/sbin/rhn_check >/dev/null 2>&1; then
    ## Fails initial check with newer subscription method, nested if to check for subscription and proceed correctly if found -SR
    if [ -x /usr/bin/subscription-manager ] && [[ -z `subscription-manager list | grep Status: | grep -qF Subscribed` ]]; then
        echo "Passed New Subscription Check"
    else
        echo "Your Red Hat Installation is not registered or does not have proper entitlements." >&2
        echo "Please register or enable entitlements at rhn.redhat.com." >&2
        exit 1
    fi
fi

# If we are running an OracleServer, make sure optional is added
if [ "$distro" == "OracleServer" ]; then
    if [ "$dist" == "el7" ]; then

        # Remove this crap in el7 so that we can actually install any mysql
        yum remove -y mysql-community-libs mysql-community-common

        if `yum repolist all | grep optional | grep -qF enabled`; then
            echo "Passed Optional Repo Check"
        else
            echo "ERROR: Please enable the 'Optional' repo for your system in:" >&2
            if [ "$dist" == "el7" ]; then
                echo "       /etc/yum.repos.d/public-yum-ol7.repo" >&2
            fi
            exit 1
        fi
    fi
fi

# Make sure everything needed is still present, and we should be working
# with a current system anyway

if [ "$distro" == "Ubuntu" ]; then
    #./ubuntumods
    echo "Repos configured OK"
    touch installed.repos
    exit 0
fi

if [ "$distro" == "Debian" ]; then
    #./debianmods
    echo "Repos configured OK"
    touch installed.repos
    exit 0
fi

if [ "$distro" == "CentOS" ]; then
    if [ "$1" != noupdate ]; then
        echo "Updating system..."
        yum -y update
    fi
fi

if [ "$distro" == "RedHatEnterpriseServer" ]; then
    echo "Updating system..."
    yum -y update
fi

echo "Repo configured OK"
touch installed.repos