#!/bin/bash -e

. ./fusion-sys.cfg

# Check whether we have sufficient privileges
if [ $(id -u) -ne 0 ]; then
    echo "This script needs to be run as root/superuser." >&2
    exit 1
fi

# Was previous step completed?
if [ ! -f installed.repos ]; then
    echo "Repos were not properly prepared - run previous script" >&2
    exit 1
fi

# Are prereqs already installed?
if [ -f installed.prereqs ]; then
    echo "Prereqs already installed - skipping..."
    echo "Note: If you want to update the RPMs installed on this system, run your package manager's update command"
    exit 0
fi

# =======================
# Install Prereqs
# -----------------------

# Run basic yum install for now
unsupported="false"
if [ "$distro" == "RedHatEnterpriseServer" ] || [ "$distro" == "CentOS" ] || [ "$distro" == "OracleServer" ] || [ "$distro" == "CloudLinux" ]; then
	if [ "$dist" == "el6" ]; then
        yum install httpd mysql mysql-server php php-pdo php-devel php-mysql php-mbstring php-ldap zip unzip ntp sysstat -y
	elif [ "$dist" == "el7" ]; then

        # Install MySQL or MariaDB because why not make life difficult
        # - practical reasoning is someone might need to use MySQL and/or mariadb doesn't even
        #   exist on their installation due to reasons...

        # Check if both MySQL and MariaDB are available in yum
        av_mariadb=0
        av_mysql=0

        set +e
        if yum info mysql-community-server &>/dev/null; then
            av_mysql=1
        fi
        if yum info mariadb-server &>/dev/null; then
            av_mariadb=1
        fi
        set -e

        # Default to MariaDB unless it doesn't exist
        mysql_res="1"
        if [ $av_mariadb -eq 0 ]; then
            mysql_res="2"
        fi

        # IF we need to ask which MySQL/MariaDB to use...
        if [ "$INTERACTIVE" != "False" ] && [ $av_mysql -eq 1 ] && [ $av_mariadb -eq 1 ]; then
            cat <<-EOF
----------------------------------------

It looks like MySQL and MariaDB are available on your system. Please pick one:
1 - MariaDB (Default)
2 - MySQL Community

----------------------------------------
EOF
            read -p "Which DB server would you like to use? [1]: " mysql_res
        fi

        # Install the package
        if [ "$mysql_res" == "1" ] || [ "$mysql_res" == "" ]; then
            yum install -y mariadb mariadb-server mariadb-devel
        elif [ "$mysql_res" == "2" ]; then
            yum install -y mysql mysql-community-server mysql-community-devel
        else
            echo "You must select one of the options, or nothing and use the default."
            exit 1;
        fi

		yum install httpd php php-pdo  php-devel php-mysql php-mbstring php-ldap zip unzip ntp sysstat -y
    elif [ "$dist" == "el8" ]; then
        yum install -y mariadb mariadb-server mariadb-devel httpd php php-pdo php-devel php-json php-xml php-mysqlnd php-mbstring php-ldap zip unzip chrony sysstat sendmail
    elif [ "$dist" == "el9" ]; then
        yum install -y mariadb mariadb-server httpd php php-pdo php-devel php-json php-xml php-mysqlnd php-mbstring php-ldap zip unzip chrony sysstat sendmail
	else
        unsupported="true"
	fi
elif [ "$dist" == "ubuntu16" ] || [ "$dist" == "ubuntu18" ] || [ "$dist" == "ubuntu20" ] || [ "$dist" == "ubuntu22" ]; then
    apt-get update
    DEBIAN_FRONTEND=noninteractive apt-get install apache2 libapache2-mod-php mysql-server php php-mysql php-pdo php-curl php-xml php-dev php-mbstring php-ldap zip unzip ntp sysstat uuid-runtime locales selinux-utils cron sendmail -y
elif [ "$dist" == "ubuntu24" ]; then
    apt-get update
    DEBIAN_FRONTEND=noninteractive apt-get install apache2 libapache2-mod-php mysql-server php php-mysql php-pdo php-curl php-xml php-dev php-mbstring php-ldap zip unzip ntpsec sysstat uuid-runtime locales tzdata-legacy selinux-utils cron sendmail -y
elif [ "$dist" == "debian9" ]; then
    apt-get update
    DEBIAN_FRONTEND=noninteractive apt-get install -y apache2 libapache2-mod-php mysql-server php php-mysql php-dev php-xml php-curl php-ldap zip unzip ntp sysstat uuid-runtime sudo net-tools locales
elif [ "$dist" == "debian10" ] || [ "$dist" == "debian11" ]; then
    apt-get update
    DEBIAN_FRONTEND=noninteractive apt-get install -y apache2 libapache2-mod-php mariadb-server php php-mysql php-dev php-mbstring php-xml php-curl php-ldap zip unzip ntp sysstat uuid-runtime sudo net-tools locales sendmail
elif [ "$dist" == "debian12" ]; then
    apt-get update
    DEBIAN_FRONTEND=noninteractive apt-get install -y apache2 libapache2-mod-php mariadb-server php php-mysql php-dev php-mbstring php-xml php-curl php-ldap zip unzip ntpsec sysstat uuid-runtime sudo net-tools locales selinux-utils sendmail
else
    unsupported="true"
fi

if [ "$unsupported" = "true" ]; then
    echo "Operating system not supported. Supported OS are Centos 9, RHEL 8/9, Ubuntu 22/24, and Debian 11/12."
fi

# Set up languages for the install
if [ "$distro" == "RedHatEnterpriseServer" ] || [ "$distro" == "CentOS" ] || [ "$distro" == "OracleServer" ] || [ "$distro" == "CloudLinux" ]; then
    # I would prefer to have this reworked in the future to be a bit more elegant -AC
    if [ "$dist" == "el8" ] || [ "$dist" == "el9" ]; then
        yum install -y glibc-langpack-de glibc-langpack-es glibc-langpack-fr glibc-langpack-it glibc-langpack-ko glibc-langpack-pt glibc-langpack-ru glibc-langpack-zh glibc-langpack-ja glibc-langpack-pl
    fi
    
elif [ "$distro" == "Raspbian" ] || [ "$distro" == "Ubuntu" ] || [ "$distro" == "Debian" ]; then
    (
        cd ../fusion/html/includes/lang/locale
        for lang in *; do
            if [ -d "$lang" ]; then
                if [ "$distro" == "Debian" ]; then
                    sed -i "/#[[:blank:]]*$lang/s/#//" /etc/locale.gen
                else
                    locale-gen "$lang" &> /dev/null
                    locale-gen "$lang.UTF-8" &> /dev/null
                    if [ "$distro" == "Ubuntu" ]; then
                        if [ "$lang" == "ja_JP" ]; then
                            locale-gen "$lang.EUC-JP" &> /dev/null
                        elif [ "$lang" == "ko_KR" ]; then
                            locale-gen "$lang.EUC-KR" &> /dev/null
                        fi
                    fi
                fi
            fi
        done
        echo "Configuring locale... this might take a minute..."
        if [ `command -v locale-update` ]; then
            locale-update &> /dev/null
        else
            locale-gen &> /dev/null
        fi
    )
fi

echo "Prerequisites installed OK"
touch installed.prereqs