#!/bin/bash -e

. ./fusion-sys.cfg

# Was previous step completed?
if [ ! -f installed.usergroups ]; then
    echo "User and Groups were not properly prepared" >&2
    exit 1
fi

# Is database already installed?
if [ -f installed.database ]; then
    echo "Database already installed - skipping..."
    exit 0
fi

##########################################
# INITIALIZE MYSQL
##########################################

if [ $ver -ge 7 ] || [ "$distro" == "Ubuntu" ] || [ "$distro" == "Debian" ]; then
    my_verify="/usr/bin/mysql"
else
    my_verify="/etc/init.d/$mysqld"
fi

# Check that mysql is installed
if [ ! -x $my_verify ]; then
    echo "ERROR: MySQL not installed - exiting." >&2
    exit 1
fi
echo "MySQL installed OK - continuing..."

echo "Starting MySQL..."
if ! service "$mysqld" restart; then
    echo "ERROR: MySQL failed to start - exiting." >&2
    exit 1
fi

echo "Initializing MySQL..."

# Test for pre-saved password (from ./fullinstall script)
if mysqlshow -u root -p"$mysqlpass" &>/dev/null; then
    echo "Saved password '$mysqlpass' worked..."
# Test for null MySQL root password
elif mysqlshow -u root &>/dev/null; then
    # Set the password to "fusion"
    mysqlpass=fusion  
    mysqladmin -u root password "$mysqlpass"
    echo "MySQL root password is now set to: $mysqlpass"
else
    for i in 1 2 3; do
        if [ "$INTERACTIVE" = "True" ]; then
            # Ask for the password
            echo "Enter the MySQL root password to continue..."
            read -p "MySQL Root Password: " pass
        fi

        # Test the password
        if mysqlshow -u root -p"$pass" &>/dev/null; then
            echo "Password validated."
            mysqlpass="$pass"
            break
        else
            echo "Password failed." >&2
            [ $i -eq 3 ] && exit 1
        fi
    done
fi

mysqlversionflag=""
if [ "$dist" == "ubuntu20" ] || [ "$dist" == "ubuntu22" ] || [ "$dist" == "ubuntu24" ]; then
    mysqlversionflag="--mysql8"
fi

# Configure and tune MySQL/MariaDB
if ! ./03-database-configuration.php $mysqlversionflag; then
    echo "ERROR: Unable to configure MySQL/MariaDB. Did it install properly?" >&2
    exit 1
fi

# Special case for ubuntu 14
if [ "$dist" == "debian8" ] || [ "$dist" == "ubuntu14" ]; then
    sed -i 's/innodb_log_file_size/#innodb_log_file_size/' /etc/mysql/my.cnf
fi

# Update fusion-sys.cfg with MySQL password for later use by subcomponent install
if ! ./set-fusion-var.sh mysqlpass "$mysqlpass"; then
    echo "ERROR: Failed to update fusion-sys.cfg with MySQL password - exiting." >&2
    exit 1
fi

# If on Ubuntu 16.04 LTS, we need to disable the NO_ZERO_IN_DATE and NO_ZERO_DATE directives
if [ "$dist" == "ubuntu14" ] || [ "$dist" == "ubuntu16" ] || [ "$dist" == "ubuntu18" ]; then
    # Make sure the file doesn't already exist
    if [ ! -f /etc/mysql/conf.d/disable_no_zero.cnf ]; then
        echo "[mysqld]" >> /etc/mysql/conf.d/disable_no_zero.cnf
        echo "sql_mode='ONLY_FULL_GROUP_BY,ERROR_FOR_DIVISION_BY_ZERO,NO_AUTO_CREATE_USER,NO_ENGINE_SUBSTITUTION'" >> /etc/mysql/conf.d/disable_no_zero.cnf
    fi
fi

# Restart MySQL on all OS
if [ ! `command -v systemctl` ]; then
    service $mysqld restart
else
    systemctl restart $mysqld
fi

# TODO:
# Update automysqlbackup script for fusion
#if ! sed -i -e "s/PASSWORD=/PASSWORD=$mysqlpass/g" nagiosfusion/automysqlbackup; then
#    echo "ERROR: Failed to update automysqlbackup with MySQL password - exiting." >&2
#    exit 1
#fi

# Setup the database schema
echo "Setting up the Nagios Fusion database..."
mysql -p$mysqlpass < db/schemas/fusion.sql >/dev/null

echo "Database installed OK"
touch installed.database
