<?php
//
// Copyright (c) 2025 Nagios Enterprises, LLC. All rights reserved.
//

////////////////////////////////////////////////////////////////////////////////
// SNMPWALK UTILITY FUNCTIONS
////////////////////////////////////////////////////////////////////////////////


/**
 * Returns information about completed and running SNMP Walk Jobs
 * Pulls hostname and job start date (unix)
 * Determines status by checking for corresponding .pid file
 * 
 * @return  mixed  JSON encoded array containing job data values     
 */
function get_all_snmpwalk_jobs()
{
    $job_file_dir = get_tmp_dir();
    $job_pattern = $job_file_dir . '/snmpwalk_*.log';
    $job_files = glob($job_pattern);
    $job_data = [];
    if (!empty($job_files)) {
        foreach ($job_files as $job) {            
            $file_contents = file_get_contents($job);
            $job_status = check_job_status($job, $file_contents);
            $oid_count = count_job_oids($job, $file_contents);
            $command_values = get_job_command_values($file_contents);

            $file_name_parts = explode("_", $job);
            $data = [
                'hostname'  => $file_name_parts[1],
                'date'      => substr($file_name_parts[2], 0, strlen($file_name_parts[2]) - 4),
                'status'    => $job_status,
                'oid_count' => $oid_count,
                'command_values' => $command_values,
            ];
            $job_data[] = $data;
        }
    }    
    return json_encode($job_data);
}

function check_job_status($job, $file_contents)
{
    if (!file_exists($job) || !is_readable($job)) {
        return "Error";
    }

    $pid_file = substr($job, 0, strlen($job) - 4) . ".pid";

    if (file_exists($pid_file)) {
        return "Pending";
    }

    if ($file_contents === false) {
        return "Error";
    }

    // only scan the file header for errors, we don't want to trigger false positives based on OID names
    $file_head = implode("\n", array_slice(explode("\n", $file_contents), 0, 3));

    $simple_error_patterns = [
        'No Such Object',
        'No Such Instance',
        'Authentication failure',
        'Connection refused',
        'Cannot find module',
        'Unknown Object Identifier',
        'snmpwalk: ',
        'passphrase chosen is below the length requirements',
        'Error generating a key',
        'The supplied password length is too short',
        'Invalid authentication',
    ];

    $word_boundary_patterns = [
        'Timeout',
        'Error',
    ];

    foreach ($simple_error_patterns as $pattern) {
        if (stripos($file_head, $pattern)) {
            return "Error";
        }
    }

    foreach ($word_boundary_patterns as $pattern) {
        $regex = '/\b' . preg_quote($pattern, '/') . '\b:?/i';
        if (preg_match($regex, $file_head)) {
            return "Error";
        }
    }

    return "Finished";
}

function count_job_oids($job, $file_contents)
{
    if (!file_exists($job) || !is_readable($job)) {
        return "Error";
    }

    $line_count = substr_count($file_contents, "\n");

    if ($line_count > 0) {
        // assuming file has data, first oid should be on the second line, so don't count the first line
        $line_count--;
    }
    return $line_count;
}

function get_job_command_values($file_contents) {
    $command = trim(explode("\n", $file_contents)[0]);
    $command_values = [
        "device_address"   => "",
        "device_port"      => "161",
        "snmp_version"     => "",
        "snmp_community"   => "",
        "security_level"   => "",
        "username"         => "",
        "auth_password"    => "",
        "auth_protocol"    => "",
        "privacy_password" => "",
        "privacy_protocol" => "",
        "oid"              => "",
        "timeout"          => 15,
        "max_results"      => null,
    ];

    if (preg_match('/^Running:\s*snmpwalk\s+(.*)$/i', $command, $matches)) {
        $command = $matches[1];

        // Check for Max Results string, capture value and slice it off if sent to preserve parsing
        if (preg_match('/\s+with\s+Max\s+Results\s+set\s+to:\s*(\d+)\s*$/i', $command, $m)) {
            $command_values['max_results'] = (int)$m[1];
            $command = preg_replace('/\s+with\s+Max\s+Results\s+set\s+to:\s*\d+\s*$/i', '', $command);
        }

        $tokens = preg_split('/\s+/', $command);

        for ($i = 0; $i < count($tokens); $i++) {
            switch ($tokens[$i]) {
                case '-v':
                    $command_values['snmp_version'] = $tokens[$i + 1] ?? null;
                    $i++;
                    break;
                case '-c':
                    $command_values['snmp_community'] = $tokens[$i + 1] ?? null;
                    $i++;
                    break;
                case '-l':
                    $command_values['security_level'] = $tokens[$i + 1] ?? null;
                    $i++;
                    break;
                case '-u':
                    $command_values['username'] = $tokens[$i + 1] ?? null;
                    $i++;
                    break;
                case '-A':
                    $command_values['auth_password'] = $tokens[$i + 1] ?? null;
                    $i++;
                    break;
                case '-a':
                    $command_values['auth_protocol'] = $tokens[$i + 1] ?? null;
                    $i++;
                    break;
                case '-X':
                    $command_values['privacy_password'] = $tokens[$i + 1] ?? null;
                    $i++;
                    break;
                case '-x':
                    $command_values['privacy_protocol'] = $tokens[$i + 1] ?? null;
                    $i++;
                    break;
                case '-p':
                    $command_values['device_port'] = $tokens[$i + 1] ?? "161";
                    $i++;
                    break;
                default:
                    // catch values without explicit flags
                    if (preg_match('/^([a-zA-Z0-9\.\-]+)(?::(\d+))?$/', $tokens[$i], $host_matches)) {
                        $command_values['device_address'] = $host_matches[1];
                        if (!empty($host_matches[2])) {
                            $command_values['device_port'] = $host_matches[2];
                        }
                        // check next token for OID
                        if (($i + 1) < count($tokens)) {
                            $candidate = $tokens[$i + 1];
                            if (
                                preg_match('/^\d+(\.\d+)+$/', $candidate) ||
                                preg_match('/^[a-zA-Z][a-zA-Z0-9\-]+$/', $candidate)
                            ) {
                                $command_values['oid'] = $candidate;
                                $i++;
                            }
                        }
                    }
                    break;
            }
        }
    }

    return $command_values;
}
