<?php
//
// TFTP Server Config Wizard
// Copyright (c) 2016-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__).'/../configwizardhelper.inc.php');

tftp_configwizard_init();

function tftp_configwizard_init()
{
    $name = "tftp";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.1",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor a TFTP Server Connection or Specific File"),
        CONFIGWIZARD_DISPLAYTITLE => _("TFTP"),
        CONFIGWIZARD_FUNCTION => "tftp_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "tftp.png",
        CONFIGWIZARD_FILTER_GROUPS => array('network', 'server'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @return array
 */
function tftp_configwizard_check_prereqs()
{
    $errors = array();
    exec("which tftp 2>&1", $output, $return_var);

    if ($return_var != 0) {
        $errors[] = _('It looks like you are missing tftp on your Nagios XI server.').'<br><br> Run: &nbsp; <b>yum install tftp -y</b> &nbsp; as root user on your Nagios XI server.';
    }

    return $errors;
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function tftp_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "tftp";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $errors = tftp_configwizard_check_prereqs();

            if ($errors) {
                $output .= '<div class="message"><ul class="errorMessage">';

                foreach ($errors as $error) {
                    $output .= "<li><p>$error</p></li>";
                }

                $output .= '</ul></div>';

            } else {

                $address = grab_array_var($inargs, "ip_address", "");

                // Save data from clicking "back" in stage 2
                $services = grab_array_var($inargs, "services", array());
                $serviceargs = grab_array_var($inargs, "serviceargs", array());

                # Get the existing host/node configurations.
                # TODO: Include passwords/secrets?
                $nodes = get_configwizard_hosts($wizard_name);

                ########################################################################################
                # Load the html
                # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
                #   to load the PHP from the Step1 file into the $output string.
                ########################################################################################
                ob_start();
                include __DIR__.'/steps/step1.php';
                $output = ob_get_clean();
            }

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $address = nagiosccm_replace_user_macros($address);

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($address) == false) {
                $errmsg[$errors++] = _("No address specified.");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $filename = grab_array_var($inargs, "filename", "");
            $filesize = grab_array_var($inargs, "filesize", "");

            $services = grab_array_var($inargs, "services", array(
                "tftpconnect" => "on",
                "id" => "on"
            ));

            $serviceargs = grab_array_var($inargs, "serviceargs", array(
                "filename" => "",
                "filesize" => ""
            ));

            $services_serial = grab_array_var($inargs, "services_serial");

            if (!empty($services_serial)) {
                $services = json_decode(base64_decode($services_serial), true);
            }

            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial");

            if (!empty($serviceargs_serial)) {
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            }

            if (!array_key_exists('id', $services)) $services['id'] = array();

            if (!isset($services['id']) || !is_array($services['id'])) {
                $services['id'] = array();
            }
            if (!array_key_exists('id', $serviceargs)) $serviceargs['id'] = array();
            
            if (!isset($serviceargs['id']) || !is_array($serviceargs['id'])) {
                $serviceargs['id'] = array();
            }

            for ($x = 0; $x < 2; $x++) {
                if (!array_key_exists($x, $services['id'])) {
                    $services['id'][$x] = '';
                }

                if (!array_key_exists($x, $serviceargs['id'])) {
                    $serviceargs['id'][$x] = array(
                        'id' => '',
                        'filename' => '',
                        'filesize' => ''
                    );
                }
            }

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $filename = grab_array_var($inargs, "filename");
            $filesize = grab_array_var($inargs, "filesize");
            $services = grab_array_var($inargs, "services", array());
            $serviceargs = grab_array_var($inargs, "serviceargs", array());

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (is_valid_domain_name($address) == false) {
                $errmsg[$errors++] = _("Invalid domain name.");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE3OPTS:

            $output = '<div style="margin-bottom: 20px;">'._('The selected TFTP server will be queried to return a check value. Click Finish to continue.').'</div>';
            $result = CONFIGWIZARD_HIDE_OPTIONS;

            break;

        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $filename = grab_array_var($inargs, "filename");
            $filesize = grab_array_var($inargs, "filesize");
            $services = grab_array_var($inargs, "services", array());
            $serviceargs = grab_array_var($inargs, "serviceargs", array());
            $services_serial = grab_array_var($inargs, "services_serial", base64_encode(json_encode($services)));
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", base64_encode(json_encode($serviceargs)));

            $output = '
                <input type="hidden" name="ip_address" value="'.encode_form_val($address).'">
                <input type="hidden" name="filename" value="'.encode_form_val($filename).'">
                <input type="hidden" name="filesize" value="'.encode_form_val($filesize).'">
                <input type="hidden" name="services_serial" value="'.$services_serial.'">
                <input type="hidden" name="serviceargs_serial" value="'.$serviceargs_serial.'">';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            $address = grab_array_var($inargs, "ip_address");
            $filename = grab_array_var($inargs, "filename");
            $filesize = grab_array_var($inargs, "filesize");
            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $output = '
                <input type="hidden" name="ip_address" value="'.encode_form_val($address).'">
                <input type="hidden" name="filename" value="'.encode_form_val($filename).'">
                <input type="hidden" name="filesize" value="'.encode_form_val($filesize).'">
            ';

            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $address = grab_array_var($inargs, "ip_address", "");
            $filename = grab_array_var($inargs, "filename", "");
            $filesize = grab_array_var($inargs, "filesize", "");
            $services = grab_array_var($inargs, "services", array());
            $serviceargs = grab_array_var($inargs, "serviceargs", array());
            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            $servicename = "";

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["ip_address"] = $address;
            $meta_arr["filename"] = $filename;
            $meta_arr["filesize"] = $filesize;
            save_configwizard_object_meta($wizard_name, $address, "", $meta_arr);

            $objs = array();

            if (!host_exists($address)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_tftp_host",
                    "host_name" => $address,
                    "check_command" => "check-host-alive-tftp!" . $address,
                    "icon_image" => "tftp.png",
                    "statusmap_image" => "tftp.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            // see which services we should monitor
            foreach ($services as $svc => $svcstate) {
                switch ($svc) {

                    case "tftpconnect":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $address,
                            "service_description" => "TFTP Connect",
                            "use" => "xiwizard_tftp_service_connect",
                            "check_command" => "check_tftp_connect!" . $address,
                            "icon_image" => "tftp.png",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "id":
                        // create service for each file we are getting
                        foreach ($serviceargs[$svc] as $k => $i) {
                            // make sure id is checked "on" to run
                            if ($services['id'][$k] != "on")
                                continue;

                            $servicename = escapeshellarg($i['filename']);
                            $filename = escapeshellarg($i['filename']);
                            $filesize = escapeshellarg($i['filesize']);

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $address,
                                "service_description" => "TFTP Get:" . $servicename,
                                "use" => "xiwizard_tftp_service_get",
                                "check_command" => "check_tftp_get!" . $address . "!" . $filename  . "!" . $filesize,
                                "icon_image" => "tftp.png",
                                "_xiwizard" => $wizard_name,
                            );
                        }
                        break;

                    default:
                        break;
                }
            }

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS]=$objs;

            break;

        default:
            break;
        }

    return $output;
}