<?php
//
// MTR Component
// Copyright (c) 2018-2024 Nagios Enterprises, LLC. All rights reserved.
//

require_once(dirname(__FILE__) . '/../componenthelper.inc.php');

mtr_component_init();


////////////////////////////////////////////////////////////////////////
// COMPONENT INIT FUNCTIONS
////////////////////////////////////////////////////////////////////////


function mtr_component_init()
{
    $component_name = "mtr";
    $args = array(
        COMPONENT_NAME => $component_name,
        COMPONENT_VERSION => "1.0.6",
        COMPONENT_AUTHOR => "Nagios Enterprises, LLC",
        COMPONENT_DESCRIPTION => _("Provides integration with the MutliTech MTR devices."),
        COMPONENT_TITLE => _("MultiTech MTR Integration"),
        COMPONENT_CONFIGFUNCTION => "mtr_component_config_func"
    );
    register_component($component_name, $args);

    $args = array(
        NOTIFICATIONMETHOD_FUNCTION => 'mtr_component_notificationmethod_func',
    );
    register_notificationmethod('mtr', $args);

    register_callback(CALLBACK_USER_NOTIFICATION_METHODS_TABS_INIT, 'mtr_component_methods_addtab');
    register_callback(CALLBACK_USER_NOTIFICATION_MESSAGES_TABS_INIT, 'mtr_component_messages_addtab');
}


///////////////////////////////////////////////////////
// TAB FUNCTIONS
///////////////////////////////////////////////////////


function mtr_component_messages_addtab($cbtype, &$cbdata)
{
    $settings_raw = get_option("mtr_component_options");
    if ($settings_raw == "")
        $settings = array();
    else
        $settings = unserialize($settings_raw);
    $enabled = grab_array_var($settings, "enabled", 0);
    if ($enabled != 1)
        return;

    $newtab = array(
        "id" => "mtr",
        "title" => "MTR",
    );

    $cbdata["tabs"][] = $newtab;
}


function mtr_component_methods_addtab($cbtype, &$cbdata)
{
    $settings_raw = get_option("mtr_component_options");
    if ($settings_raw == "")
        $settings = array();
    else
        $settings = unserialize($settings_raw);
    $enabled = grab_array_var($settings, "enabled", 0);
    if ($enabled != 1)
        return;

    $newtab = array(
        "id" => "mtr",
        "title" => "MTR",
    );

    $cbdata["tabs"][] = $newtab;
}


///////////////////////////////////////////////////////
// NOTIFICATION METHOD FUNCTIONS
///////////////////////////////////////////////////////


function mtr_component_notificationmethod_func($mode, $inargs, &$outargs, &$result)
{
    $component_name = "mtr";

    // Initialize return values
    $result = 0;
    $outargs = array();
    $output = '';

    // Bail if this component has been disabled by the admin
    $settings = array();
    $settings_raw = get_option("mtr_component_options", array());
    if (!empty($settings_raw)) {
        $settings = unserialize($settings_raw);
    }
    $mtr_enabled = grab_array_var($settings, "enabled", 0);
    if ($mtr_enabled != 1) {
        return $output;
    }

    switch ($mode) {

        case NOTIFICATIONMETHOD_MODE_GETCONFIGOPTIONS:

            // Defaults
            $mtr_notifications_enabled = get_user_meta(0, 'mtr_notifications_enabled');
            $mtr_mobile_number = get_user_meta(0, 'mtr_mobile_number', get_user_meta(0, 'mobile_number'));

            // Get values from form submission
            $mtr_notifications_enabled = grab_request_var("mtr_notifications_enabled", $mtr_notifications_enabled);
            $mtr_notifications_enabled = checkbox_binary($mtr_notifications_enabled);
            $mtr_mobile_number = grab_request_var("mtr_mobile_number", $mtr_mobile_number);

            $component_url = get_component_url_base($component_name);

            $output = "
            
            <p><img src='" . $component_url . "/images/multitech.png' alt='MultiTech MTR' title='MultiTech MTR'></p>
            
            <table class='table table-condensed table-no-border table-auto-width'>
                <tr>
                    <td class='vt'>
                        <input type='checkbox' class='checkbox' name='mtr_notifications_enabled' " . is_checked($mtr_notifications_enabled, 1) . ">
                    </td>
                    <td>
                        <b>"._('SMS Text Message')."</b><br>
                        " . _("Receive out-of-band SMS alerts via the") . " <a href='http://www.multitech.com/en_US/PRODUCTS/Families/MultiModemiSMS/' target='_top'>MultiTech MTR</a>.
                        <div class='pad-t5'>
                            <input type='text' size='15' name='mtr_mobile_number'value='" . $mtr_mobile_number . "' class='form-control condensed' placeholder='"._('Phone number')."'>
                        </div>
                    </td>
                </tr>
                <tr>
                    <td>
                        <input type='checkbox' class='checkbox' name='mtr_send_test' id='ts'>
                    </td>
                    <td>
                        <label for='ts' style='font-weight: normal; color: #000;'>" . _("Send a test SMS message to the number specified above.") . "</label>
                    </td>
                </tr>
            </table>";
            break;

        case NOTIFICATIONMETHOD_MODE_SETCONFIGOPTIONS:

            $mtr_notifications_enabled = grab_array_var($inargs, "mtr_notifications_enabled", 0);
            $mtr_notifications_enabled = checkbox_binary($mtr_notifications_enabled);
            $mtr_mobile_number = grab_array_var($inargs, "mtr_mobile_number", "");

            // Check for errors
            $errors = 0;
            $errmsg = array();
            $okmsg = array();
            if ($mtr_notifications_enabled == 1) {
                if ($mtr_mobile_number == "") {
                    $errmsg[$errors++] = _("Mobile phone number for SMS alerts is blank.");
                }
            }

            // handle errors
            if ($errors > 0) {
                $outargs[NOTIFICATIONMETHOD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            // Save values
            set_user_meta(0, 'mtr_notifications_enabled', $mtr_notifications_enabled);
            set_user_meta(0, 'mtr_mobile_number', $mtr_mobile_number);

            // Send a test message
            $testnumber = grab_array_var($inargs, "mtr_send_test", "");
            $smsoutput = array();
            $smsresult = -1;
            if ($testnumber != "" && $errors == 0) {
                $args = array(
                    "number" => $mtr_mobile_number,
                    "message" => "This is a test SMS message from Nagios XI\n" . get_option('url'),
                );
                $smsresult = mtr_component_send_sms($args, $smsoutput);

                // handle errors
                if ($smsresult == 1) {
                    $errmsg[$errors++] = _("An error occurred sending the test SMS message.");
                    $outargs[NOTIFICATIONMETHOD_ERROR_MESSAGES] = $errmsg;
                    $result = 1;
                    return '';
                }

                // Success message
                $okmsg = array();
                if ($smsresult == 0) {
                    $okmsg[] = _("Test SMS message sent to ") . $mtr_mobile_number . _(" successfully.");
                    $outargs[NOTIFICATIONMETHOD_INFO_MESSAGES] = $okmsg;
                }
            }

            break;

        case NOTIFICATIONMETHOD_MODE_GETMESSAGEFORMAT:

            // Defaults/saved values
            $mtr_notifications_host_message = mtr_component_get_host_message(0);
            $mtr_notifications_service_message = mtr_component_get_service_message(0);

            // Newly submitted values
            $mtr_notifications_host_message = grab_array_var($inargs, "mtr_notifications_host_message", $mtr_notifications_host_message);
            $mtr_notifications_service_message = grab_array_var($inargs, "mtr_notifications_service_message", $mtr_notifications_service_message);

            $component_url = get_component_url_base($component_name);

            // Warn user about notifications being disabled
            if (get_user_meta(0, 'mtr_notifications_enabled') == 0) {
                $msg = "<div>" . _("Note: You currently have SMS notifications disabled.") . "  <a href='notifymethods.php#tab-custom-mtr'>" . _("Change settings") . "</a>.</div>";
                $output .= get_message_text(true, false, $msg);
            }

            $output .= '

<p><img src="' . $component_url . '/images/multitech.png" alt="MultiTech MTR" title="MultiTech MTR"></p>

<h5 class="ul">' . _('SMS Message Settings') . '</h5>
    
<p>' . _('Specify the format of the SMS messages you want to receive.') . '<br><b>' . _('NOTE') . ':</b> ' . _('The maximum length of SMS text messages is 160 characters.  Messages longer than this limit will be trimmed.') . '</p>
    
<table class="table table-condensed table-no-border table-auto-width">
    <tr>
        <td class="vt">
            <label>' . _('Host Alert Message') . ':</label>
        </td>
        <td>
            <textarea name="mtr_notifications_host_message" style="width: 400px; height: 100px;" class="form-control">' . htmlentities($mtr_notifications_host_message) . '</textarea>
        </td>
    </tr>
    <tr>
        <td class="vt">
            <label>' . _('Service Alert Message') . ':</label>
        </td>
        <td>
            <textarea name="mtr_notifications_service_message" style="width: 400px; height: 100px;" class="form-control">' . htmlentities($mtr_notifications_service_message) . '</textarea>
        </td>
    </tr>
</table>';

            break;

        case NOTIFICATIONMETHOD_MODE_SETMESSAGEFORMAT:

            // Newly submitted values
            $mtr_notifications_host_message = grab_array_var($inargs, "mtr_notifications_host_message", "");
            $mtr_notifications_service_message = grab_array_var($inargs, "mtr_notifications_service_message", "");

            // Save options
            set_user_meta(0, "mtr_notifications_host_message", $mtr_notifications_host_message);
            set_user_meta(0, "mtr_notifications_service_message", $mtr_notifications_service_message);

            break;

        default:
            $output = "";
            break;
    }

    return $output;
}


///////////////////////////////////////////////////////////////////////////////////////////
//CONFIG FUNCTIONS
///////////////////////////////////////////////////////////////////////////////////////////


function mtr_component_config_func($mode, $inargs, &$outargs, &$result)
{
    $result = 0;
    $output = "";
    $component_name = "mtr";

    switch ($mode) {

        case COMPONENT_CONFIGMODE_GETSETTINGSHTML:

            $settings_raw = get_option("mtr_component_options", array());
            if (!empty($settings_raw)) {
                $settings = unserialize($settings_raw);
            }

            // Initial values
            $address = grab_array_var($settings, "address", "");
            $http_port = grab_array_var($settings, "http_port", "443");
            $username = grab_array_var($settings, "username", "nagiosadmin");
            $password = grab_array_var($settings, "password", "");
            $enabled = grab_array_var($settings, "enabled", "");

            // Values passed to us
            $address = grab_array_var($inargs, "address", $address);
            $http_port = grab_array_var($inargs, "http_port", $http_port);
            $username = grab_array_var($inargs, "username", $username);
            $password = grab_array_var($inargs, "password", $password);
            $enabled = checkbox_binary(grab_array_var($inargs, "enabled", $enabled));

            $component_url = get_component_url_base($component_name);

            $output = '
            
<p><a href="http://www.multitech.com" target="_blank"><img src="' . $component_url . '/images/multitech.png"></a></p>
<p>' . _('Allows integration between Nagios XI and a MultiTech MTR device.') . ' <strong>' . _('Requires') . '</strong> ' . _('MultiTech MTR') . ' <a href="https://assets.nagios.com/downloads/nagiosxi/docs/MultiTech-MTR-API-Integration-in-Nagios-XI-2024.pdf" target="_blank" rel="nofollow" title="Documentation">' . _('setup') . '</a>.</p>

<h5 class="ul">' . _('Integration Settings') . '</h5>

<table class="table table-condensed table-no-border table-auto-width">
    <tr>
        <td></td>
        <td class="checkbox">
            <label>
                <input type="checkbox" class="checkbox" id="enabled" name="enabled" ' . is_checked($enabled, 1) . '>
                ' . _('Enable integration') . '
            </label>
        </td>
    </tr>
</table>

<h5 class="ul">' . _('Sender Settings') . '</h5>

<p>' . _('These settings relate to sending SMS alerts from Nagios XI through the MTR API.') . '</p>

<table class="table table-condensed table-no-border table-auto-width">
    <tr>
        <td class="vt">
            <label>' . _('Address') . ':</label>
        </td>
        <td>
            <input type="text" size="40" name="address" id="address" value="' . encode_form_val($address) . '" class="form-control">
            <div class="subtext">' . _('The IP address or hostname of the MTR.') . '</div>
        </td>
    </tr>
    <tr>
        <td class="vt">
            <label>' . _('HTTPS Port:') . '</label>
        </td>
        <td>
            <input type="text" size="4" name="http_port" id="http_port" value="' . intval($http_port) . '" class="form-control">
            <div class="subtext">' . _('The HTTPS port used to access the MTR API') . '.</div>
        </td>
    </tr>
    <tr>
        <td class="vt">
            <label>' . _('Username') . ':</label>
        </td>
        <td>
            <input type="text" size="20" name="username" id="username" value="' . encode_form_val($username) . '" class="form-control">
            <div class="subtext">' . _('The username used to authenticate to the MTR API.') . '</div>
        </td>
    </tr>
    <tr>
        <td class="vt">
            <label>' . _('Password:') . '</label>
        </td>
        <td>
            <input type="password" size="20" name="password" id="password" value="' . encode_form_val($password) . '" class="form-control">
            <div class="subtext">' . _('The password used to authenticate to the MTR API.') . '</div>
        </td>
    </tr>
</table>';

            $output .= '

<h5 class="ul">' . _('Test Message') . '</h5>

<table class="table table-condensed table-no-border table-auto-width">
    <tr>
        <td valign="top">
        <label>' . _('Phone Number') . ':</label>
        </td>
        <td>
            <input type="text" size="20" name="testnumber" id="testnumber" value="" class="form-control">
            <div class="subtext">' . _('Enter a mobile phone number to send a test SMS message to.  This is used for testing the Nagios XI and MultiTech MTR integration.') . '</div>
        </td>
    </tr>
</table>

<p style="font-size: 7pt;">' . _('MultiTech and the MultiTech logo are trademarks or registered trademarks of') . ' <a href="http://www.multitech.com" target="_blank">Multi-Tech Systems, Inc.</a></p>
            ';
            break;

        case COMPONENT_CONFIGMODE_SAVESETTINGS:

            // Get variables
            $address = grab_array_var($inargs, "address", "");
            $http_port = grab_array_var($inargs, "http_port", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $enabled = checkbox_binary(grab_array_var($inargs, "enabled", ""));

            // Validate variables
            $errors = 0;
            $errmsg = array();
            if ($enabled == 1) {
                if (have_value($address) == false) {
                    $errmsg[$errors++] = "No address specified.";
                }
                if (have_value($http_port) == false) {
                    $errmsg[$errors++] = "No HTTP port specified.";
                }
                if (have_value($username) == false) {
                    $errmsg[$errors++] = "No username specified.";
                }
                if (have_value($password) == false) {
                    $errmsg[$errors++] = "No password specified.";
                }
            }

            // Handle errors
            if ($errors > 0) {
                $outargs[COMPONENT_ERROR_MESSAGES] = $errmsg;
                $result = 1;
                return '';
            }

            // Save settings
            $settings = array(
                "address" => $address,
                "http_port" => $http_port,
                "username" => $username,
                "password" => $password,
                "enabled" => $enabled,
            );
            set_option("mtr_component_options", serialize($settings));

            // Send a test message
            $testnumber = grab_array_var($inargs, "testnumber", "");
            $smsoutput = array();
            $smsresult = -1;
            if ($testnumber != "") {
                $args = array(
                    "number" => $testnumber,
                    "message" => "Test MultiTech MTR message from Nagios XI\n" . get_option('url'),
                );
                $smsresult = mtr_component_send_sms($args, $smsoutput);

                // handle errors
                if ($smsresult == 1) {
                    $errmsg[$errors++] = "An error occurred sending the test SMS message.";
                    $outargs[COMPONENT_ERROR_MESSAGES] = $errmsg;
                    $result = 1;
                    return '';
                }
            }

            // Info messages
            $okmsg = array();
            $okmsg[] = _("MultiTech MTR settings updated.");
            if ($smsresult == 0) {
                $okmsg[] = _("Test SMS message sent to ") . $testnumber . _(" successfully.");
            }
            $outargs[COMPONENT_INFO_MESSAGES] = $okmsg;

            break;

        default:
            break;

    }

    return $output;
}


function mtr_component_get_host_message($user_id)
{
    $txt = get_user_meta($user_id, 'mtr_notifications_host_message');
    if ($txt == "")
        $txt = "%host% %type% (%hoststate%) %hostoutput% Addr: %hostaddress%  Time: %datetime% Nagios URL: %xiserverurl%";
    return $txt;
}


function mtr_component_get_service_message($user_id)
{
    $txt = get_user_meta($user_id, 'mtr_notifications_service_message');
    if ($txt == "")
        $txt = "%host% / %service% %type% (%servicestate%) %serviceoutput% Time: %datetime% Nagios URL: %xiserverurl%";
    return $txt;
}


////////////////////////////////////////////////////////////////////////
// EVENT HANDLER AND NOTIFICATION FUNCTIONS
////////////////////////////////////////////////////////////////////////


register_callback(CALLBACK_EVENT_PROCESSED, 'mtr_component_eventhandler');


function mtr_component_eventhandler_register_callbacks()
{

    $settings_raw = get_option("mtr_component_options", array());
    $settings = array();
    if (!empty($settings_raw)) {
        $settings = unserialize($settings_raw);
    }
    
    if (empty($settings)){
        return;
    }
    
    // Bail out of component is not enabled
    $enabled = grab_array_var($settings, "enabled", "");
    if ($enabled != 1) {
        return;
    }

    register_callback(CALLBACK_EVENT_PROCESSED, 'mtr_component_eventhandler');
}


function mtr_component_eventhandler($cbtype, $args)
{

    $settings_raw = get_option("mtr_component_options", array());
    $settings = array();
    if (!empty($settings_raw)) {
        $settings = unserialize($settings_raw);
    }

    // Bail out of component is not enabled
    $enabled = grab_array_var($settings, "enabled", "");
    if ($enabled != 1) {
        return;
    }

    switch ($args["event_type"]) {
        case EVENTTYPE_NOTIFICATION:
            mtr_component_handle_notification_event($args);
            break;
        default:
            break;
    }
}


function mtr_component_handle_notification_event($args)
{

    $meta = $args["event_meta"];
    $contact = $meta["contact"];
    $nt = $meta["notification-type"];

    // Find the XI user
    $user_id = get_user_id($contact);
    if ($user_id <= 0) {
        return;
    }

    echo "==== MTR Notification Handler ====\n";

    // Bail if user has notifications disabled completely
    $notifications_enabled = get_user_meta($user_id, 'enable_notifications');
    if ($notifications_enabled != 1) {
        echo "ERROR: User has (global) notifications disabled!\n";
        return;
    }

    // Set user id session variable - used later in date/time, preference, etc. functions
    if (!defined("NAGIOSXI_USER_ID")) {
        define("NAGIOSXI_USER_ID", $user_id);
    }

    echo " MTR: CONTACT=$contact, USERID=$user_id\n";

    // Get settings
    $mtr_notifications_enabled = get_user_meta($user_id, "mtr_notifications_enabled");
    $mtr_mobile_number = get_user_meta($user_id, 'mtr_mobile_number');

    // Not enabled for this user
    if ($mtr_notifications_enabled != 1) {
        echo " MTR: User has SMS notifications disabled\n";
        return 1;
    }

    // Don't have a mobile number
    if ($mtr_mobile_number == "") {
        echo "MTR: User does not have a mobile number specified\n";
        return 1;
    }

    // Support for SMS notification options
    if (get_product_release() > 407) {

        // Get SMS notification options for user 
        $notify_sms_host_recovery = get_user_meta($user_id, 'notify_sms_host_recovery', get_user_meta($user_id, 'notify_host_recovery'));
        $notify_sms_host_down = get_user_meta($user_id, 'notify_sms_host_down', get_user_meta($user_id, 'notify_host_down'));
        $notify_sms_host_unreachable = get_user_meta($user_id, 'notify_sms_host_unreachable', get_user_meta($user_id, 'notify_host_unreachable'));
        $notify_sms_host_flapping = get_user_meta($user_id, 'notify_sms_host_flapping', get_user_meta($user_id, 'notify_host_flapping'));
        $notify_sms_host_downtime = get_user_meta($user_id, 'notify_sms_host_downtime', get_user_meta($user_id, 'notify_host_downtime'));
		$notify_sms_host_acknowledgement = get_user_meta($user_id, 'notify_sms_host_acknowledgement', get_user_meta($user_id, 'notify_host_acknowledgement'));
		
		
        $notify_sms_service_recovery = get_user_meta($user_id, 'notify_sms_service_recovery', get_user_meta($user_id, 'notify_service_recovery'));
        $notify_sms_service_warning = get_user_meta($user_id, 'notify_sms_service_warning', get_user_meta($user_id, 'notify_service_warning'));
        $notify_sms_service_unknown = get_user_meta($user_id, 'notify_sms_service_unknown', get_user_meta($user_id, 'notify_service_unknown'));
        $notify_sms_service_critical = get_user_meta($user_id, 'notify_sms_service_critical', get_user_meta($user_id, 'notify_service_critical'));
        $notify_sms_service_flapping = get_user_meta($user_id, 'notify_sms_service_flapping', get_user_meta($user_id, 'notify_service_flapping'));
        $notify_sms_service_downtime = get_user_meta($user_id, 'notify_sms_service_downtime', get_user_meta($user_id, 'notify_service_downtime'));
		$notify_sms_service_acknowledgement = get_user_meta($user_id, 'notify_sms_service_acknowledgement', get_user_meta($user_id, 'notify_service_acknowledgement'));


        // Service
        if ($nt == "service") {
            switch ($meta['type']) {
				case "ACKNOWLEDGMENT":
					if($notify_sms_service_acknowledgement !=1)
						return 1;
					break;
                case "PROBLEM":
                    if (($notify_sms_service_warning != 1) && ($meta['servicestateid'] == 1))
                        return 1;
                    else if (($notify_sms_service_critical != 1) && ($meta['servicestateid'] == 2))
                        return 1;
                    else if (($notify_sms_service_unknown != 1) && ($meta['servicestateid'] == 3))
                        return 1;
                    break;
                case "RECOVERY":
                    if ($notify_sms_service_recovery != 1)
                        return 1;
                    break;
                case "FLAPPINGSTART":
                case "FLAPPINGSTOP":
                    if ($notify_sms_service_flapping != 1)
                        return 1;
                    break;
                case "DOWNTIMESTART":
                case "DOWNTIMECANCELLED":
                case "DOWNTIMEEND":
                    if ($notify_sms_service_downtime != 1)
                        return 1;
                    break;
            }    
        } else {
            // Host
            switch ($meta['type']) {
				case "ACKNOWLEDGMENT":
					if($notify_sms_host_acknowledgement !=1)
						return 1;
					break;
                case "PROBLEM":
                    if (($notify_sms_host_down != 1) && ($meta['hoststateid'] == 1))
                        return 1;
                    else if (($notify_sms_host_unreachable != 1) && ($meta['hoststateid'] == 2))
                        return 1;
                break;
                case "RECOVERY":
                    if ($notify_sms_host_recovery != 1)
                        return 1;
                break;
                case "FLAPPINGSTART":
                    case "FLAPPINGSTOP":
                    if ($notify_sms_host_flapping != 1)
                        return 1;
                break;
                case "DOWNTIMESTART":
                    case "DOWNTIMECANCELLED":
                    case "DOWNTIMEEND":
                    if ($notify_sms_host_downtime != 1)
                        return 1;
                break;
            }    
        }
    }
    
    // get the SMS message
    if ($meta["notification-type"] == "service") {
        $message = mtr_component_get_service_message($user_id);
    } else {
        $message = mtr_component_get_host_message($user_id);
    }

    echo " MTR: RAW MESSAGE='" . $message . "'\n";

    // Process notification text (replace variables)
    $message = process_notification_text($message, $meta);

    // Trim the message
    $message = substr($message, 0, 159);

    echo " MTR: SMS MESSAGE='" . $message . "'\n";

    $args = array(
        "number" => $mtr_mobile_number,
        "message" => $message,
    );

    // Send the SMS message
    $outargs = array();
    $smsresult = mtr_component_send_sms($args, $outargs);

    echo "SMS RESULT=$smsresult\n";

    return 0;
}


////////////////////////////////////////////////////////////////////////
// MTR NOTIFICATION FUNCTIONS
////////////////////////////////////////////////////////////////////////


function mtr_component_send_sms($args, &$outargs)
{

    $number = grab_array_var($args, "number");
    $message = grab_array_var($args, "message");

    // Bail if empty number or message
    if ($number == "" || $message == "") {
        return 1;
    }

    // Load settings
    $settings_raw = get_option("mtr_component_options", array());
    $settings = array();
    if (!empty($settings_raw)) {
        $settings = unserialize($settings_raw);
    }

    $address = grab_array_var($settings, "address");
    $http_port = grab_array_var($settings, "http_port");
    $username = grab_array_var($settings, "username");
    $password = grab_array_var($settings, "password");

    // Bail out if we don't have the required info
    if ($address == "" || $http_port == "" || $username == "" || $password == "") {
        return 1;
    }

    $auth_url = 'https://'.$address.':'.$http_port.'/api/login';

    $curl = curl_init();
    curl_setopt($curl, CURLOPT_URL, $auth_url);
    curl_setopt($curl, CURLOPT_POST, true);
    curl_setopt($curl, CURLOPT_POSTFIELDS, '{"username":"'.$username.'","password":"'.$password.'"}');
    curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($curl, CURLOPT_HTTPHEADER, array('Expect:'));
    curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($curl, CURLOPT_HTTPHEADER, array('Content-Type: application/x-www-form-urlencoded'));

    $authorization = curl_exec($curl);  
    curl_close($curl);
    
    // Get auth token
    $authorization = json_decode($authorization, true);
    $token = $authorization['result']['token'];


    // Send the SMS request
    $send_url = 'https://'.$address.':'.$http_port.'/api/sms/outbox';
    $curl = curl_init();
    curl_setopt($curl, CURLOPT_URL, $send_url);
    curl_setopt($curl, CURLOPT_POST, true);
    curl_setopt($curl, CURLOPT_POSTFIELDS, '{"recipients":["'.$number.'"],"message":"'.$message.'"}');
    curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($curl, CURLOPT_COOKIE, "Cookie: token=$token");
    curl_setopt($curl, CURLOPT_HTTPHEADER, array('Content-Type: application/json'));
    curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
    $sent_response = curl_exec($curl);
    curl_close($curl);


    // Log out
    $logout_url = 'https://'.$address.':'.$http_port.'/api/logout?token='.$token;
    $curl = curl_init();
    curl_setopt($curl, CURLOPT_URL, $logout_url);
    curl_setopt($curl, CURLOPT_GET, true);
    curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
    $logout_result = curl_exec($curl);
    curl_close($curl);

    $sent_response = json_decode($sent_response, true);
    $response_code = $sent_response['code'];


    if ($response_code != "200")  {
        return 1;
    }

    return 0;
}
