<?php
//
// MongoDB Database Config Wizard
// Copyright (c) 2013-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

mongodb_database_configwizard_init();

function mongodb_database_configwizard_init()
{
    $name = "mongodb_database";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.1",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor a MongoDB Database"),
        CONFIGWIZARD_DISPLAYTITLE => _("MongoDB Database"),
        CONFIGWIZARD_FUNCTION => "mongodb_database_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "mongodb_db.png",
        CONFIGWIZARD_FILTER_GROUPS => array('database'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @return int
 *          0 = good to go
 *          1 = prereqs non-existent
 *          2 = may need to upgrade pymongo
 */
function mongodb_database_configwizard_check_prereqs()
{
    // Plugin doesn't exist
    if (!file_exists("/usr/local/nagios/libexec/check_mongodb.py") || !file_exists("/usr/local/nagios/libexec/check_mongodb_3.py")) {
        return 1; // prereqs non-existent
    }

    $preferred_version = '3.4.0';

    $found_yum = false;
    $yum_output = array();

    $found_pip = false;
    $pip_output = array();

    // See if pymongo is installed and get the version number(s)
    $yum_command = "yum list installed | grep pymongo | awk '{ print $2 }'";
    exec($yum_command, $yum_output, $yum_return);

    // See if pymongo is installed via pip and get the version number
    $pip_command = "pip show pymongo --no-cache-dir";
    exec($pip_command, $pip_output, $pip_return);

    // If neither yum nor pip returned anything, there is no need to continue
    if ($yum_return !== 0 || $pip_return !== 0) {
        return 1; // prereqs non-existent
    }

    // Check pip output
    if ($pip_return === 0) {
        foreach ($pip_output as $line) {

            // We're only interested in lines that start with "Version:"
            if (strpos($line, "Version:") === 0) {
                $pip_version = trim(str_replace("Version:", "", $line));

                if (version_compare($preferred_version, $pip_version) < 1) {
                    return 0; // good to go!
                }
            }
        }
    }

    // Check yum output
    if ($yum_return === 0) {
        foreach ($yum_output as $line) {
            $yum_version = trim($line);

            if (version_compare($preferred_version, $yum_version) < 1) {
                return 0; // good to go
            }
        }
    }

    return 2; // need to upgrade pymongo
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function mongodb_database_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "mongodb_database";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    // See what python version is installed
    $v_command = "python --version | grep -o '\s[0-9]\.[0-9]\+'";
    exec($v_command, $v_output, $v_return);

    // decide which python version
    $c_script = "check_mongodb_database_3";
    $service = "xiwizard_mongodbdatabase_service_3";
    if (floatval($v_output[0]) < 3.0) {
        $service = "xiwizard_mongodbdatabase_service";
        $c_script = "check_mongodb_database";
    }

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $check_prereqs = mongodb_database_configwizard_check_prereqs();

            if ($check_prereqs == 1) {
                // Redirect output to buffer -- we can load all 'printed' text into a string by calling ob_get_clean() right before we 'break;' for this case.
                ob_start();
?>
            <div class="alert alert-danger w-50">
                <div class="align-items-center d-flex pb-3">
                    <i class="material-symbols-outlined md-middle md-padding">warning</i>
                    <span class="fw-bold pe-2"><?= _('Error') ?>:</span> <?= _('The python pymongo module is missing from the Nagios XI server.') ?>
                </div>
                <div class="card card-p">
                    <p class="text-start"><?= _('To use this wizard you must install pymongo on your server. If you are using CentOS or RHEL you can run:') ?></p>
                    <pre><code>pip install pymongo</code></pre>
                </div>
            </div>
<?php
                $output = ob_get_clean();

            } else {
                $address = grab_array_var($inargs, "ip_address", "");
                $port = grab_array_var($inargs, "port", "27017");
                $username = grab_array_var($inargs, "username", "");
                $password = grab_array_var($inargs, "password", "");
                $database = grab_array_var($inargs, "database", "test");

                if (!empty($services_serial)) {
                    $services = json_decode(base64_decode($services_serial), true);
                }

                $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial");

                if (!empty($serviceargs_serial)) {
                    $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
                }

                # Get the existing host/node configurations.
                # TODO: Include passwords/secrets?
                $nodes = get_configwizard_hosts($wizard_name);

                ########################################################################################
                # Load the html
                # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
                #   to load the PHP from the Step1 file into the $output string.
                ########################################################################################
                ob_start();
                include __DIR__.'/steps/step1.php';
                $output = ob_get_clean();
            }

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $hostname = grab_array_var($inargs, "hostname");
            $port = grab_array_var($inargs, "port", "27017");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $database = grab_array_var($inargs, "database", "test");

            // Swap out eligible user macros if detected
            $address = nagiosccm_replace_user_macros($address);
            $port = nagiosccm_replace_user_macros($port);
            $username = nagiosccm_replace_user_macros($username);
            $password = nagiosccm_replace_user_macros($password);

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($address) == false)
                $errmsg[$errors++] = _("No address specified.");
            if (have_value($username) == false)
                $errmsg[$errors++] = _("No username specified.");
            if (have_value($password) == false)
                $errmsg[$errors++] = _("No password specified.");
            if (have_value($database) == false)
                $errmsg[$errors++] = _("No database specified.");

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $hostname = grab_array_var($inargs, "hostname");
            $port = grab_array_var($inargs, "port", "27017");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $database = grab_array_var($inargs, "database", "test");

            $ha = @gethostbyaddr($address);

            if ($ha == "") {
                $ha = $address;
            }

            $hostname = grab_array_var($inargs, "hostname", $ha);

            $services = grab_array_var($inargs, "services", array(
                "collections" => "on",
                "database_indexes" => "on",
                "database_size" => "on"
            ));

            $serviceargs = grab_array_var($inargs, "serviceargs", array(
                "collections_warning" => "50",
                "collections_critical" => "200",
                "database_indexes_warning" => "33554432", // 32 MB
                "database_indexes_critical" => "67108864", // 64 MB
                "database_size_warning" => "33554432", // 32 MB
                "database_size_critical" => "67108864" // 64 MB
            ));

            // Replace defaults with the given values
            $services_serial = grab_array_var($inargs, "services_serial");

            if (!empty($services_serial)) {
                $services = json_decode(base64_decode($services_serial), true);
            }

            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial");

            if (!empty($serviceargs_serial)) {
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            }

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $hostname = grab_array_var($inargs, "hostname");
            $hostname = nagiosccm_replace_user_macros($hostname);
            $port = grab_array_var($inargs, "port", "27017");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $database = grab_array_var($inargs, "database", "test");

            $services = grab_array_var($inargs, "services", array());
            $serviceargs = grab_array_var($inargs, "serviceargs", array());

            // check for errors
            $errors = 0;
            $errmsg = array();

            if (is_valid_host_name($hostname) == false) {
                $errmsg[$errors++] = "Invalid host name.";
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $hostname = grab_array_var($inargs, "hostname");
            $port = grab_array_var($inargs, "port", "27017");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $database = grab_array_var($inargs, "database", "test");

            $services = grab_array_var($inargs, "services", array());
            if (empty($services)) {
                $services_serial = grab_array_var($inargs, "services_serial");
            } else {
                $services_serial = base64_encode(json_encode($services));
            }
            $serviceargs = grab_array_var($inargs, "serviceargs", array());
            if (empty($serviceargs)) {
                $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial");
            } else {
                $serviceargs_serial = base64_encode(json_encode($serviceargs));
            }

            $output = '

                <input type="hidden" name="ip_address" value="' . encode_form_val($address) . '" />
                <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '" />
                <input type="hidden" name="port" value="' . encode_form_val($port) . '" />
                <input type="hidden" name="username" value="' . encode_form_val($username) . '" />
                <input type="hidden" name="password" value="' . encode_form_val($password) . '" />
                <input type="hidden" name="database" value="' . encode_form_val($database) . '" />
                <input type="hidden" name="services_serial" value="' . $services_serial . '" />
                <input type="hidden" name="serviceargs_serial" value="' . $serviceargs_serial . '" />

            ';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $hostname = grab_array_var($inargs, "hostname");
            $port = grab_array_var($inargs, "port", "27017");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $database = grab_array_var($inargs, "database", "test");

            $services = grab_array_var($inargs, "services", array());
            if (empty($services)) {
                $services_serial = grab_array_var($inargs, "services_serial");
            } else {
                $services_serial = base64_encode(json_encode($services));
            }
            $serviceargs = grab_array_var($inargs, "serviceargs", array());
            if (empty($serviceargs)) {
                $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial");
            } else {
                $serviceargs_serial = base64_encode(json_encode($serviceargs));
            }

            $output = '

                <input type="hidden" name="ip_address" value="' . encode_form_val($address) . '" />
                <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '" />
                <input type="hidden" name="port" value="' . encode_form_val($port) . '" />
                <input type="hidden" name="username" value="' . encode_form_val($username) . '" />
                <input type="hidden" name="password" value="' . encode_form_val($password) . '" />
                <input type="hidden" name="database" value="' . encode_form_val($database) . '" />
                <input type="hidden" name="services_serial" value="' . $services_serial . '" />
                <input type="hidden" name="serviceargs_serial" value="' . $serviceargs_serial . '" />

            ';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            // Get the vars
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $port = escapeshellarg(grab_array_var($inargs, "port"));
            $username = escapeshellarg(grab_array_var($inargs, "username"));
            $password = escapeshellarg(grab_array_var($inargs, "password"));
            $database = escapeshellarg(grab_array_var($inargs, "database"));

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["ip_address"] = $address;
            $meta_arr["port"] = $port;
            $meta_arr["database"] = $database;
            $meta_arr["username"] = $username;
            $meta_arr["services"] = $services;
            $meta_arr["serviceargs"] = $serviceargs;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = array();

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_mongodbdatabase_host",
                    "host_name" => $hostname,
                    "address" => $address,
                    "icon_image" => "mongodb.png",
                    "statusmap_image" => "mongodb.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            foreach ($services as $svcvar => $svcval) {

                switch ($svcvar) {

                    case "collections":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MongoDB Database Collections",
                            "use" => $service,
                            "check_command" => $c_script . "!collections!" . $port . "!" . escapeshellarg($serviceargs["collections_warning"]) . "!" . escapeshellarg($serviceargs["collections_critical"]) . "!" . $username . "!" . $password . "!'" . $database . "'",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "database_indexes":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MongoDB Database Indexes Size",
                            "use" => $service,
                            "check_command" => $c_script . "!database_indexes!" . $port . "!" . escapeshellarg($serviceargs["database_indexes_warning"]) . "!" . escapeshellarg($serviceargs["database_indexes_critical"]) . "!" . $username . "!" . $password . "!'" . $database . "'",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "database_size":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MongoDB Database Size",
                            "use" => $service,
                            "check_command" => $c_script . "!database_size!" . $port . "!" . escapeshellarg($serviceargs["database_size_warning"]) . "!" . escapeshellarg($serviceargs["database_size_critical"]) . "!" . $username . "!" . $password . "!'" . $database . "'",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    default:
                        break;
                }
            }

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}
