<?php
//
// Printer Config Wizard
// Copyright (c) 2008-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

printer_configwizard_init();

function printer_configwizard_init()
{
    $name = "printer";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.2",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor an HP JetDirect&reg; compatible network printer."),
        CONFIGWIZARD_DISPLAYTITLE => _("Printer"),
        CONFIGWIZARD_FUNCTION => "printer_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "printer2.png",
        CONFIGWIZARD_FILTER_GROUPS => array('network'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function printer_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "printer";

    // initialize return code and output
    $result = 0;
    $output = "";

    // initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            // Remove any existing session variables if this is the first time on the page
            if (isset($_POST['backButton']) == false) {
                unset($_SESSION['printer_wizard_address']);
                unset($_SESSION['printer_wizard_hostname']);
                unset($_SESSION['printer_wizard_snmpcommunity']);
                $_SESSION['printer_wizard_services'] = array('0' => 'ping',
                                                            '1' => 'printerstatus');
            }

            $address = grab_array_var($inargs, "ip_address", "");

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $address = nagiosccm_replace_user_macros($address);
            $services = grab_array_var($inargs, "services");

            if (array_key_exists('printer_wizard_address', $_SESSION) && $address == "") {
                $address = $_SESSION['printer_wizard_address'];
            }

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($address) == false) {
                $errmsg[$errors++] = _("No address specified.");
            } else if (!valid_ip($address)) {
                $errmsg[$errors++] = _("Invalid IP address.");
            } 

            if ($address) {
                $_SESSION['printer_wizard_address'] = $address;
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $services = grab_array_var($inargs, "services");
            $snmpcommunity = grab_array_var($inargs, "snmpcommunity");

            if (isset($_POST['submit']) == true && $services === '') {
                $services = $_SESSION['printer_wizard_services'];
            }

            if (array_key_exists('printer_wizard_address', $_SESSION) && $address == "") {
                $address = $_SESSION['printer_wizard_address'];
            }

            if (array_key_exists('printer_wizard_services', $_SESSION) && !$services) {
                $services = $_SESSION['printer_wizard_services'];
            }

            if (array_key_exists('printer_wizard_snmpcommunity', $_SESSION) && !$snmpcommunity) {
                $snmpcommunity = $_SESSION['printer_wizard_snmpcommunity'];
            }
            
            if ($snmpcommunity == "" && isset($_POST['backButton']) == false) {
                $snmpcommunity = 'public';
            }

            $hostname = @gethostbyaddr($address);

            if ($hostname) {
                $_SESSION['printer_wizard_hostname'] = $hostname;
            }

            if ($snmpcommunity) {
                $_SESSION['printer_wizard_snmpcommunity'] = $snmpcommunity;
            }


            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $hostname = grab_array_var($inargs, "hostname", "");
            $services = grab_array_var($inargs, "services");
            $snmpcommunity = grab_array_var($inargs, "snmpcommunity", "");

            if (isset($_POST['submitButton2']) && empty($services)) {
                unset($_SESSION['printer_wizard_services']);
            }

            if (isset($_POST['submitButton2']) && $snmpcommunity == '') {
                unset($_SESSION['printer_wizard_snmpcommunity']);
            }

            if (array_key_exists('printer_wizard_address', $_SESSION) && $address == "") {
                $address = $_SESSION['printer_wizard_address'];
            }

            if (array_key_exists('printer_wizard_hostname', $_SESSION) && $hostname == "") {
                $hostname = $_SESSION['printer_wizard_hostname'];
            }

            if (array_key_exists('printer_wizard_services', $_SESSION) && !$services) {
                $services = $_SESSION['printer_wizard_services'];
            }

            if (array_key_exists('printer_wizard_snmpcommunity', $_SESSION) && $snmpcommunity == "") {
                $snmpcommunity = $_SESSION['printer_wizard_snmpcommunity'];
            }

            $hostname = nagiosccm_replace_user_macros($hostname);

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (is_valid_host_name($hostname) == false) {
                $errmsg[$errors++] = _("Invalid host name.");
            }

            if (empty($services) == true) {
                $errmsg[$errors++] = _("No services specified.");       
            } else if (in_array('printerstatus', $services) && $snmpcommunity == '') {
                $errmsg[$errors++] = _("Specify a SNMP Community String.");
            }

            if ($hostname) {
                $_SESSION['printer_wizard_hostname'] = $hostname;
            }

            if ($services) {
                $_SESSION['printer_wizard_services'] = $services;
            }

            if ($snmpcommunity) {
                $_SESSION['printer_wizard_snmpcommunity'] = $snmpcommunity;
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $hostname = grab_array_var($inargs, "hostname", "");
            $services = grab_array_var($inargs, "services");
            $snmpcommunity = grab_array_var($inargs, "snmpcommunity", "");

            if (array_key_exists('printer_wizard_services', $_SESSION) && !$services) {
                $services = $_SESSION['printer_wizard_services'];
            }

            if (array_key_exists('printer_wizard_snmpcommunity', $_SESSION) && $snmpcommunity == "") {
                $snmpcommunity = $_SESSION['printer_wizard_snmpcommunity'];
            }

            if (array_key_exists('printer_wizard_hostname', $_SESSION) && $hostname == "") {
                $hostname = $_SESSION['printer_wizard_hostname'];
            }

            $services_serial = grab_array_var($inargs, "services_serial", base64_encode(json_encode($services)));

            if ($services) {
                $_SESSION['printer_wizard_services'] = $services;
            }

            if ($snmpcommunity) {
                $_SESSION['printer_wizard_snmpcommunity'] = $snmpcommunity;
            }
            
            $output = '

        <input type="hidden" name="ip_address" value="' . encode_form_val($address) . '">
        <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '">
        <input type="hidden" name="hostname" value="' . encode_form_val($snmpcommunity) . '">
        <input type="hidden" name="services_serial" value="' . $services_serial . '">
';
            break;


        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            $output = '

            ';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $hostname = grab_array_var($inargs, "hostname", "");
            $address = grab_array_var($inargs, "ip_address", "");
            $hostaddress = $address;

            $services_serial = grab_array_var($inargs, "services_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $snmpcommunity = grab_array_var($inargs, "snmpcommunity", "");

            if (array_key_exists('printer_wizard_services', $_SESSION)) {
                $services = $_SESSION['printer_wizard_services'];
            }
            if (array_key_exists('printer_wizard_address', $_SESSION)) {
                $address = $_SESSION['printer_wizard_address'];
            }
            if (array_key_exists('printer_wizard_hostname', $_SESSION)) {
                $hostname = $_SESSION['printer_wizard_hostname'];
            }
            if (array_key_exists('printer_wizard_snmpcommunity', $_SESSION)) {
                $snmpcommunity = $_SESSION['printer_wizard_snmpcommunity'];
            }

            if ($snmpcommunity) {
                $_SESSION['printer_wizard_snmpcommunity'] = $snmpcommunity;
            }

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["ip_address"] = $address;
            $meta_arr["services"] = $services;
            $meta_arr["snmpcommunity"] = $snmpcommunity;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = [];

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_printer_host",
                    "host_name" => $hostname,
                    "address" => $hostaddress,
                    "icon_image" => "printer2.png",
                    "statusmap_image" => "printer2.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            // see which services we should monitor
            foreach ($services as $service) {

                switch ($service) {

                    case ('ping'):
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Ping",
                            "use" => "xiwizard_printer_ping_service",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case ('printerstatus'):
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Printer Status",
                            "use" => "xiwizard_printer_hpjd_service",
                            "check_command" => "check_xi_service_hpjd!" . $snmpcommunity,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    default:
                        break;
                }
            }

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}
