<?php
//
// Metrics Component
// Copyright (c) 2011-2022 Nagios Enterprises, LLC. All rights reserved.
//

require_once(dirname(__FILE__) . '/../../common.inc.php');
include_once(dirname(__FILE__) . '/dashlet.inc.php');

// Initialization stuff
pre_init();
init_session();

// Grab GET or POST variables and pre-reqs
grab_request_vars();
check_prereqs();
check_authentication(false);

route_request();

function route_request()
{
    global $request;
    $mode = grab_request_var("mode");

    switch ($mode) {
        case "highchart":
            generate_graph_data();
            break;
        case "pnpchart":
            generate_graph_data(1);
            break;
        default:
            display_metrics();
            break;
    }
}

function display_metrics()
{

    $showperfgraphs = 0;
    $sortorder = "desc";
    $metric = "disk";
    $maxitems = 20;
    $type = "";
    $tab = "";
    $mc_args = array();

    // use saved values
    $prefs_raw = get_user_meta(0, "metrics_prefs");
    if (have_value($prefs_raw)) {
        $prefs = unserialize($prefs_raw);

        $showperfgraphs = grab_array_var($prefs, "showperfgraphs");
        $sortorder = grab_array_var($prefs, "sortorder");
        $metric = grab_array_var($prefs, "metric");
        $maxitems = grab_array_var($prefs, "maxitems");
        $type = grab_array_var($prefs, "type");
        $tab = grab_array_var($prefs, "tab");
    }

    // grab request vars
    $type = grab_request_var("type", $type);
    $host = grab_request_var("host", "");
    $hostgroup = grab_request_var("hostgroup", "");
    $servicegroup = grab_request_var("servicegroup", "");
    $maxitems = grab_request_var("maxitems", $maxitems);
    $metric = grab_request_var("metric", $metric);
    $sortorder = grab_request_var("sortorder", $sortorder);
    $showperfgraphs = grab_request_var("showperfgraphs", $showperfgraphs);
    $tab = grab_request_var("tab", $tab);
    $details = grab_request_var("details", 0);
    $graphtype = grab_request_var("graphtype", 0);
    $advanced = grab_request_var("advanced", 0);
    $pdfrender = grab_request_var("pdfrender", 0);

    // Settings for auto-running
    $manual_run = grab_request_var("manual_run", 0);

    // Hacky fix for maxitems being set to nothing...
    if ($maxitems == "") {
        $maxitems = 0;
    }

    // save values
    $prefs = array(
        "showperfgraphs" => $showperfgraphs,
        "sortorder" => $sortorder,
        "metric" => $metric,
        "maxitems" => $maxitems,
        "type" => $type,
        "tab" => $tab,
        "details" => $details
    );
    set_user_meta(0, "metrics_prefs", serialize($prefs), false);


    // makes sure user has appropriate license level
    licensed_feature_check();

    // start the HTML page
    do_page_start(array("page_title" => "Metrics"), true);
?>

<script type="text/javascript">
$(document).ready(function () {

    if (!is_neptune()) {
        $('#hostgroupList').searchable({maxMultiMatch: 9999});
        $('#servicegroupList').searchable({maxMultiMatch: 9999});
    }

    $('#servicegroupList').change(function () {
        $('#hostgroupList').val('');
    });

    $('#hostgroupList').change(function () {
        $('#servicegroupList').val('');
    });

    // Add the ability to show the advanced options section
    $('#advanced-options-btn').click(function () {
        if ($('#advanced-options').is(":visible")) {
            $('#advanced-options').hide();
            $('#advanced').val(0);
            $('#advanced-options-btn').html('<?php echo _("Advanced"); ?> <i class="fa fa-chevron-down"></i>');
        } else {
            $('#advanced-options').show();
            $('#advanced').val(1);
            $('#advanced-options-btn').html('<?php echo _("Advanced"); ?> <i class="fa fa-chevron-up"></i>');
        }
    });
});
</script>


<?php if (!$pdfrender) { ?>
<form method="get" id="metricsForm">
    <div class="well report-options form-inline neptune-metric-well">
        <div class="input-group" style="margin-right: 5px;">
            <label class="h-9 input-group-addon border-right"><?php echo _("Limit To"); ?></label>
            <select name="hostgroup" id="hostgroupList" style="width: 150px;" class="neptune-select h-9 rounded-l-none border-l form-control">
                <option value=""><?php echo _("Hostgroup"); ?>:</option>
                <?php
                $args = array('orderby' => 'hostgroup_name:a');
                $xml = get_xml_hostgroup_objects($args);
                if ($xml) {
                    foreach ($xml->hostgroup as $hg) {
                        $name = strval($hg->hostgroup_name);
                        echo "<option value='" . $name . "' " . is_selected($hostgroup, $name) . ">$name</option>\n";
                    }
                }
                ?>
            </select>
            <select name="servicegroup" id="servicegroupList" style="width: 175px; margin-right: 5px;" class="neptune-select h-9 rounded-l-none rounded-r form-control">
                <option value=""><?php echo _("Servicegroup"); ?>:</option>
                <?php
                $args = array('orderby' => 'servicegroup_name:a');
                $xml = get_xml_servicegroup_objects($args);
                if ($xml) {
                    foreach ($xml->servicegroup as $sg) {
                        $name = strval($sg->servicegroup_name);
                        echo "<option value='" . $name . "' " . is_selected($servicegroup, $name) . ">$name</option>\n";
                    }
                }
                ?>
            </select>
        </div>

        <div class="input-group">
            <label class="input-group-addon"><?php echo _("Metric"); ?></label>
            <select name="metric" id="metricList" class="neptune-select form-control" style="width: 150px; margin-right: 5px;">
                <?php
                $metrics = get_metric_names();
                foreach ($metrics as $mn => $md) {
                    $name = $mn;
                    echo "<option value='" . $name . "' " . is_selected($metric, $name) . ">$md</option>\n";
                }
                ?>
            </select>
        </div>

        <div class="input-group">
            <label class="input-group-addon"><?php echo _("Show"); ?></label>
            <select name="sortorder" id="sortorderList" class="neptune-select rounded-r form-control" style="width: 150px; margin-right: 5px;">
                <?php
                $sortorders = array(
                    "desc" => "Top",
                    "asc" => "Bottom",
                );
                foreach ($sortorders as $sn => $sd) {
                    $name = $sn;
                    echo "<option value='" . $name . "' " . is_selected($sortorder, $name) . ">$sd</option>\n";
                }
                ?>
            </select>
        </div>

        <div class="input-group ">
                <?php if (is_neptune()) {?>
                    <div class="flex gap-2">
                        <input type="text" name="maxitems" class="rounded-md  form-control" value="<?php echo htmlentities($maxitems); ?>" style="width: 53px;">
                        <input type="hidden" value="<?php echo intval($advanced); ?>" id="advanced" name="advanced">
                        <button type="submit" id="goButton" class='btn btn-sm btn-primary' name='goButton'><?php echo _("Run"); ?></button>
                        <a id="advanced-options-btn" class="tt-bind flex items-center text-medium" data-placement="bottom" title="<?php echo _('Toggle advanced options'); ?>"><span class="flex items-center gap-2"><?php echo _('Advanced'); ?>  <?php if (!$advanced) { echo '<i class="fa fa-chevron-down"></i>'; } else { echo '<i class="fa fa-chevron-up"></i>'; } ?></span></a>
                    </div>
                <?php } else {?>
                    <input type="text" name="maxitems" class="rounded-md  form-control" value="<?php echo htmlentities($maxitems); ?>" style="width: 53px;">
                    <a id="advanced-options-btn" class="tt-bind" data-placement="bottom" title="<?php echo _('Toggle advanced options'); ?>"><?php echo _('Advanced'); ?>  <?php if (!$advanced) { echo '<i class="fa fa-chevron-down"></i>'; } else { echo '<i class="fa fa-chevron-up"></i>'; } ?></a>
                    <input type="hidden" value="<?php echo intval($advanced); ?>" id="advanced" name="advanced">
                    <button type="submit" id="goButton" class='btn btn-sm btn-primary' name='goButton'><?php echo _("Run"); ?></button>
                <?php }?>

        </div>

<?php if (is_neptune()) {?>
        <div id="advanced-options" style="<?php if (!$advanced) { echo 'display: none;'; } ?>">
                <div class="flex gap-2">
                        <div class="flex">
                            <label class="neptune-checkbox-container">
                                <span class="neptune-checkbox-check"><input type="checkbox" class="m-0" name="details" value="1" <?php echo is_checked($details, 1); ?>></span>
                                <span class="ml-2"><?php echo _("Remove Details"); ?></span>
                            </label> 
                        </div>
                        <div class="flex">
                            <label class="neptune-checkbox-container">
                                <span class="neptune-checkbox-check"><input type="checkbox"  class="m-0"  name="graphtype" value="1" <?php echo is_checked($graphtype, 1); ?>></span>
                                <span class="ml-2"> <?php echo _("Use Old Graphs"); ?></span>
                            </label> 
                        </div>
                </div>
            <div style="clear: both;"></div>
            </div>
            <input type="hidden" name="manual_run" value="1">
            <input type="hidden" name="disable_metrics_auto_run" value="1">
        </div>
<?php } else {?>
                <div id="advanced-options" style="<?php if (!$advanced) { echo 'display: none;'; } ?>">
            <div class="floatbox">
                <div class="checkbox">
                    <label>
                        <input type="checkbox" name="details" value="1" <?php echo is_checked($details, 1); ?>> <?php echo _("Remove Details"); ?>
                    </label>
                </div>
            </div>
            <div class="floatbox">
                <div class="checkbox">
                    <label>
                        <input type="checkbox" name="graphtype" value="1" <?php echo is_checked($graphtype, 1); ?>> <?php echo _("Use Old Graphs"); ?>
                    </label>
                </div>
            </div>

            <div style="clear: both;"></div>
        </div>
        <input type="hidden" name="manual_run" value="1">
        <input type="hidden" name="disable_metrics_auto_run" value="1">
    </div>
        <?php }?>
</form>
<?php } ?>

    <h1 style="padding-top: 0; margin-top:20px;"><?php echo _("Metrics"); ?></h1>
    <div class="clear"></div>

    <!-- Disable metrics auto-run settings -->
    <input type="hidden" name="tab_hash" id="tab_hash" value="metricsFormtabname" />
    <input type="hidden" name="tab" class="form-control" value="<?php echo encode_form_val($tab); ?>" id="metricsFormtabname">
    <input type="hidden" name="graphtype" value="<?php echo encode_form_val($graphtype); ?>">
    <input type="hidden" name="details" value="<?php echo encode_form_val($details); ?>">

    <?php
    // Disable running if they disabled running metrics automatically
    $disable_metrics_auto_run = get_option("disable_metrics_auto_run", 0);
    if ($disable_metrics_auto_run == 0 || ($disable_metrics_auto_run == 1 && $manual_run == 1)) {
    ?>

    <div id="tabs" style="display: none;">
        <ul>
            <li><a href="#tab-summary"><?php echo _("Summary"); ?></a></li>
            <li><a href="#tab-graphs" class="graphs"><?php echo _("Graphs"); ?></a></li>
            <li><a href="#tab-gauges"><?php echo _("Gauges"); ?></a></li>
        </ul>

        <div id="tab-summary">
            <?php
            $dargs = array(
                DASHLET_ARGS => array(
                    "type" => $type,
                    "host" => $host,
                    "hostgroup" => $hostgroup,
                    "servicegroup" => $servicegroup,
                    "maxitems" => $maxitems,
                    "metric" => $metric,
                    "sortorder" => $sortorder,
                    "showperfgraphs" => $showperfgraphs,
                    "details" => $details
                ),
            );
            display_dashlet("metrics", "", $dargs, DASHLET_MODE_OUTBOARD);
            ?>
        </div>

        <?php

        // get service metrics
        $args = array(
            "type" => $type,
            "host" => $host,
            "hostgroup" => $hostgroup,
            "servicegroup" => $servicegroup,
            "maxitems" => $maxitems,
            "metric" => $metric,
            "sortorder" => $sortorder,
            "details" => $details
        );
        $metricdata = get_service_metrics($args);

        // Get dropdowns
        $opts = get_report_timeperiod_options();
        unset($opts['last24hours']);
        unset($opts['custom']);
        ?>

        <div id="tab-graphs">

            <div class="form-inline" style="margin: 10px 0 20px 0;">
                <div class="input-group" style="margin-right: 10px;">
                    <label class="input-group-addon"><?php echo _("Time Period"); ?></label>
                    <select id="timeperiod" name="timepriod" class=" neptune-select form-control">
                        <option value="-4h"><?php echo _("Last 4 Hours"); ?></option>
                        <option value="-24h"><?php echo _("Last 24 Hours"); ?></option>
                        <option value="-48h"><?php echo _("Last 48 Hours"); ?></option>
                        <option value="-1w"><?php echo _("Last 7 Days"); ?></option>
                        <option value="-1m"><?php echo _("Last 30 Days"); ?></option>
                        <option value="-1y"><?php echo _("Last 365 Days"); ?></option>
                        <?php
                        foreach ($opts as $shortname => $longname) {
                            echo "<option value='" . $shortname . "'>" . $longname . "</option>";
                        }
                        ?>
                    </select>
                </div>
                <div id="customdates" class="cal hide">
                    <div style="margin-bottom: 5px;">
                        <span style="width: 32px; display: inline-block;"><?php echo _("From"); ?></span>
                        <input class="textfield form-control" type="text" id='startdateBox' name="startdate" value="" size="16">
                        <div id="startdatepickercontainer"></div>
                        <div class="reportstartdatepicker"><i class="fa fa-calendar fa-cal-btn"></i></div>
                    </div>
                    <div>
                        <span style="width: 32px; display: inline-block;"><?php echo _("To"); ?></span>
                        <input class="textfield form-control" type="text" id='enddateBox' name="enddate" value="" size="16">
                        <div id="enddatepickercontainer"></div>
                        <div class="reportenddatepicker"><i class="fa fa-calendar fa-cal-btn"></i></div>
                    </div>
                </div>
            </div>

            <script type="text/javascript">
            $(document).ready(function () {

                if ($("#highcharts").is(":visible") || $("#chart").is(":visible")) {
                    load_perfgraphs_panel();
                }

                // Timeframe selection
                $("#timeperiod").change(function() {
                    load_perfgraphs_panel();
                });

                $(".graphs").click(function() {
                    if ($('#highcharts .pd-container').length < 1) {
                        load_perfgraphs_panel();
                    }
                });

            });
        
            function load_perfgraphs_panel() {
                var base_url = "<?php echo get_base_url(); ?>";
                var graphtype = <?php echo json_encode(encode_form_val($graphtype)); ?>;
                var metricdata = "<?php echo base64_encode(json_encode($metricdata)); ?>";
                var mc_args = "<?php echo base64_encode(json_encode($mc_args)); ?>";

                // Load default time settings

                var start = $("#timeperiod option:selected").val();

                if (graphtype == 1) {
                    $.ajax({
                        type: "POST",
                        url: base_url + "includes/components/metrics/?cmd=getxicoreajax&tab=tab-graphs&mode=pnpchart",
                        data: { metricdata: metricdata, mc_args: mc_args, start: start, json: 1 },
                        success: function(data) {
                            $("#chart").html(data);
                        }
                    });
                } else {
                    $.ajax({
                        type: "POST",
                        url: base_url + "includes/components/metrics/?tab=tab-graphs&mode=highchart",
                        data: { metricdata: metricdata, mc_args: mc_args, start: start, json: 1 },
                        success: function(data) {
                            $("#highcharts").html(data);
                        }
                    });
                }
            }
            </script>

            <?php
            if ($graphtype == 1) {
                // create pnp graph container
                echo '<div id="chart"></div>';
            } else {
                $json = 0;
                // create highcharts container div
                echo '<div id="highcharts"></div>';
            }
            ?>
        </div>

        <div id="tab-gauges">
            <?php
            foreach ($metricdata as $id => $arr) {

                $hostname = $arr["host_name"];
                $servicename = $arr["service_name"];
                $command = grab_array_var($arr, "command", "");

                ?>
                <div style="margin: 0 25px 25px 0; float: left;">
                    <?php

                    $dargs = array(
                        DASHLET_ARGS => array(
                            "host" => $hostname,
                            "service" => $servicename,
                            "metric" => $metric,
                            "percent" => $arr["sortval"],
                            "current" => $arr["current"],
                            "uom" => $arr["uom"],
                            "warn" => $arr["warn"],
                            "crit" => $arr["crit"],
                            "min" => $arr["min"],
                            "max" => $arr["max"],
                            "plugin_output" => $arr["output"],
                            "command" => $command
                        ),
                    );

                    display_dashlet("metricsguage", "", $dargs, DASHLET_MODE_OUTBOARD);

                    ?>
                </div>
            <?php

            }

            ?>

        </div>
    </div>

    <?php
    }

    // closes the HTML page
    do_page_end(true);
}


///////////////////////////////////////////////////////////////////
// FUNCTIONS
///////////////////////////////////////////////////////////////////

// Get a list of host and services
function get_host_services_list()
{
    // Get the actual service/hostnames for the reports
    $str = get_service_status_xml_output(array());
    $x = simplexml_load_string($str);

    $services = array();
    foreach ($x->servicestatus as $service) {
        $c = explode("!", strval($service->check_command));

        if ($c[0] == "check_xi_service_mrtgtraf") {

            // Create their address and port
            $c_clean = str_replace(".rrd", "", $c[1]);
            list($address, $port) = explode("_", $c_clean);

            $s = array("display_name" => $service->host_name . " - " . $service->display_name,
                       "host_name" => strval($service->host_name),
                       "service_name" => strval($service->display_name));

            $services[$c_clean] = $s;
        }
    }

    return $services;
}

// Select the actual display name using the list given (this way it's only gotta make one XML call)
function find_bwselect_display_name($list, $fullname)
{
    if (array_key_exists($fullname, $list)) {
        return $list[$fullname]['display_name'];
    }
}

function generate_graph_data($old = 0)
{
    $metricdata = grab_request_var("metricdata", "");
    $mc_args = grab_request_var("mc_args", "");
    $start = grab_request_var("start", "");
    $end = grab_request_var("end", "");
    $json = grab_request_var("json", "");

    if ($json > 0) {
        $metricdata = json_decode(base64_decode($metricdata), true);
        $mc_args = json_decode(base64_decode($mc_args), true);
    }

    foreach ($metricdata as $id => $arr) {
        $hostname = $arr["host_name"];
        $servicename = $arr["service_name"];

        echo "<div class='serviceperfgraphcontainer pd-container'>";

        $dargs = array(
            DASHLET_ADDTODASHBOARDTITLE => _("Add to Dashboard"),
            DASHLET_CLASS => "fullwidth",
            DASHLET_ARGS => array(
                "hostname" => $hostname,
                "host_id" => get_host_id($hostname),
                "servicename" => $servicename,
                "service_id" => get_service_id($hostname, $servicename),
                "start" => $start,
                "old" => $old,
                "mode" => PERFGRAPH_MODE_SERVICEDETAIL
            ),
            DASHLET_TITLE => $hostname . " " . $servicename . " " . _("Performance Graph"),
        );

        display_dashlet("xicore_perfdata_chart", "", $dargs, DASHLET_MODE_OUTBOARD);
    }
}
