<?php
//
// LDAP Server Config Wizard
// Copyright (c) 2010-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

ldapserver_configwizard_init();
activedirectory_configwizard_init();

function ldapserver_configwizard_init()
{
    $name = "ldapserver";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.1",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor an LDAP server."),
        CONFIGWIZARD_DISPLAYTITLE => _("LDAP Server"),
        CONFIGWIZARD_FUNCTION => "ldapserver_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "directory_services3.png",
        CONFIGWIZARD_FILTER_GROUPS => array('network', 'server'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

function activedirectory_configwizard_init()
{
    $name = "activedirectory";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.1",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor an Active Directory server via LDAP connection."),
        CONFIGWIZARD_DISPLAYTITLE => _("Active Directory"),
        CONFIGWIZARD_FUNCTION => "ldapserver_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "activedirectory.png",
        CONFIGWIZARD_FILTER_GROUPS => array('network'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function ldapserver_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = array_key_exists('wizard', $inargs) ? $inargs['wizard'] : '';

    switch ($wizard_name) {
        case 'ldapserver':
            $short_name = _("LDAP");
            $logo_filename = 'directory_services3.png';
            $address_subtext = _("The IP Address or FQDNS name associated with the LDAP server.");
            $ldap_base_subtext = _('The LDAP base to use');
            $bind_dn_subtext = _('LDAP bind DN (if required)');
            break;
        case 'activedirectory':
            $short_name = _("Active Directory");
            $logo_filename = 'activedirectory.png';
            $address_subtext = _("The IP Address or FQDNS name associated with a Domain Controller.");
            $ldap_base_subtext = _('The distinguishedName of the container for this user.');
            $bind_dn_subtext = _('The distinguishedName of the user.');
            break;
        default:
            return;
    }

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $address = grab_array_var($inargs, "ip_address", "");

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $address = nagiosccm_replace_user_macros($address);

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($address) == false) {
                $errmsg[$errors++] = _("No address specified.");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");

            $ha = @gethostbyaddr($address);

            if ($ha == "") {
                $ha = $address;
            }

            $hostname = grab_array_var($inargs, "hostname", $ha);
            $port = grab_array_var($inargs, "port");
            $password = grab_array_var($inargs, "password");

            $base = grab_array_var($inargs, "base", "");
            $bind_dn = grab_array_var($inargs, "bind_dn");
            $security = grab_array_var($inargs, "security");
            $version = grab_array_var($inargs, "version");
            $search = grab_array_var($inargs, "search");

            $services = grab_array_var($inargs, "services", array("server" => "on", "transfer" => ""));

            $operation = "";
            $selectedhostconfig = "";

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $hostname = nagiosccm_replace_user_macros($hostname);

            $port = grab_array_var($inargs, "port");
            $port = nagiosccm_replace_user_macros($port);
            $base = grab_array_var($inargs, "base");
            $bind_dn = grab_array_var($inargs, "bind_dn");
            $password = grab_array_var($inargs, "password");
            $password = nagiosccm_replace_user_macros($password);
            $security = grab_array_var($inargs, "security");
            $version = grab_array_var($inargs, "version");
            $search = grab_array_var($inargs, "search");
            $services = grab_array_var($inargs, "services", array());

            // check for errors
            $errors = 0;
            $errmsg = array();

            if (is_valid_host_name($hostname) == false) {
                $errmsg[$errors++] = "Invalid host name.";
            }

            if ($base == "") {
                $errmsg[$errors++] = "LDAP base is blank.";
            }

            //if ($port=="")
            //$errmsg[$errors++]="Invalid port number.";

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");

            $port = grab_array_var($inargs, "port");
            $base = grab_array_var($inargs, "base");
            $bind_dn = grab_array_var($inargs, "bind_dn");
            $password = grab_array_var($inargs, "password");
            $security = grab_array_var($inargs, "security");
            $version = grab_array_var($inargs, "version");
            $search = grab_array_var($inargs, "search");

            $services = grab_array_var($inargs, "services");
            $serviceargs = grab_array_var($inargs, "serviceargs");

            $output = '

        <input type="hidden" name="ip_address" value="' . encode_form_val($address) . '">
        <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '">
        <input type="hidden" name="port" value="' . encode_form_val($port) . '">
        <input type="hidden" name="base" value="' . encode_form_val($base) . '">
        <input type="hidden" name="bind_dn" value="' . encode_form_val($bind_dn) . '">
        <input type="hidden" name="password" value="' . encode_form_val($password) . '">
        <input type="hidden" name="security" value="' . encode_form_val($security) . '">
        <input type="hidden" name="version" value="' . encode_form_val($version) . '">
        <input type="hidden" name="search" value="' . encode_form_val($search) . '">
        <input type="hidden" name="services_serial" value="' . base64_encode(json_encode($services)) . '">
        <input type="hidden" name="serviceargs_serial" value="' . base64_encode(json_encode($serviceargs)) . '">
';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:


            $output = '
            ';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $hostname = grab_array_var($inargs, "hostname", "");
            $address = grab_array_var($inargs, "ip_address", "");
            $hostaddress = $address;

            $port = grab_array_var($inargs, "port");
            $base = grab_array_var($inargs, "base");
            $bind_dn = grab_array_var($inargs, "bind_dn");
            $password = grab_array_var($inargs, "password");
            $security = grab_array_var($inargs, "security");
            $version = grab_array_var($inargs, "version");
            $search = grab_array_var($inargs, "search");

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["ip_address"] = $address;
            $meta_arr["port"] = $port;
            $meta_arr["base"] = $base;
            $meta_arr["bind_dn"] = $bind_dn;
            $meta_arr["password"] = $password;
            $meta_arr["security"] = $security;
            $meta_arr["version"] = $version;
            $meta_arr["search"] = $search;
            $meta_arr["services"] = $services;
            $meta_arr["serivceargs"] = $serviceargs;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            // Escape values for check_command line
            if (function_exists('nagiosccm_replace_command_line')) {
                $password = nagiosccm_replace_command_line($password);
            }

            $objs = array();

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_ldapserver_host",
                    "host_name" => $hostname,
                    "address" => $hostaddress,
                    "icon_image" => "$logo_filename",
                    "statusmap_image" => "$logo_filename",
                    "_xiwizard" => $wizard_name,
                );
            }

            $pluginopts = "";
            $pluginopts .= "-b \"" . $base . "\"";
            if ($bind_dn != "")
                $pluginopts .= " -D \"" . $bind_dn . "\"";
            if ($password != "")
                $pluginopts .= " -P \"" . $password . "\"";
            if ($version == "2")
                $pluginopts .= " -2";
            else if ($version == "3")
                $pluginopts .= " -3";
            if ($security == "ssl")
                $pluginopts .= " -S";
            else if ($security == "starttls")
                $pluginopts .= " -T";
            if ($port != "")
                $pluginopts .= " -p " . $port;

            $objs[] = array(
                "type" => OBJECTTYPE_SERVICE,
                "host_name" => $hostname,
                "service_description" => "$short_name Server",
                "use" => "xiwizard_ldapserver_ldap_service",
                "check_command" => "check_xi_service_ldap!" . $pluginopts,
                "_xiwizard" => $wizard_name,
                "icon_image" => "$logo_filename",
            );

            //echo "OBJECTS:<BR>";
            //print_r($objs);
            //exit();

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}
