<?php
//
// Custom URL Dashlet 
// Copyright (c) 2008-2019 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__).'/../dashlethelper.inc.php');

// Run the initialization function
custom_dashlet_init();

function custom_dashlet_init()
{
	$name = "custom_dashlet";
	
	$args = array(
		DASHLET_NAME => $name,
		DASHLET_VERSION => "1.0.9",
		DASHLET_DATE => "07/03/2019",
		DASHLET_AUTHOR => "Nagios Enterprises, LLC",
		DASHLET_DESCRIPTION => _("Custom URL Dashlet. Define your own embedded URL, frame size, and opacity.<br><b>Ensure the URL is accessible and is not blocked by Cross-Origin Policy.</b>"),
		DASHLET_COPYRIGHT => "Copyright (c) 2010-2019 Nagios Enterprises, LLC",
		DASHLET_LICENSE => "BSD",
		DASHLET_HOMEPAGE => "https://www.nagios.com",
		DASHLET_REFRESHRATE => 60,
		DASHLET_FUNCTION => "custom_dashlet_func",
        DASHLET_FILTER_GROUPS => array("custom"),
		DASHLET_TITLE => _("Custom URL Dashlet"),
		DASHLET_OUTBOARD_CLASS=> "custom_outboardclass",
		DASHLET_INBOARD_CLASS => "custom_inboardclass",
		DASHLET_PREVIEW_CLASS => "custom_previewclass",
		DASHLET_WIDTH => "300",
		DASHLET_HEIGHT => "200",
		DASHLET_OPACITY => "1.0",
        DASHLET_ISCUSTOM => false
	);
		
	register_dashlet($name, $args);
}

function custom_dashlet_func($mode=DASHLET_MODE_PREVIEW, $id="", $args=null)
{
    $output = "";

    switch ($mode) {

		case DASHLET_MODE_GETCONFIGHTML:
			//input form for dashlet vars 
			$output='
			<label for="height">'._('Height').'</label>
			<br class="nobr" />
			<input type="text" class="form-control" name="height" id="height" value="600"/>
			<br class="nobr" />
			<label for="width">'._('Width').'</label>
			<br class="nobr" />
			<input type="text" class="form-control" name="width" id="width" value="800"/>
			<br class="nobr" />
			
			<label for="url">'._('Dashlet URL').'</label>
			<br class="nobr" />
			<input type="text" class="form-control" name="url" id="url" value="http://">
			<br class="nobr" />
			<label for="refresh">'._('Refresh Rate').'</label>
			<br class="nobr" />
			<input type="text" class="form-control" name="refresh" id="refresh" value="60"> '._('seconds').'
              <br class="nobr" />
            <label for="opacity">'._('Opacity').'</label>
            <br class="nobr" />
            <select class="form-control" name="opacity" id="opacity">
            <option value="1.0">100%</option>
            <option value=".75" selected>75%</option>
            <option value=".50">50%</option>
            <option value=".25">25%</option>
            </select>
            <br class="nobr" />
            <label>&nbsp;</label>
            ';
            break;

        case DASHLET_MODE_OUTBOARD:
        case DASHLET_MODE_INBOARD:

            // Vars from input form (or saved dashlet info)
            $opacity = $args["opacity"];
            $url = urldecode($args["url"]);
            $height = $args['height'];
            $width = $args['width'];
            $refresh = (!empty($args["refresh"])) ? $args["refresh"] : 9999;

            $headers = @get_headers($url, 1);
            if ($headers) $headers = array_change_key_case($headers, CASE_LOWER);

            $frame_options_norestrict = get_option('frame_options_norestrict', 0);
            $frame_src_norestrict = get_option('frame_src_norestrict', 0);
            $url_host = parse_url($url, PHP_URL_HOST);
            $error_message = "";

            if ($url_host !== $_SERVER['HTTP_HOST']) {
                if (isset($headers['x-frame-options']) && (strtolower($headers['x-frame-options']) === 'sameorigin' || strtolower($headers['x-frame-options']) === 'deny')) {
                    if ($frame_options_norestrict) {
                        $error_message .= "<div class='alert alert-danger'>" . _("Error: This website prevents embedding in frames for security reasons (X-Frame-Options: SAMEORIGIN).<br>This restriction is a security feature of modern websites and cannot be bypassed from Nagios XI.") . "</div>";
                    } else {
                        $error_message .= "<div class='alert alert-danger'>" . _("Error: Restricting pages from being opened in iframes is enabled. This setting can be modified by an administrator under Page Security Settings in <a href='/nagiosxi/admin/globalconfig.php#security'>System Settings</a>.") . "</div>";
                    }
                }

                if (isset($headers['content-security-policy']) && (strtolower($headers['content-security-policy']) === "frame-ancestors 'none'" || strtolower($headers['content-security-policy']) === "frame-ancestors 'self'")) {
                   if ($frame_src_norestrict) {
                        $error_message .= "<div class='alert alert-danger'>" . _("Error: This website has security policies that prevent embedding (Content-Security-Policy).<br>This restriction is a security feature of modern websites and cannot be bypassed from Nagios XI.") . "</div>";
                    } else {
                        $error_message .= "<div class='alert alert-danger'>" . _("Error: Restricting iframe sources is enabled. This setting can be modified by an administrator under Page Security Settings in <a href='/nagiosxi/admin/globalconfig.php#security'>System Settings</a>.") . "</div>";
                    }
                }

                if ($error_message) return $error_message;
            }

            // Fix issue with old version of wkhtmltopdf not working properly with new jQuery
            // so we have to set it to use the old version instead when building PDFs
            if (grab_request_var('old_browser_compat', 0)) {
                if (strpos($url, '?') !== false) {
                    $url .= "&old_browser_compat=1";
                } else {
                    $url .= "?old_browser_compat=1";
                }
            }

            // Random dashlet id
            $rand = rand();

            $refresh_rate = $args["refresh"] * 1000;

            // HTML output (heredoc string syntax)
            $output = <<<OUTPUT
            <div id='customdashlet{$rand}' style='opacity: {$opacity};'>
                <iframe id='dashletIframe{$rand}' src='{$url}'></iframe>
            </div>
            <script type="text/javascript">
            var dashcontainer{$rand} = $('#customdashlet{$rand}').parent().parent();

            $(document).ready(function() {
                var h = dashcontainer{$rand}.height();
                var w = dashcontainer{$rand}.width();
                $('#dashletIframe{$rand}').width(w-5).height(h-25);


                $("#customdashlet{$rand}").everyTime({$refresh_rate}, "timer-customdashlet{$rand}", function(i) {
                    $('#dashletIframe{$rand}').attr('src', $('#dashletIframe{$rand}').attr('src'));
                });

            });

            // Bind resize handlers
            dashcontainer{$rand}.resize(function() {
                nh = $(this).height();
                nw = $(this).width();
                $('#dashletIframe{$rand}').width(nw-5).height(nh-25);
            });
            </script>
OUTPUT;
            break;

        case DASHLET_MODE_PREVIEW:

            if(!is_neptune()) {
                $output="<img src='/nagiosxi/includes/dashlets/custom-dashlet/preview.png' alt='No Preview Available' />";
            } else if(get_theme() == 'neptunelight') {
                $output="<img src='/nagiosxi/includes/dashlets/custom-dashlet/custom_dashlet_neptune_light_preview.png' alt='No Preview Available' />";
            } else {
                $output="<img src='/nagiosxi/includes/dashlets/custom-dashlet/neptune_preview.png' alt='No Preview Available' />";
            }

            break;
        }

    return $output;}