#!/usr/bin/env python3

#region Initialization
import argparse
import sys
import ssl
from socket import socket
from pyVim import connect
from pyVmomi import vim
from datetime import timedelta, datetime
import os

try:
   from importlib.metadata import distribution
except ImportError:
   import warnings
   warnings.filterwarnings("ignore", category=DeprecationWarning)
   import pkg_resources

# ---------------------------------------------------------------------------- #
#                                   Constants                                  #
# ---------------------------------------------------------------------------- #
VERSION="1.0.5"

# Possible return values
STATUSES = {"OK": 0, "WARNING": 1, "CRITICAL": 2, "UNKNOWN": 3}
UNK = STATUSES["UNKNOWN"]
CRIT = STATUSES["CRITICAL"]
WARN = STATUSES["WARNING"]
OK = STATUSES["OK"]

# Allowed commands for each domain
ALLOWED_COMMANDS = {
    "VM": {
        "CPU": ["USAGE", "USAGEMHZ", "WAIT", "READY"],
        "MEM": ["USAGE", "SWAP", "SWAPIN", "SWAPOUT", "OVERHEAD", "ACTIVE", "MEMCTL"],
        "NET": ["USAGE", "RECEIVE", "SEND"],
        "IO": ["USAGE", "READ", "WRITE"],
        "RUNTIME": ["CON", "CPU", "MEM", "STATE", "STATUS", "CONSOLECONNECTIONS", "GUEST", "TOOLS", "ISSUES"],
        "VMFS": ["FREE", "USAGE", "USED", "CAPACITY"],
    },
    "HOST": {
        "CPU": ["USAGE", "USAGEMHZ"],
        "MEM": ["USAGE", "SWAP", "OVERHEAD", "OVERALL", "ACTIVE", "MEMCTL"],
        "NET": ["USAGE", "RECEIVE", "SEND", "NIC"],
        "IO": ["USAGE", "READ", "WRITE"],
        "VMFS": ["NAME"],
        "RUNTIME": ["CON", "HEALTH", "STORAGEHEALTH", "TEMPERATURE", "SENSOR", "MAINTENANCE", "LISTVM", "STATUS", "ISSUES"],
        "SERVICE": ["NAMES"],
        "STORAGE": ["ADAPTERS", "LUNS", "PATHS"],
        "UPTIME": [],
        "DEVICE": ["CD"],
    },
    "CLUSTER": {
        "CPU": ["USAGE", "USAGEMHZ"],
        "MEM": ["USAGE", "SWAP", "OVERHEAD", "OVERALL", "ACTIVE", "MEMCTL"],
        "CLUSTER": ["EFFECTIVECPU", "EFFECTIVEMEM", "FAILOVER", "CPUFAIRNESS", "MEMFAIRNESS"],
        "RUNTIME": ["LIST", "LISTHOST", "STATUS", "ISSUES"],
        "VMFS": ["NAME"],
    },
    "DATACENTER": {
        "RECOMMENDATIONS": [],
        "CPU": ["USAGE", "USAGEMHZ"],
        "MEM": ["USAGE", "SWAP", "OVERHEAD", "OVERALL", "ACTIVE", "MEMCTL"],
        "NET": ["USAGE", "RECEIVE", "SEND"],
        "IO": ["ABORTED", "RESETS", "READ", "WRITE", "KERNEL", "DEVICE", "QUEUE"],
        "VMFS": ["NAME"],
        "RUNTIME": ["LIST", "LISTVM", "LISTHOST", "LISTCLUSTER", "STATUS", "ISSUES", "TOOLS"],
    },
}

# ---------------------------------------------------------------------------- #
#                               Global Variables                               #
# ---------------------------------------------------------------------------- #
_address = ""
_content = vim.ServiceInstanceContent
_container = vim.ManagedEntity
_virt_time = datetime
_perf_dict = dict
_verbose = False
_noCheck = False
_default_interval = 300

#region Helper Functions
# ---------------------------------------------------------------------------- #
#                               Helper Functions                               #
# ---------------------------------------------------------------------------- #
def Log(msg, domain="GENERAL"):
    """Log a message"""
    timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    if _verbose:
        print(f"{timestamp} - {domain} - {msg}")

def nagios_exit(result, output):
    """Exit with a STATUS and output message"""
    if result == STATUSES["OK"]:
        output = "OK: " + output
    elif result == STATUSES["WARNING"]:
        output = "WARNING: " + output
    elif result == STATUSES["CRITICAL"]:
        output = "CRITICAL: " + output
    elif result == STATUSES["UNKNOWN"]:
        output = "UNKNOWN: " + output
    print(output)
    sys.exit(result)


def evaluate_threshold(value, threshold):
    """
    Evaluate if a value should trigger an alert based on the threshold
    0 = OK 
    1 = WARN/CRIT   
    """
    if threshold is None or len(threshold) == 0:
        return 0
    value = float(value)

    alert_range = [None, None] # [lower_bound, upper_bound]
    inverted_range = False

    # Determine the type of range and parse boundaries per the guidelines (https://nagios-plugins.org/doc/guidelines.html#THRESHOLDFORMAT)
    if threshold.startswith('@'):
        inverted_range = True
        threshold = threshold[1:]

    if ':' not in threshold:
        # number - Lower bound is 0
        alert_range[0] = 0.0
        alert_range[1] = float(threshold)
    else:
        parts = threshold.split(':')
        if threshold.startswith('~:') or parts[0] == '':
            # ~:number or :number - Lower bound is negative infinity
            alert_range[0] = float('-inf')
            alert_range[1] = float(parts[1]) if parts[1] else float('inf')
        elif parts[1] == '':
            # number: - Upper bound is positive infinity
            alert_range[0] = float(parts[0])
            alert_range[1] = float('inf')
        else:
            # number:number - Both bounds are specified
            alert_range[0] = float(parts[0])
            alert_range[1] = float(parts[1])

    in_alert_range = alert_range[0] <= value <= alert_range[1]

    if inverted_range:
        in_alert_range = not in_alert_range
    
    Log(f"Value {value} {'is' if in_alert_range else 'is not'} in the OK range {'@' if inverted_range else ''}{alert_range}", domain="EVALUATE_THRESHOLD")
    return 0 if in_alert_range else 1
        
# Check a value against the thresholds
# Returns OK, WARN, or CRIT
def check_thresholds(perfdata_dict):
    """Check the value against the thresholds"""
    has_warn = hasattr(args, "warn_range") and args.warn_range
    has_crit = hasattr(args, "crit_range") and args.crit_range
    warn_ok = False
    crit_ok = False

    statuses={}
    if not has_warn and not has_crit:
        Log("No thresholds specified - returning OK", domain="THRESHOLDS")
        return OK

    global ignore_warn_if_any_ok
    global ignore_crit_if_any_warn_ok
    if has_warn and len(args.warn_range) == 1:
        args.warn_range = [args.warn_range[0] for _ in range(len(perfdata_dict) + 1)]
        ignore_warn_if_any_ok = True
    if has_crit and len(args.crit_range) == 1:
        args.crit_range = [args.crit_range[0] for _ in range(len(perfdata_dict) + 1)]
        ignore_crit_if_any_warn_ok = True
    
    warn = None
    crit = None
    for key, value in perfdata_dict.items():
        idx = list(perfdata_dict.keys()).index(key)
        value = value[0]
        if not isinstance(value, (int, float)):
            old_value = value
            try:
                Log(f"Attempting to convert {value} to a number", domain="THRESHOLDS")
                value = float(value)
            except:
                Log(f"{value} is not a number - attempting to pull a number out of it", domain="THRESHOLDS")
                try:
                    Log(f"Attempting to split {value} on '/'", domain="THRESHOLDS")
                    value = float(value.split("/")[0])
                    Log(f"Converted {old_value} to {value}", domain="THRESHOLDS")
                except:
                    try:
                        Log("Couldn't split on '/' - attempting to split on ' '", domain="THRESHOLDS")
                        value = float(value.split(" ")[0])
                        Log(f"Converted {old_value} to {value}", domain="THRESHOLDS")
                    except: 
                        try:
                            Log("Couldn't split on ' ' - attempting to split on '%'", domain="THRESHOLDS")
                            value = float(value.split("%")[0])
                            Log(f"Converted {old_value} to {value}", domain="THRESHOLDS")
                        except:
                            Log(f"Could not convert {value} to a number - skipping", domain="THRESHOLDS")
                            continue
        value = float(value)
        
        if has_crit:
            crit_list = args.crit_range
            if len(crit_list) > idx:
                crit = crit_list[idx]
            else:
                crit = ""

            Log(f"Checking {key}={value} against critical range {crit}", domain="THRESHOLDS")
            if evaluate_threshold(value, crit) == 1:
                statuses[key] = CRIT
                if not ignore_crit_if_any_warn_ok:
                    has_warn = False
            elif ignore_crit_if_any_warn_ok:
                crit_ok = True
        if has_warn:
            warn_list = args.warn_range
            if len(warn_list) > idx:
                warn = warn_list[idx]
            else:
                warn = ""

            Log(f"Checking {key}={value} against warning range {warn}", domain="THRESHOLDS")
            if evaluate_threshold(value, warn) == 1:
                if statuses.get(key) != CRIT:
                    statuses[key] = WARN
                elif ignore_crit_if_any_warn_ok: # We may ignore CRIT if OK/WARN is found, but WARN is still validated per-check, so if all others are OK, but this is WARN, we will need to return WARN
                    warn_found = True
            elif ignore_warn_if_any_ok:
                    warn_ok = True

        if key not in statuses:
            Log("No thresholds specified - OK", domain="THRESHOLDS")
            statuses[key] = OK

        if has_warn:
            has_warn = len(args.warn_range) > idx
        if has_crit:
            has_crit = len(args.crit_range) > idx
    if len(statuses) == 0: 
        Log("Was not caught by any checks - returning OK", domain="THRESHOLDS")
        return OK
    
    most_severe = max(statuses.values())
    least_severe = min(statuses.values())

    if ignore_warn_if_any_ok and ignore_crit_if_any_warn_ok: # -w [range applied to all metrics] -c [range applied to all metrics], use the least severe status
        if warn_ok and crit_ok:
            least_severe = OK
        Log(f"Least severe status is {least_severe} - Returning {least_severe}", domain="THRESHOLDS")
        return least_severe
    elif ignore_warn_if_any_ok:
        if most_severe == CRIT:
            Log("Most severe status is CRIT - Returning CRIT", domain="THRESHOLDS")
            return CRIT
        elif least_severe == OK:
            Log("Least severe status is OK - Returning OK", domain="THRESHOLDS")
            return OK
        else:
            Log("Returning WARN", domain="THRESHOLDS")
            return WARN
    elif ignore_crit_if_any_warn_ok:
        if WARN in statuses.values() or warn_found:
            Log("WARN found - Returning WARN", domain="THRESHOLDS")
            return WARN
        elif least_severe == OK:
            Log("Least severe status is OK - Returning OK", domain="THRESHOLDS")
            return OK
        else:
            Log("Returning CRIT", domain="THRESHOLDS")
            return CRIT
    else:
        Log(f"Most severe status is {most_severe} - Returning {most_severe}", domain="THRESHOLDS")
        return most_severe

# Create a view from the root folder targeting the highest level object specified
def create_root_view(content, args):
    viewType = None
    if args.vm is not None:
        viewType = [vim.VirtualMachine]
        Log("View Type: Virtual Machine", domain="MAIN")
    elif args.host is not None and (args.address is not None or args.datacenter is not None):
        viewType = [vim.HostSystem]
        Log("View Type: Host System", domain="MAIN")
    elif args.cluster is not None:
        viewType = [vim.ClusterComputeResource]
        Log("View Type: Cluster Compute Resource", domain="MAIN")
    elif args.datacenter is not None:
        viewType = [vim.Datacenter]
        Log("View Type: Datacenter", domain="MAIN")
    elif args.address is not None and args.host is None:
        viewType = [vim.Datacenter, vim.ClusterComputeResource, vim.HostSystem, vim.VirtualMachine]
        Log("vSphere address specified, but no object specified, creating view for all objects", domain="MAIN")
    else:
        viewType = [vim.HostSystem]
        Log("No object specified, creating view for all objects", domain="MAIN")

    try:
        view = content.viewManager.CreateContainerView(content.rootFolder, viewType, True)
        Log(f"View: {view}", domain="MAIN")
        return view.view
        # view.Destroy() # TODO: add this to the end of the script
    except Exception as e:
        nagios_exit("UNKNOWN", "Unable to create view: %s" % e)
    return None

def traverse_obj(view, instance, name):
    Log(f"Traversing {view} for {instance} {name}", domain="TRAVERSE_OBJ")
    view = [o for o in view if (o.name == name and isinstance(o, instance))]
    if view is None or len(view) == 0:
        nagios_exit("UNKNOWN", f"Object `{name}` not found")
    elif len(view) > 1:
        nagios_exit("UNKNOWN", f"Multiple objects found with the name `{name}`")
    else:
        Log(f"view: {view}", domain="TRAVERSE_OBJ")
        Log(f"Returning {instance} {view[0]} with name {view[0].name}", domain="TRAVERSE_OBJ")
        return view[0]

# Return a list of objects that match the specified criteria
def traverse_view_for_objs(view, args):
    Log(f"Traversing view {view} for objects", domain="TRAVERSE_VIEW_FOR_OBJS")
    
    if args.vm is not None:
        view = traverse_obj(view, vim.VirtualMachine, args.vm)
    elif args.host is not None and args.address is not None:
        view = traverse_obj(view, vim.HostSystem, args.host)
    elif args.cluster is not None:
        view = traverse_obj(view, vim.ClusterComputeResource, args.cluster)
    elif args.datacenter is not None:
        if args.address is not None:
            view = traverse_obj(view, vim.Datacenter, args.datacenter)
        else:
            view = view[0]
    Log(f"return view: {view}", domain="TRAVERSE_VIEW_FOR_OBJS")
    return view

# This will return a search domain to give to CreateContainerView
# The search domain will be the smallest domain specified: host, then cluster, then datacenter
### TODO: This may not work when there is are multiple domains with the same name - need to test and fix
def get_container_view(content: vim.ServiceInstanceContent) -> vim.ManagedEntity:
    # """Get the container view of the content"""
    root_view = create_root_view(content, args)
    return traverse_view_for_objs(root_view, args)


def QueryPerformanceMetrics(content, counterId, entity, interval):
    global service_instance
    # perfmanager is a reference to the performance manager that we get from the content (instance snapshot)
    perfManager = content.perfManager
    # metricId is the metric we want to query. We pass in the counterId and an empty instance
    metricId = vim.PerformanceManager.MetricId(counterId=counterId, instance="")
    # This determines the range of time we want to query. We want to query the last 20 minutes
    startTime = _virt_time - timedelta(minutes=(interval + 1)) # type: ignore
    Log(f"Start time: {startTime}", domain="QUERY_PERF_METRICS")
    endTime = _virt_time - timedelta(minutes=1) # type: ignore
    Log(f"End time: {endTime}", domain="QUERY_PERF_METRICS")
    Log(f"(Interval: {interval} minutes)", domain="QUERY_PERF_METRICS")



    # # List all the historical intervals
    # historical_intervals = perf_manager.historicalInterval
    # for interval in historical_intervals:
    #     print(f"Interval ID: {interval.key}, Length: {interval.samplingPeriod}")
    # else:
    #     print("No historical intervals found")

    # This is the query we will pass to the performance manager
    query = vim.PerformanceManager.QuerySpec(
        intervalId=interval,
        entity=entity,
        metricId=[metricId],
        startTime=startTime,
        endTime=endTime,
        maxSample=1,
    )
    Log("Querying performance manager...", domain="QUERY_PERF_METRICS")
    # We pass the query to the performance manager
    perfResults = vim.PerformanceManager.QueryPerf(perfManager, querySpec=[query])
    if perfResults:
        # If we get results, we return them
        Log(f"Successfully queried performance manager for {entity.name}", domain="QUERY_PERF_METRICS")
        return perfResults
    else:
        # If we don't get results, we exit with an UNKNOWN status
        Log("Failed to query performance manager", domain="QUERY_PERF_METRICS")
        Log("Failed to get performance metrics", domain="QUERY_PERF_METRICS")
        Log("This may be due to a time sync issue. Check that NTP is configured and working on the vCenter server and ESXi hosts", domain="QUERY_PERF_METRICS")
        Log("If NTP is configured and working, try changing the interval value. If there is no historical data this will fail.", domain="QUERY_PERF_METRICS")
    
        # nagios_exit(UNK, "Failed to get performance metrics")

def FormatPerfResults(perfResults):
    if not perfResults or not perfResults[0] or not perfResults[0].value or not perfResults[0].value[0].value:
        Log("Warning: Performance results returned empty", domain="FORMAT_PERF_RESULTS")
        return 0
    # We return the value of the counter. This is a list of values, so we average them. We get a range because we are querying a range of time
    # If output of the old plugin vs this one is different, this is likely where the math is slightly different
    if not perfResults[0].value:
        Log("Warning: Performance results returned empty", domain="FORMAT_PERF_RESULTS")
        return 0

    Log(f"Returning {float(sum(perfResults[0].value[0].value)) / len(perfResults[0].value[0].value)}", domain="FORMAT_PERF_RESULTS")
    return float(sum(perfResults[0].value[0].value)) / len( perfResults[0].value[0].value )

def GetPerfResults(counter_name: str, obj, interval = None, avg = False, max = False, percent = False):
    if args.max:
        max = True
        avg = False

    if not interval:
        if hasattr(args, "interval") and args.interval is not None:
            if not interval or args.interval is not None:
                for hInterval in _content.perfManager.historicalInterval:
                    if str(hInterval.samplingPeriod) == str(args.interval):
                        interval = int(args.interval)
                        break
            if not interval:
                periodList = [str(i.samplingPeriod) for i in _content.perfManager.historicalInterval]
                if len(periodList) == 0:
                    nagios_exit(UNK, "No historical intervals found, please enable performance statistics and try again")
                else:
                    nagios_exit(UNK, f"Interval {args.interval} not found in historical intervals, please choose from the available intervals: {', '.join(periodList)}")
        else:
            if type(obj) == vim.HostSystem or type(obj) == vim.VirtualMachine:
                interval = 20
            else:
                interval = _content.perfManager.historicalInterval[0].samplingPeriod

    if type(obj) == vim.Datacenter:
        datacenterHosts = _content.viewManager.CreateContainerView(obj, [vim.HostSystem], True).view
        Log(f"Datacenter hosts: {datacenterHosts}", domain="GET_PERF_RESULTS")
        total = 0
        totalWithValues = 0
        for host in datacenterHosts:
            perfVal = GetPerfResults(counter_name, host, interval)
            if perfVal > 0:
                totalWithValues += 1
            if max:
                if perfVal > total:
                    total = perfVal
            else:
                total += perfVal
            if perfVal > 0:
                totalWithValues += 1
        if percent or avg:
            return total / totalWithValues
        return total
    elif type(obj) == vim.ClusterComputeResource:
        usageSummary = obj.summaryEx.usageSummary
        if percent:
            perfVal = GetPerfResults(counter_name, obj, interval)
            return perfVal / usageSummary.totalVmCount
    elif type(obj) == vim.VirtualMachine:
        if obj.runtime.powerState != "poweredOn":
            Log(f"vm {obj.name} is not powered on - skipping", domain="GET_PERF_RESULTS")
            return 0
        if obj.runtime.connectionState != "connected":
            Log(f"vm {obj.name} is not connected - skipping", domain="GET_PERF_RESULTS")
            return 0

    # Look up the counter key by name (e.g. cpu.usage.average)
    # We built this dictionary earlier (look for _perf_dict, it's a global variable)
    counter_key = _perf_dict[counter_name]  # type: ignore
    # We pass the counter key to QueryPerformanceMetrics (see above)
    Log(f"Querying performance results for {counter_name} from {obj.name}", domain="GET_PERF_RESULTS")

    perfResults = QueryPerformanceMetrics(
        _content, counter_key, obj, interval
    )
    
    return FormatPerfResults(perfResults)
        



def format_output(values_dict, delimiter="="):
    """Format the output"""
    output = ""
    # Loop through the dictionary
    for key, value in values_dict.items():
        Log(f"Formatting {key}={value}", domain="FORMAT_OUTPUT")
        # Keep 2 decimal places
        if isinstance(value, float):
            value = round(value, 2)
            Log(f"Rounded value: {value}", domain="FORMAT_OUTPUT")
        output += f"{key}{delimiter}{value}, "
    # Remove the trailing comma but leave the space
    output = output[:-2]
    output += " "
    Log(f"Formatted output: {output}", domain="FORMAT_OUTPUT")
    return output


def format_perfdata(values_dict):
    """Format the perfdata"""
    perfdata = ""
    warn = ""
    crit = ""

    if hasattr(args, "warn_range") and args.warn_range and len(args.warn_range) == 1:
        args.warn_range = [args.warn_range[0] for _ in range(len(values_dict) + 1)]
    if hasattr(args, "crit_range") and args.crit_range and len(args.crit_range) == 1:
        args.crit_range = [args.crit_range[0] for _ in range(len(values_dict) + 1)]
    
    Log(f"values_dict: {values_dict}", domain="FORMAT_PERFDATA")
    for key, value in values_dict.items():
        Log(f"Formatting {key}={value}", domain="FORMAT_PERFDATA")
        index = list(values_dict.keys()).index(key)
        if hasattr(args, "warn_range") and args.warn_range and len(args.warn_range) > index:
            warn = [args.warn_range[index]]
        else:
            warn = ""
        if hasattr(args, "crit_range") and args.crit_range and len(args.crit_range) > index:
            crit = [args.crit_range[index]]
        else:
            crit = ""
        if isinstance(value[0], float):
            if len(warn) > 0 and len(crit) > 0:
                perfdata += f"{key}={value[0]:.2f}{value[1]};{warn[0]};{crit[0]} "
            elif len(warn) > 0:
                perfdata += f"{key}={value[0]:.2f}{value[1]};{warn[0]}; "
            elif len(crit) > 0:
                perfdata += f"{key}={value[0]:.2f}{value[1]};;{crit[0]} "
            elif len(warn) == 0 and len(crit) == 0:
                perfdata += f"{key}={value[0]:.2f}{value[1]};; "
        else:
            if len(warn) > 0 and len(crit) > 0:
                perfdata += f"{key}={value[0]}{value[1]};{warn[0]};{crit[0]} "
            elif len(warn) > 0:
                perfdata += f"{key}={value[0]}{value[1]};{warn[0]}; "
            elif len(crit) > 0:
                perfdata += f"{key}={value[0]}{value[1]};;{crit[0]} "
            elif len(warn) == 0 and len(crit) == 0:
                perfdata += f"{key}={value[0]}{value[1]};; "
    Log(f"Formatted perfdata: {perfdata}", domain="FORMAT_PERFDATA")
    return perfdata

def cfb(bytes, to, rounded=True, decimal_places=2):
    """Convert a value from bytes to another unit"""
    ret=0
    Log(f"Converting {bytes} bytes to {to}", domain="CONVERSION")
    try:
        if to == "B":
            ret = bytes
        elif to == "KB":
            ret = bytes / 1000
        elif to == "Kib":
            ret = bytes / 1024
        elif to == "MB":
            ret = bytes / 1000 / 1000
        elif to == "Mib":
            ret = bytes / 1024 / 1024
        elif to == "GB":
            ret = bytes / 1000 / 1000 / 1000
        elif to == "Gib":
            ret = bytes / 1024 / 1024 / 1024
        elif to == "TB":
            ret = bytes / 1000 / 1000 / 1000 / 1000
        elif to == "Tib":
            ret = bytes / 1024 / 1024 / 1024 / 1024
        else:
            nagios_exit(UNK, f"Invalid unit {to} - must be one of B, KB, Kib, MB, Mib, GB, Gib, TB or TiB")
    except:
        nagios_exit(UNK, f"Error converting {bytes} bytes to " + to)
    if rounded:
        Log(f"Rounding {ret} to {decimal_places} decimal places", domain="CONVERSION")
        return round(ret, decimal_places)
    Log(f"Returning {ret} without rounding", domain="CONVERSION")
    return ret
    
def ctb(value, from_unit):
    """Convert a value to bytes from another unit"""
    Log(f"Converting {value} {from_unit} to bytes", domain="CONVERSION")
    try:
        if from_unit == "B":
            return value
        elif from_unit == "KB":
            return value * 1024
        elif from_unit == "Kib":
            return value * 1000
        elif from_unit == "MB":
            return value * 1024 * 1024
        elif from_unit == "Mib":
            return value * 1000 * 1000
        elif from_unit == "GB":
            return value * 1024 * 1024 * 1024
        elif from_unit == "Gib":
            return value * 1000 * 1000 * 1000
        elif from_unit == "TB":
            return value * 1024 * 1024 * 1024 * 1024
        elif from_unit == "Tib":
            return value * 1000 * 1000 * 1000 * 1000
        else:
            nagios_exit(UNK, f"Invalid unit {from_unit} - must be one of B, KB, Kib, MB, Mib, GB, Gib, TB or TiB")
    except:
        nagios_exit(UNK, f"Error converting {value} {from_unit} to bytes")
        
def list_commands():
    """List the commands"""
    print("Commands:")
    for domain in ALLOWED_COMMANDS:
        print(f"{domain}:")
        for command in ALLOWED_COMMANDS[domain]:
            print(f"\t{command}:")
            for subcommand in ALLOWED_COMMANDS[domain][command]:
                print(f"\t\t{subcommand}")

def NOT_IMPLEMENTED():
    nagios_exit(UNK, "Not implemented yet")
#endregion

#region Argument Parsing
# ---------------------------------------------------------------------------- #
#                              Argument Definition                             #
# ---------------------------------------------------------------------------- #

usage = """Usage: check_vsphere.py  -A <server_name> | -H <host_name> | -A <server_name> -H <host_name>
                    [ -D <datacenter> ] [ -C <cluster_name> ] [ -N <vm_name> ]
                    -u <user> -p <password> | -f <authfile>
                    -l <command> [ -s <subcommand> ] [ -T <timeshift> ] [ -i <interval> ]
                    [ -G <get_guests> ]
                    [ -x <blacklist> ] [ -o <additional_options> ]
                    [ -t <timeout> ] [ -w <warn_range> ] [ -c <crit_range> ]
                    [ -V <version> ] [ -v <verbose> ]"""
                    
epilog = """*** vSphere Wizard Help ***
            When connecting to either an ESXi host or a vCenter server, you must specify a host name (-H) and either a username and password (-u and -p) or an authfile (-f).
            
            You may also specify a datacenter (-D) and/or a cluster (-C). When you specify only a datacenter or a cluster with no host or VM specified, the plugin will
            check the resource usage of the datacenter or cluster. When you specify a host or VM, the search for that resource will be limited to the datacenter or cluster
            specified.

            You must also specify a command (-l) and optionally a subcommand (-s).
            
            Examples: 
            $ ./check_vsphere.py -H vcenter.example.com -D datacenter1 -C cluster1 -l CPU 
                This will check the CPU usage of the compute cluster resource since no host or VM was specified.
                
            $ ./check_vsphere.py -H esxi.domain.local -D datacenter1 -C some -N vm1 -l CPU
                This will search for the VM named 'vm1' in the cluster named 'some' in the datacenter named 'datacenter1' and check the CPU usage. If either the cluster or 
                VM are not found when searching in the datacenter, the check will fail.
                
            $ ./check_vsphere.py -H esxi.domain.local -N some_vm -l MEM
                This will search for the VM named 'some_vm' on the ESXi host and return all memory usage checks.
                
            $ ./check_vsphere.py -H vcenter.example.com -C some_cluster -N some_vm -l MEM -s usage
                This will search for the VM named 'some_vm' in the cluster named 'some_cluster' and check the memory usage.
                
            *** Thresholds ***
            Thresholds are specified using the -w and -c options. You may specify multiple thresholds for each check. The thresholds are checked in the order they are specified.
            If a threshold is not specified, it will not be checked. If a threshold is specified but the check does not return a value, the check will fail. Note that the value
            must be of the same units as the check. For example, if you are checking the CPU usage in MHz, the thresholds must be in MHz. If you are checking the CPU usage in
            percentage, the thresholds must be in percentage.
            
            Examples:
            $ ./check_vsphere.py -H vcenter.example.com -D datacenter1 -C some -N vm1 -l CPU -s USAGE -w 10 -c 20
                This will check the CPU usage of the VM named 'vm1' in the cluster named 'some' in the datacenter named 'datacenter1'. If the CPU usage is >20, the check will
                return CRITICAL. If the CPU usage is between 10 and 20, the check will return WARNING. If the CPU usage is between 0 and 10, the check will return OK.
                
            $ ./check_vsphere.py -H vcenter.example.com -D datacenter1 -C some -N vm1 -l CPU -w ,30 -c @500:3000,40
                This will check the CPU usage of the VM named 'vm1' in the cluster named 'some' in the datacenter named 'datacenter1'. Since no subcommand was specified, all CPU
                metrics will be returned in one check. 
                Commands can be split by metric using a comma. The first metric will be tested against the first set of thresholds.
                The first metric has no warning threshold, so it will not be checked. The first metric has a critical threshold of @500:3000, so it will be checked against an
                inverted range due to the preceding symbol '@'. Thus, if the value is between 500 and 3000, the check will return CRITICAL. If the value is outside of that range, 
                the check will return OK. 
                The second metric has a warning threshold of 30, so it will be checked against a range of 0 to 30. If the value is between 0 and 30, the check will return WARNING. 
                The second metric has a critical threshold of 40, so it will be checked against a range of 30 to 40. If the value is between 30 and 40, the check will return CRITICAL.
                
            $ ./check_vsphere.py -H vcenter.example.com -D datacenter1 -C some -N vm1 -l CPU -s USAGE -w 20 -c 10
                This will check the CPU usage of the VM named 'vm1' in the cluster named 'some' in the datacenter named 'datacenter1'. If the CPU usage is >10, the check will
                return CRITICAL. If the CPU usage is between 10 and 20, the check will still return CRITICAL because CRITICAL values will always override WARNING values.
                
            Further clarification on thresholds can be found at https://nagios-plugins.org/doc/guidelines.html#THRESHOLDFORMAT under `Threshold and Ranges`
            
            *** Commands ***
                
            Supported commands(^ - blank or not specified parameter, o - options, T - timeshift value, b - blacklist) :"
		    VM specific :"
		        * cpu - shows cpu info"
		            + usage - CPU usage in percentage"
		            + usagemhz - CPU usage in MHz"
		            + wait - CPU wait time in ms"
		            + ready - CPU ready time in ms"
		            ^ all cpu info(no thresholds)"
		        * mem - shows mem info"
		            + usage - mem usage in percentage"
                    + active - active mem usage in Mib"
		            + swap - swap mem usage in Mib"
		            + swapin - swapin mem usage in Mib"
		            + swapout - swapout mem usage in Mib"
		            + overhead - additional mem used by VM Server in Mib"
		            + overall - overall mem used by VM Server in Mib"
		            + memctl - mem used by VM memory control driver(vmmemctl) that controls ballooning"
		            ^ all mem info(except overall and no thresholds)"
		        * net - shows net info"
		            + usage - overall network usage in KBps(Kilobytes per Second)"
		            + receive - receive in KBps(Kilobytes per Second)"
		            + send - send in KBps(Kilobytes per Second)"
		            ^ all net info(except usage and no thresholds)"
		        * io - shows disk I/O info"
		            + usage - overall disk usage in Mib/s"
		            + read - read disk usage in Mib/s"
		            + write - write disk usage in Mib/s"
		            ^ all disk io info(no thresholds)"
		        * runtime - shows runtime info"
		            + con - connection state"
		            + cpu - allocated CPU in MHz"
		            + mem - allocated mem in Mib"
		            + state - virtual machine state (UP, DOWN, SUSPENDED)"
		            + status - overall object status (gray/green/red/yellow)"
		            + consoleconnections - console connections to VM"
		            + guest - guest OS status, needs VMware Tools"
		            + tools - VMware Tools status"
		            + issues - all issues for the host"
		            ^ all runtime info(except con and no thresholds)"
		    Host specific :"
		        * cpu - shows cpu info"
		            + usage - CPU usage in percentage"
		                o quickstats - switch for query either PerfCounter values or Runtime info"
		            + usagemhz - CPU usage in MHz"
		                o quickstats - switch for query either PerfCounter values or Runtime info"
		            ^ all cpu info"
		                o quickstats - switch for query either PerfCounter values or Runtime info"
		        * mem - shows mem info"
		            + usage - mem usage in percentage"
		                o quickstats - switch for query either PerfCounter values or Runtime info"
                    + active - active mem usage in Mib"
		            + swap - swap mem usage in Mib"
		                o listvm - turn on/off output list of swapping VM's"
		            + overhead - additional mem used by VM Server"
		            + overall - overall mem used by VM Server"
		            + memctl - mem used by VM memory control driver(vmmemctl) that controls ballooning"
		                o listvm - turn on/off output list of ballooning VM's"
		            ^ all mem info(except overall and no thresholds)"
		        * net - shows net info"
		            + usage - overall network usage in KBps(Kilobytes per Second)"
		            + receive - receive in KBps(Kilobytes per Second)"
		            + send - send in KBps(Kilobytes per Second)"
		            + nic - makes sure all active NICs are plugged in"
		            ^ all net info(except usage and no thresholds)"
		        * io - shows disk io info"
		            + aborted - aborted commands count"
		            + resets - bus resets count"
		            + read - read latency in ms (totalReadLatency.average)"
		            + write - write latency in ms (totalWriteLatency.average)"
		            + kernel - kernel latency in ms"
		            + device - device latency in ms"
		            + queue - queue latency in ms"
		            ^ all disk io info"
		        * vmfs - shows Datastore info"
		            + (name) - free space info for datastore with name (name)"
		                o used - output used space instead of free"
		                o brief - list only alerting volumes"
		                o regexp - whether to treat name as regexp"
		                o blacklistregexp - whether to treat blacklist as regexp"
		                b - blacklist VMFS's"
		                T (value) - timeshift to detemine if we need to refresh"
		            ^ all datastore info"
		                o used - output used space instead of free"
		                o brief - list only alerting volumes"
		                o blacklistregexp - whether to treat blacklist as regexp"
		                b - blacklist VMFS's"
		                T (value) - timeshift to detemine if we need to refresh"
		        * runtime - shows runtime info"
		            + con - connection state"
		            + health - checks cpu/storage/memory/sensor status and propagates worst state"
		                o listitems - list all available sensors(use for listing purpose only)"
		                o blackregexpflag - whether to treat blacklist as regexp"
		                b - blacklist status objects"
		            + storagehealth - storage status check"
		                o blackregexpflag - whether to treat blacklist as regexp"
		                b - blacklist status objects"
		            + temperature - temperature sensors"
		                o blackregexpflag - whether to treat blacklist as regexp"
		                b - blacklist status objects"
		            + sensor - threshold specified sensor"
		            + maintenance - shows whether host is in maintenance mode"
		                o maintwarn - sets warning state when host is in maintenance mode"
		                o maintcrit - sets critical state when host is in maintenance mode"
		            + list(vm) - list of VMware machines and their statuses"
		            + status - overall object status (gray/green/red/yellow)"
		            + issues - all issues for the host"
		                b - blacklist issues"
		            ^ all runtime info(health, storagehealth, temperature and sensor are represented as one value and no thresholds)"
		        * service - shows Host service info"
		            + (names) - check the state of one or several services specified by (names), syntax for (names):<service1>,<service2>,...,<serviceN>"
		            ^ show all services"
		        * storage - shows Host storage info"
		            + adapter - list bus adapters"
		                b - blacklist adapters"
		            + lun - list SCSI logical units"
		                b - blacklist LUN's"
		            + path - list logical unit paths"
		                b - blacklist paths"
		            ^ show all storage info"
		        * uptime - shows Host uptime"
		                o quickstats - switch for query either PerfCounter values or Runtime info"
		    DC specific :"
		        * cpu - shows cpu info"
		            + usage - CPU usage in percentage"
		                o quickstats - switch for query either PerfCounter values or Runtime info"
		            + usagemhz - CPU usage in MHz"
		                o quickstats - switch for query either PerfCounter values or Runtime info"
		            ^ all cpu info"
		                o quickstats - switch for query either PerfCounter values or Runtime info"
		        * mem - shows mem info"
		            + usage - mem usage in percentage"
		                o quickstats - switch for query either PerfCounter values or Runtime info"
                    + active - active mem usage in MB"
		            + swap - swap mem usage in MB"
		            + overhead - additional mem used by VM Server in MB"
		            + overall - overall mem used by VM Server in MB"
		            + memctl - mem used by VM memory control driver(vmmemctl) that controls ballooning"
		            ^ all mem info(except overall and no thresholds)"
		        * net - shows net info"
		            + usage - overall network usage in KBps(Kilobytes per Second)"
		            + receive - receive in KBps(Kilobytes per Second)"
		            + send - send in KBps(Kilobytes per Second)"
		            ^ all net info(except usage and no thresholds)"
		        * io - shows disk io info"
		            + aborted - aborted commands count"
		            + resets - bus resets count"
		            + read - read latency in ms (totalReadLatency.average)"
		            + write - write latency in ms (totalWriteLatency.average)"
		            + kernel - kernel latency in ms"
		            + device - device latency in ms"
		            + queue - queue latency in ms"
		            ^ all disk io info"
		        * vmfs - shows Datastore info"
		            + (name) - free space info for datastore with name (name)"
		                o used - output used space instead of free"
		                o brief - list only alerting volumes"
		                o regexp - whether to treat name as regexp"
		                o blacklistregexp - whether to treat blacklist as regexp"
		                b - blacklist VMFS's"
		                T (value) - timeshift to detemine if we need to refresh"
		            ^ all datastore info"
		                o used - output used space instead of free"
		                o brief - list only alerting volumes"
		                o blacklistregexp - whether to treat blacklist as regexp"
		                b - blacklist VMFS's"
		                T (value) - timeshift to detemine if we need to refresh"
		        * runtime - shows runtime info"
		            + list(vm) - list of VMware machines and their statuses"
		            + listhost - list of VMware esx host servers and their statuses"
		            + listcluster - list of VMware clusters and their statuses"
		            + tools - VMware Tools status"
		                b - blacklist VM's"
		            + status - overall object status (gray/green/red/yellow)"
		            + issues - all issues for the host"
		                b - blacklist issues"
		            ^ all runtime info(except cluster and tools and no thresholds)"
		        * recommendations - shows recommendations for cluster"
		            + (name) - recommendations for cluster with name (name)"
		            ^ all clusters recommendations"
		    Cluster specific :"
		        * cpu - shows cpu info"
		            + usage - CPU usage in percentage"
		            + usagemhz - CPU usage in MHz"
		            ^ all cpu info"
		        * mem - shows mem info"
		            + usage - mem usage in percentage"
		            + active - active mem usage in MB"
		            + swap - swap mem usage in MB"
		                o listvm - turn on/off output list of swapping VM's"
		            + memctl - mem used by VM memory control driver(vmmemctl) that controls ballooning"
		                o listvm - turn on/off output list of ballooning VM's"
		            ^ all mem info(plus overhead and no thresholds)"
		        * cluster - shows cluster services info"
		            + effectivecpu - total available cpu resources of all hosts within cluster"
		            + effectivemem - total amount of machine memory of all hosts in the cluster"
		            + failover - VMware HA number of failures that can be tolerated"
		            + cpufairness - fairness of distributed cpu resource allocation"
		            + memfairness - fairness of distributed mem resource allocation"
		            ^ only effectivecpu and effectivemem values for cluster services"
		        * runtime - shows runtime info"
		            + list(vm) - list of VMware machines in cluster and their statuses"
		            + listhost - list of VMware esx host servers in cluster and their statuses"
		            + status - overall cluster status (gray/green/red/yellow)"
		            + issues - all issues for the cluster"
		                b - blacklist issues"
		            ^ all cluster runtime info"
		        * vmfs - shows Datastore info"
		            + (name) - free space info for datastore with name (name)"
		                o used - output used space instead of free"
		                o brief - list only alerting volumes"
		                o regexp - whether to treat name as regexp"
		                o blacklistregexp - whether to treat blacklist as regexp"
		                b - blacklist VMFS's"
		                T (value) - timeshift to detemine if we need to refresh"
		            ^ all datastore info"
		                o used - output used space instead of free"
		                o brief - list only alerting volumes"
		                o blacklistregexp - whether to treat blacklist as regexp"
		                b - blacklist VMFS's"
		                T (value) - timeshift to detemine if we need to refresh"
            """

parser = argparse.ArgumentParser(
    description="Nagios XI VMware ESX/vSphere Plugin", usage=usage, epilog=epilog, formatter_class=argparse.RawDescriptionHelpFormatter
)
parser.add_argument("-A", "--address", help="The vCenter VM IP", action="store", type=str)
parser.add_argument("-P", "--port", help="The port to connect to. Default is 443", action="store", type=int, default=443)
parser.add_argument("--unverified-ssl", help="Do not verify SSL. SSL is verified by default.", action="store_true", default=False)
parser.add_argument("--no-ssl", help="Do not use SSL. SSL is enabled by default.", action="store_true", default=False)
parser.add_argument("-D", "--datacenter", help="Datacenter name to check", action="store", type=str)
parser.add_argument("-C", "--cluster", help="Cluster name to check", action="store", type=str)
parser.add_argument("-H", "--host", help="The ESXi host", action="store", type=str)
parser.add_argument("-N", "--vm", help="VM name to check", action="store", type=str)
parser.add_argument("-u", "--user", help="Username", action="store", type=str)
parser.add_argument("-p", "--password", help="Password", action="store", type=str)
parser.add_argument("-f", "--authfile", help="Authentication file", action="store", type=str)
parser.add_argument("-l", "--command", help="Command", action="store", type=str, default=None)
parser.add_argument("-s", "--subcommand", help="Subcommand", action="store", type=str, default=None)
parser.add_argument("-G", "--get-guests", help="Get guests", action="store_true")
parser.add_argument("-L", "--list-datastores", help="List datastores", action="store_true")
parser.add_argument("--list-datacenters", help="List datacenters", action="store_true")
parser.add_argument("--list-clusters", help="List clusters", action="store_true")
parser.add_argument("--list-commands", help="List commands", action="store_true")
parser.add_argument("-T", "--timeshift", help="Timeshift (NOT IMPLEMENTED)")
parser.add_argument("-i", "--interval", help="Interval", default=None) # default 1 day since it should always have this interval
parser.add_argument("-x", "--blacklist", help="Blacklist (NOT IMPLEMENTED)")
parser.add_argument("-o", "--additional_options", help="Additional options (NOT IMPLEMENTED)")
parser.add_argument("-t", "--timeout", help="Timeout (NOT IMPLEMENTED)")
parser.add_argument("-w", "--warn_range", help="Warning range", action="append", nargs="*", type=str)
parser.add_argument("-c", "--crit_range", help="Critical range", action="append", nargs="*", type=str)
parser.add_argument("-U", "--unit", help="Desired SI Unit (B, KB, Kib, MB, Mib, GB, Gib, TB or Tib) Default=MB", action="store", type=str, default="MB")
parser.add_argument("-V", "--version", help="Version", action="version", version="%(prog)s {}".format(VERSION))
parser.add_argument("-v", "--verbose", help="Verbose", action="store_true", default=False)
parser.add_argument("-e", "--extra-data", help="Extra data", action="store_true", default=False)
parser.add_argument("-m", "--max", help="Max value", action="store_true", default=False)
args, unknownargs = parser.parse_known_args()
if unknownargs and unknownargs is not None:
    Log(f"Unknown arguments: {unknownargs}", domain="MAIN")

# Set a flag if we are listing datacenters, clusters, or datastores
if args.list_datacenters or args.list_clusters or args.list_datastores or args.get_guests or args.list_commands:
    _noCheck = True
    
# Set the verbose flag
_verbose = args.verbose

def validate_threshold(value):
    if not value or len(value) == 0 or len(value[0]) == 0:
        return None
    if not any(char.isdigit() for char in value[0][0]):
        return None
    return [x.split(",") for x in value[0]][0]

if args.warn_range and len(args.warn_range) > 0:
    args.warn_range = [[",".join(warn[0] for warn in args.warn_range)]]
if args.crit_range and len(args.crit_range) > 0:
    args.crit_range = [[",".join(crit[0] for crit in args.crit_range)]]
ignore_warn_if_any_ok = True
ignore_crit_if_any_warn_ok = True
if args.warn_range and ',' in args.warn_range[0]:
        ignore_warn_if_any_ok = False
if args.crit_range and ',' in args.crit_range[0]:
        ignore_crit_if_any_warn_ok = False
args.warn_range = validate_threshold(args.warn_range)
args.crit_range = validate_threshold(args.crit_range)

# Convert the subcommand to uppercase
if args.subcommand:
    args.subcommand = args.subcommand.upper()

if args.command:
    args.command = args.command.upper()
#endregion

#region Check Functions
# ---------------------------------------------------------------------------- #
#                                Check Functions                               #
# ---------------------------------------------------------------------------- #

# Check functions take a vim object and a subcommand
# The subcommand is optional and is used to filter the output
# If the subcommand is not specified, the check function will return all values
# Some checks use "performance counters" and some use sdk endpoints
# The performance counters are defined in _perf_dict
# The sdk endpoints are not actively documented anywhere - you can look
# at the VMware API documentation, but they are implemented/named
# differently in the pyvmomi SDK. You can also look at the old plugin
# to see what endpoints are used, they are *mostly* the same. 
# These performance counters are very unintuitive - datadog has a good
# list of them here: https://docs.datadoghq.com/integrations/vsphere/#metrics

#region VM Checks
# Matches (-H <host> | -D <datacenter>) [ -C <cluster> ] -N <vm> -l CPU
def vm_cpu_info(vm, subcommand):
    Log("Getting CPU info for VM {}...".format(vm.name), domain="VM_CPU CHECK")
    cpu_mhz = GetPerfResults("cpu.usagemhz.average", vm)
    cpu_usage_prct = GetPerfResults("cpu.usage.average", vm)
    cpu_wait_ms = GetPerfResults("cpu.wait.summation", vm)
    cpu_ready_ms = GetPerfResults("cpu.ready.summation", vm)
    
    results_dict = {}
    perfdata_dict = {}
    if subcommand and subcommand in ALLOWED_COMMANDS["VM"]["CPU"]:
        Log("Filtering by subcommand {}".format(subcommand), domain="VM_CPU CHECK")
        if subcommand == "USAGE":
            results_dict["usage"] = f"{cpu_usage_prct:.2f}%"
            perfdata_dict["cpu_usage"] = (cpu_usage_prct, "%")
        elif subcommand == "USAGEMHZ":
            results_dict["usagemhz"] = f"{cpu_mhz:.2f} MHz"
            perfdata_dict["cpu_usagemhz"] = (cpu_mhz, "")
        elif subcommand == "WAIT":
            results_dict["wait"] = f"{cpu_wait_ms:.2f} ms"
            perfdata_dict["cpu_wait"] = (cpu_wait_ms, "ms")
        elif subcommand == "READY":
            results_dict["ready"] = f"{cpu_ready_ms:.2f} ms"
            perfdata_dict["cpu_ready"] = (cpu_ready_ms, "ms")
    elif subcommand and subcommand not in ALLOWED_COMMANDS["VM"]["CPU"]:
        nagios_exit(UNK, f"Invalid subcommand {subcommand} for VM_CPU")
    else:
        results_dict["cpu usage"] = f"{cpu_mhz} Mhz ({cpu_usage_prct:.2f}%)"
        results_dict["wait"] = f"{cpu_wait_ms:.2f} ms"
        results_dict["ready"] = f"{cpu_ready_ms:.2f} ms"
        perfdata_dict["cpu_usagemhz"] = (f"{cpu_mhz:.2f}", "")
        perfdata_dict["cpu_usage"] = (f"{cpu_usage_prct:.2f}", "%")
        perfdata_dict["cpu_wait"] = (f"{cpu_wait_ms:.2f}", "ms")
        perfdata_dict["cpu_ready"] = (f"{cpu_ready_ms:.2f}", "ms")
            
    output = format_output(
        results_dict
    )
    perfdata = format_perfdata(
        perfdata_dict
    )
    nagios_exit(check_thresholds(perfdata_dict), f"{vm.name}" + output + "| " + perfdata)

# Matches (-H <host> | -D <datacenter>) [ -C <cluster> ] -N <vm> -l MEM
def vm_mem_info(vm, subcommand):
    Log("Getting memory info for VM {}...".format(vm.name), domain="VM_MEM CHECK")
    # All units are in Kib
    mem_pct         = round(GetPerfResults("mem.usage.average", vm) / 100, 2)
    mem_active_a    = GetPerfResults("mem.active.average", vm)
    Log(f"mem.active.average: {mem_active_a}", domain="VM_MEM CHECK")
    mem_active_bytes = ctb(mem_active_a, "Kib")
    Log(f"mem.active.average in bytes: {mem_active_bytes}", domain="VM_MEM CHECK")
    mem_active      = cfb(mem_active_bytes, args.unit)
    Log(f"mem.active.average in {args.unit}: {mem_active}", domain="VM_MEM CHECK")
    mem_active      = cfb(ctb(GetPerfResults("mem.active.average", vm), "Kib"), args.unit)
    mem_overhead    = cfb(ctb(GetPerfResults("mem.overhead.average", vm), "Kib"), args.unit)
    mem_swapped     = cfb(ctb(GetPerfResults("mem.swapped.average", vm), "Kib"), args.unit)
    mem_swapin      = cfb(ctb(GetPerfResults("mem.swapin.average", vm), "Kib"), args.unit)
    mem_swapout     = cfb(ctb(GetPerfResults("mem.swapout.average", vm), "Kib"), args.unit)
    mem_consumed    = cfb(ctb(GetPerfResults("mem.consumed.average", vm), "Kib"), args.unit)
    mem_memctl      = cfb(ctb(GetPerfResults("mem.vmmemctl.average", vm), "Kib"), args.unit)
    results_dict = {}
    perfdata_dict = {}
    if subcommand and subcommand in ALLOWED_COMMANDS["VM"]["MEM"]:
        # Subcommand specified - return only that value
        Log("Filtering by subcommand {}".format(subcommand), domain="VM_MEM CHECK")
        if subcommand == "USAGE":
            Log("In subcommand usage", domain="VM_MEM CHECK")
            results_dict["usage"] = f"{mem_pct:.2f}%"
            perfdata_dict["mem_usage"] = (mem_pct, "%")
        elif subcommand == "USAGEMB":
            results_dict["usage"] = f"{mem_consumed:.2f} {args.unit}"
            perfdata_dict["mem_usagemb"] = (mem_consumed, args.unit)
        elif subcommand == "ACTIVE":
            results_dict["active"] = f"{mem_active:.2f} {args.unit}"
            perfdata_dict["mem_active"] = (mem_active, args.unit)
        elif subcommand == "OVERHEAD":
            results_dict["overhead"] = f"{mem_overhead:.2f} {args.unit}"
            perfdata_dict["mem_overhead"] = (mem_overhead, args.unit)
        elif subcommand == "SWAP":
            results_dict["swap"] = f"{mem_swapped:.2f} {args.unit}"
            perfdata_dict["mem_swap"] = (mem_swapped, args.unit)
        elif subcommand == "SWAPIN":
            results_dict["swapin"] = f"{mem_swapin:.2f} {args.unit}"
            perfdata_dict["mem_swapin"] = (mem_swapin, args.unit)
        elif subcommand == "SWAPOUT":
            results_dict["swapout"] = f"{mem_swapout:.2f} {args.unit}"
            perfdata_dict["mem_swapout"] = (mem_swapout, args.unit)
        elif subcommand == "MEMCTL":
            results_dict["memctl"] = f"{mem_memctl:.2f} {args.unit}"
            perfdata_dict["mem_memctl"] = (mem_memctl, args.unit)
    elif subcommand and subcommand not in ALLOWED_COMMANDS["VM"]["MEM"]:
        nagios_exit(UNK, f"Invalid subcommand {subcommand} for VM_MEM CHECK")
    else:
        # No subcommand specified - return all values
        results_dict["mem usage"] = f"{mem_consumed} {args.unit} ({mem_pct:.2f}%)"
        results_dict["overhead"] = f"{mem_overhead:.2f} {args.unit}"
        results_dict["active"] = f"{mem_active:.2f} {args.unit}"
        results_dict["swapped"] = f"{mem_swapped:.2f} {args.unit}"
        results_dict["swapin"] = f"{mem_swapin:.2f} {args.unit}"
        results_dict["swapout"] = f"{mem_swapout:.2f} {args.unit}"
        results_dict["memctl"] = f"{mem_memctl:.2f} {args.unit}"
        perfdata_dict["mem_usagemb"] = (mem_consumed, args.unit)
        perfdata_dict["mem_usage"] = (mem_pct, "%")
        perfdata_dict["mem_overhead"] = (mem_overhead, args.unit)
        perfdata_dict["mem_active"] = (mem_active, args.unit)
        perfdata_dict["mem_swap"] = (mem_swapped, args.unit)
        perfdata_dict["mem_swapin"] = (mem_swapin, args.unit)
        perfdata_dict["mem_swapout"] = (mem_swapout, args.unit)
        perfdata_dict["mem_memctl"] = (mem_memctl, args.unit)
    Log(f"Results dict: {results_dict}", domain="VM_MEM CHECK")
    output = format_output(
        results_dict
    )
    perfdata = format_perfdata(
        perfdata_dict
    )
    nagios_exit(check_thresholds(perfdata_dict), output + "| " + perfdata)

# Matches (-H <host> | -D <datacenter>) [ -C <cluster> ] -N <vm> -l NET
def vm_net_info(vm, subcommand):
    Log("Getting network info for VM {}...".format(vm.name), domain="VM_NET CHECK")
    # Net values are in B/s
    net_rx = cfb(ctb(GetPerfResults("net.received.average", vm), "KB"), args.unit)
    net_tx = cfb(ctb(GetPerfResults("net.transmitted.average", vm), "KB"), args.unit)
    perfunit = (f"{args.unit}ps" if args.extra_data else "")
    results_dict = {}
    perfdata_dict = {}
    if subcommand and subcommand in ALLOWED_COMMANDS["VM"]["NET"]:
        # Subcommand specified - return only that value
        Log("Filtering by subcommand {}".format(subcommand), domain="VM_NET CHECK")
        if subcommand == "RECEIVE":
            results_dict["receive"] = f"{net_rx:.2f} {args.unit}ps"
            perfdata_dict["net_receive"] = (net_rx, perfunit)
        elif subcommand == "SEND":
            results_dict["send"] = f"{net_tx:.2f} {args.unit}ps"
            perfdata_dict["net_send"] = (net_tx, perfunit)
        elif subcommand == "USAGE":
            net_usage = cfb(ctb(GetPerfResults("net.usage.average", vm), "B"), args.unit)
            results_dict["usage"] = f"{net_usage:.2f} {args.unit}ps"
            perfdata_dict["net_usage"] = (net_usage, perfunit)
    elif subcommand and subcommand not in ALLOWED_COMMANDS["VM"]["NET"]:
        nagios_exit(UNK, f"Invalid subcommand {subcommand} for VM_NET CHECK")
    else:
        # No subcommand specified - return all values
        results_dict["receive"] = f"{net_rx:.2f} {args.unit}ps"
        results_dict["send"] = f"{net_tx:.2f} {args.unit}ps"
        perfdata_dict["net_receive"] = (net_rx, perfunit)
        perfdata_dict["net_send"] = (net_tx, perfunit)
    Log(f"Results dict: {results_dict}", domain="VM_NET CHECK")
    output = format_output(
        results_dict
    )
    perfdata = format_perfdata(
        perfdata_dict
    )
    nagios_exit(check_thresholds(perfdata_dict), output + "| " + perfdata)

# Matches (-H <host> | -D <datacenter>) [ -C <cluster> ] -N <vm> -l IO
def vm_io_info(vm, subcommand):
    Log("Getting IO info for VM {}...".format(vm.name), domain="VM_IO CHECK")
    disk_read = cfb(ctb(GetPerfResults("virtualDisk.read.average", vm), "B"), args.unit)
    disk_write = cfb(ctb(GetPerfResults("virtualDisk.write.average", vm), "B"), args.unit)
    disk_usage = disk_read + disk_write

    results_dict = {}
    perfdata_dict = {}
    if subcommand and subcommand in ALLOWED_COMMANDS["VM"]["IO"]:
        # Subcommand specified - return only that value
        Log("Filtering by subcommand {}".format(subcommand), domain="VM_IO CHECK")
        if subcommand == "USAGE":
            results_dict["usage"] = f"{disk_usage:.2f} {args.unit}/s"
            perfdata_dict["io_usage"] = (disk_usage, f"{args.unit}/s")
        elif subcommand == "READ":
            results_dict["read"] = f"{disk_read:.2f} {args.unit}/s"
            perfdata_dict["io_read"] = (disk_read, f"{args.unit}/s")
        elif subcommand == "WRITE":
            results_dict["write"] = f"{disk_write:.2f} {args.unit}/s"
            perfdata_dict["io_write"] = (disk_write, f"{args.unit}/s")
    elif subcommand and subcommand not in ALLOWED_COMMANDS["VM"]["IO"]:
        nagios_exit(UNK, f"Invalid subcommand {subcommand} for VM_IO CHECK")
    else:
        # No subcommand specified - return all values
        results_dict["usage"] = f"{disk_usage:.2f} {args.unit}/s"
        results_dict["read"] = f"{disk_read:.2f} {args.unit}/s"
        results_dict["write"] = f"{disk_write:.2f} {args.unit}/s"

        perfdata_dict["io_usage"] = (disk_usage, "")
        perfdata_dict["io_read"] = (disk_read, "")
        perfdata_dict["io_write"] = (disk_write, "")

    Log(f"Results dict: {results_dict}", domain="VM_IO CHECK")
    output = format_output(
        results_dict
    )
    perfdata = format_perfdata(
        perfdata_dict
    )
    nagios_exit(check_thresholds(perfdata_dict), output + "| " + perfdata)

# Matches (-H <host> | -D <datacenter>) [ -C <cluster> ] -N <vm> -l RUNTIME
def vm_runtime_info(vm, subcommand):
    Log("Getting runtime info for VM {}...".format(vm.name), domain="VM_RUNTIME CHECK")
    runtime_state = vm.summary.runtime.powerState
    if runtime_state == "poweredOn":
        runtime_state = "UP"
    elif runtime_state == "poweredOff":
        runtime_state = "DOWN"
    elif runtime_state == "suspended":
        runtime_state = "SUSPENDED"
        
    overall_status = vm.summary.overallStatus
    connection_state = vm.summary.runtime.connectionState
    if connection_state == "connected":
        connection_state = "Running"
        
    for metric in [vm.summary.runtime.numMksConnections, vm.summary.runtime.maxCpuUsage, vm.summary.runtime.maxMemoryUsage, vm.summary.guest.toolsStatus]:
        if metric == None:
            nagios_exit(UNK, f"Unable to get runtime info for VM {vm.name}")
    console_connections = vm.summary.runtime.numMksConnections
    max_cpu_mhz = round(vm.summary.runtime.maxCpuUsage, 2)
    max_mem = cfb(ctb(vm.summary.runtime.maxMemoryUsage, "Mib"), args.unit)
    tools_status = vm.summary.guest.toolsStatus.replace("tools", "").lower()
    tools_message = vm.summary.guest.toolsRunningStatus
    tools_message = tools_message.replace("guestTools", "")
    if tools_message == "Running":
        tools_message = "has no config issues"
    else:
        tools_message = "has config issues"
    results_dict = {}
    perfdata_dict = {}
    if subcommand and subcommand in ALLOWED_COMMANDS["VM"]["RUNTIME"]:
        # Subcommand specified - return only that value
        Log("Filtering by subcommand {}".format(subcommand), domain="VM_RUNTIME CHECK")
        if subcommand == "CON":
            results_dict["connection state"] = f"{connection_state}"
            perfdata_dict["connection_state"] = (connection_state, "")
        elif subcommand == "CPU":
            results_dict["max cpu"] = f"{max_cpu_mhz:.0f} MHz"
            perfdata_dict["max_cpu"] = (max_cpu_mhz, "MHz")
        elif subcommand == "MEM":
            results_dict["max mem"] = f"{max_mem:.0f} {args.unit}"
            perfdata_dict["max_mem"] = (max_mem, f"{args.unit}")
        elif subcommand == "STATE":
            results_dict["run state"] = f"{runtime_state}"
            perfdata_dict["runtime_state"] = (runtime_state, "")
        elif subcommand == "STATUS":
            results_dict["status"] = f"{overall_status}"
            perfdata_dict["runtime_status"] = (overall_status, "")
        elif subcommand == "CONSOLECONNECTIONS":
            results_dict["console connections"] = f"{console_connections}"
            perfdata_dict["console_connections"] = (console_connections, "")
        elif subcommand == "GUEST":
            results_dict["guest state"] = f"{tools_status}, {tools_message}"
            perfdata_dict["tools_status"] = (tools_status, "")
        elif subcommand == "TOOLS":
            tools_status = vm.summary.guest.toolsStatus.replace("tools", "").lower()
            tools_message = vm.summary.guest.toolsRunningStatus
            tools_version = vm.summary.guest.toolsVersionStatus
            tools_message = tools_message.replace("guestTools", "")
            tools_version = tools_version.replace("guestTools", "")
            results_dict["tools"] = f"{tools_status}, {tools_message}, {tools_version}"
        elif subcommand == "ISSUES":
            results_dict["issues"] = f"{tools_message}"
    elif subcommand and subcommand not in ALLOWED_COMMANDS["VM"]["RUNTIME"]:
        nagios_exit(UNK, f"Invalid subcommand {subcommand} for VM_RUNTIME CHECK")
    else:
        # No subcommand specified - return all values
        results_dict["state"] = f"{runtime_state}"
        results_dict["status"] = f"{overall_status}"
        results_dict["connection"] = f"{connection_state}"
        results_dict["console"] = f"{console_connections}"
        results_dict["maxcpu"] = f"{max_cpu_mhz:.0f} MHz"
        results_dict["maxmem"] = f"{max_mem:.0f} {args.unit}"
        results_dict["tools"] = f"{tools_status}, {tools_message}"
        
    Log(f"Results dict: {results_dict}", domain="VM_RUNTIME CHECK")
    output = format_output(
        results_dict
    )
    perfdata = format_perfdata(
        perfdata_dict
    )

    nagios_exit(check_thresholds(perfdata_dict), output)

# Matches (-H <host> | -D <datacenter>) [ -C <cluster> ] -N <vm> -l VMFS
def vm_vmfs_info(vm, subcommand):
    Log("Getting VMFS info for VM {}...".format(vm.name), domain="VM_VMFS CHECK")
    # Get the VM's datastores
    datastores = vm.datastore
    results_dict = {}
    perfdata_dict = {}
    if subcommand and subcommand in ALLOWED_COMMANDS["VM"]["VMFS"]:
        # Subcommand specified - return only that value
        Log("Filtering by subcommand {}".format(subcommand), domain="VM_VMFS CHECK")
        if subcommand == "FREE":
            for ds in datastores:
                ds_free = cfb(ctb(ds.summary.freeSpace, "B"), args.unit)
                results_dict[ds.name] = f"{ds_free:.2f} {args.unit} free"
                perfdata_dict[ds.name] = (ds_free, f"{args.unit}")
        elif subcommand == "USAGE" or subcommand == "USED":
            for ds in datastores:
                ds_used = cfb(ctb(ds.summary.capacity - ds.summary.freeSpace, "B"), args.unit)
                results_dict[ds.name] = f"{ds_used:.2f} {args.unit} used"
                perfdata_dict[ds.name] = (ds_used, f"{args.unit}")
        elif subcommand == "CAPACITY":
            for ds in datastores:
                ds_capacity = cfb(ctb(ds.summary.capacity, "B"), args.unit)
                results_dict[ds.name] = f"{ds_capacity:.2f} {args.unit} capacity"
                perfdata_dict[ds.name] = (ds_capacity, f"{args.unit}")
    elif subcommand and subcommand not in ALLOWED_COMMANDS["VM"]["VMFS"]:
        nagios_exit(UNK, f"Invalid subcommand {subcommand} for VM_VMFS CHECK")
    else:
        # No subcommand specified - return all values
        for ds in datastores:
            ds_free = cfb(ctb(ds.summary.freeSpace, "B"), args.unit)
            results_dict[ds.name] = f"{ds_free:.2f} {args.unit} free"
            perfdata_dict[ds.name] = (ds_free, f"{args.unit}")
    Log(f"Results dict: {results_dict}", domain="VM_VMFS CHECK")
    output = format_output(
        results_dict
    )
    perfdata = format_perfdata(
        perfdata_dict
    )
    nagios_exit(check_thresholds(perfdata_dict), output + "| " + perfdata)

#endregion VM Checks

#region Host Checks
def iterate_hosts(metric, hostList, avg=False, max=False, percent=False):
    totalResults = 0
    totalWithValues = 0
    Log(f"Iterating through {hostList}", domain="ITERATING HOSTS")
    for host in hostList:
        try:
            # if datacenter/cluster, results are avg/max/percent of their subsidiary hosts
            # if host, results are the actual values
            if type(host) == vim.Datacenter or type(host) == vim.ClusterComputeResource:
                result = GetPerfResults(metric, host, avg=avg, max=max, percent=percent)
            elif type(host) == vim.HostSystem:
                result = GetPerfResults(metric, host)
            else:
                nagios_exit(UNK, f"Invalid host type {type(host)}")
            
            if result > 0:
                totalWithValues += 1
            if max:
                if result > totalResults:
                    totalResults = result
            else:
                totalResults += result
        except:
            Log(f"Failed to get {metric} for {host.name}", domain="ITERATING HOSTS")
            continue
    if avg:
        totalResults = totalResults / len(hostList)
    if percent:
        totalResults = totalResults / totalWithValues
    return totalResults

# Matches -H <host> -l CPU
def host_cpu_info(host, subcommand, no_exit=False):
    global _content
    if type(host) == vim.HostSystem:
        Log("Getting CPU info for host {}...".format(host.name), domain="HOST_CPU CHECK")
        cpu_usage_mhz = GetPerfResults("cpu.usagemhz.average", host)
        cpu_usage_prct = GetPerfResults("cpu.usage.average", host, percent=True)/100
    else:
        for h in host:
            Log("Getting CPU info for host {}...".format(h.name), domain="HOST_CPU CHECK")
        cpu_usage_mhz = iterate_hosts("cpu.usagemhz.average", host)
        cpu_usage_prct = iterate_hosts("cpu.usage.average", host, avg=True)/100
        cpu_usage_max = iterate_hosts("cpu.usage.average", host, max=True)/100
        

    results_dict = {}
    perfdata_dict = {}
    if subcommand and subcommand in ALLOWED_COMMANDS["HOST"]["CPU"]:
        if subcommand == "USAGE":
            results_dict["usage"] = f"{cpu_usage_prct:.2f}%"
            perfdata_dict["cpu_usage"] = (cpu_usage_prct, "%")
        elif subcommand == "USAGEMHZ":
            results_dict["usagemhz"] = f"{cpu_usage_mhz:.2f} MHz"
            perfdata_dict["cpu_usagemhz"] = (cpu_usage_mhz, "")
    elif subcommand and subcommand not in ALLOWED_COMMANDS["HOST"]["CPU"]:
        nagios_exit(UNK, f"Invalid subcommand {subcommand} for HOST_CPU CHECK")
    else:
        results_dict["cpu usage"] = f"{cpu_usage_mhz:.2f} MHz ({cpu_usage_prct:.2f}%)"
        perfdata_dict["cpu_usagemhz"] = (f"{cpu_usage_mhz:.2f}", "")
        perfdata_dict["cpu_usage"] = (f"{cpu_usage_prct:.2f}", "%")
        try:
            results_dict["highest host CPU usage"] = f"{cpu_usage_max:.2f}%"
            if args.extra_data:
                perfdata_dict["cpu_usage_max"] = (f"{cpu_usage_max:.2f}", "%")
        except:
            pass
    if no_exit:
        return perfdata_dict
    output = format_output(
        results_dict
    )
    perfdata = format_perfdata(
        perfdata_dict
    )

    nagios_exit(check_thresholds(perfdata_dict), output + "| " + perfdata)
    
# Matches -H <host> -l MEM
def host_mem_info(host, subcommand, no_exit=False):
    if type(host) == vim.HostSystem:
        Log("Getting memory info for host {}...".format(host.name), domain="HOST_MEM CHECK")
        mem_pct = round(GetPerfResults("mem.usage.average", host, percent=True) / 100, 2)
        # All units are in Kib
        mem_overhead = cfb(ctb(GetPerfResults("mem.overhead.average", host), "Kib"), args.unit)
        mem_consumed = cfb(ctb(GetPerfResults("mem.consumed.average", host), "Kib"), args.unit)
        mem_memctl = cfb(ctb(GetPerfResults("mem.vmmemctl.average", host), "Kib"), args.unit)
        mem_swapped = cfb(ctb(GetPerfResults("mem.swapused.average", host), "Kib"), args.unit)
    else:
        for h in host:
            Log("Getting memory info for host {}...".format(h.name), domain="HOST_MEM CHECK")
        mem_pct = iterate_hosts("mem.usage.average", host, max=True)/100
        mem_overhead = cfb(ctb(iterate_hosts("mem.overhead.average", host, max=True), "Kib"), args.unit)
        mem_consumed = cfb(ctb(iterate_hosts("mem.consumed.average", host, max=True), "Kib"), args.unit)
        mem_memctl = cfb(ctb(iterate_hosts("mem.vmmemctl.average", host, max=True), "Kib"), args.unit)
        mem_swapped = cfb(ctb(iterate_hosts("mem.swapused.average", host, max=True), "Kib"), args.unit)

    results_dict = {}
    perfdata_dict = {}
    if hasattr(args, "subcommand") and subcommand in ALLOWED_COMMANDS["HOST"]["MEM"]:
        # Subcommand specified - return only that value
        Log("Filtering by subcommand {}".format(subcommand), domain="HOST_MEM CHECK")
        if subcommand == "USAGE":
            results_dict["usage"] = f"{mem_pct:.2f}%"
            perfdata_dict["mem_usage"] = (mem_pct, "%")
            if no_exit:
                return perfdata_dict
        elif subcommand == "USAGEMHZ":
            results_dict["usage"] = f"{mem_consumed:.2f} {args.unit}"
            perfdata_dict["mem_usagemhz"] = (mem_consumed, args.unit)
        elif subcommand == "OVERHEAD":
            results_dict["overhead"] = f"{mem_overhead:.2f} {args.unit}"
            perfdata_dict["mem_overhead"] = (mem_overhead, args.unit)
        elif subcommand == "SWAP":
            results_dict["swap"] = f"{mem_swapped:.2f} {args.unit}"
            perfdata_dict["mem_swap"] = (mem_swapped, args.unit)
        elif subcommand == "MEMCTL":
            results_dict["memctl"] = f"{mem_memctl:.2f} {args.unit}"
            perfdata_dict["mem_memctl"] = (mem_memctl, args.unit)
    elif subcommand and subcommand not in ALLOWED_COMMANDS["HOST"]["MEM"]:
        nagios_exit(UNK, f"Invalid subcommand {subcommand} for HOST_MEM CHECK")
    else:
        # No subcommand specified - return all values
        results_dict["usage"] = f"{mem_pct:.2f}%"
        results_dict["overhead"] = f"{mem_overhead:.2f} {args.unit}"
        results_dict["swapped"] = f"{mem_swapped:.2f} {args.unit}"
        results_dict["memctl"] = f"{mem_memctl:.2f} {args.unit}"
        results_dict["consumed"] = f"{mem_consumed:.2f} {args.unit}"

        perfdata_dict["mem_usagemb"] = (f"{mem_consumed:.2f}", args.unit)
        perfdata_dict["mem_usage"] = (f"{mem_pct:.2f}", "%")
        perfdata_dict["mem_overhead"] = (f"{mem_overhead:.2f}", args.unit)
        perfdata_dict["mem_swap"] = (f"{mem_swapped:.2f}", args.unit)
        perfdata_dict["mem_memctl"] = (f"{mem_memctl:.2f}", args.unit)

        
    Log(f"Results dict: {results_dict}", domain="HOST_MEM CHECK")
    output = format_output(
        results_dict
    )
    perfdata = format_perfdata(
        perfdata_dict
    )
    nagios_exit(check_thresholds(perfdata_dict), output + "| " + perfdata)

# Matches -H <host> -l NET
def host_net_info(host, subcommand):
    if type(host) == vim.HostSystem:
        Log("Getting network info for host {}...".format(host.name), domain="HOST_NET CHECK")
        net_rx = cfb(ctb(GetPerfResults("net.received.average", host), "B"), args.unit)
        net_tx = cfb(ctb(GetPerfResults("net.transmitted.average", host), "B"), args.unit)
        net_usage = cfb(ctb(GetPerfResults("net.usage.average", host), "B"), args.unit)
        num_nics = len(host.config.network.pnic)
        num_nics_connected = 0
        nics_str = ""
        for nic in host.config.network.pnic:
                num_nics_connected += 1
        if num_nics_connected == num_nics:
            nics_str = f"all {num_nics} NICs connected"
        else:
            nics_str = f"{num_nics_connected}/{num_nics} NICs connected"
    else:
        net_rx = iterate_hosts("net.received.average", host, max=True)
        net_tx = iterate_hosts("net.transmitted.average", host, max=True)
        net_usage = iterate_hosts("net.usage.average", host)
        num_nics = 0
        num_nics_connected = 0
        nics_str = ""
        if type(host) == vim.Datacenter or type(host) == vim.ClusterComputeResource:
            host = host.hostFolder
        else:
            objset = set()
            if type(host) == vim.HostSystem or type(host) == vim.VirtualMachine:
                objset.add(host)
            else:
                for h in host:
                    if not hasattr(h, "hostFolder") or h.hostFolder is not None:
                        if type(h) == vim.HostSystem:
                            objset.add(h)
                            continue
                    else:
                        view = _content.viewManager.CreateContainerView(h.hostFolder, [vim.HostSystem], True).view
                        for item in view:
                            objset.add(item)
            host = list(objset)

        for h in host:
            Log("Getting network info for host {}...".format(h.name), domain="HOST_NET CHECK")
            if h.config is not None:
                for nic in h.config.network.pnic:
                    num_nics += 1
                    num_nics_connected += 1
        if num_nics_connected == num_nics:
            nics_str = f"all {num_nics} NICs connected"
        else:
            nics_str = f"{num_nics_connected}/{num_nics} NICs connected"
    perfunit = (f"{args.unit}ps" if args.extra_data else "")

    results_dict = {}
    perfdata_dict = {}
    if subcommand and subcommand in ALLOWED_COMMANDS["HOST"]["NET"]:
        # Subcommand specified - return only that value
        Log("Filtering by subcommand {}".format(subcommand), domain="HOST_NET CHECK")
        if subcommand == "RECEIVE":
            results_dict["receive"] = f"{net_rx:.2f} {args.unit}ps"
            perfdata_dict["net_receive"] = (net_rx, perfunit)
        elif subcommand == "SEND":
            results_dict["send"] = f"{net_tx:.2f} {args.unit}ps"
            perfdata_dict["net_send"] = (net_tx, perfunit)
        elif subcommand == "NICS":
            results_dict["nics"] = f"{nics_str}"
            perfdata_dict["OK_NICs"] = (num_nics_connected, "")
            perfdata_dict["Bad_NICs"] = (num_nics - num_nics_connected, "")
        elif subcommand == "USAGE":
            results_dict["usage"] = f"{net_usage:.2f}"
            perfdata_dict["net_usage"] = (net_usage, perfunit)
    elif subcommand and subcommand not in ALLOWED_COMMANDS["HOST"]["NET"]:
        nagios_exit(UNK, f"Invalid subcommand {subcommand} for HOST_NET CHECK")
    else:
        # No subcommand specified - return all values
        results_dict["receive"] = f"{net_rx:.2f} {args.unit}ps"
        results_dict["send"] = f"{net_tx:.2f} {args.unit}ps"
        results_dict["nics"] = f"{nics_str}"
        perfdata_dict["net_receive"] = (f"{net_rx:.2f}", perfunit)
        perfdata_dict["net_send"] = (f"{net_tx:.2f}", perfunit)
        perfdata_dict["OK_NICs"] = (f"{num_nics_connected}", "")
        perfdata_dict["Bad_NICs"] = (f"{num_nics - num_nics_connected}", "")
 
    Log(f"Results dict: {results_dict}", domain="HOST_NET CHECK")
    output = format_output(
        results_dict
    )
    perfdata = format_perfdata(
        perfdata_dict
    )
    nagios_exit(check_thresholds(perfdata_dict), output + "| " + perfdata)
    
# Matches -H <host> -l IO
def host_io_info(host, subcommand):    
    if type(host) == vim.HostSystem:
        Log("Getting IO info for host {}...".format(host.name), domain="HOST_IO CHECK")
        commands_aborted = GetPerfResults("disk.commandsAborted.summation", host)
        bus_resets = GetPerfResults("disk.busResets.summation", host)
        disk_usage = cfb(ctb(GetPerfResults("disk.usage.average", host), "Kib"), args.unit)
        disk_read = cfb(ctb(GetPerfResults("disk.read.average", host), "Kib"), args.unit)
        disk_write = cfb(ctb(GetPerfResults("disk.write.average", host), "Kib"), args.unit)
    else:
        for h in host:
            Log("Getting IO info for host {}...".format(h.name), domain="HOST_IO CHECK")
        commands_aborted = iterate_hosts("disk.commandsAborted.summation", host)
        bus_resets = iterate_hosts("disk.busResets.summation", host)
        disk_usage = cfb(ctb(iterate_hosts("disk.usage.average", host, max=True), "Kib"), args.unit)
        # disk_read = cfb(ctb(iterate_hosts("disk.read.average", host, max=True), "Kib"), args.unit)
        # disk_write = cfb(ctb(iterate_hosts("disk.write.average", host, max=True), "Kib"), args.unit)
        disk_read = iterate_hosts("disk.read.average", host, max=True)
        disk_write = iterate_hosts("disk.write.average", host, max=True)

    results_dict = {}
    perfdata_dict = {}
    if subcommand and subcommand in ALLOWED_COMMANDS["HOST"]["IO"]:
        # Subcommand specified - return only that value
        Log("Filtering by subcommand {}".format(subcommand), domain="HOST_IO CHECK")
        if subcommand == "USAGE":
            results_dict["usage"] = f"{disk_usage:.2f} {args.unit}/s"
            perfdata_dict["io_usage"] = (disk_usage, f"{args.unit}/s")
        elif subcommand == "READ":
            results_dict["read"] = f"{disk_read:.2f} {args.unit}/s"
            perfdata_dict["io_read"] = (disk_read, f"{args.unit}/s")
        elif subcommand == "WRITE":
            results_dict["write"] = f"{disk_write:.2f} {args.unit}/s"
            perfdata_dict["io_write"] = (disk_write, f"{args.unit}/s")
    elif subcommand and subcommand not in ALLOWED_COMMANDS["HOST"]["IO"]:
        nagios_exit(UNK, f"Invalid subcommand {subcommand} for HOST_IO CHECK")
    else:
        # No subcommand specified - return all values
        results_dict["usage"] = f"{disk_usage:.2f} {args.unit}/s"

        # Legacy data - not supported # TODO: Figure out if it's possible to get this data or remove it
        # results_dict["read"] = f"{disk_read:.2f} ms"
        # results_dict["write"] = f"{disk_write:.2f} ms"
        # results_dict["commands_aborted"] = f"{commands_aborted}"
        # results_dict["bus_resets"] = f"{bus_resets}"

        # perfdata_dict["io_aborted"] = (commands_aborted, "")
        # perfdata_dict["io_busresets"] = (bus_resets, "")
        # perfdata_dict["io_read"] = (disk_read, f"ms")
        # perfdata_dict["io_write"] = (disk_write, f"ms")
        # perfdata_dict["io_kernel"] = ("0", "ms")
        # perfdata_dict["io_device"] = ("0", "ms")
        # perfdata_dict["io_queue"] = ("0", "ms")

        perfdata_dict["io_usage"] = (disk_usage, f"{args.unit}/s")
        

    Log(f"Results dict: {results_dict}", domain="HOST_IO CHECK")
    output = format_output(
        results_dict
    )
    perfdata = format_perfdata(
        perfdata_dict
    )
    nagios_exit(check_thresholds(perfdata_dict), output + "| " + perfdata)

# Matches -H <host> -l VMFS
def host_vmfs_info(host, subcommand):
    datastore = {}
    if type(host) == vim.HostSystem:
        Log("Getting VMFS info for host {}...".format(host.name), domain="HOST_VMFS CHECK")
        for ds in host.datastore:
            if ds.summary.type == "VMFS":
                if ds.summary.capacity != 0:
                    datastore[ds] = {
                        "name": ds.name,
                        "capacity": cfb(ctb(ds.summary.capacity, "B"), args.unit),
                        "free": cfb(ctb(ds.summary.freeSpace, "B"), args.unit),
                        "pct_free": round(
                            (ds.summary.freeSpace / ds.summary.capacity) * 100, 2
                        ),
                    }
                else:
                    datastore[ds] = {
                        "name": ds.name,
                        "capacity": cfb(ctb(ds.summary.capacity, "B"), args.unit),
                        "free": cfb(ctb(ds.summary.freeSpace, "B"), args.unit),
                        "pct_free": 0,
                    }
    else:
        for h in host:
            Log("Getting VMFS info for host {}...".format(h.name), domain="HOST_VMFS CHECK")
            for ds in h.datastore:
                if ds.summary.type == "VMFS":
                    if ds.summary.capacity != 0:
                        datastore[h.name + "-" + ds.name] = {
                            "name": h.name + "-" + ds.name,
                            "capacity": cfb(ctb(ds.summary.capacity, "B"), args.unit),
                            "free": cfb(ctb(ds.summary.freeSpace, "B"), args.unit),
                            "pct_free": round(
                                (ds.summary.freeSpace / ds.summary.capacity) * 100, 2 #
                            ),
                        }
                    else:
                        datastore[h.name + "-" + ds.name] = {
                            "name": h.name + "-" + ds.name,
                            "capacity": cfb(ctb(ds.summary.capacity, "B"), args.unit),
                            "free": cfb(ctb(ds.summary.freeSpace, "B"), args.unit),
                            "pct_free": 0,
                        }
    results_dict = {}
    perfdata_dict = {}
    if subcommand and subcommand in ALLOWED_COMMANDS["HOST"]["VMFS"]:
        # Subcommand specified - return only that value
        Log("Filtering by subcommand {}".format(subcommand), domain="HOST_VMFS CHECK")
        if subcommand == "USAGE":
            for ds in datastore:
                results_dict[datastore[ds]["name"]] = f"{datastore[ds]['pct_free']}%"
                perfdata_dict[datastore[ds]["name"]] = (datastore[ds]['pct_free'], "%")
        elif subcommand == "FREE":
            for ds in datastore:
                results_dict[datastore[ds]["name"]] = f"{datastore[ds]['free']:.2f} {args.unit}"
                perfdata_dict[datastore[ds]["name"]] = (datastore[ds]['free'], f"{args.unit}")
        elif subcommand == "CAPACITY":
            for ds in datastore:
                results_dict[datastore[ds]["name"]] = f"{datastore[ds]['capacity']:.2f} {args.unit}"
                perfdata_dict[datastore[ds]["name"]] = (datastore[ds]['capacity'], f"{args.unit}")
    elif subcommand and subcommand not in ALLOWED_COMMANDS["HOST"]["VMFS"]:
        # Assume this is a datastore name
        Log(f"Checking for datastore {subcommand}", domain="HOST_VMFS CHECK")
        found = False
        subcommand = subcommand.upper()
        for ds in datastore:
            if datastore[ds]["name"].upper() in subcommand:
                results_dict[datastore[ds]["name"]] = f"{datastore[ds]['pct_free']}%"
                perfdata_dict[datastore[ds]["name"]] = (datastore[ds]['pct_free'], "%")
                results_dict[datastore[ds]["name"]] = f"{datastore[ds]['free']:.2f} {args.unit}"
                perfdata_dict[datastore[ds]["name"]] = (datastore[ds]['free'], f"{args.unit}")
                results_dict[datastore[ds]["name"]] = f"{datastore[ds]['capacity']:.2f} {args.unit}"
                perfdata_dict[datastore[ds]["name"]] = (datastore[ds]['capacity'], f"{args.unit}")
                found = True
        if not found:
            nagios_exit(CRIT, f"Datastore {subcommand} not found on host {host.name}")
    else:
        # No subcommand specified - return all values

        Log("No subcommand specified - returning all values", domain="HOST_VMFS CHECK")
        for ds in datastore:
            percent_free = f"{datastore[ds]['pct_free']}"
            results_dict[datastore[ds]["name"] + "(free)"] = f"{datastore[ds]['free']:.2f} {args.unit} ({percent_free}%)"
            perfdata_dict[datastore[ds]["name"]] = (datastore[ds]['free'], f"{args.unit}")
            
    Log(f"Results dict: {results_dict}", domain="HOST_VMFS CHECK")
    output = format_output(
        results_dict
    )
    perfdata = format_perfdata(
        perfdata_dict
    )
    nagios_exit(check_thresholds(perfdata_dict), output + "| " + perfdata)
        
# Matches -H <host> -n <host> -l RUNTIME
def host_runtime_info(host, subcommand, no_exit=False):
    results_dict = {}
    perfdata_dict = {}
    if type(host) == vim.HostSystem:
        vmcount = len(host.vm)

        Log("Getting runtime info for host {}...".format(host.name), domain="HOST_RUNTIME CHECK")
        status = host.summary.runtime.powerState
        connection_state = host.summary.runtime.connectionState
        uptime_seconds = host.summary.quickStats.uptime
        uptime=uptime_seconds
        for item in ["status", "connection_state", "uptime"]:
            if vars().get(item) is None:
                Log(f"Failed to get {item} for {host.name}", domain="HOST_RUNTIME CHECK")
                return
        result = OK
        if status == "poweredOn":
            status = "UP"
        elif status == "poweredOff":
            status = "DOWN"
            result = CRIT
        elif status == "suspended":
            status = "SUSPENDED"
        else:
            status = "UNKNOWN"
            result = CRIT
        if connection_state == "connected":
            connection_state = "Running"
        # Get uptime in in only whole numbers (9 days, 2 hours, 3 minutes, 32 seconds)
        uptime_str = ""
        if uptime > 60 * 60 * 24:
            uptime_str += f"{uptime // (60 * 60 * 24)}d, "
        if uptime > 60 * 60:
            uptime_str += f"{uptime // (60 * 60) % 24}h, "
        if uptime > 60:
            uptime_str += f"{uptime // 60 % 60}m, "
        uptime_str += f"{uptime % 60}s"
    else: # Top level Object
        tempout_dict = {}
        tempperf_dict = {}

        vmcount = 0
        for h in host:
            Log(f"Getting runtime info for {h} {h.name}", domain="HOST_RUNTIME CHECK")
            if type(h) == vim.Datacenter:
                host_runtime_list = ["CON", "HEALTH", "STORAGEHEALTH", "TEMPERATURE"]
                if subcommand and subcommand in host_runtime_list:
                    output, perf_dict = {}, {}
                    host_list = _content.viewManager.CreateContainerView(h.hostFolder, [vim.HostSystem], True).view
                    for subhost in host_list:
                        outputs = host_runtime_info(subhost, subcommand, no_exit=True)
                        if outputs is None:
                            continue
                        host_out, host_perf_dict = outputs
                        for out in host_out:
                            output[subhost.name + "-" + out] = host_out[out]
                        for perf in host_perf_dict:
                            perf_dict[subhost.name + "-" + perf] = host_perf_dict[perf]
                else:
                    output, perf_dict = datacenter_runtime_info(h, subcommand, no_exit=True)
                for out in output:
                    if out not in tempout_dict:
                        tempout_dict[out] = output[out]
                    else:
                        tempout_dict[out] += output[out]
                for perf in perf_dict:
                    if perf not in tempperf_dict:
                        tempperf_dict[perf] = perf_dict[perf]
                    else:
                        tempperf_dict[perf] += " " + h.name + "-" + perf_dict[perf]
            elif type(h) == vim.HostSystem:
                vmcount += len(h.vm)

                output, perf_dict = host_runtime_info(h, subcommand, no_exit=True)
                for out in output:
                    if out not in tempout_dict:
                        tempout_dict[h.name + "-" + out] = output[out]
                    else:
                        tempout_dict[h.name + "-" + out] = output[out]

        results_dict = tempout_dict
        perfdata_dict = tempperf_dict
                
    if subcommand and subcommand in ALLOWED_COMMANDS["HOST"]["RUNTIME"]:
        # Subcommand specified - return only that value
        try:
            Log("Filtering by subcommand {}".format(subcommand), domain="HOST_RUNTIME CHECK")
            if subcommand == "CON":
                results_dict["connection"] = f"{connection_state}"
                perfdata_dict["runtime_connection"] = (connection_state, "")
            elif subcommand == "HEALTH":
                results_dict["status"] = f"{status}"
                perfdata_dict["runtime_state"] = (status, "")
            elif subcommand == "STORAGEHEALTH":
                results_dict["storage status"] = f"{status}"
                perfdata_dict["runtime_storage"] = (status, "")
            elif subcommand == "TEMPERATURE":
                sensorInfo = host.runtime.healthSystemRuntime.systemHealthInfo.numericSensorInfo
                for sensor in sensorInfo:
                    if sensor.name == "Temp":
                        results_dict["temperature"] = f"{sensor.currentReading} C"
                        perfdata_dict["runtime_temp"] = (sensor.currentReading, "C")
                if "temperature" not in results_dict:
                    Log(f"Failed to get temperature for {host}", domain="HOST_RUNTIME CHECK")
            elif subcommand == "SENSOR":
                sensorInfo = host.runtime.healthSystemRuntime.systemHealthInfo.numericSensorInfo
                if args.additional_options is None:
                    nagios_exit(UNK, "No sensor specified for HOST_RUNTIME CHECK, please specify a sensor with -o")
                for sensor in sensorInfo:
                    if sensor.name == args.additional_options:
                        results_dict[sensor.name] = f"{sensor.currentReading} {sensor.baseUnits}"
                        perfdata_dict[f"runtime_{sensor.name}"] = (sensor.currentReading, sensor.baseUnits)
                if sensor.name not in results_dict:
                    Log(f"Failed to get sensor {args.additional_options} for {host}", domain="HOST_RUNTIME CHECK")
            elif subcommand == "MAINTENANCE":
                inMaintenance = host.runtime.inMaintenanceMode
                if inMaintenance:
                    results_dict["maintenance"] = f"{inMaintenance}"
                    perfdata_dict["runtime_maintenance"] = (inMaintenance, "")
                else:
                    Log(f"Failed to get maintenance mode for {host}", domain="HOST_RUNTIME CHECK")
            elif subcommand == "LISTVM":
                try:
                    vmList = host.vm
                    results_dict["vms"] = f"{vmList}"
                except:
                    Log(f"Failed to get VM list for {host}", domain="HOST_RUNTIME CHECK")
            elif subcommand == "STATUS":
                try:
                    status = host.runtime.healthSystemRuntime.systemHealthInfo.numericSensorInfo
                    results_dict["status"] = f"{status}"
                except:
                    Log(f"Failed to get status for {host}", domain="HOST_RUNTIME CHECK")
            elif subcommand == "ISSUES":
                try:
                    issues = host.runtime.healthSystemRuntime.systemHealthInfo.numericSensorInfo
                    results_dict["issues"] = f"{issues}"
                except:
                    Log(f"Failed to get issues for {host}", domain="HOST_RUNTIME CHECK")
        except:
            Log(f"Failed to get {subcommand} for {host}", domain="HOST_RUNTIME CHECK")
    elif subcommand and subcommand not in ALLOWED_COMMANDS["HOST"]["RUNTIME"]:
        nagios_exit(UNK, f"Invalid subcommand {subcommand} for HOST_RUNTIME CHECK")
    else:
        # No subcommand specified - return all values   
        if type(host) == vim.HostSystem:
            results_dict["state"] = f"{status}"
            results_dict["connection"] = f"{connection_state}"
            results_dict["uptime"] = f"{uptime_str}"
        results_dict["vms"] = f"{vmcount}"
        perfdata_dict["vmcount"] = (vmcount, "units")

    if no_exit:
        return results_dict, perfdata_dict
    output = format_output(
        results_dict
    )
    perfdata = format_perfdata(
        perfdata_dict
    )
    if output.strip() == "":
        nagios_exit(UNK, "No data found for HOST_RUNTIME CHECK")
    if type(host) == vim.HostSystem:
        result = check_thresholds(perfdata_dict)
        nagios_exit(result, output)
    else:
        nagios_exit(OK, output)

# Matches -H <host> -n <host> -l SERVICE
# Gets running services (DCUI, TSM, TSM-SSH, attestd, dpd, entropyd, kmxd, lbtd, lwsmd, ntpd, pcscd, ptpd sfcbd-watchdog, slpd, snmpd, vdtc, vltd, vmsyslogd, vpxa, xorg)
# Output should be DCUI (up), TSM (down) etc
def host_service_info(host, service_names, no_exit=False):
    results_dict = {}
    output_dict = {}
    if type(host) == vim.HostSystem:
        services = {}
        Log("Getting service info for host {}...".format(host.name), domain="HOST_SERVICE CHECK")
        for service in host.configManager.serviceSystem.serviceInfo.service:
            services[service.key] = service.running
        for service in services:
            if services[service]:
                output_dict[service] = "(up)"
            else:
                output_dict[service] = "(down)"
        if service_names:
            # Services specified - return only those values
            Log("Filtering by services {}".format(service_names), domain="HOST_SERVICE CHECK")
            services = service_names.split(",")
            for service in services:
                if service in output_dict:
                    results_dict[service] = output_dict[service]
                else:
                    results_dict[service] = "(unknown)"
        else:
            # No services specified - return all values
            results_dict = output_dict
    else:
        services = {}
        for h in host:
            host_services = {}
            if type(h) == vim.Datacenter:
                for host in _content.viewManager.CreateContainerView(h.hostFolder, [vim.HostSystem], True).view:
                    if hasattr(host.configManager, "serviceSystem") and hasattr(host.configManager.serviceSystem, "serviceInfo"):
                        host_services = host_service_info(host, service_names, no_exit=True)
                    else:
                        Log(f"No service info found for host {host.name}", domain="HOST_SERVICE CHECK")
                    if host_services:
                        services[host.name] = host_services
            else:
                if hasattr(h.configManager, "serviceSystem") and hasattr(h.configManager.serviceSystem, "serviceInfo"):
                    host_services = host_service_info(h, service_names, no_exit=True)
                else:
                    Log(f"No service info found for host {h.name}", domain="HOST_SERVICE CHECK")
                if host_services:
                    services[h.name] = host_services
        # create string "host1: service1 (up), service2 (down), ... \nhost2: service1 (up), service2 (down), ..."
        service_str = ""
        for host in services:
            service_str += f"{host}: {services[host]}, \n"
        service_str = service_str[:-3]
        nagios_exit(OK, service_str)

    Log(f"Results dict: {results_dict}", domain="HOST_SERVICE CHECK")
    output = format_output(results_dict, delimiter=" ")
    if no_exit:
        return output
    nagios_exit(OK, output)

# Matches -H <host> -n <host> -l STORAGE
def host_storage_info(host, subcommand):
    if type(host) == vim.HostSystem:
        Log("Getting storage info for host {}...".format(host.name), domain="HOST_STORAGE CHECK")
        adapters = host.config.storageDevice.hostBusAdapter
        LUNs = host.config.storageDevice.scsiLun
        paths = host.config.storageDevice.multipathInfo.lun
        good_adapters = 0
        total_adapters = 0
        good_LUNs = 0
        total_LUNs = 0
        good_paths = 0
        total_paths = 0
        # Find out what adapters are connected
        for adapter in adapters:
            total_adapters += 1
            if adapter.status == "ok":
                good_adapters += 1
        Log(f"Found {good_adapters}/{total_adapters} adapters online", domain="HOST_STORAGE CHECK")
        # Find out what LUNs are connected
        for lun in LUNs:
            total_LUNs += 1
            if lun.operationalState[0] == "ok":
                good_LUNs += 1
        Log(f"Found {good_LUNs}/{total_LUNs} LUNs ok", domain="HOST_STORAGE CHECK")
        # Find out what paths are connected
        for path in paths:
            if hasattr(path, "path"):
                for p in path.path:
                    total_paths += 1
                    if p.state == "active":
                        good_paths += 1
        Log(f"Found {good_paths}/{total_paths} paths active", domain="HOST_STORAGE CHECK")
    else:
        good_adapters = 0
        total_adapters = 0
        good_LUNs = 0
        total_LUNs = 0
        good_paths = 0
        total_paths = 0
        if not hasattr(host[0], "hostFolder"):
            hostList = [h for h in host]
        else:
            hostList = _content.viewManager.CreateContainerView(host[0].hostFolder, [vim.HostSystem], True).view
        for h in hostList:
            Log("Getting storage info for host {}...".format(h.name), domain="HOST_STORAGE CHECK")
            if hasattr(h.config, "storageDevice") and hasattr(h.config.storageDevice, "hostBusAdapter"):
                adapters = h.config.storageDevice.hostBusAdapter
                LUNs = h.config.storageDevice.scsiLun
                paths = h.config.storageDevice.multipathInfo.lun
                for adapter in adapters:
                    total_adapters += 1
                    if adapter.status == "ok":
                        good_adapters += 1
                for lun in LUNs:
                    total_LUNs += 1
                    if lun.operationalState[0] == "ok":
                        good_LUNs += 1
                for path in paths:
                    if hasattr(path, "path"):
                        for p in path.path:
                            total_paths += 1
                            if p.state == "active":
                                good_paths += 1
    results_dict = {}
    perfdata_dict = {}
    if subcommand and subcommand in ALLOWED_COMMANDS["HOST"]["STORAGE"]:
        # Subcommand specified - return only that value
        Log("Filtering by subcommand {}".format(subcommand), domain="HOST_STORAGE CHECK")
        if subcommand == "ADAPTERS":
            results_dict["adapters"] = f"{good_adapters}/{total_adapters} adapters online"
            perfdata_dict["adapters"] = (good_adapters, "")
        elif subcommand == "LUNS":
            results_dict["LUNs"] = f"{good_LUNs}/{total_LUNs} LUNs ok"
            perfdata_dict["LUNs"] = (good_LUNs, "")
        elif subcommand == "PATHS":
            results_dict["paths"] = f"{good_paths}/{total_paths} paths active"
            perfdata_dict["paths"] = (good_paths, "")
    elif subcommand and subcommand not in ALLOWED_COMMANDS["HOST"]["STORAGE"]:
        nagios_exit(UNK, f"Invalid subcommand {subcommand} for HOST_STORAGE CHECK")
    else:
        # No subcommand specified - return all values
        # results_dict[""] = []
        if type(host) == vim.HostSystem:
            results_dict["adapters"] = f"{good_adapters}/{total_adapters} adapters online"
            results_dict["LUNs"] = f"{good_LUNs}/{total_LUNs} LUNs ok"
            results_dict["paths"] = f"{good_paths}/{total_paths} paths active"
            perfdata_dict["adapters"] = (good_adapters, "units")
            perfdata_dict["LUNs"] = (good_LUNs, "units")
            perfdata_dict["paths"] = (good_paths, "units")
        else:
            results_dict = {
                "adapters": f"{good_adapters}/{total_adapters} adapters online",
                "LUNs": f"{good_LUNs}/{total_LUNs} LUNs ok",
                "paths": f"{good_paths}/{total_paths} paths active"
            }
            perfdata_dict = {
                "adapters": (good_adapters, "units"),
                "LUNs": (good_LUNs, "units"),
                "paths": (good_paths, "units")
            }
    Log(f"Results dict: {results_dict}", domain="HOST_STORAGE CHECK")
    output = format_output(
        results_dict
    )
    perfdata = format_perfdata(
        perfdata_dict
    )
    nagios_exit(check_thresholds(perfdata_dict), output + "| " + perfdata)
    #output_str = f"{good_adapters}/{total_adapters} adapters online, {good_LUNs}/{total_LUNs} LUNs ok, {good_paths}/{total_paths} paths active"

# Matches -H <host> -n <host> -l UPTIME
def host_uptime_info(host, subcommand):
    if type(host) == vim.HostSystem:
        uptime_seconds = host.summary.quickStats.uptime
        uptime=uptime_seconds
        # Get uptime in format of 9 days, 4:43:58
        uptime_str = ""
        if uptime > 60 * 60 * 24:
            uptime_str += f"{uptime // (60 * 60 * 24)} days, "
        uptime_str += f"{uptime // (60 * 60) % 24}:{uptime // 60 % 60}:{uptime % 60}"
    else:
        total_uptime = 0
        max_uptime = 0
        hosts_with_uptime = 0
        if not hasattr(host[0], "hostFolder"):
            hostList = [h for h in host]
        else:
            hostList = _content.viewManager.CreateContainerView(host[0].hostFolder, [vim.HostSystem], True).view
        for h in hostList:
            Log("Getting uptime info for host {}...".format(h.name), domain="HOST_UPTIME CHECK")
            uptime_seconds = h.summary.quickStats.uptime
            if uptime_seconds:
                total_uptime += uptime_seconds
                if uptime_seconds > max_uptime:
                    max_uptime = uptime_seconds
                hosts_with_uptime += 1
        average_uptime = total_uptime / hosts_with_uptime
        # Get times in format of 9 days, 4:43:58
        total_uptime_str = ""
        if total_uptime > 60 * 60 * 24:
            total_uptime_str += f"{total_uptime // (60 * 60 * 24)} days, "
        total_uptime_str += f"{total_uptime // (60 * 60) % 24}:{total_uptime // 60 % 60}:{total_uptime % 60}"
        average_uptime_str = ""
        if average_uptime > 60 * 60 * 24:
            average_uptime_str += f"{average_uptime // (60 * 60 * 24):.0f} days, "
        average_uptime_str += f"{average_uptime // (60 * 60) % 24:.0f}:{average_uptime // 60 % 60:.0f}:{average_uptime % 60:.0f}"
        if max_uptime > 60 * 60 * 24:
            max_uptime_str = f"{max_uptime // (60 * 60 * 24):.0f} days, "
        max_uptime_str += f"{max_uptime // (60 * 60) % 24:.0f}:{max_uptime // 60 % 60:.0f}:{max_uptime % 60:.0f}"

    results_dict = {}
    perfdata_dict = {}
    if subcommand and subcommand in ALLOWED_COMMANDS["HOST"]["UPTIME"]:
        # Subcommand specified - return only that value
        Log("Filtering by subcommand {}".format(subcommand), domain="HOST_UPTIME CHECK")
        if subcommand == "UPTIME":
            results_dict["uptime"] = f"{uptime_str}"
            perfdata_dict["uptime"] = (uptime_seconds, "s")
    elif subcommand and subcommand not in ALLOWED_COMMANDS["HOST"]["UPTIME"]:
        nagios_exit(UNK, f"Invalid subcommand {subcommand} for HOST_UPTIME CHECK")
    else:
        # No subcommand specified - return all values
        try:
            if args.extra_data:
                results_dict["total_uptime"] = f"{total_uptime_str}"
                perfdata_dict["uptime"] = (total_uptime, "s")
                results_dict["average_uptime"] = f"{average_uptime_str}"
                perfdata_dict["average_uptime"] = (average_uptime, "s")
            else:
                results_dict["total_uptime"] = f"{average_uptime_str}"
                perfdata_dict["uptime"] = (max_uptime, "s")
        except:
            results_dict["maximum host uptime"] = f"{max_uptime_str}"
            perfdata_dict["uptime"] = (uptime_seconds, "s") 

    Log(f"Results dict: {results_dict}", domain="HOST_UPTIME CHECK")
    output = format_output(
        results_dict
    )
    perfdata = format_perfdata(
        perfdata_dict
    )
    nagios_exit(OK, output + "| " + perfdata)
#endregion Host checks

#region Cluster checks
# Matches (-H <host> | -D <datacenter>) -C <cluster> -l RECOMMENDATIONS
def cluster_cpu_info(cluster, subcommand):
    if type(cluster) == vim.ClusterComputeResource:
        host_list = cluster.host
        cpu_usage_mhz = GetPerfResults("cpu.usagemhz.average", cluster)
        cpu_usage_prct = GetPerfResults("cpu.usage.average", cluster, percent = True)
        cpu_usage_max = 0
        for host in host_list:
            host_cpu_usage = host_cpu_info(host, "USAGE", no_exit=True)["cpu_usage"][0]
            if host_cpu_usage > cpu_usage_max:
                cpu_usage_max = host_cpu_usage
    else:
        for c in cluster:
            Log("Getting CPU info for cluster {}...".format(c.name), domain="CLUSTER_CPU CHECK")
        host_list = _content.viewManager.CreateContainerView(cluster[0].hostFolder, [vim.HostSystem], True).view
        cpu_usage_mhz = iterate_hosts("cpu.usagemhz.average", host_list, max=True)
        cpu_usage_prct = iterate_hosts("cpu.usage.average", host_list, max=True)
    results_dict = {}
    perfdata_dict = {}
    if subcommand and subcommand in ALLOWED_COMMANDS["CLUSTER"]["CPU"]:
        if subcommand == "USAGE":
            results_dict["usage"] = f"{cpu_usage_prct:.2f}%"
            perfdata_dict["cpu_usage"] = (cpu_usage_prct, "%")
        elif subcommand == "USAGEMHZ":
            results_dict["usagemhz"] = f"{cpu_usage_mhz:.2f} MHz"
            perfdata_dict["cpu_usagemhz"] = (cpu_usage_mhz, "")
    elif subcommand and subcommand not in ALLOWED_COMMANDS["CLUSTER"]["CPU"]:
        nagios_exit(UNK, f"Invalid subcommand {subcommand} for CLUSTER_CPU CHECK")
    else:
        results_dict["cpu usage"] = f"{cpu_usage_mhz:.2f} Mhz ({cpu_usage_prct:.2f}%)"
        perfdata_dict["cpu_usagemhz"] = (f"{cpu_usage_mhz:.2f}", "")
        perfdata_dict["cpu_usage"] = (f"{cpu_usage_prct:.2f}", "%")
        if cpu_usage_max and cpu_usage_max > 0 and args.extra_data:
            results_dict["highest host cpu usage"] = f"{cpu_usage_max:.2f}%"
            perfdata_dict["cpu_host_max_avg"] = (f"{cpu_usage_max:.2f}", "%")
    output = format_output(
        results_dict
    )
    perfdata = format_perfdata(
        perfdata_dict
    )
    nagios_exit(check_thresholds(perfdata_dict), output + "| " + perfdata)

# Matches (-H <host> | -D <datacenter>) -C <cluster> -l MEM
def cluster_mem_info(cluster, subcommand):
    Log("Getting memory info for cluster {}...".format(cluster.name), domain="CLUSTER_MEM CHECK")
    mem_pct = round(GetPerfResults("mem.usage.average", cluster, percent=True) / 100, 2)
    # All units are in Kib
    mem_consumed = cfb(ctb(GetPerfResults("mem.consumed.average", cluster), "Kib"), args.unit)
    mem_overhead = cfb(ctb(GetPerfResults("mem.overhead.average", cluster), "Kib"), args.unit)
    # mem_swapped = cfb(ctb(GetPerfResults("mem.swapused.average", cluster), "Kib"), args.unit)
    mem_memctl = cfb(ctb(GetPerfResults("mem.vmmemctl.average", cluster), "Kib"), args.unit)
    # mem_active = cfb(ctb(GetPerfResults("mem.active.average", cluster), "Kib"), args.unit)
    host_list = cluster.host
    mem_max = 0
    for host in host_list:
        host_mem = host_mem_info(host, "USAGE", no_exit=True)["mem_usage"][0]
        if host_mem > mem_max:
            mem_max = host_mem
    results_dict = {}
    perfdata_dict = {}
    if subcommand and subcommand in ALLOWED_COMMANDS["CLUSTER"]["MEM"]:
        # Subcommand specified - return only that value
        Log("Filtering by subcommand {}".format(subcommand), domain="CLUSTER_MEM CHECK")
        if subcommand == "USAGE":
            results_dict["usage"] = f"{mem_pct:.2f}%"
            perfdata_dict["mem_usage"] = (mem_pct, "%")
        elif subcommand == "USAGEMB":
            results_dict["mem usage"] = f"{mem_consumed:.2f} {args.unit}"
            perfdata_dict["mem_usagemb"] = (mem_consumed, args.unit)
        elif subcommand == "ACTIVE":
            nagios_exit(UNK, "ACTIVE subcommand not implemented for CLUSTER_MEM CHECK")
        elif subcommand == "OVERHEAD":
            results_dict["overhead"] = f"{mem_overhead:.2f} {args.unit}"
            perfdata_dict["mem_overhead"] = (mem_overhead, args.unit)
        elif subcommand == "SWAP":
            nagios_exit(UNK, "SWAP subcommand not implemented for CLUSTER_MEM CHECK")
        elif subcommand == "MEMCTL":
            results_dict["memctl"] = f"{mem_memctl:.2f} {args.unit}"
            perfdata_dict["mem_memctl"] = (mem_memctl, args.unit)
        elif subcommand == "OVERALL":
            overall = mem_consumed + mem_overhead
            results_dict["overall"] = f"{overall:.2f} {args.unit}"
            perfdata_dict["mem_overall"] = (overall, args.unit)
    elif subcommand and subcommand not in ALLOWED_COMMANDS["CLUSTER"]["MEM"]:
        nagios_exit(UNK, f"Invalid subcommand {subcommand} for CLUSTER_MEM CHECK")
    else:
        # No subcommand specified - return all values
        results_dict["usage"] = f"{mem_pct:.2f}%"
        # results_dict["active"] = f"{mem_active:.2f} {args.unit}"
        results_dict["overhead"] = f"{mem_overhead:.2f} {args.unit}"
        # results_dict["swapped"] = f"{mem_swapped:.2f} {args.unit}"
        results_dict["memctl"] = f"{mem_memctl:.2f} {args.unit}"
        results_dict["consumed"] = f"{mem_consumed:.2f} {args.unit}"

        perfdata_dict["mem_usagemb"] = (mem_consumed, args.unit)
        perfdata_dict["mem_usage"] = (mem_pct, "%")
        perfdata_dict["mem_overhead"] = (mem_overhead, args.unit)
        # perfdata_dict["mem_swapped"] = (mem_swapped, args.unit)
        perfdata_dict["mem_swap"] = ("0.00", args.unit) # NOTE: This is a dummy because the API doesn't return swap usage. TODO: should this be NaN?
        perfdata_dict["mem_memctl"] = (mem_memctl, args.unit)
        # perfdata_dict["mem_active"] = (mem_active, args.unit)
        if mem_max and mem_max > 0 and args.extra_data:
            results_dict["highest host memory usage"] = f"{mem_max:.2f}%"
            perfdata_dict["mem_host_max_avg"] = (f"{mem_max:.2f}", "%")

    Log(f"Results dict: {results_dict}", domain="CLUSTER_MEM CHECK")
    output = format_output(
        results_dict
    )
    perfdata = format_perfdata(
        perfdata_dict
    )
    nagios_exit(check_thresholds(perfdata_dict), output + "| " + perfdata)

# Matches (-H <host> | -D <datacenter>) -C <cluster> -l CLUSTER
def cluster_cluster_info(cluster, subcommand):
    Log("Getting cluster info for cluster {}...".format(cluster.name), domain="CLUSTER_CLUSTER CHECK")
    results_dict = {}
    perfdata_dict = {}
    if subcommand and subcommand in ALLOWED_COMMANDS["CLUSTER"]["CLUSTER"]:
        # Subcommand specified - return only that value
        Log("Filtering by subcommand {}".format(subcommand), domain="CLUSTER_CLUSTER CHECK")
        if subcommand == "CLUSTER":
            results_dict["cluster"] = f"{cluster.name}"
            perfdata_dict["cluster"] = (cluster.name, "")
    elif subcommand and subcommand not in ALLOWED_COMMANDS["CLUSTER"]["CLUSTER"]:
        nagios_exit(UNK, f"Invalid subcommand {subcommand} for CLUSTER_CLUSTER CHECK")
    else:
        # No subcommand specified - return all values
        results_dict["cluster"] = f"{cluster.name}"
        perfdata_dict["cluster"] = (cluster.name, "")

    Log(f"Results dict: {results_dict}", domain="CLUSTER_CLUSTER CHECK")
    output = format_output(
        results_dict
    )
    perfdata = format_perfdata(
        perfdata_dict
    )
    nagios_exit(OK, output + "| " + perfdata)

# Matches (-H <host> | -D <datacenter>) -C <cluster> -l VMFS
def cluster_vmfs_info(cluster, subcommand):
    datastore = {}
    if type(cluster) == vim.ClusterComputeResource:
        Log("Getting VMFS info for cluster {}...".format(cluster.name), domain="CLUSTER_VMFS CHECK")
        for ds in cluster.datastore:
            if ds.summary.type == "VMFS" and ds.summary.accessible:
                if ds.summary.capacity != 0:
                    datastore[ds] = {
                        "name": ds.name,
                        "capacity": cfb(ctb(ds.summary.capacity, "B"), args.unit),
                        "free": cfb(ctb(ds.summary.freeSpace, "B"), args.unit),
                        "pct_free": round(
                            (ds.summary.freeSpace / ds.summary.capacity) * 100, 2
                        ),
                    }
                else:
                    datastore[ds] = {
                        "name": ds.name,
                        "capacity": cfb(ctb(ds.summary.capacity, "B"), args.unit),
                        "free": cfb(ctb(ds.summary.freeSpace, "B"), args.unit),
                        "pct_free": 0,
                    }
                    
    else:
        for c in cluster:
            Log("Getting VMFS info for cluster {}...".format(c.name), domain="CLUSTER_VMFS CHECK")
            for ds in c.datastore:
                if ds.summary.type == "VMFS" and ds.summary.accessible:
                    if ds.summary.capacity != 0:
                        datastore[c.name + "-" + ds.name] = {
                            "name": c.name + "-" + ds.name,
                            "capacity": cfb(ctb(ds.summary.capacity, "B"), args.unit),
                            "free": cfb(ctb(ds.summary.freeSpace, "B"), args.unit),
                            "pct_free": round(
                                (ds.summary.freeSpace / ds.summary.capacity) * 100, 2
                            ),
                        }
                    else:
                        datastore[c.name + "-" + ds.name] = {
                            "name": c.name + "-" + ds.name,
                            "capacity": cfb(ctb(ds.summary.capacity, "B"), args.unit),
                            "free": cfb(ctb(ds.summary.freeSpace, "B"), args.unit),
                            "pct_free": 0,
                        }
    results_dict = {}
    perfdata_dict = {}
    if subcommand and subcommand in ALLOWED_COMMANDS["CLUSTER"]["VMFS"]:
        # Subcommand specified - return only that value
        Log("Filtering by subcommand {}".format(subcommand), domain="CLUSTER_VMFS CHECK")
        if subcommand == "USAGE":
            for ds in datastore:
                results_dict[datastore[ds]["name"]] = f"{datastore[ds]['pct_free']}%"
                perfdata_dict[datastore[ds]["name"]] = (datastore[ds]['pct_free'], "%")
        elif subcommand == "FREE":
            for ds in datastore:
                results_dict[datastore[ds]["name"]] = f"{datastore[ds]['free']:.2f} {args.unit}"
                perfdata_dict[datastore[ds]["name"]] = (datastore[ds]['free'], f"{args.unit}")
        elif subcommand == "CAPACITY":
            for ds in datastore:
                results_dict[datastore[ds]["name"]] = f"{datastore[ds]['capacity']:.2f} {args.unit}"
                perfdata_dict[datastore[ds]["name"]] = (datastore[ds]['capacity'], f"{args.unit}")
    elif subcommand and subcommand not in ALLOWED_COMMANDS["CLUSTER"]["VMFS"]:
        nagios_exit(UNK, f"Invalid subcommand {subcommand} for CLUSTER_VMFS CHECK")
    else:
        # No subcommand specified - return all values
        Log("No subcommand specified - returning all values", domain="CLUSTER_VMFS CHECK")
        for ds in datastore:
            percent_free = f"{datastore[ds]['pct_free']}"
            results_dict[datastore[ds]["name"] + "(free)"] = f"{datastore[ds]['free']:.2f} {args.unit} ({percent_free}%)"
            perfdata_dict[datastore[ds]["name"]] = (datastore[ds]['free'], f"{args.unit}")
            
    Log(f"Results dict: {results_dict}", domain="CLUSTER_VMFS CHECK")
    output = format_output(
        results_dict
    )
    perfdata = format_perfdata(
        perfdata_dict
    )
    nagios_exit(check_thresholds(perfdata_dict), output + "| " + perfdata)

# Matches (-H <host> | -D <datacenter>) -C <cluster> -l RUNTIME
def cluster_runtime_info(cluster, subcommand):
    Log("Getting runtime info for cluster {}...".format(cluster.name), domain="CLUSTER_RUNTIME CHECK")
    status = cluster.summary.overallStatus
    connection_state = str(cluster.summary.usageSummary.totalVmCount - cluster.summary.usageSummary.poweredOffVmCount) + "/" + str(cluster.summary.usageSummary.totalVmCount)
    results_dict = {}
    perfdata_dict = {}
    if subcommand and subcommand in ALLOWED_COMMANDS["CLUSTER"]["RUNTIME"]:
        # Subcommand specified - return only that value
        Log("Filtering by subcommand {}".format(subcommand), domain="CLUSTER_RUNTIME CHECK")
        if subcommand == "LIST":
            nagios_exit(UNK, "LIST subcommand not implemented for CLUSTER_RUNTIME CHECK")
        elif subcommand == "LISTHOST":
            nagios_exit(UNK, "LISTHOST subcommand not implemented for CLUSTER_RUNTIME CHECK")
        elif subcommand == "STATUS":
            results_dict["state"] = f"{status}"
            perfdata_dict["runtime_state"] = (status, "")
        elif subcommand == "ISSUES":
            nagios_exit(UNK, "ISSUES subcommand not implemented for CLUSTER_RUNTIME CHECK")
    elif subcommand and subcommand not in ALLOWED_COMMANDS["CLUSTER"]["RUNTIME"]:
        nagios_exit(UNK, f"Invalid subcommand {subcommand} for CLUSTER_RUNTIME CHECK")
    else:
        # No subcommand specified - return all values
        results_dict["state"] = f"{status}"
        results_dict["VMs up"] = f"{connection_state}"
        perfdata_dict["vmcount"] = (str(cluster.summary.usageSummary.totalVmCount - cluster.summary.usageSummary.poweredOffVmCount), "units")
        
    Log(f"Results dict: {results_dict}", domain="CLUSTER_RUNTIME CHECK")
    output = format_output(
        results_dict
    )
    perfdata = format_perfdata(
        perfdata_dict
    )
    nagios_exit(check_thresholds(perfdata_dict), output + "| " + perfdata)
#endregion Cluster checks

#region Datacenter checks
# Matches -D <datacenter> -l RECOMMENDATIONS
def datacenter_recommendations_info(datacenter, subcommand):
    NOT_IMPLEMENTED()

# Matches -D <datacenter> -l CPU
def datacenter_cpu_info(datacenter, subcommand):
    Log("Getting CPU info for datacenter {}...".format(datacenter.name), domain="DATACENTER_CPU CHECK")
    cpu_usage_mhz = GetPerfResults("cpu.usagemhz.average", datacenter)
    cpu_usage_prct = GetPerfResults("cpu.usage.average", datacenter, percent=True)/100
    results_dict = {}
    perfdata_dict = {}
    if subcommand and subcommand in ALLOWED_COMMANDS["DATACENTER"]["CPU"]:
        if subcommand == "USAGE":
            results_dict["usage"] = f"{cpu_usage_prct:.2f}%"
            perfdata_dict["cpu_usage"] = (cpu_usage_prct, "%")
        elif subcommand == "USAGEMHZ":
            results_dict["usagemhz"] = f"{cpu_usage_mhz:.2f} MHz"
            perfdata_dict["cpu_usagemhz"] = (cpu_usage_mhz, "")
    elif subcommand and subcommand not in ALLOWED_COMMANDS["DATACENTER"]["CPU"]:
        nagios_exit(UNK, f"Invalid subcommand {subcommand} for DATACENTER_CPU CHECK")
    else:
        results_dict["usagemhz"] = f"{cpu_usage_mhz:.2f} MHz"
        results_dict["usage"] = f"{cpu_usage_prct:.2f}%"
        perfdata_dict["cpu_usagemhz"] = (cpu_usage_mhz, "")
        perfdata_dict["cpu_usage"] = (cpu_usage_prct, "%")
    output = format_output(
        results_dict
    )
    perfdata = format_perfdata(
        perfdata_dict
    )
    nagios_exit(check_thresholds(perfdata_dict), output + "| " + perfdata)

# Matches -D <datacenter> -l MEM
def datacenter_mem_info(datacenter, subcommand):
    if type(datacenter) == vim.Datacenter:
        Log("Getting memory info for datacenter {}...".format(datacenter.name), domain="DATACENTER_MEM CHECK")
        mem_pct = round(GetPerfResults("mem.usage.average", datacenter, percent=True) / 100, 2)
        # All units are in Kib
        mem_overhead = cfb(ctb(GetPerfResults("mem.overhead.average", datacenter), "Kib"), args.unit)
        mem_consumed = cfb(ctb(GetPerfResults("mem.consumed.average", datacenter), "Kib"), args.unit)
        mem_memctl = cfb(ctb(GetPerfResults("mem.vmmemctl.average", datacenter), "Kib"), args.unit)
        mem_swapped = cfb(ctb(GetPerfResults("mem.swapused.average", datacenter), "Kib"), args.unit)
    else:
        mem_pct = []
        mem_overhead = 0
        mem_consumed = 0
        mem_memctl = 0
        mem_swapped = 0
        for dc in datacenter:
            Log("Getting memory info for datacenter {}...".format(dc.name), domain="DATACENTER_MEM CHECK")
            mem_pct.append(round(GetPerfResults("mem.usage.average", dc) / 100, 2))
            mem_overhead += cfb(ctb(GetPerfResults("mem.overhead.average", dc), "Kib"), args.unit)
            mem_consumed += cfb(ctb(GetPerfResults("mem.consumed.average", dc), "Kib"), args.unit)
            mem_memctl += cfb(ctb(GetPerfResults("mem.vmmemctl.average", dc), "Kib"), args.unit)
            mem_swapped += cfb(ctb(GetPerfResults("mem.swapused.average", dc), "Kib"), args.unit)
        # do we want averages?
        # avg_mem_pct = mem_pct / len(datacenter)
        # avg_mem_overhead = mem_overhead / len(datacenter)
        # avg_mem_consumed = mem_consumed / len(datacenter)
        # avg_mem_memctl = mem_memctl / len(datacenter)
        # avg_mem_swapped = mem_swapped / len(datacenter)
    results_dict = {}
    perfdata_dict = {}
    if subcommand and subcommand in ALLOWED_COMMANDS["DATACENTER"]["MEM"]:
        # Subcommand specified - return only that value
        Log("Filtering by subcommand {}".format(subcommand), domain="DATACENTER_MEM CHECK")
        if subcommand == "USAGE":
            results_dict["usage"] = f"{mem_pct:.2f}%"
            perfdata_dict["mem_usage"] = (mem_pct, "%")
        elif subcommand == "USAGEMB":
            results_dict["usagemb"] = f"{mem_consumed:.2f} {args.unit}"
            perfdata_dict["mem_usagemb"] = (mem_consumed, args.unit)
        elif subcommand == "OVERHEAD":
            results_dict["overhead"] = f"{mem_overhead:.2f} {args.unit}"
            perfdata_dict["mem_overhead"] = (mem_overhead, args.unit)
        elif subcommand == "SWAP":
            results_dict["swapped"] = f"{mem_swapped:.2f} {args.unit}"
            perfdata_dict["mem_swap"] = (mem_swapped, args.unit)
        elif subcommand == "MEMCTL":
            results_dict["memctl"] = f"{mem_memctl:.2f} {args.unit}"
            perfdata_dict["mem_memctl"] = (mem_memctl, args.unit)
        elif subcommand == "OVERALL":
            overall = mem_consumed + mem_overhead
            results_dict["overall"] = f"{overall:.2f} {args.unit}"
            perfdata_dict["mem_overall"] = (overall, args.unit)
    elif subcommand and subcommand not in ALLOWED_COMMANDS["DATACENTER"]["MEM"]:
        nagios_exit(UNK, f"Invalid subcommand {subcommand} for DATACENTER_MEM CHECK")
    else:
        # No subcommand specified - return all values
        results_dict["mem usage"] = f"{mem_consumed} {args.unit} ({mem_pct:.2f}%)"
        results_dict["overhead"] = f"{mem_overhead:.2f} {args.unit}"
        results_dict["swapped"] = f"{mem_swapped:.2f} {args.unit}"
        results_dict["memctl"] = f"{mem_memctl:.2f} {args.unit}"
        perfdata_dict["mem_usagemb"] = (f"{mem_consumed:.2f}", args.unit)
        perfdata_dict["mem_usage"] = (mem_pct, "%")
        perfdata_dict["mem_overhead"] = (mem_overhead, args.unit)
        perfdata_dict["mem_swap"] = (mem_swapped, args.unit)
        perfdata_dict["mem_memctl"] = (mem_memctl, args.unit)
        
    Log(f"Results dict: {results_dict}", domain="DATACENTER_MEM CHECK")
    output = format_output(
        results_dict
    )
    perfdata = format_perfdata(
        perfdata_dict
    )
    nagios_exit(check_thresholds(perfdata_dict), output + "| " + perfdata)

# Matches -D <datacenter> -l NET
def datacenter_net_info(datacenter, subcommand):
    Log("Getting network info for datacenter {}...".format(datacenter.name), domain="DATACENTER_NET CHECK")
    results_dict = {}
    perfdata_dict = {}
    perfunit = (f"{args.unit}ps" if args.extra_data else "")

    if subcommand and subcommand in ALLOWED_COMMANDS["DATACENTER"]["NET"]:
        # Subcommand specified - return only that value
        Log("Filtering by subcommand {}".format(subcommand), domain="DATACENTER_NET CHECK")
        if subcommand == "USAGE":
            results_dict["net usage"] = f"{cfb(ctb(GetPerfResults('net.usage.average', datacenter), 'KB'), args.unit):.2f} {args.unit}"
            perfdata_dict["net_usage"] = (cfb(ctb(GetPerfResults('net.usage.average', datacenter), 'KB'), args.unit), perfunit)
        elif subcommand == "RECEIVE":
            results_dict["net received"] = f"{cfb(ctb(GetPerfResults('net.received.average', datacenter), 'KB'), args.unit):.2f} {args.unit}"
            perfdata_dict["net_receive"] = (cfb(ctb(GetPerfResults('net.received.average', datacenter), 'KB'), args.unit), perfunit)
        elif subcommand == "SEND":
            results_dict["send"] = f"{cfb(ctb(GetPerfResults('net.transmitted.average', datacenter), 'KB'), args.unit):.2f} {args.unit}"
            perfdata_dict["net_send"] = (cfb(ctb(GetPerfResults('net.transmitted.average', datacenter), 'KB'), args.unit), perfunit)
    elif subcommand and subcommand not in ALLOWED_COMMANDS["DATACENTER"]["NET"]:
        nagios_exit(UNK, f"Invalid subcommand {subcommand} for DATACENTER_NET CHECK")
    else:
        # No subcommand specified - return all values
        results_dict["net received"] = f"{cfb(ctb(GetPerfResults('net.received.average', datacenter), 'KB'), args.unit):.2f} {args.unit}ps"
        results_dict["send"] = f"{cfb(ctb(GetPerfResults('net.transmitted.average', datacenter), 'KB'), args.unit):.2f} {args.unit}ps"
        perfdata_dict["net_receive"] = (cfb(ctb(GetPerfResults('net.received.average', datacenter), 'KB'), args.unit), perfunit)
        perfdata_dict["net_send"] = (cfb(ctb(GetPerfResults('net.transmitted.average', datacenter), 'KB'), args.unit), perfunit)
        
    Log(f"Results dict: {results_dict}", domain="DATACENTER_NET CHECK")
    
    output = format_output(
        results_dict
    )
    perfdata = format_perfdata(
        perfdata_dict
    )
    nagios_exit(check_thresholds(perfdata_dict), output + "| " + perfdata)

# Matches -D <datacenter> -l IO
def datacenter_io_info(datacenter, subcommand):
    results_dict = {}
    perfdata_dict = {}
    if type(datacenter) == vim.Datacenter:  
        Log("Getting IO info for datacenter {}...".format(datacenter.name), domain="DATACENTER_IO CHECK")

        commands_aborted = f"{GetPerfResults('disk.commandsAborted.summation', datacenter)}"
        bus_resets = f"{GetPerfResults('disk.busResets.summation', datacenter)}"
        read_latency = f"{GetPerfResults('disk.totalReadLatency.average', datacenter)}"
        write_latency = f"{GetPerfResults('disk.totalWriteLatency.average', datacenter)}"
        kernel_latency = f"{GetPerfResults('disk.kernelLatency.average', datacenter)}"
        device_latency = f"{GetPerfResults('disk.deviceLatency.average', datacenter)}"
        queue_latency = f"{GetPerfResults('disk.queueLatency.average', datacenter)}"
    else:
        commands_aborted = 0
        bus_resets = 0
        read_latency = 0
        write_latency = 0
        kernel_latency = 0
        device_latency = 0
        queue_latency = 0
        for dc in datacenter:
            Log("Getting IO info for datacenter {}...".format(dc.name), domain="DATACENTER_IO CHECK")
            commands_aborted += GetPerfResults('disk.commandsAborted.summation', dc)
            bus_resets += GetPerfResults('disk.busResets.summation', dc)
            read_latency += GetPerfResults('disk.totalReadLatency.average', dc)
            write_latency += GetPerfResults('disk.totalWriteLatency.average', dc)
            kernel_latency += GetPerfResults('disk.kernelLatency.average', dc)
            device_latency += GetPerfResults('disk.deviceLatency.average', dc)
            queue_latency += GetPerfResults('disk.queueLatency.average', dc)
        # do we want averages?
        # avg_commands_aborted = commands_aborted / len(datacenter)
        # avg_bus_resets = bus_resets / len(datacenter)
        # avg_read_latency = read_latency / len(datacenter)
        # avg_write_latency = write_latency / len(datacenter)
        # avg_kernel_latency = kernel_latency / len(datacenter)
        # avg_device_latency = device_latency / len(datacenter)
        # avg_queue_latency = queue_latency / len(datacenter)
        # do we want maximums?
            
    if subcommand and subcommand in ALLOWED_COMMANDS["DATACENTER"]["IO"]:
        # Subcommand specified - return only that value
        Log("Filtering by subcommand {}".format(subcommand), domain="DATACENTER_IO CHECK")
        if subcommand == "ABORTED":
            results_dict["commands aborted"] = f"{commands_aborted}"
            perfdata_dict["io_aborted"] = (commands_aborted, "")
        elif subcommand == "RESETS":
            results_dict["bus resets"] = f"{bus_resets}"
            perfdata_dict["io_busresets"] = (bus_resets, "")
        elif subcommand == "READ":
            results_dict["read latency"] = f"{read_latency} ms"
            perfdata_dict["io_read"] = (read_latency, "ms")
        elif subcommand == "WRITE":
            results_dict["write latency"] = f"{write_latency} ms"
            perfdata_dict["io_write"] = (write_latency, "ms")
        elif subcommand == "KERNEL":
            results_dict["kernel latency"] = f"{kernel_latency} ms"
            perfdata_dict["io_kernel"] = (kernel_latency, "ms")
        elif subcommand == "DEVICE":
            results_dict["device latency"] = f"{device_latency} ms"
            perfdata_dict["io_device"] = (device_latency, "ms")
        elif subcommand == "QUEUE":
            results_dict["queue latency"] = f"{queue_latency} ms"
            perfdata_dict["io_queue"] = (queue_latency, "ms")
    elif subcommand and subcommand not in ALLOWED_COMMANDS["DATACENTER"]["IO"]:
        nagios_exit(UNK, f"Invalid subcommand {subcommand} for DATACENTER_IO CHECK")
    else:
        # No subcommand specified - return all values

        # Legacy data - not supported # TODO: figure out if it's possible to get this data and if not, remove it
        # results_dict["commands aborted"] = f"{commands_aborted}"
        # results_dict["bus resets"] = f"{bus_resets}"
        # results_dict["read latency"] = f"{read_latency} ms"
        # results_dict["write latency"] = f"{write_latency} ms"
        # results_dict["kernel latency"] = f"{kernel_latency} ms"
        # results_dict["device latency"] = f"{device_latency} ms"
        # results_dict["queue latency"] = f"{queue_latency} ms"

        # perfdata_dict["io_aborted"] = (commands_aborted, "")
        # perfdata_dict["io_busresets"] = (bus_resets, "")
        # perfdata_dict["io_read"] = (read_latency, "ms")
        # perfdata_dict["io_write"] = (write_latency, "ms")
        # perfdata_dict["io_kernel"] = (kernel_latency, "ms")
        # perfdata_dict["io_device"] = (device_latency, "ms")
        # perfdata_dict["io_queue"] = (queue_latency, "ms")
        # nagios_exit(CRIT, "Extra data must be enabled to display new IO values (-e). Legacy data is not supported.")
        nagios_exit(CRIT, "IO data is not supported for datacenter level checks at this time.")

    Log(f"Results dict: {results_dict}", domain="DATACENTER_IO CHECK")
    output = format_output(
        results_dict
    )
    perfdata = format_perfdata(
        perfdata_dict
    )
    nagios_exit(check_thresholds(perfdata_dict), output + "| " + perfdata)

# Matches -D <datacenter> -l VMFS
def datacenter_vmfs_info(datacenter, subcommand):
    if type(datacenter) == vim.Datacenter:
        Log("Getting VMFS info for datacenter {}...".format(datacenter.name), domain="DATACENTER_VMFS CHECK")
        datastore = {}
        for ds in datacenter.datastore:
            if ds.summary.type == "VMFS":
                datastore[ds] = {
                    "name": ds.name,
                    "capacity": cfb(ctb(ds.summary.capacity, "B"), args.unit),
                    "free": cfb(ctb(ds.summary.freeSpace, "B"), args.unit),
                    "pct_free": round(
                        (ds.summary.freeSpace / ds.summary.capacity) * 100, 2
                    ),
                }
    else:
        for dc in datacenter:
            Log("Getting VMFS info for datacenter {}...".format(dc.name), domain="DATACENTER_VMFS CHECK")
            datastore = {}
            for ds in dc.datastore:
                if ds.summary.type == "VMFS":
                    datastore[ds] = {
                        "name": ds.name,
                        "capacity": cfb(ctb(ds.summary.capacity, "B"), args.unit),
                        "free": cfb(ctb(ds.summary.freeSpace, "B"), args.unit),
                        "pct_free": round(
                            (ds.summary.freeSpace / ds.summary.capacity) * 100, 2
                        ),
                    }
    results_dict = {}
    perfdata_dict = {}
    if subcommand and subcommand in ALLOWED_COMMANDS["DATACENTER"]["VMFS"]:
        print("Warning: subcommands are not tested for datacenter level VMFS checks")
        # Subcommand specified - return only that value
        Log("Filtering by subcommand {}".format(subcommand), domain="DATACENTER_VMFS CHECK")
        if subcommand == "USAGE":
            for ds in datastore:
                results_dict[datastore[ds]["name"]] = f"{datastore[ds]['pct_free']}%"
                perfdata_dict[datastore[ds]["name"]] = (datastore[ds]['pct_free'], "%")
        elif subcommand == "FREE":
            for ds in datastore:
                results_dict[datastore[ds]["name"]] = f"{datastore[ds]['free']:.2f} {args.unit}"
                perfdata_dict[datastore[ds]["name"]] = (datastore[ds]['free'], f"{args.unit}")
        elif subcommand == "CAPACITY":
            for ds in datastore:
                results_dict[datastore[ds]["name"]] = f"{datastore[ds]['capacity']:.2f} {args.unit}"
                perfdata_dict[datastore[ds]["name"]] = (datastore[ds]['capacity'], f"{args.unit}")
    elif subcommand and subcommand not in ALLOWED_COMMANDS["DATACENTER"]["VMFS"]:
        nagios_exit(UNK, f"Invalid subcommand {subcommand} for DATACENTER_VMFS CHECK")
    else:
        # No subcommand specified - return all values
        Log("No subcommand specified - returning all values", domain="DATACENTER_VMFS CHECK")
        for ds in datastore:
            percent_free = f"{datastore[ds]['pct_free']}"
            results_dict[datastore[ds]["name"] + "(free)"] = f"{datastore[ds]['free']:.2f} {args.unit} ({percent_free}%)"
            perfdata_dict[datastore[ds]["name"]] = (datastore[ds]['free'], f"{args.unit}")
            
    Log(f"Results dict: {results_dict}", domain="DATACENTER_VMFS CHECK")
    output = format_output(
        results_dict
    )
    perfdata = format_perfdata(
        perfdata_dict
    )
    nagios_exit(check_thresholds(perfdata_dict), output + "| " + perfdata)

# Matches -D <datacenter> -l RUNTIME
def datacenter_runtime_info(datacenter, subcommand, no_exit=False):
    if type(datacenter) == vim.Datacenter:
        Log("Getting runtime info for datacenter {}...".format(datacenter.name), domain="DATACENTER_RUNTIME CHECK")
        config_issues = len(datacenter.configIssue)
        dc_overall_status = datacenter.overallStatus
        host_view = _content.viewManager.CreateContainerView(datacenter.hostFolder, [vim.HostSystem], True)
        vm_view = _content.viewManager.CreateContainerView(datacenter.vmFolder, [vim.VirtualMachine], True)
        hosts = host_view.view
        vms = vm_view.view
        def human_power_state(powerState): # maybe move this outside
            if powerState == 'poweredOn': return '(UP)'
            if powerState == 'poweredOff': return '(DOWN)'
            return '(%s)' % powerState
        vmslist = [vm.name + ' ' + human_power_state(vm.runtime.powerState) for vm in vms]
        vms_up = len([1 for vm in vms if vm.runtime.powerState == 'poweredOn'])
        vms_down = len([1 for vm in vms if vm.runtime.powerState == 'poweredOff'])
        hosts_up = len([1 for host in hosts if host.runtime.connectionState == 'connected'])
        hosts_down = len(hosts) - hosts_up
    else:
        vms_up = 0
        vms_down = 0
        hosts_up = 0
        hosts_down = 0
        config_issues = 0
        dc_overall_status = []
        for dc in datacenter:
            Log("Getting runtime info for datacenter {}...".format(dc.name), domain="DATACENTER_RUNTIME CHECK")
            status = dc.overallStatus
            config_issues += len(dc.configIssue)
            host_view = _content.viewManager.CreateContainerView(dc.hostFolder, [vim.HostSystem], True)
            vm_view = _content.viewManager.CreateContainerView(dc.vmFolder, [vim.VirtualMachine], True)
            hosts = host_view.view
            vms = vm_view.view
            vmslist = []
            hostlist = []
            for vm in vms:
                if vm.runtime.powerState == "poweredOn":
                    vms_up += 1
                    vmslist.append(vm.name + " (UP)")
                else:
                    vms_down += 1
                    if vm.runtime.powerState == "poweredOff":
                        vmslist.append(vm.name + " (DOWN)")
                    else:
                        vmslist.append(vm.name + " (" + vm.runtime.powerState + ")")
            for host in hosts:
                if host.runtime.connectionState == "connected":
                    hosts_up += 1
                    hostlist.append(host.name + " (UP)")
                else:
                    hosts_down += 1
                    hostlist.append(host.name + " (" + host.runtime.connectionState + ")")
            dc_overall_status.append(status)
            
            
    results_dict = {}
    perfdata_dict = {}
    if subcommand and subcommand in ALLOWED_COMMANDS["DATACENTER"]["RUNTIME"]:
        # Subcommand specified - return only that value
        Log("Filtering by subcommand {}".format(subcommand), domain="DATACENTER_RUNTIME CHECK")
        if subcommand == "LIST" or subcommand == "LISTVM":
            vmslist = ", ".join(vmslist)
            results_dict["VMs"] = f"{vmslist}"
            perfdata_dict["vmcount"] = (vms_up, "units")
        elif subcommand == "LISTHOST":
            hostlist = ", ".join(hostlist)
            results_dict["Hosts"] = f"{hostlist}"
            perfdata_dict["hostcount"] = (hosts_up, "units")
        elif subcommand == "LISTCLUSTER":
            results_dict["state"] = f"{status}"
            perfdata_dict["runtime_state"] = (status, "")
        elif subcommand == "STATUS":
            nagios_exit(UNK, f"Subcommand {subcommand} not implemented for DATACENTER_RUNTIME CHECK")
        elif subcommand == "ISSUES":
            nagios_exit(UNK, f"Subcommand {subcommand} not implemented for DATACENTER_RUNTIME CHECK")
        elif subcommand == "TOOLS":
            nagios_exit(UNK, f"Subcommand {subcommand} not implemented for DATACENTER_RUNTIME CHECK")
    elif subcommand and subcommand not in ALLOWED_COMMANDS["DATACENTER"]["RUNTIME"]:
        nagios_exit(UNK, f"Invalid subcommand {subcommand} for DATACENTER_RUNTIME CHECK")
    else:
        # No subcommand specified - return all values
        results_dict["VMs up"] = f"{vms_up}/{vms_up + vms_down}"
        results_dict["Hosts up"] = f"{hosts_up}/{hosts_up + hosts_down}"
        results_dict["Datacenter overall statuses"] = f"{dc_overall_status}"
        results_dict["config issues"] = f"{config_issues}"
        perfdata_dict["vmcount"] = (vms_up, "units")
        if args.datacenter is not None or args.extra_data is True:
            perfdata_dict["hostcount"] = (hosts_up, "units")
            perfdata_dict["config_issues"] = (config_issues, "")
        
    Log(f"Results dict: {results_dict}", domain="DATACENTER_RUNTIME CHECK")
    output = format_output(
        results_dict
    )
    perfdata = format_perfdata(
        perfdata_dict
    )
    if no_exit:
        return results_dict, perfdata_dict
    nagios_exit(check_thresholds(perfdata_dict), output + "| " + perfdata)
#endregion Datacenter checks

def list_datastores(host):
    datastore = {}
    try:
        for h in host:
            Log("Getting VMFS info for host {}...".format(h.name), domain="HOST_VMFS CHECK")
            for ds in h.datastore:
                if ds.summary.type == "VMFS" and ds.summary.accessible:
                    datastore[h.name + "-" + ds.name] = {
                        "name": h.name + "-" + ds.name,
                        "capacity": cfb(ctb(ds.summary.capacity, "B"), args.unit),
                        "free": cfb(ctb(ds.summary.freeSpace, "B"), args.unit),
                        "pct_free": round(
                            (ds.summary.freeSpace / ds.summary.capacity) * 100, 2
                        ),
                    }
    except:
        Log("Getting VMFS info for host {}...".format(host.name), domain="HOST_VMFS CHECK")
        for ds in host.datastore:
            if ds.summary.type == "VMFS":
                datastore[ds] = {
                    "name": ds.name,
                    "capacity": cfb(ctb(ds.summary.capacity, "B"), args.unit),
                    "free": cfb(ctb(ds.summary.freeSpace, "B"), args.unit),
                    "pct_free": round(
                        (ds.summary.freeSpace / ds.summary.capacity) * 100, 2
                    ),
                }
    outputs = []
    for ds in datastore:
        outputs.append(
                f"{datastore[ds]['name']}\t{datastore[ds]['free']:.2f} {args.unit}\t{datastore[ds]['capacity']:.2f} {args.unit}\t({datastore[ds]['pct_free']}%)",
            )
    output = "\n".join(outputs)
    print(output)         

def list_datastores_helper(host):
    datastore = {}
    for ds in host.datastore:
        if ds.summary.type == "VMFS":
            datastore[ds] = {
                "name": ds.name,
                "capacity_mb": ds.summary.capacity / 1024 / 1024,
                "free_mb": ds.summary.freeSpace / 1024 / 1024,
                "used_mb": ds.summary.capacity / 1024 / 1024 - ds.summary.freeSpace / 1024 / 1024,
                "pct_used": round(
                    ((ds.summary.capacity - ds.summary.freeSpace) / ds.summary.capacity) * 100, 2
                ),
            }
    return datastore
    
# TODO: make these work dynamically so it can work properly for other structures and with multiple datacenters
def list_datacenters(content):
    datacenters = []
    datacentersview = content.viewManager.CreateContainerView(content.rootFolder, [vim.Datacenter], True)
    for datacenter in datacentersview.view:
        datacenters.append(datacenter.name)
    print("\n".join(datacenters))
    
def list_clusters(content, datacenter):
    clusters = []
    if datacenter is not None:
        datacenterview = content.viewManager.CreateContainerView(content.rootFolder, [vim.Datacenter], True)
        for dc in datacenterview.view:
            if dc.name == datacenter:
                datacenterview = dc
                break
        clusterviews = content.viewManager.CreateContainerView(datacenterview.hostFolder, [vim.ClusterComputeResource], True)
        for cluster in clusterviews.view:
            clusters.append(cluster.name)
    else:
        clusterviews = content.viewManager.CreateContainerView(content.rootFolder, [vim.ClusterComputeResource], True)
        for cluster in clusterviews.view:
            clusters.append(cluster.name)
    print("\n".join(clusters))



# If help is requested, print it and exit
if hasattr(args, "help"):
    print(usage)
    sys.exit(0)
#endregion

#region Required Arguments
# ---------------------------------------------------------------------------- #
#                              Required Arguments                              #
# ---------------------------------------------------------------------------- #

# Make sure we have a host or a datacenter
if args.address:
    Log("Setting address to match address: {}".format(args.address), domain="INIT")
    _address = args.address
elif args.host:
    Log("Setting address to match host: {}".format(args.host), domain="INIT")
    _address = args.host
elif args.datacenter:
    Log("Setting address to match datacenter: {}".format(args.datacenter), domain="INIT")
    _address = args.datacenter
else:
    Log("Failed to get host - tried {}".format(args.host), domain="INIT")
    nagios_exit(UNK, "No host specified")
if not _address:
    Log("Failed to get address - tried {}".format(args.address), domain="INIT")
    nagios_exit(UNK, "No address specified")

# Make sure we have either a username and password or an authfile
if not args.authfile and (not args.user or not args.password):
    nagios_exit(UNK, "No authentication information specified")

# Make sure we have a command unless we are getting guests
if not args.command and not _noCheck:
    nagios_exit(UNK, "No command specified")
else:  # Validate the command depending on the domain (VM vs Host vs Cluster)
    if not _noCheck:
        args.command = args.command.upper()
        if hasattr(args, "vm") and args.vm:
            Log("Received VM command - {}".format(args.command), domain="ARG VALIDATION")
            if args.command not in ALLOWED_COMMANDS["VM"]:
                Log("Error - Command: {}".format(args.command), domain="ARG VALIDATION")
                Log("Allowed commands: {}".format(ALLOWED_COMMANDS["VM"]), domain="ARG VALIDATION")
                nagios_exit(UNK, "Unknown HOST-VM command specified")
        elif hasattr(args, "cluster") and args.cluster:
            Log("Received cluster command - {}".format(args.command), domain="ARG VALIDATION")
            if args.command not in ALLOWED_COMMANDS["CLUSTER"]:
                Log("Error - Command: {}".format(args.command), domain="ARG VALIDATION")
                Log("Allowed commands: {}".format(ALLOWED_COMMANDS["CLUSTER"]), domain="ARG VALIDATION")
                nagios_exit(UNK, "Unknown HOST-CLUSTER command specified")
        elif hasattr(args, "host") and args.host:
            Log("Received host command - {}".format(args.command), domain="ARG VALIDATION")
            if args.command not in ALLOWED_COMMANDS["HOST"]:
                Log("Error - Command: {}".format(args.command), domain="ARG VALIDATION")
                Log("Allowed commands: {}".format(ALLOWED_COMMANDS["HOST"]), domain="ARG VALIDATION")
                nagios_exit(UNK, "Unknown HOST command specified")
        else:
            Log("Received datacenter command - {}".format(args.command), domain="ARG VALIDATION")
            if args.command not in ALLOWED_COMMANDS["DATACENTER"]:
                Log("Error - Command: {}".format(args.command), domain="ARG VALIDATION")
                Log("Allowed commands: {}".format(ALLOWED_COMMANDS["DATACENTER"]), domain="ARG VALIDATION")
                nagios_exit(UNK, "Unknown command specified")


# If we were given an authfile, make sure it exists and then read it into the
# username and password variables
if args.authfile:
    Log("Authfile specified - {}".format(args.authfile), domain="AUTHFILE")
    Log("Reading authfile", domain="AUTHFILE")
    if not os.path.isfile(args.authfile):
        nagios_exit(UNK, "Authfile does not exist")
    else:
        with open(args.authfile) as f:
            read = 0
            for line in f:
                if line.startswith("username="):
                    username = line.split("=", 1)[1].rstrip()
                    if username == "":
                        nagios_exit(UNK, "No username in authfile")
                    else:
                        Log("Found username - {}".format(username), domain="AUTHFILE")
                        read += 1
                        args.user = username
                elif line.startswith("password="):
                    password = line.split("=", 1)[1].rstrip()
                    if password == "":
                        nagios_exit(UNK, "No password in authfile")
                    else:
                        Log("Found password", domain="AUTHFILE")
                        read += 1
                        args.password = password
            if read != 2:
                nagios_exit(
                    UNK,
                    "Authfile must contain username and password in the format of: \n\tusername=<USERNAME>\n\tpassword=<PASSWORD>",
                )
#endregion

#region Setup
# ---------------------------------------------------------------------------- #
#                                     Setup                                    #
# ---------------------------------------------------------------------------- #
# If SSL is enabled, try to get the certificate from the host and add it to
# the SSL context
cert=None
ssl_context=None
if args.unverified_ssl:
    Log("SSL enabled", domain="SSL")
    Log("Creating unverified SSL context", domain="SSL")
    ssl_context = ssl._create_unverified_context()
    Log("SSL context created", domain="SSL")
    if _verbose:
        Log("SSL context: {}".format(ssl_context), domain="SSL")
elif not args.no_ssl:
    Log("SSL enabled", domain="SSL")
    Log("Creating secure connection to vCenter server", domain="SSL")
    Log("Server: {}".format(_address), domain="SSL")
    try:
        ssl_context = ssl.create_default_context()
        ssl_context.check_hostname = False
        ssl_context.verify_mode = ssl.CERT_NONE
        s = ssl_context.wrap_socket(socket(), server_hostname=_address)
        try:
            Log("Attempting to connect to vCenter server", domain="SSL")
            s.connect((_address, args.port))
        except:
            nagios_exit(CRIT, "Could not connect to the specified vCenter server.")
        try:
            Log("Attempting to get SSL certificate from vCenter server", domain="SSL")
            bcert = s.getpeercert(binary_form=True)
            cert = ssl.DER_cert_to_PEM_cert(bcert)
        except:
            nagios_exit(CRIT, "Could not get SSL certificate from vCenter server.")
        Log("Adding SSL certificate to SSL context", domain="SSL")
        ssl_context.load_verify_locations(cadata=cert)
        if _verbose:
            # Display certificate information - issuer, subject, expiry
            Log("Certificate information:", domain="SSL")
            f = open("cert.pem", "w")
            f.write(cert)
            f.close()
            cert = ssl._ssl._test_decode_cert("cert.pem")
            subject = dict(x[0] for x in cert["subject"])
            issued_to = subject["commonName"]
            issuer = dict(x[0] for x in cert["issuer"])
            issuer_country = issuer["countryName"]
            issuer_org = issuer["organizationName"]
            issued_by = issuer["commonName"]
            issuer_name = issuer["organizationalUnitName"]
            expiry = datetime.strptime(cert["notAfter"], "%b %d %H:%M:%S %Y %Z")
            Log(f"Issued to: {issued_to}", domain="SSL")
            Log(f"Issued by: {issued_by}", domain="SSL")
            Log(f"Issuer name: {issuer_name}", domain="SSL")
            Log(f"Issuer country: {issuer_country}", domain="SSL")
            Log(f"Issuer org: {issuer_org}", domain="SSL")
            Log(f"Expiry: {expiry}\n", domain="SSL")
            if issuer_org == "localhost":
                Log("This certificate was issued by localhost. This is likely a self-signed certificate.\n", domain="SSL")
    except Exception as e:
        Log(f"Exception during SSL setup: {e}", domain="SSL")
        nagios_exit(
            CRIT,
            f"Failed to create secure connection to vCenter server: {e}. "
            "Try the --unverified-ssl or --no-ssl flag if you are using a self-signed certificate, "
            "or verify that the server's SSL certificate is valid and trusted."
        )
else:
    Log("--no-ssl specified, not using SSL", domain="SSL")
    ssl_context = None

# Test authentication and connection with vSphere
Log("Creating service instance", domain="CONNECTION")
service_instance = vim.ServiceInstance("ServiceInstance")
try:
    Log(f"Attempting connection with the following parameters: \nhost={_address}\nuser={args.user}\npwd=********\nport={args.port}\ndisableSslCertValidation={(not args.no_ssl)}\n", domain="CONNECTION")
    Log("Getting pyvmomi version", domain="CONNECTION")
    try:
        pyvmomiVersion = distribution('pyVmomi').version
    except:
        pyvmomiVersion = pkg_resources.get_distribution("pyvmomi").version

    if (int(pyvmomiVersion.split(".")[0]) < 8):
        if (args.no_ssl):
            service_instance = connect.SmartConnectNoSSL(
                host=_address,
                user=args.user,
                pwd=args.password,
                port=args.port,
            )
        else:
            service_instance = connect.SmartConnect(
                host=_address,
                user=args.user,
                pwd=args.password,
                port=args.port,
                sslContext=ssl_context,
            )
    else:
        service_instance = connect.SmartConnect(
            host=_address,
            user=args.user,
            pwd=args.password,
            port=args.port,
            disableSslCertValidation=(args.no_ssl),
            sslContext=ssl_context,
        )

    if not service_instance:
        Log("Failed to create service instance - no service instance returned from SmartConnect", domain="CONNECTION")
        nagios_exit(CRIT, "Could not connect to the specified vCenter server.")
except Exception as e:
    Log("Failed to create service instance - {}".format(e), domain="CONNECTION")
    if "Permission to perform this operation was denied" in str(e):
        nagios_exit(CRIT, "Permission denied. Check username and password.")
    elif "Cannot complete login due to an incorrect user name or password" in str(e):
        nagios_exit(CRIT, "Incorrect username or password. Check credentials.")
    elif "certificate verify failed" in str(e):
        nagios_exit(CRIT, "Certificate verification failed. Check certificate.")
    elif "timed out" in str(e):
        nagios_exit(CRIT, "Connection timed out. Check host and port.")
    elif "No route to host" in str(e):
        nagios_exit(CRIT, "No route to host. Check host and port.")
    else:    
        import re
        match = re.search(r"msg\s*=\s*'([^']*)'", str(e))
        msg_text = match.group(1) if match else str(e)
        nagios_exit(CRIT, f"Could not connect to the specified vCenter server, '{msg_text}'")

# Get the content of the service instance
Log("Retrieving content from service instance", domain="CONNECTION")
_content = service_instance.RetrieveContent()
# Log("Content retrieved - {}".format(_content), domain="CONNECTION")
    
Log("Retrieving container view from content...", domain="CONTAINER")
_container = get_container_view(_content)

if not _container:
    Log(f"Failed to get container using host {args.host} - container is empty.", domain="CONTAINER")
    Log(f"This is not necessarily an error, it may just mean that the specified host is a vCenter VM.", domain="CONTAINER")
else:
    Log(f"Container retrieved.", domain="CONTAINER")
    
Log(f"Getting current time from service instance", domain="VIRT_TIME")
_virt_time = service_instance.CurrentTime()
Log(f"Current time set - {_virt_time}", domain="VIRT_TIME")

# Get a list of all performance counters
_perf_dict = {}
perfList = _content.perfManager.perfCounter
for counter in perfList:
    counter_full = "{}.{}.{}".format(
        counter.groupInfo.key, counter.nameInfo.key, counter.rollupType
    )
    _perf_dict[counter_full] = counter.key
num_perf_counters = len(_perf_dict)
Log(f"Built dictionary of {num_perf_counters} performance counters", domain="PERF")


if not _container:
    Log("Container is empty, attempting to correct. This generally implies that the specified host is a vCenter VM.", domain="CONTAINER")
    # We get here if we are using the vCenter VM as Host, what we really
    # want is the host that vCenter is running on. Find the vCenter VM
    # and get the host it is running on
    vm_view = _content.viewManager.CreateContainerView(
        _content.rootFolder, [vim.VirtualMachine], True
    )
    Log("Retrieving VM view from root folder", domain="CONTAINER")
    vms = vm_view.view
    Log("Looping through VMs to find vCenter VM", domain="CONTAINER")
    found=False
    for vm in vms:
        if found:
            break
        Log("Found a VM - {}".format(vm.name), domain="CONTAINER")
        # Match based on IP of Host
        if vm.summary.guest.ipAddress == _address:
            found=True
            Log("Found vCenter VM by IP address - {}".format(vm.name), domain="CONTAINER")
            _container = vm.runtime.host.parent
            Log("Set container to parent of vCenter VM - {}".format(_container), domain="CONTAINER")
            # Set the args.host to the IP of the host
            args.host = vm.runtime.host.name
            Log("Set args.host to the IP of the vCenter VM - {}".format(args.host), domain="CONTAINER")
            break
    if not _container:
        Log("Still unable to find a container, exiting", domain="CONTAINER")
        nagios_exit(CRIT, "Could not find the specified resource.")
#endregion

# ---------------------------------------------------------------------------- #
#                                 Match Command                                #
# ---------------------------------------------------------------------------- #

if _noCheck:
# If a list of guests is requested, get it and exit
    if hasattr(args, "get_guests") and args.get_guests:
        Log("Getting list of guests", domain="GUESTS")
        # Get a list of all VMs
        vm_view = _content.viewManager.CreateContainerView(
            _content.rootFolder, [vim.VirtualMachine], True
        )
        vms = vm_view.view
        Log("Looping through VMs to get guest information", domain="GUESTS")
        for vm in vms:
            Log("Found a VM - {} - outputting guest information...".format(vm.name), domain="GUESTS")
            print(vm.summary.config.name, end="\t")
            print(vm.summary.config.guestFullName, end="\t")
            print(vm.summary.guest.ipAddress, end="\t")
            print(vm.summary.runtime.powerState, end="\n")
        sys.exit(0)
    elif hasattr(args, "list_datastores") and args.list_datastores:
        Log("Getting list of datastores", domain="LIST_DATASTORES")
        list_datastores(_container)
        sys.exit(0)
    elif hasattr(args, "list_datacenters") and args.list_datacenters:
        Log("Getting list of datacenters", domain="LIST_DATACENTERS")
        list_datacenters(_content)
        sys.exit(0)
    elif hasattr(args, "list_clusters") and args.list_clusters:
        Log("Getting list of clusters", domain="LIST_CLUSTERS")
        if hasattr(args, "datacenter") and args.datacenter:
            list_clusters(_content, args.datacenter)
        else:
            list_clusters(_content, None)
        sys.exit(0)
    elif hasattr(args, "list_commands") and args.list_commands:
        list_commands()
    else:
        Log("No command specified, exiting", domain="NO CHECK")
        nagios_exit(UNK, "No command specified")

def call_command(object, command):
    if hasattr(args, "list_datastores"):
        list_datastores(object)
        sys.exit(0)
    function_name = f"{object}_{command.lower()}_info"
    function = globals().get(function_name)
    if function:
        function(object, args.subcommand)
    else:
        nagios_exit(UNK, f"Unknown command specified - {command}")

def check_command():
    for domain in ["VM", "HOST", "CLUSTER", "DATACENTER"]:
        if hasattr(args, domain.lower()) and getattr(args, domain.lower()):
            Log(f"{domain} specified - {getattr(args, domain.lower())}", domain=domain)
            if domain == "HOST" and args.address is not None:
                try:
                    hostname = _container.name
                except:
                    nagios_exit(UNK, f"Could not find the specified {domain}")
            elif getattr(args, domain.lower()) != _container.name:
                nagios_exit(UNK, f"Could not find the specified {domain}")
            call_command(_container, args.command)
            break

# check_command() # not tested yet and the other method is working for now

# VM
if hasattr(args, "vm") and args.vm:
    Log("VM specified - {}".format(args.vm), domain="VM")
    vm = _container
    if args.vm != vm.name:
        nagios_exit(UNK, f"Could not find the specified VM - {args.vm}, found {vm.name} instead.")
    
    vm_power_state = vm.runtime.powerState
    if vm_power_state == "poweredOff":
        nagios_exit(CRIT, f"VM {vm.name} is powered off.")

    # Match commands
    if args.command == "CPU":
        Log("Matched CPU command", domain="VM")
        vm_cpu_info(vm, args.subcommand)
    elif args.command == "MEM":
        Log("Matched MEM command", domain="VM")
        vm_mem_info(vm, args.subcommand)
    elif args.command == "NET":
        Log("Matched NET command", domain="VM")
        vm_net_info(vm, args.subcommand)
    elif args.command == "IO":
        Log("Matched IO command", domain="VM")
        vm_io_info(vm, args.subcommand)
    elif args.command == "RUNTIME":
        Log("Matched RUNTIME command", domain="VM")
        vm_runtime_info(vm, args.subcommand)
    elif args.command == "VMFS":
        Log("Matched VMFS command", domain="VM")
        vm_vmfs_info(vm, args.subcommand)
    else:
        Log("Unknown command specified - {}".format(args.command), domain="VM")
        nagios_exit(UNK, "Unknown command specified")

elif hasattr(args, "cluster") and args.cluster:
    Log("Cluster specified - {}".format(args.cluster), domain="CLUSTER")
    cluster = _container

    # Check if the cluster is empty or power off
    if not cluster:
        nagios_exit(UNK, "Could not find the specified cluster.")
    if cluster.summary.effectiveCpu == 0:
        nagios_exit(CRIT, f"Cluster {cluster.name} is empty.")

    # TODO: Need to make sure we have a cluster
    if args.cluster != cluster.name:
        nagios_exit(UNK, "Could not find the specified cluster.")
    if args.command ==  "CPU":
        Log("Matched CPU command", domain="CLUSTER")
        cluster_cpu_info(cluster, args.subcommand)
    elif args.command == "MEM":
        Log("Matched MEM command", domain="CLUSTER")
        cluster_mem_info(cluster, args.subcommand)
    elif args.command == "CLUSTER":
        Log("Matched CLUSTER command", domain="CLUSTER")
        cluster_cluster_info(cluster, args.subcommand)
    elif args.command == "VMFS":
        Log("Matched VMFS command", domain="CLUSTER")
        cluster_vmfs_info(cluster, args.subcommand)
    elif args.command == "RUNTIME":
        Log("Matched RUNTIME command", domain="CLUSTER")
        cluster_runtime_info(cluster, args.subcommand)
    else:
        Log("Unknown command specified - {}".format(args.command), domain="CLUSTER")
        nagios_exit(UNK, "Cluster - Unknown command specified")

# Datacenter
elif hasattr(args, "datacenter") and args.datacenter:
    Log("Root folder: {}".format(_content.rootFolder), domain="DATACENTER")
    Log("_container: {}".format(_container), domain="DATACENTER")
    datacenter = _container
    if args.address is not None and args.datacenter != datacenter.name:
        nagios_exit(UNK, "Could not find the specified datacenter.")
    if args.command == "RECOMMENDATIONS":
        datacenter_recommendations_info(datacenter, args.subcommand)
    elif args.command == "CPU":
        datacenter_cpu_info(datacenter, args.subcommand)
    elif args.command == "MEM":
        datacenter_mem_info(datacenter, args.subcommand)
    elif args.command == "NET":
        datacenter_net_info(datacenter, args.subcommand)
    elif args.command == "IO":
        datacenter_io_info(datacenter, args.subcommand)
    elif args.command == "VMFS":
        datacenter_vmfs_info(datacenter, args.subcommand)
    elif args.command == "RUNTIME":
        datacenter_runtime_info(datacenter, args.subcommand)
    else:
        nagios_exit(UNK, "Datacenter - Unknown command specified")
# Host
elif hasattr(args, "host") and args.host:
    Log("Host specified - {}".format(args.host), domain="HOST")
    host = _container
    if args.address is not None:
        try:
            hostname = host.name
        except:
            nagios_exit(UNK, "Could not find the specified host.")
    if hasattr(args, "list_datastores") and args.list_datastores:
        Log("List datastores specified, listing datastores for host and exiting", domain="HOST")
        list_datastores(host)
        sys.exit(0)

    if args.command == "CPU":
        Log("Matched CPU command", domain="HOST")
        host_cpu_info(host, args.subcommand)
    elif args.command == "MEM":
        Log("Matched MEM command", domain="HOST")
        host_mem_info(host, args.subcommand)
    elif args.command == "NET":
        Log("Matched NET command", domain="HOST")
        host_net_info(host, args.subcommand)
    elif args.command == "IO":
        Log("Matched IO command", domain="HOST")
        host_io_info(host, args.subcommand)
    elif args.command == "VMFS":
        Log("Matched VMFS command", domain="HOST")
        host_vmfs_info(host, args.subcommand)
    elif args.command == "RUNTIME":
        Log("Matched RUNTIME command", domain="HOST")
        host_runtime_info(host, args.subcommand)
    elif args.command == "SERVICE":
        Log("Matched SERVICE command", domain="HOST")
        host_service_info(host, args.subcommand)
    elif args.command == "STORAGE":
        Log("Matched STORAGE command", domain="HOST")
        host_storage_info(host, args.subcommand)
    elif args.command == "UPTIME":
        Log("Matched UPTIME command", domain="HOST")
        host_uptime_info(host, args.subcommand)
    else:
        nagios_exit(UNK, "Host - Unknown command specified")


Log("No command specified, exiting", domain="CATASTROPHIC")
nagios_exit(UNK, "Failed to match search domain. Please specify a server [-A] and/or host [-H] along with any other specifiers and a command [-l].")
