<?php
//
// Mass Immediate Check Component
// Copyright (c) 2019-2020 Nagios Enterprises, LLC. All rights reserved.
//

require_once(dirname(__FILE__) . '/../../common.inc.php');

// Initialization stuff
pre_init();
init_session();

// Grab GET or POST variables and do prereq/auth checks
grab_request_vars();
check_prereqs();
check_authentication(false);

$title = _("Mass Immediate Check");
do_page_start(array("page_title" => $title), true);

$hostgroup = grab_request_var("hostgroup", "");
$servicegroup = grab_request_var("servicegroup", "");
$select_state = grab_request_var("selectstate", "");
$host = grab_request_var("host", "");

/////////////////////////////////////
// Limit by host/service group HTML
/////////////////////////////////////
?>

<script type="text/javascript">
$(document).ready(function () {

    if (!is_neptune()) {
        $('#hostList').searchable({maxMultiMatch: 9999});
        $('#hostgroupList').searchable({maxMultiMatch: 9999});
        $('#servicegroupList').searchable({maxMultiMatch: 9999});
    }

    $('#hostList').change(function () {
        $('#servicegroupList').val('');
        $('#hostgroupList').val('');
    });

    $('#hostgroupList').change(function () {
        $('#servicegroupList').val('');
        $('#hostList').val('');
    });

    $('#servicegroupList').change(function () {
        $('#hostgroupList').val('');
        $('#hostList').val('');
    });

    $('#selectAllButton:button').toggle(function(){
        $('input:checkbox').attr('checked','checked');
        $(this).val('Unselect All');
    },function(){
        $('input:checkbox').removeAttr('checked');
        $(this).val('Select All');
    });

    // change a parent checkbox
    $('.host.parent').change(function() {
        // grab the id and checked value
        const id = $(this).data('id');
        const checked = $(this).prop('checked');
        // toggle all the children with the same id
        $(`.host.child[data-id=${id}]`).prop('checked', checked || false);
    });

});
</script>

<form method="get">
    <div class="well report-options form-inline flex pl-0">
        <div class="input-group" style="margin-right: 10px;">
            <label class="input-group-addon"><?php echo _("Limit To"); ?></label>
            <select name="host" id="hostList" style="width: 150px;" class="rounded-l-none border-l form-control">
                <option value=""><?php echo _("Host"); ?>:</option>
                <?php
                $args = array('brevity' => 1, 'orderby' => 'host_name:a');
                $oxml = get_xml_host_objects($args);
                if ($oxml) {
                    foreach ($oxml->host as $hostobject) {
                        $name = strval($hostobject->host_name);
                        echo "<option value='" . $name . "' " . is_selected($host, $name) . ">$name</option>\n";
                    }
                }
                ?>
            </select>        
            <select name="hostgroup" id="hostgroupList" style="width: 150px;" class="rounded-l-none  form-control">
                <option value=""><?php echo _("Hostgroup"); ?>:</option>
                <?php
                $args = array('orderby' => 'hostgroup_name:a');
                $oxml = get_xml_hostgroup_objects($args);
                if ($oxml) {
                    foreach ($oxml->hostgroup as $hg) {
                        $name = strval($hg->hostgroup_name);
                        echo "<option value='" . $name . "' " . is_selected($hostgroup, $name) . ">$name</option>";
                    }
                }
                ?>
            </select>
            <select name="servicegroup" id="servicegroupList" style="width: 175px;" class="rounded-l-none rounded-r form-control">
                <option value=""><?php echo _("Servicegroup"); ?>:</option>
                <?php
                $args = array('orderby' => 'servicegroup_name:a');
                $oxml = get_xml_servicegroup_objects($args);
                if ($oxml) {
                    foreach ($oxml->servicegroup as $sg) {
                        $name = strval($sg->servicegroup_name);
                        echo "<option value='" . $name . "' " . is_selected($servicegroup, $name) . ">$name</option>";
                    }
                }
                ?>
            </select>
        </div>
        <div class="input-group flex" style="margin-right: 10px;">
            <label class="input-group-addon"><?php echo _("Status"); ?></label>
            <select name="selectstate" id="selectstate" style="width: 150px;" class="rounded-l-none rounded-r form-control">
                <?php
                echo "<option value='0' " . is_selected($select_state, 0) . ">"._("Show Problems")."</option>";
                echo "<option value='1' " . is_selected($select_state, 1) . ">"._("Show All")."</option>";
                ?>
            </select>
            <input type="submit" style="margin-left: 10px;" id="runButton" class='btn btn-sm btn-primary' name='runButton' value="<?php echo _("Update"); ?>">
        </div>
    </div>
</form>
<?php

/////////////////////////////////
// Filtering by group/state
/////////////////////////////////

$selectstate = grab_request_var("selectstate", 0);
$hostgroup = grab_request_var("hostgroup", "");
$servicegroup = grab_request_var("servicegroup", "");
$host = grab_request_var("host", "");

$hosts = mic_get_hosts();
$services = mic_get_services($selectstate);

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['forceCheckButton'])) {
        force_check();
    }
}

if (is_readonly_user(0)) {
    $html = _("You are not authorized to use this component.");
} else {
    $html = mic_build_html($hosts, $services);
}

print $html;

////////////////////////////////////////////////////////////////////////
//  FUNCTIONS
////////////////////////////////////////////////////////////////////////

function force_check()
{    
    global $cfg;
    $pipe = $cfg['component_info']['nagioscore']['cmd_file'];
    $output = array();
    $now = time();
    $service_command = 'SCHEDULE_FORCED_SVC_CHECK';
    $host_command = 'SCHEDULE_FORCED_HOST_CHECK';
        
    // Check if file exists before trying
    if (!file_exists($pipe)) {        
        echo feedback_html('The Nagios process is likely not running. Cannot connect to nagios.cmd pipe.', true);
        return;
    }

    // Set our locale for PHP
    // If we don't do this, escapeshellarg will remove all unicode.
    $locale = grab_array_var($_SESSION, 'language', 'en_US');
    if ($locale == 'en_EN' || $locale == 'en') {
        $locale = 'en_US';
    }

    if (FALSE == setlocale(LC_ALL, $locale.".UTF-8")) {
        return FALSE;
    }

    if (!empty($_POST['services'])) {
        foreach($_POST['services'] as $selected) {
            send_to_audit_log(_('Immediate check command sent for service').": ".str_replace(';', ' - ', $selected), AUDITLOGTYPE_INFO);
            $sc_string = "/bin/echo " . escapeshellarg("[$now] $service_command;$selected;$now") . " > $pipe";            
            exec($sc_string, $output, $returncode);
        }
    }

    if (!empty($_POST['hosts'])) {
        foreach($_POST['hosts'] as $selected) {
            send_to_audit_log(_('Immediate check command sent for host').": ".$selected, AUDITLOGTYPE_INFO);
            $hc_string = "/bin/echo " . escapeshellarg("[$now] $host_command;$selected;$now") . " > $pipe";            
            exec($hc_string, $output, $returncode);
        }
    }

    if (!empty($_POST['services']) || (!empty($_POST['hosts']))) {
        echo feedback_html('Check command submitted successfully.', false);
    } else {
        echo feedback_html('Please select at least one host/service.', true);
    }
}

function mic_build_html($hosts, $services)
{   
    $html = "
    <h1>"._("Mass Immediate Check")."</h1>
    <div class='text-medium'>"._("Use this tool to force an immediate check on large groups of hosts/services.")."</div>
    <form id='form_masscheck' action='index.php' method='post'>

    <div style='margin: 10px 0 0 0;'> 
    <input type='hidden' id='submitted' name='submitted' value='true' />
        <div style='padding: 20px 0 10px 0;'>                
            <input type='button' style='margin-right: 20px;' class='btn btn-sm btn-default fl' id='selectAllButton'  title='" . _('Select All Hosts and Services') . "' value='" . _("Select All") . "'>
            <input type='submit' class='btn btn-sm btn-primary' id='forceCheckButton' name='forceCheckButton' title='" . _('Force Check Selected') . "' value='" . _("Force Check") . "'>
        </div> 
        <div class='w-full'>
                

            <table class='table table-condensed table-striped table-bordered table-auto-width w-full' id='massforce_table' style='min-width: 750px;'>
                <thead>
                    <tr class='center'>
                        <th class='center'>" . _("Host") . "</th>
                        <th class='center'>" . _("Service") . "</th>
                        <th class='center'>" . _("Last Check") . "</th>
                        <th class='center'>" . _("Status Information") . "</th>
                    </tr>
                </thead>
                <tbody>";

    $hostcount = 0;
    foreach ($hosts as $host) {
        $host_class = host_class($host['host_state']);

        $service_html = "";

        // Verify we have services
        if (isset($services[$host['host_name']])) {
            foreach ($services[$host['host_name']] as $service) {
                if(is_neptune()){
                    $service_html .= "
                <tr>  
                    <td></td>
                    <td class=''>
                        <div class='checkbox'>
                                <label class='flex items-center'>
                                    <input class='mass-ack-service-box host child host{$hostcount}' data-id='{$hostcount}' type='checkbox' name='services[]' value='{$service['host_name']};{$service['service_description']}'>
                                    <div class='flex items-center'>
                                        <span class='status-dot dot-10 " . service_class($service['current_state']) . "'></span>
                                    </div>
                                    {$service['service_description']}
                            </label>
                        </div>
                    </td>
                    <td>
                        <div class='last_check'>{$service['last_check']}</div>
                    </td>
                    <td>
                        <div class='plugin_output scrollable-30'>{$service['plugin_output']}</div>
                    </td>
                </tr>";
                } else {
                    $service_html .= "
                <tr>  
                    <td></td>
                    <td class='" . service_class($service['current_state']) . "'>
                        <div class='checkbox'>
                            <label>
                                <input class='host child host{$hostcount}' data-id='{$hostcount}' type='checkbox' name='services[]' value='{$service['host_name']};{$service['service_description']}'>
                                {$service['service_description']}
                            </label>
                        </div>
                    </td>
                    <td>
                        <div class='last_check'>{$service['last_check']}</div>
                    </td>
                    <td>
                        <div class='plugin_output'>{$service['plugin_output']}</div>
                    </td>
                </tr>";
                }
            
            }
        }

        if ($host['problem'] || !empty($service_html)) {
            if(is_neptune()){
                $html .= "
                <tr>
                    <td>
                        <div class='checkbox'>
                            <label class='flex items-center'>
                                <div class='flex items-center'>
                                    <span class='status-dot dot-10 {$host_class}'></span> 
                                </div>
                                <input class='massack-checkbox' type='checkbox' name='hosts[]' value='{$host['host_name']}'>{$host['host_name']}
                            </label>
                        </div>
                    </td>
                    <td>";
            } else {
                $html .= "
                <tr>
                    <td class='{$host_class}'>
                        <div class='checkbox'>
                            <label>
                                <input type='checkbox' name='hosts[]' value='{$host['host_name']}'>{$host['host_name']}
                            </label>
                        </div>
                    </td>
                    <td>";
            }


            if (!empty($service_html)) {
                $html .= "<div class='checkbox'>
                        <label>
                            <input class='host parent host{$hostcount}' data-id='{$hostcount}' id='selectAllHost' type='checkbox' name='hosts[]' value='{$host['host_name']}'><a>Select all {$host['host_name']} services</a>
                        </label>
                    </div>";
            }

            $html .= "</td>
                <td>
                    {$host['last_check']}
                </td>
                <td>
                    <div>
                        {$host['plugin_output']}
                    </div>
                </td>
            </tr>
            ";

            $html .= $service_html;

            $hostcount++;

        }
    }

    if ($hostcount == 0) {
        $html .= "<tr>
            <td colspan='4'>"._("No problems were found. To force checks for non-problem hosts and services, change the status filter above.")."</td>
        </tr>";
    }

    $html .= "</tbody></table></div><div class='clear'></div></form>";

    return $html;
}

function feedback_html($msg, $error)
{
    if ($error) {
        $class = 'errorMessage align-items-center-flex';
        $icon = "<i class='material-symbols-outlined md-400 md-20 md-padding'>error</i>";
    } else {
        $class = 'actionMessage align-items-center-flex';
        $icon = "<i class='material-symbols-outlined md-400 md-20 md-padding'>info</i>";
    }

    $feedback = "<div class='{$class} standalone'>{$icon} {$msg}</div>";

    return $feedback;
}

function mic_get_hosts()
{
    $hostgroup = grab_request_var("hostgroup", "");
    $servicegroup = grab_request_var("servicegroup", "");
    $host = grab_request_var("host", "");
    $backendargs = array();
    $hosts = array();
    $host_ids = array();

    if (!empty($host)) {
        $hosts[] = $host;
    }

    if (!empty($hostgroup)) {
        $host_ids = get_hostgroup_member_ids($hostgroup);
    }

    if (!empty($servicegroup)) {
        $host_ids = get_servicegroup_host_member_ids($servicegroup);
    }

    if (!empty($hosts)) {
        $backendargs['host_name'] = "in:".implode(',', $hosts);
    }

    if (!empty($host_ids)) {
        $backendargs['host_id'] = "in:".implode(',', $host_ids);
    }

    $xml = get_xml_host_status($backendargs);
    $hosts = array();
    if ($xml) {
        foreach ($xml->hoststatus as $x) {
            $problem = true;
            if (("$x->current_state" == 0 && "$x->has_been_checked" == 1) || "$x->scheduled_downtime_depth" > 0 || "$x->problem_acknowledged" > 0) {
                $problem = false;
            }

            $hosts["$x->name"] = array(
                'host_state' => "$x->current_state", 
                'host_name' => "$x->name",
                'plugin_output' => "$x->status_text",
                'last_check' => "$x->last_check",
                'host_id' => "$x->host_id",
                'problem' => $problem
            );
        }
    } 

    return $hosts;
}

function mic_get_services($select_state = 0)
{
    $servicegroup = grab_request_var("servicegroup", "");
    $host = grab_request_var("host", "");
    $backendargs = array();

    if ($select_state == 0) {
        $backendargs["current_state"] = "in:1,2,3";
        $backendargs["has_been_checked"] = 1;
        $backendargs["problem_acknowledged"] = 0;
        $backendargs["scheduled_downtime_depth"] = 0;
        $backendargs['is_active'] = 1;
    }

    if (!empty($host)) {
        $backendargs["host_name"] = $host;
    }

    if (!empty($servicegroup)) {
        $service_ids = get_servicegroup_member_ids($servicegroup);
    }

    if (!empty($service_ids)) {
        $backendargs['service_id'] = "in:".implode(',', $service_ids);
    }

    $xml = get_xml_service_status($backendargs);
    if ($xml) {
        foreach ($xml->servicestatus as $x) {
            $host_state = intval($x->host_current_state);
            $service = array(
                'host_name' => "$x->host_name",
                'service_description' => "$x->name",
                'current_state' => "$x->current_state",
                'plugin_output' => "$x->status_text",
                'last_check' => "$x->status_update_time",
                'service_id' => "$x->service_id"
            );
            $services["$x->host_name"][] = $service;
        }
    }

    return $services;  
}

function host_class($code)
{
    switch ($code) {
        case 0:
            return "hostup";
        case 1:
            return 'hostdown';
        default:
            return 'hostunreachable';
    }
}

function service_class($code)
{
    switch ($code) {
        case 0:
            return "serviceok";
        case 1:
            return 'servicewarning';
        case 2:
            return 'servicecritical';
        default:
            return 'serviceunknown';
    }
}
