<?php
//
// Nagios Cross-Platform Agent (NCPA) Config Wizard
// Copyright (c) 2014-2025 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');
require_once(dirname(__FILE__) . '/../../../config/deployment/includes/utils-deployment.inc.php');

ncpa_configwizard_init();

function ncpa_configwizard_init()
{
    $name = "ncpa";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "4.1.1",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor a host (Windows, Linux, OS X, Solaris, or AIX) using NCPA.") . ' <div class="hide">centos rhel sles suse opensuse oracle cloudlinux ubuntu debian redhat mac</div>',
        CONFIGWIZARD_DISPLAYTITLE => _("NCPA"),
        CONFIGWIZARD_FUNCTION => "ncpa_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "ncpa.png",
        CONFIGWIZARD_FILTER_GROUPS => array('nagios', 'windows','linux','otheros'),
        CONFIGWIZARD_REQUIRES_VERSION => 70000
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function ncpa_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null, $extra = array())
{
    // Grab core wizard info
    $wizard_name = grab_array_var($inargs, "wizard");
    $wiz_args = get_configwizard_by_name($wizard_name);
    $wizard_title = isset($wiz_args[CONFIGWIZARD_DISPLAYTITLE]) ? $wiz_args[CONFIGWIZARD_DISPLAYTITLE] : '';

    /* Prerequisite software */
    $NCPA_download_url = "https://www.nagios.org/ncpa/#downloads";
    $NCPA_docs_url = "https://www.nagios.org/ncpa/#docs";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {

        case CONFIGWIZARD_MODE_GETSTAGE1HTML:
            $ip_address = grab_array_var($inargs, "ip_address", "");
            $port = grab_array_var($inargs, "port", "5693");
            $token = grab_array_var($inargs, "token", "");
            $no_ssl_verify = grab_array_var($inargs, "no_ssl_verify", 1);
            $os = grab_array_var($inargs, "os", grab_array_var($extra, "os", ""));
            $selectedhostconfig = grab_array_var($inargs, "selectedhostconfig", "");
            $operation = grab_array_var($inargs, "operation", "new");

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:
            // Get variables that were passed to us
            $ip_address = grab_array_var($inargs, "ip_address", "");
            $port = grab_array_var($inargs, "port", "5693");
            $token = grab_array_var($inargs, "token", "");
            $no_ssl_verify = grab_array_var($inargs, "no_ssl_verify", 1);

            // For multi-host ncpa run (from deployment)
            $hosts = grab_request_var("hosts", array());
            $job_id = grab_request_var("job_id", 0);

            // Grab deploy id if it exists
            $deploy_id = grab_request_var('deploy_id', 0);

            if (!empty($deploy_id)) {
                $agent = deploy_get_agent($deploy_id);
                $metadata = json_decode(base64_decode($agent['metadata']), true);
                $token = decrypt_data($metadata['token']);
                $ip_address = $agent['address'];
            }

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($ip_address) == false && empty($hosts)) {
                $errmsg[$errors++] = _("No address specified.");
            }

            if (have_value($port) == false) {
                $errmsg[$errors++] = _("No port number specified.");
            }

            // Test the connection if no errors
            if (empty($errors) && empty($hosts)) {
                $ip_address_replaced = nagiosccm_replace_user_macros($ip_address);
                $port_replaced = nagiosccm_replace_user_macros($port);
                $token_replaced = nagiosccm_replace_user_macros($token);

                // The URL we will use to query the NCPA agent, and do a walk
                // of all monitorable items.
                $query_url = "https://{$ip_address}:{$port}/testconnect/?token=".urlencode($token);
                $query_url_replaced = "https://{$ip_address_replaced}:{$port_replaced}/testconnect/?token=".urlencode($token_replaced);

                // Remove SSL verification or not
                $context = array("ssl" => array("verify_peer" => true, "verify_peer_name" => true));

                if ($no_ssl_verify) {
                    $context['ssl']['verify_peer'] = false;
                    $context['ssl']['verify_peer_name'] = false;
                }

                // All we want to do is test if we can hit this URL.
                // Error Control Operator - @ - http://php.net/manual/en/language.operators.errorcontrol.php

                // For a simple request, make timeout shorter than php default or typical gateway timeout of 60
                ini_set("default_socket_timeout", 10);
                $raw_json = @file_get_contents($query_url_replaced, false, stream_context_create($context));

                # This will be displayed, so hide the token, for "security"
                $safe_url = str_replace('token='.urlencode($token_replaced), "token=&lt;your_token&gt;", $query_url);

                if ($raw_json === FALSE || empty($raw_json)) {
                    $errmsg[$errors++] = _("Unable to contact server at")." $safe_url";

                } else {
                    $json = json_decode($raw_json, true);

                    # This should not happen, but it might.
                    if (!is_array($json)) {
                        $errmsg[$errors++] = _("Bad data received")." $safe_url<br>"._('This may be a temporary condition, please check that NCPA is running and try resubmitting.');
                    } else if (!array_key_exists('value', $json)) {
                        $errmsg[$errors++] = _("Bad token for connection.");
                    }
                }
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:
            // Get variables that were passed in.
            $ip_address = grab_array_var($inargs, "ip_address", "");
            $as_host = @gethostbyaddr($ip_address);
            if ($as_host === false) {
                $as_host = $ip_address;
            }
            $port = grab_array_var($inargs, "port", "5693");
            $token = grab_array_var($inargs, "token", "");
            $no_ssl_verify = grab_array_var($inargs, "no_ssl_verify", 1);
            $hostname = grab_array_var($inargs, 'hostname', $as_host);
            $default_mem_units = grab_array_var($inargs, 'default_mem_units', 'Gi');
            $selectedhostconfig = grab_array_var($inargs, "selectedhostconfig", "");
            $operation = grab_array_var($inargs, "operation", "new");

            // Set os to extra os
            $os = grab_array_var($inargs, "os", grab_array_var($extra, "os", ""));

            // For multi-host ncpa run (from deployment)
            $hosts = grab_request_var("hosts", array());
            $job_id = grab_request_var("job_id", 0);

            if (!empty($job_id)) {
                if (count($hosts) == 1) {
                    $job = deploy_get_job($job_id);
                    $token = decrypt_data($job['ncpa_token']);
                    $ip_address = $hosts[0];
                    $hostname = gethostbyaddr($ip_address);
                    if ($hostname === false) {
                        $hostname = $ip_address;
                    }
                    $hosts = array();
                    $os = $job['os'];
                } else {
                    // Don't set the normal things because it's multi-host...
                }
            }

            // Grab deploy id if it exists
            $deploy_id = grab_request_var('deploy_id', 0);

            if (!empty($deploy_id)) {
                $agent = deploy_get_agent($deploy_id);
                $metadata = json_decode(base64_decode($agent['metadata']), true);
                $token = decrypt_data($metadata['token']);
                $ip_address = $agent['address'];
                $hostname = $agent['hostname'];
                $os = $agent['os'];
            }

            $rp_address = nagiosccm_replace_user_macros($ip_address);
            $rp_port = nagiosccm_replace_user_macros($port);
            $rp_token = nagiosccm_replace_user_macros($token);

            $services = grab_array_var($inargs, "services", array());
            $services_serial = grab_array_var($inargs, "services_serial", "");

            if ($services_serial != "" && isset($_POST['submitButton2']) == false) {
                $services = json_decode(base64_decode($services_serial), true);
            }
            
            // Initialize the data arrays.
            $disk_data = [ '' => '' ];
            $interface_data = [ '' => '' ];
            $service_data = [];
            $process_data = [];
            $plugin_data = [];

            /*
             * Single Host
             *
             * If we are running the wizard on a single host get the data from NCPA.
             *
             * Use generic data for multiple hosts.
             */
            if (empty($hosts)) {
                $api_url = "https://{$rp_address}:{$rp_port}/api";

                try {
                    $disk_data = get_data_from_ncpa_api('disk', $api_url, $rp_token, $no_ssl_verify);
                    $interface_data = get_data_from_ncpa_api('interface', $api_url, $rp_token, $no_ssl_verify);
                    $service_data = get_data_from_ncpa_api('services', $api_url, $rp_token, $no_ssl_verify);
                    $process_data = get_data_from_ncpa_api('processes', $api_url, $rp_token, $no_ssl_verify);
                    $plugin_data = get_data_from_ncpa_api('plugins', $api_url, $rp_token, $no_ssl_verify);

                    // These are used in step2.php
                    $cpu_data = get_data_from_ncpa_api('cpu/percent', $api_url, $rp_token, $no_ssl_verify, array('aggregate' => 'avg'));
                    $user_data = get_data_from_ncpa_api('user/count', $api_url, $rp_token, $no_ssl_verify);
                    $mem_data = get_data_from_ncpa_api('memory/virtual/percent', $api_url, $rp_token, $no_ssl_verify);
                    $swap_data = get_data_from_ncpa_api('memory/swap/percent', $api_url, $rp_token, $no_ssl_verify);

                // Would this be more "clear", as one exception block?  PHP 7.1.0+ can use '|'.
                } catch (NCPAConnectionException $e) {
                    show_step2(true, $e->getMessage());
                } catch (NCPAEmptyDataException $e) {
                    show_step2(true, $e->getMessage());
                } catch (NCPAInvalidDataException $e) {
                    show_step2(true, $e->getMessage());
                } catch (NCPAErrorException $e) {
                    show_step2(true, $e->getMessage());
                }
            }

            // Setup the data retrieved from NCPA in the $root array
            $root = [];
            $root['disk'] = $disk_data;
            $root['interface'] = $interface_data;
            $root['services'] = $service_data;

            // This is for setting up the windows counters
            $tmp = [];

            foreach ($process_data as $data) {
                if (array_key_exists($data['name'], $tmp)) {
                    $tmp[$data['name']]++;
                } else {
                    $tmp[$data['name']] = 1;
                }
            }

            $root['processes'] = $tmp;

            // Sort plugin data
            if (!empty($plugin_data)) {
                sort($plugin_data);
            }

            $root['plugins'] = $plugin_data;
            $icon = _("Generic");

            if (!empty($os)) {
                $icon = '<img src="'.nagioscore_get_ui_url()."images/logos/".ncpa_configwizard_get_os_icon($os).'" title="'.$os.'">';
            }

            $defaults = grab_array_var($extra, "defaults", "");

            ksort($root['services']);
            ksort($root['processes']);
            ksort($root['plugins']);

            /*
             * SETUP the default values to display in Step 2.
             */

            if (empty($services)) {
                $services = setup_defaults_for_services($root, $defaults);
            }

            /*
             * Build services multi-select - Except for macosx
             */

            if ($wizard_name != 'macosx') {
                // Used in step2.php
                $services_array = [];
                $services_by_name = [];

                $service_select_html = setup_multi_select('services', $services, $root, $services_by_name, $services_array);
            }

            /*
             * Build processes multi-select
             */

            // Used in step2.php
            $processes_array = [];
            $processes_by_name = [];

            $process_select_html = setup_multi_select('processes', $services, $root, $processes_by_name, $processes_array);

            /*
             * Build plugins multi-select
             */

            // Used in step2.php
            $plugins_array = $services['plugins'];  # We can have duplicates setup (Back from Step3), so load here.
            $plugins_by_name = [];

            $plugin_select_html = setup_multi_select('plugins', $services, $root, $plugins_by_name, $plugins_array);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:
            // Get variables that were passed to us
            $ip_address = grab_array_var($inargs, 'ip_address');
            $hostname = grab_array_var($inargs, 'hostname');
            $port = grab_array_var($inargs, 'port');
            $token = grab_array_var($inargs, 'token');
            $default_mem_units = grab_array_var($inargs, 'default_mem_units');
            $services_serial = grab_array_var($inargs, "services_serial", "");

            $services = json_decode(base64_decode($services_serial), true);

            // Use current request data if available
            if (empty($services)) {
                $services = grab_array_var($inargs, "services", array());
            }

            #
            # Data passed back from "next" steps.
            #
            # TODO: This should be handled in monitoringwizards.php
            #
            $passbackdata = grab_array_var($inargs, "passbackdata", "");

            if ($passbackdata) {
                $decodedpassbackdata = json_decode(base64_decode($passbackdata), true);

                $ip_address = $decodedpassbackdata['ip_address'];
                $hostname = $decodedpassbackdata['hostname'];
                $port = $decodedpassbackdata['port'];
                $token = $decodedpassbackdata['token'];

                $services_serial = $decodedpassbackdata['services_serial'];
                $services = json_decode(base64_decode($services_serial), true);
            }

            $hosts = grab_array_var($inargs,  "hosts", array());
            $job_id = grab_array_var($inargs, "job_id", 0);

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (empty($hostname)) {
                $errmsg[$errors++] = _("You must enter a host name");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE3HTML:
            // Get variables that were passed to us
            $ip_address = grab_array_var($inargs, 'ip_address');
            $hostname = grab_array_var($inargs, 'hostname');
            $port = grab_array_var($inargs, 'port');
            $token = grab_array_var($inargs, 'token');
            $services = grab_array_var($inargs, 'services', array());

            $services_serial = (!empty($services) ? base64_encode(json_encode($services)) : grab_array_var($inargs, "services_serial", ''));

            $default_mem_units = grab_array_var($inargs, 'default_mem_units');
            $os = grab_array_var($inargs, 'os', '');

            $job_id = grab_array_var($inargs, 'job_id', 0);
            $hosts = grab_array_var($inargs, 'hosts', array());

            #
            # Data passed back from "next" steps.
            #
            # TODO: This should be handled in monitoringwizards.php
            #
            $passbackdata = grab_array_var($inargs, "passbackdata", "");

            if ($passbackdata) {
                $decodedpassbackdata = json_decode(base64_decode($passbackdata), true);

                $ip_address = $decodedpassbackdata['ip_address'];
                $hostname = $decodedpassbackdata['hostname'];
                $port = $decodedpassbackdata['port'];
                $token = $decodedpassbackdata['token'];

                $services_serial = $decodedpassbackdata['services_serial'];
            }

            $output = '
            <input type="hidden" name="ip_address" value="' . encode_form_val($ip_address) . '" />
            <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '" />
            <input type="hidden" name="port" value="' . encode_form_val($port) . '" />
            <input type="hidden" name="token" value="' . encode_form_val($token) . '" />
            <input type="hidden" name="os" value="' . encode_form_val($os) . '" />
            <input type="hidden" name="default_mem_units" value="' . encode_form_val($default_mem_units) . '" />
            <input type="hidden" name="services_serial" value="' . $services_serial . '" />
            <input type="hidden" name="job_id" value="' . intval($job_id) . '" />';

            if (!empty($hosts)) {
                foreach ($hosts as $h) {
                    $output .= '<input type="hidden" name="hosts[]" value="' . $h . '" />';
                }
            }

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:
            // Get variables that were passed to us
            $ip_address = grab_array_var($inargs, 'ip_address');
            $hostname = grab_array_var($inargs, 'hostname');
            $port = grab_array_var($inargs, 'port');
            $token = grab_array_var($inargs, 'token');
            $services = grab_array_var($inargs, 'services', array());

            if (empty($services)) {
                $services_serial = grab_array_var($inargs, "services_serial", "");
                $services = json_decode(base64_decode($services_serial), true);
            }

            $default_mem_units = grab_array_var($inargs, 'default_mem_units');
            $os = grab_array_var($inargs, 'os', '');

            $job_id = grab_array_var($inargs, 'job_id', 0);
            $hosts = grab_array_var($inargs, 'hosts', array());

            $output = '
            <input type="hidden" name="ip_address" value="' . encode_form_val($ip_address) . '" />
            <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '" />
            <input type="hidden" name="port" value="' . encode_form_val($port) . '" />
            <input type="hidden" name="token" value="' . encode_form_val($token) . '" />
            <input type="hidden" name="os" value="' . encode_form_val($os) . '" />
            <input type="hidden" name="default_mem_units" value="' . encode_form_val($default_mem_units) . '" />
            <input type="hidden" name="services_serial" value="' . base64_encode(json_encode($services)) . '" />
            <input type="hidden" name="job_id" value="' . intval($job_id) . '" />';

            if (!empty($hosts)) {
                foreach ($hosts as $h) {
                    $output .= '<input type="hidden" name="hosts[]" value="' . $h . '" />';
                }
            }

            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:
            $hostname = grab_array_var($inargs, "hostname", "");
            $ip_address = grab_array_var($inargs, "ip_address", "");
            $port = grab_array_var($inargs, "port", "");
            $token = grab_array_var($inargs, "token", "");
            $default_mem_units = grab_array_var($inargs, 'default_mem_units');
            $services_serial = grab_array_var($inargs, "services_serial", "");
            $services = json_decode(base64_decode($services_serial), true);
            $os = grab_array_var($inargs, "os", "");

            $hosts = grab_array_var($inargs,  "hosts", array());
            $job_id = grab_array_var($inargs, "job_id", 0);

            // Get job info and token
            if (!empty($job_id) && !empty($hosts)) {
                $job = deploy_get_job($job_id);
                $token = decrypt_data($job['ncpa_token']);
            }

            // Save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["ip_address"] = $ip_address;
            $meta_arr["port"] = $port;
            $meta_arr["token"] = $token;
            $meta_arr["services"] = $services;
            $meta_arr["job_id"] = $job_id;
            $meta_arr["hosts"] = $hosts;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            // Escape values for check_command line
            if (function_exists('nagiosccm_replace_command_line')) {
                $token = nagiosccm_replace_command_line($token, '$', '!');
            }

            $objs = array();

            if (empty($hosts)) {

                if (!host_exists($hostname)) {
                    $objs[] = array(
                        "type" => OBJECTTYPE_HOST,
                        "use" => "xiwizard_ncpa_host",
                        "host_name" => $hostname,
                        "address" => $ip_address,
                        "icon_image" => ncpa_configwizard_get_os_icon($os),
                        "statusmap_image" => ncpa_configwizard_get_os_icon($os),
                        "_xiwizard" => $wizard_name);
                }
            } else {

                foreach ($hosts as $h) {
                    if (!host_exists($h)) {
                        $objs[] = array(
                            "type" => OBJECTTYPE_HOST,
                            "use" => "xiwizard_ncpa_host",
                            "host_name" => $h,
                            "address" => $h,
                            "_xiwizard" => $wizard_name);
                    }
                }
            }

            // Common plugin opts
            $commonopts = "-t ".escapeshellarg_nowrap($token)." ";
            if ($port) {
                $commonopts .= "-P ".intval($port)." ";
            }

            // If we don't already have an array of hosts, make it
            $hostnames = array($hostname);
            if (!empty($hosts)) {
                $hostnames = $hosts;
            }

            foreach ($hostnames as $hostname) {
                foreach ($services as $type => $args) {
                    $pluginopts = "";
                    $pluginopts .= $commonopts;

                    switch ($type) {

                        case "cpu_usage":
                            if (!array_key_exists('monitor', $args)) {
                                break;
                            }
                            $pluginopts .= "-M cpu/percent";

                            if (!empty($args['warning'])) {
                                $pluginopts .= " -w " . escapeshellarg($args["warning"]);
                            }
                            if (!empty($args['critical'])) {
                                $pluginopts .= " -c " . escapeshellarg($args["critical"]);
                            }

                            if (!empty($args['average'])) {
                                $pluginopts .= " -q 'aggregate=avg'";
                            }

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => "CPU Usage",
                                "use" => "xiwizard_ncpa_service",
                                "check_command" => "check_xi_ncpa!" . $pluginopts,
                                "_xiwizard" => $wizard_name);
                            break;

                        case "users":
                            if (!array_key_exists('monitor', $args)) {
                                break;
                            }
                            $pluginopts .= "-M user/count";

                            if (!empty($args['warning'])) {
                                $pluginopts .= " -w " . escapeshellarg($args["warning"]);
                            }
                            if (!empty($args['critical'])) {
                                $pluginopts .= " -c " . escapeshellarg($args["critical"]);
                            }

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => "User Count",
                                "use" => "xiwizard_ncpa_service",
                                "check_command" => "check_xi_ncpa!" . $pluginopts,
                                "_xiwizard" => $wizard_name);
                            break;

                        case "memory_usage":
                            if (!array_key_exists('monitor', $args)) {
                                break;
                            }
                            $pluginopts .= "-M memory/virtual";

                            if (!empty($default_mem_units)) {
                                $pluginopts .= " -u " . escapeshellarg($default_mem_units);
                            }
                            if (!empty($args['warning'])) {
                                $pluginopts .= " -w " . escapeshellarg($args["warning"]);
                            }
                            if (!empty($args['critical'])) {
                                $pluginopts .= " -c " . escapeshellarg($args["critical"]);
                            }

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => "Memory Usage",
                                "use" => "xiwizard_ncpa_service",
                                "check_command" => "check_xi_ncpa!" . $pluginopts,
                                "_xiwizard" => $wizard_name);
                            break;

                        case "swap_usage":
                            if (!array_key_exists('monitor', $args)) {
                                break;
                            }
                            $pluginopts .= "-M memory/swap";

                            if (!empty($default_mem_units)) {
                                $pluginopts .= " -u " . escapeshellarg($default_mem_units);
                            }
                            if (!empty($args['warning'])) {
                                $pluginopts .= " -w " . escapeshellarg($args["warning"]);
                            }
                            if (!empty($args['critical'])) {
                                $pluginopts .= " -c " . escapeshellarg($args["critical"]);
                            }

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => "Swap Usage",
                                "use" => "xiwizard_ncpa_service",
                                "check_command" => "check_xi_ncpa!" . $pluginopts,
                                "_xiwizard" => $wizard_name);
                            break;

                        case "disk":
                            foreach ($args as $id => $metrics) {
                                if (!array_key_exists('monitor', $metrics)) {
                                    continue;
                                }

                                // Ensure we are using the right name/title for the disk API (and verify it has | instead of / or \)
                                $title = $metrics['name'];
                                if (isset($metrics['title'])) {
                                    $title = $metrics['title'];
                                }
                                $title = str_replace(array('\\', '/'), '|', $title);

                                $disk = "disk/logical/{$title}";
                                $theseopts = "{$pluginopts} -M " . escapeshellarg($disk);

                                if (!empty($metrics["warning"])) {
                                    $theseopts .= " -w " . escapeshellarg($metrics["warning"]);
                                }
                                if (!empty($metrics["critical"])) {
                                    $theseopts .= " -c " . escapeshellarg($metrics["critical"]);
                                }

                                // Make sure back slash doesn't escape service description line
                                $description = str_replace(array('\\', ','), array('/', ' '), $metrics['name']);

                                $objs[] = array(
                                    "type" => OBJECTTYPE_SERVICE,
                                    "host_name" => $hostname,
                                    "service_description" => "Disk Usage on " . $description,
                                    "use" => "xiwizard_ncpa_service",
                                    "check_command" => "check_xi_ncpa!" . $theseopts,
                                    "_xiwizard" => $wizard_name);
                            }
                            break;

                        case "interface":
                            foreach ($args as $id => $metrics) {
                                if (!array_key_exists('monitor', $metrics) || empty($metrics['name'])) {
                                    continue;
                                }

                                $title = $metrics['name'];

                                $theseopts = "{$pluginopts} -M 'interface/{$title}/bytes_sent' -d -u M";

                                if (!empty($metrics["warning"])) {
                                    $theseopts .= " -w " . escapeshellarg($metrics["warning"]);
                                }
                                if (!empty($metrics["critical"])) {
                                    $theseopts .= " -c " . escapeshellarg($metrics["critical"]);
                                }

                                $objs[] = array(
                                    "type" => OBJECTTYPE_SERVICE,
                                    "host_name" => $hostname,
                                    "service_description" => "{$title} Bandwidth - Outbound",
                                    "use" => "xiwizard_ncpa_service",
                                    "check_command" => "check_xi_ncpa!" . $theseopts,
                                    "_xiwizard" => $wizard_name);

                                $theseopts = "{$pluginopts} -M 'interface/{$title}/bytes_recv' -d -u M";

                                if (!empty($metrics["warning"])) {
                                    $theseopts .= " -w " . escapeshellarg($metrics["warning"]);
                                }
                                if (!empty($metrics["critical"])) {
                                    $theseopts .= " -c " . escapeshellarg($metrics["critical"]);
                                }

                                $objs[] = array(
                                    "type" => OBJECTTYPE_SERVICE,
                                    "host_name" => $hostname,
                                    "service_description" => "{$title} Bandwidth - Inbound",
                                    "use" => "xiwizard_ncpa_service",
                                    "check_command" => "check_xi_ncpa!" . $theseopts,
                                    "_xiwizard" => $wizard_name);
                            }
                            break;

                        case "services":
                            foreach ($args as $i => $service) {
                                if (!array_key_exists('monitor', $service) || empty($service['name']) || empty($service['desc'])) {
                                    continue;
                                }
                                $query = "service=" . str_replace('$', '$$', $service["name"]) . ",status=" . $service["state"];
                                $theseopts = "{$pluginopts} -M 'services' -q " . escapeshellarg($query);
                                $description = str_replace(array('\\', ','), array('/', ' '), $service['desc']);

                                $objs[] = array(
                                    "type" => OBJECTTYPE_SERVICE,
                                    "host_name" => $hostname,
                                    "service_description" => $description,
                                    "use" => "xiwizard_ncpa_service",
                                    "check_command" => "check_xi_ncpa!" . $theseopts,
                                    "_xiwizard" => $wizard_name);
                            }
                            break;

                        case "processes":
                            foreach ($args as $i => $metrics) {
                                if (!array_key_exists('monitor', $metrics) || empty($metrics['desc'])) {
                                    continue;
                                }
                                $proc_name = $metrics['name'];
                                $theseopts = "{$pluginopts} -M 'processes' -q 'name={$proc_name}'";

                                if (!empty($metrics["warning"])) {
                                    $theseopts .= " -w " . escapeshellarg($metrics["warning"]);
                                }
                                if (!empty($metrics["critical"])) {
                                    $theseopts .= " -c " . escapeshellarg($metrics["critical"]);
                                }

                                $description = str_replace(array('\\', ','), array('/', ' '), $metrics['desc']);

                                $objs[] = array(
                                    "type" => OBJECTTYPE_SERVICE,
                                    "host_name" => $hostname,
                                    "service_description" => $description,
                                    "use" => "xiwizard_ncpa_service",
                                    "check_command" => "check_xi_ncpa!" . $theseopts,
                                    "_xiwizard" => $wizard_name);
                            }
                            break;

                        case 'counters':
                            foreach ($args as $i => $counter) {
                                if (!array_key_exists('monitor', $counter) || empty($counter['desc'])) {
                                    continue;
                                }
                                $theseopts = "{$pluginopts} -M ". escapeshellarg("windowscounters/".str_replace('\\', '/', $counter['name']));

                                if (!empty($counter["warning"])) {
                                    $theseopts .= " -w " . escapeshellarg($counter["warning"]);
                                }
                                if (!empty($counter["critical"])) {
                                    $theseopts .= " -c " . escapeshellarg($counter["critical"]);
                                }

                                if (!empty($count["sleep"])) {
                                    $theseopts .= " -q 'sleep=" . intval($counter["sleep"]) . "'";
                                }

                                $description = str_replace(array('\\', ','), array('/', ' '), $counter['desc']);

                                $objs[] = array(
                                    "type" => OBJECTTYPE_SERVICE,
                                    "host_name" => $hostname,
                                    "service_description" => $description,
                                    "use" => "xiwizard_ncpa_service",
                                    "check_command" => "check_xi_ncpa!" . $theseopts,
                                    "_xiwizard" => $wizard_name);
                            }
                            break;

                        case "plugins":
                            foreach ($args as $i => $plugin) {
                                if (!array_key_exists('monitor', $plugin) || empty($plugin['desc'])) {
                                    continue;
                                }
                                $theseopts = "{$pluginopts} -M " . escapeshellarg("plugins/".$plugin['name']);

                                if (!empty($plugin['args'])) {
                                    $theseopts .= " -q " . escapeshellarg("args=".$plugin['args']);
                                }

                                $description = str_replace(array('\\', ','), array('/', ' '), $plugin['desc']);

                                $objs[] = array(
                                    "type" => OBJECTTYPE_SERVICE,
                                    "host_name" => $hostname,
                                    "service_description" => $description,
                                    "use" => "xiwizard_ncpa_service",
                                    "check_command" => "check_xi_ncpa!" . $theseopts,
                                    "_xiwizard" => $wizard_name);
                            }
                            break;

                        default:
                            break;
                    }
                }
            }

            // Return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;
            break;

        default:
            break;
    }

    return $output;
}

/**
 * Get operating system icon based on OS name
 *
 * @param   string  $os     Operating system
 * @return  string          Icon file
 */
function ncpa_configwizard_get_os_icon($os)
{
    $icon = "ncpa.png";
    $os = strtolower($os);

    switch ($os) {
        case "rhel":
            $icon = "redhat.png";
            break;
        case "fedora":
            $icon = "fedora.png";
            break;
        case "centos":
            $icon = "centos.png";
            break;
        case "oracle":
            $icon = "oracle-linux.gif";
            break;
        case "ubuntu":
            $icon = "ubuntu.png";
            break;
        case "debian":
            $icon = "debian.png";
            break;
        case "sles":
            $icon = "suse_enterprise.png";
            break;
        case "suse":
            $icon = "opensuse.png";
            break;
        case "windows":
            $icon = "win_server.png";
            break;
        case "aix":
            $icon = "aix.png";
            break;
        case "sol":
            $icon = "solaris.png";
            break;
        case "osx":
            $icon = "osx.png";
            break;
    }

    return $icon;
}


function setup_defaults_for_services($ncpa_data, $defaults) {
    
    // Set defaults for services
    $default_services['cpu_usage']['monitor'] = 'on';
    $default_services['cpu_usage']['warning'] = 20;
    $default_services['cpu_usage']['critical'] = 40;
    $default_services['cpu_usage']['average'] = 1;
    $default_services['users']['monitor'] = 'on';
    $default_services['users']['warning'] = 2;
    $default_services['users']['critical'] = 4;
    $default_services['memory_usage']['monitor'] = 'on';
    $default_services['memory_usage']['warning'] = 50;
    $default_services['memory_usage']['critical'] = 80;
    $default_services['swap_usage']['monitor'] = 'on';
    $default_services['swap_usage']['warning'] = 5;
    $default_services['swap_usage']['critical'] = 10;

    // If we have data from NCPA - single host, use it to setup the "default_services".
    if (isset($ncpa_data['disk']['logical'])) {

        $id = 0;

        foreach ($ncpa_data['disk']['logical'] as $title => $value) {
            if (empty($value['device_name'])) {
                continue;
            }

            $default_services['disk'][$id]['monitor'] = 'on';
            $default_services['disk'][$id]['warning'] = 70;
            $default_services['disk'][$id]['critical'] = 90;
            $default_services['disk'][$id]['name'] = $value['device_name'];

            if (is_array($value['device_name'])) {
                $default_services['disk'][$id]['name'] = $value['device_name'][0];
            }

            $default_services['disk'][$id]['title'] = $title;
            $id++;
        }
    } else if (!empty($hosts)) {
        $default_services['disk'][0]['monitor'] = 'on';
        $default_services['disk'][0]['warning'] = 70;
        $default_services['disk'][0]['critical'] = 90;
        $default_services['disk'][0]['name'] = "/";
    }

    ksort($ncpa_data['interface']);
    $id = 0;

    foreach ($ncpa_data['interface'] as $title => $value) {
        if (stripos($title, "Local Area Connection") !== false || stripos($title, "eth") !== false || stripos($title, "ens") !== false || stripos($title, "Wireless") !== false) {
            $default_services['interface'][$id]['monitor'] = 'on';
        } else {
            $default_services['interface'][$id]['monitor'] = 'off';
        }

        $default_services['interface'][$id]['warning'] = 10;
        $default_services['interface'][$id]['critical'] = 100;
        $default_services['interface'][$id]['name'] = $title;
        $id++;
    }

    // Create defaults for processes
    if ($defaults == "desktop") {

        // Add explorer
        $default_services['processes'][] = array(
            'monitor' => 'off',
            'name' => 'explorer.exe',
            'desc' => 'Explorer',
            'warning' => '',
            'critical' => '1:'
        );
    }

    $default_services['processes'][] = array(
        'monitor' => 'off',
        'name' => '',
        'desc' => '',
        'warning' => '',
        'critical' => ''
    );

    $default_services['plugins'][] = array(
        'monitor' => 'off',
        'name' => '',
        'desc' => '',
        'args' => '',
    );

    // Create defaults for services
    if ($defaults == "desktop" || $defaults == "server") {

        // MSSQL server defaults
        if (array_key_exists('MSSQLSERVER', $ncpa_data['services'])) {
            $default_services['services'][] = array(
                'monitor' => 'off',
                'name' => 'MSSQLSERVER',
                'desc' => 'MSSQL Service Status',
                'state' => 'running'
            );
        } else if (array_key_exists('MSSQL$SQLEXPRESS', $ncpa_data['services'])) {
            $default_services['services'][] = array(
                'monitor' => 'off',
                'name' => 'MSSQL$SQLEXPRESS',
                'desc' => 'MSSQL Express Service Status',
                'state' => 'running'
            );
        }

        // Windows Defender running
        if (array_key_exists('WinDefend', $ncpa_data['services'])) {
            $default_services['services'][] = array(
                'monitor' => 'off',
                'name' => 'WinDefend',
                'desc' => 'Windows Defender Service',
                'state' => 'running'
            );
        }

        // MSSQL telemetry defaults
        if (array_key_exists('MSSQLTELEMETRY', $ncpa_data['services'])) {
            $default_services['services'][] = array(
                'monitor' => 'off',
                'name' => 'MSSQLTELEMETRY',
                'desc' => 'MSSQL Telemetry Service Status',
                'state' => 'running'
            );
        } else if (array_key_exists('SQLTELEMETRY$SQLEXPRESS', $ncpa_data['services'])) {
            $default_services['services'][] = array(
                'monitor' => 'off',
                'name' => 'SQLTELEMETRY$SQLEXPRESS',
                'desc' => 'MSSQL Express Telemetry Service Status',
                'state' => 'running'
            );
        }

        // IIS server defaults
        if ($defaults == "server") {
            if (array_key_exists('WAS', $ncpa_data['services'])) {
                $default_services['services'][] = array(
                    'monitor' => 'off',
                    'name' => 'WAS',
                    'desc' => 'Windows Process Activation Service Status',
                    'state' => 'running'
                );
            }
            if (array_key_exists('W3SVC', $ncpa_data['services'])) {
                $default_services['services'][] = array(
                    'monitor' => 'off',
                    'name' => 'W3SVC',
                    'desc' => 'IIS Web Publishing Service Status',
                    'state' => 'running'
                );
            }
        }
    }

    $default_services['services'][] = array(
        'monitor' => 'off',
        'name' => '',
        'desc' => '',
        'state' => 'running'
    );

    // Create defaults for counters
    if ($defaults == "desktop" || $defaults == "server") {
        $default_services['counters'][] = array(
            'monitor' => 'off',
            'name' => '\Paging File(*)\% Usage',
            'desc' => _('Paging File Usage'),
            'sleep' => '',
            'warning' => 70,
            'critical' => 90
        );
        $default_services['counters'][] = array(
            'monitor' => 'off',
            'name' => '\Event Log\Events/sec',
            'desc' => _('Event Log Events/sec'),
            'sleep' => 5,
            'warning' => 10,
            'critical' => 20
        );
        $default_services['counters'][] = array(
            'monitor' => 'off',
            'name' => '\Terminal Services\Active Sessions',
            'desc' => _('Active Terminal Sessions'),
            'sleep' => '',
            'warning' => 2,
            'critical' => 3
        );
        $default_services['counters'][] = array(
            'monitor' => 'off',
            'name' => '\System\Threads',
            'desc' => _('Number of System Threads'),
            'sleep' => '',
            'warning' => '',
            'critical' => ''
        );

        if ($defaults == "desktop") {
            $default_services['counters'][] = array(
                'monitor' => 'off',
                'name' => '\Print Queue(*)\Job Errors',
                'desc' => _('Print Queue Job Errors'),
                'sleep' => '',
                'warning' => 1,
                'critical' => 2
            );
            $default_services['counters'][] = array(
                'monitor' => 'off',
                'name' => '\Telephony\Active Lines',
                'desc' => _('Active Telephony Lines'),
                'sleep' => '',
                'warning' => '',
                'critical' => ''
            );
        } else {
            $default_services['counters'][] = array(
                'monitor' => 'off',
                'name' => '\Server\Errors Logon',
                'desc' => _('Logon Errors'),
                'sleep' => '',
                'warning' => 2,
                'critical' => 20
            );
            $default_services['counters'][] = array(
                'monitor' => 'off',
                'name' => '\Server Work Queues(0)\Queue Length',
                'desc' => _('Server Work Queues'),
                'sleep' => '',
                'warning' => 4,
                'critical' => 7
            );
        }
    }

    $default_services['counters'][] = array(
            'monitor' => 'off',
            'name' => '',
            'desc' => '',
            'sleep' => '',
            'warning' => '',
            'critical' => ''
        );

    return $default_services;
}

/*
 * Build multi-select data
 *
 * Example:
 * $service_select_html = setup_multi_select('services', $services, $root, $services_by_name, $services_array);
 *
 * TODO: This should be moved to a shared wizard file, so it can be reused for all the multi-selects.
 *       May be more appropriate to be called in Step2.
 */
function setup_multi_select($types, $services_by_type, $root, &$metrics_by_name, &$selected_metrics_array) {

    # TODO: Why are we doing this?
    foreach($services_by_type[$types] as $metrics) {
        if ($metrics['name']) {
            # TODO: This breaks the iterative arrays, because it collapses duplicate names, like multiple plugins.
            $metrics_by_name[$metrics['name']] = $metrics;
        }
    }

    $metrics_list = '';

    foreach ($root[$types] as $key => $value) {
        $selected = '';

        # Associative array
        if (!array_is_list($root[$types])) {
            $name = encode_form_val($key);
            $safe_name = clean_str_for_dom($name);
            $status = encode_form_val($value);
            $selected_metrics_array[$name] = $status;

            if (isset($metrics_by_name[$name]) && array_key_exists('monitor', $metrics_by_name[$name]) && $metrics_by_name[$name]['monitor'] == 'on') {
                $selected = ' selected';
            }

            $metrics_list .= "<option value='$safe_name' data-status='$status' $selected>$name ($status)</option>";

        # Iterative array - may have duplicate plugins setup
        } else {
            $name = encode_form_val($value);
            $safe_name = clean_str_for_dom($name);
            $selected_metrics_array[$name] = '';

            if (isset($metrics_by_name[$name]) && array_key_exists('monitor', $metrics_by_name[$name]) && $metrics_by_name[$name]['monitor'] == 'on') {
                $selected = ' selected';
            }

            $metrics_list .= "<option value='$safe_name' $selected>$name</option>";
        }
    }

    ob_start();
?>
    <select name="<?= $types ?>-select[]" id="<?= $types ?>-select" multiple class="form form-control metrics-select multiselect form-select">
        <?= $metrics_list ?>
    </select>
<?php

    $multi_select_data = ob_get_clean();

    return $multi_select_data;
}

# Function to test whether an array is associative or just iterative.
# array_is_list() is a function in PHP 8.
if (!function_exists('array_is_list')) {
    function array_is_list(array $arr) {
        if ($arr === []) {
            return true;
        }

        return array_keys($arr) === range(0, count($arr) - 1);
    }
}
