<?php
//
// Nagios XI Operations Center Component
// Copyright (c) 2010-2021 Nagios Enterprises, LLC. All rights reserved.
//

require_once(dirname(__FILE__) . '/../../common.inc.php');

// Initialization and session start
pre_init();
init_session();
grab_request_vars();
check_prereqs();

// Check authentication
check_authentication(false);

///////////////////MAIN////////////////////////

$cmd = grab_request_var('cmd', '');
$summary = grab_request_var('summary', false);

// Check for commands
if (!empty($cmd)) {
    if ($cmd == 'setfilter') {
        $show = grab_request_var('show', 0);
        set_user_meta(0, 'nocscreen_show_filters', $show);
    } else if ($cmd == 'setrefresh') {
        $refresh = intval(grab_request_var('refresh', 10));
        if ($refresh < 1) {
            $refresh = 1;
        }
        set_user_meta(0, 'nocscreen_page_refresh', $refresh);
    }
}

// Check for data summaries
if ($summary) {
    noc_summary_table();
} else {
    show_all_tables();
}

////////////////FUNCTIONS//////////////////////////

function show_all_tables()
{
    noc_host_table();
    noc_service_table();
}

function noc_get_host_problems()
{
    global $request;
    
    $host = grab_request_var("host", "");
    $service = grab_request_var("service", "");
    $hostgroup = grab_request_var("hostgroup", "");
    $servicegroup = grab_request_var("servicegroup", "");
    $filters = grab_request_var("filters", array());

    // Set orderby
    $host_orderby = grab_request_var("host_orderby", "last_state_change:d");
    $orderby_update = grab_request_var("orderby_update", 0);
    if ($orderby_update) {
        set_user_meta(0,'nocscreen_host_orderby', $host_orderby);
    }

    $backendargs = array();
    $backendargs["cmd"] = "gethoststatus";
    $backendargs["current_state"] = "in:1,2";
    $backendargs["has_been_checked"] = 1;
    $backendargs["orderby"] = $host_orderby;
    $backendargs['is_active'] = 1;
    $backendargs["problem_acknowledged"] = 0;
    
    // QUERIES HERE
    // If hide handled filter exists - set scheduled_downtime_depth and problem_acknowledged to 0
    // This only grabs items NOT in downtime OR NOT acknowledged
    if (in_array('hide_handled', $filters)) {
        $backendargs["scheduled_downtime_depth"] = 0;
        $backendargs["problem_acknowledged"] = 0;
    } else {
        $backendargs["scheduled_downtime_depth"] = "in:0,1";
        $backendargs["problem_acknowledged"] = "in:0,1";
    }

    // Hide soft states if filter exists
    if (in_array('hide_soft', $filters)) {
        $backendargs["state_type"] = 1;
    } else {
        $backendargs["state_type"] = "in:0,1";
    }

    // Hide notifications disabled if filter exists
    if (in_array('hide_not_dis', $filters)) {
        $backendargs["notifications_enabled"] = 1;
    } else {
        $backendargs["notifications_enabled"] = "in:0,1";
    }

    if ($host != "") {
        $backendargs["name"] = $host;
    } else if ($hostgroup != "" || $servicegroup != "") {
        if ($hostgroup != "") {
            $host_ids = get_hostgroup_member_ids($hostgroup);
        } //  limit by hostgroup hosts
        else if ($servicegroup != "") {
            $host_ids = get_servicegroup_host_member_ids($servicegroup);
        } //  limit by host
        else if ($host != "") {
            $host_ids[] = get_host_id($host);
        }
        $y = 0;
        $host_ids_str = "";
        foreach ($host_ids as $hid) {
            if ($y > 0)
                $host_ids_str .= ",";
            $host_ids_str .= $hid;
            $y++;
        }

        if ($host_ids_str != "") {
            $backendargs["host_id"] = "in:" . $host_ids_str;
        }
    }

    $hosts_xml = get_xml_host_status($backendargs);
    if ($hosts_xml) {
        return $hosts_xml;
    }
}

function noc_get_service_problems()
{
    global $request;

    $host = grab_request_var("host", "");
    $service = grab_request_var("service", "");
    $hostgroup = grab_request_var("hostgroup", "");
    $servicegroup = grab_request_var("servicegroup", "");
    $state = grab_request_var("state", "");
    $filters = grab_request_var("filters", array());

    // Set orderby
    $service_orderby = grab_request_var("service_orderby", "last_state_change:d");
    $orderby_update = grab_request_var("orderby_update", 0);
    if ($orderby_update) {
        set_user_meta(0,'nocscreen_service_orderby', $service_orderby);
    }

    $backendargs = array();
    $backendargs["cmd"] = "getservicestatus";
    $backendargs["combinedhost"] = 1;
    $backendargs["current_state"] = "in:1,2,3";
    $backendargs["has_been_checked"] = 1;
    $backendargs["orderby"] = $service_orderby;
    $backendargs['is_active'] = 1;

    // If hide handled filter exists - set scheduled_downtime_depth and problem_acknowledged to 0
    // This only grabs items NOT in downtime OR NOT acknowledged
    if (in_array('hide_handled', $filters)) {
        $backendargs["scheduled_downtime_depth"] = 0;
        $backendargs["problem_acknowledged"] = 0;
    } else {
        $backendargs["scheduled_downtime_depth"] = "in:0,1";
        $backendargs["problem_acknowledged"] = "in:0,1";
    }

    // Hide soft states if filter exists
    if (in_array('hide_soft', $filters)) {
        $backendargs["state_type"] = 1;
    } else {
        $backendargs["state_type"] = "in:0,1";
    }

    // Hide notifications disabled if filter exists
    if (in_array('hide_not_dis', $filters)) {
        $backendargs["notifications_enabled"] = 1;
    } else {
        $backendargs["notifications_enabled"] = "in:0,1";
    }

    if (!empty($state)) {
        $backendargs["current_state"] = "in:".$state;
    }

    $host_ids = array();
    $service_ids = array();

    if ($service != "") {
        $backendargs["host"] = $host;
        $backendargs["service"] = $service;
    } else if ($host != "") {
        $backendargs["host_name"] = $host;
    } else if ($hostgroup != "" || $servicegroup != "") {
        if ($hostgroup != "") {
            $host_ids = get_hostgroup_member_ids($hostgroup);
        } // limit by servicegroup hosts
        else if ($servicegroup != "") {
            // limit by servicegroup
            $service_ids = get_servicegroup_member_ids($servicegroup);
        } // limit by host
        else if ($host != "") {
            $host_ids[] = get_host_id($host);
        }

        $y = 0;
        $service_ids_str = "";
        foreach ($service_ids as $sid) {
            if ($y > 0)
                $service_ids_str .= ",";
            $service_ids_str .= $sid;
            $y++;
        }

        $y = 0;
        $host_ids_str = "";
        foreach ($host_ids as $hid) {
            if ($y > 0) {
                $host_ids_str .= ",";
            }
            $host_ids_str .= $hid;
            $y++;
        }

        if ($service_ids_str != "") {
            $backendargs["service_id"] = "in:" . $service_ids_str;
        }

        if ($host_ids_str != "") {
            $backendargs["host_id"] = "in:" . $host_ids_str;
        }
    }
    $xml = get_xml_service_status($backendargs);
    return $xml;
}

function host_class($code)
{
    switch ($code) {
        case 0:
            return "hostup";
        case 1:
            return 'hostdown';
        default:
            return 'hostunreachable';
    }
}

function service_class($code)
{
    switch ($code) {
        case 0:
            return "serviceok";
        case 1:
            return 'servicewarning';
        case 2:
            return 'servicecritical';
        default:
            return 'serviceunknown';
    }
}

function noc_host_table()
{
    $hide_soft = grab_request_var("hide_soft", "");
    $hide_not_dis = grab_request_var("hide_not_dis", 0);
    $hosts = noc_get_host_problems();

    $sort_header = "";
    $host_orderby = grab_request_var("host_orderby", "last_state_change:d");
    $table = "
      <div id='hostdiv'>
        <table id='hosttable' class='tablesorter hoststatustable table table-condensed table-striped table-bordered'><thead>
          <tr>";
    if ($host_orderby == 'name:a') $sort_header = "headerSortDown";
    if ($host_orderby == 'name:d') $sort_header = "headerSortUp";
    $table .= "<th class='sort-header host_order $sort_header'><div class='th_link'>" . _("Host Name") . "</div></th>";
    $sort_header = "";
    if ($host_orderby == 'last_state_change:a') $sort_header = "headerSortDown";
    if ($host_orderby == 'last_state_change:d') $sort_header = "headerSortUp";
    $table .= "<th class='sort-header duration_order $sort_header'><div class='th_link'>" . _("Duration") . "</div></th>
            <th>" . _("Status Information") . "</th>
          </thead></tr>";

    if ($hosts !== NULL && $hosts->recordcount != 0) {
        foreach ($hosts as $host) {
            if (trim("$host->name") != '') {
                noc_host_row($table, $host);
            }
        }
    } else {
        $table .= '<tr><td colspan="4">'._("No hosts matching filters.").'</td></tr>';
    }

    $table .= "</table></div>";

    print $table;
}

function noc_service_table()
{
    $hide_soft = grab_request_var("hide_soft", "");
    $hide_not_dis = grab_request_var("hide_not_dis", 0);
    $hosts = noc_get_service_problems();
    $sort_header = "";
    $service_orderby = grab_request_var("service_orderby", "last_state_change:d");

    if ($service_orderby == 'host_name:a') $sort_header = "headerSortDown";
    if ($service_orderby == 'host_name:d') $sort_header = "headerSortUp";

    $table = "
      <div id='servicediv'>
        <table id='servicetable' class='tablesorter servicestatustable table table-condensed table-striped table-bordered'><thead>
          <tr>
            <th class='sort-header host_order $sort_header'><div class='th_link'>" . _("Host Name") . "</div></th>";
        $sort_header = "";
        if ($service_orderby == 'service_description:a') $sort_header = "headerSortDown";
        if ($service_orderby == 'service_description:d') $sort_header = "headerSortUp";
        $table .= "<th class='sort-header service_order $sort_header'><div class='th_link'>" . _("Service") . "</div></th>";
        $sort_header = "";
        if ($service_orderby == 'last_state_change:a') $sort_header = "headerSortDown";
        if ($service_orderby == 'last_state_change:d') $sort_header = "headerSortUp";
        $table .= "<th><div class='sort-header duration_order $sort_header duration'><div class='th_link'>" . _("Duration") . "</div></div></th>
            <th>" . _("Status Information") . "</th>
          </thead></tr>";

        $inner_table = '';
        if (!empty($hosts)) {
            foreach ($hosts as $host) {
                if (trim("$host->name") != '') {
                    noc_service_row($inner_table, $host);
                }
            }
        }

        // Check if we actually got any services
        if (!empty($inner_table)) {
            $table .= $inner_table;
        } else {
            $table .= '<tr><td colspan="4">'._("No services matching filters.").'</td></tr>';
        }

    $table .= "</table></div>";

    print $table;
}

function noc_host_row(&$table, $host)
{
    $stateduration = get_duration_string(time() - strtotime("$host->last_state_change"));

    //allow html tags?
    $allow_html = is_null(get_option('allow_status_html')) ? false : get_option('allow_status_html');
    $status_info = ($allow_html == true) ? html_entity_decode(strval($host->status_text)) : strval($host->status_text);

    // Add icons to hosts
    $host_icons = "";

    // Flapping
    if (intval($host->is_flapping) == 1) {
        $host_icons .= get_host_status_note_image("flapping.png", _("This host is flapping"));
    }

    // Acknowledged
    if (intval($host->problem_acknowledged) == 1) {
        $host_icons .= get_host_status_note_image("ack.png", _("This host problem has been acknowledged"));
    }

    $passive_checks_enabled = intval($host->passive_checks_enabled);
    $active_checks_enabled = intval($host->active_checks_enabled);

    // Passive only
    if ($active_checks_enabled == 0 && $passive_checks_enabled == 1) {
        $host_icons .= get_host_status_note_image("passiveonly.png", _("Passive Only Check"));
    }

    // Notifications
    if (intval($host->notifications_enabled) == 0) {
        $host_icons .= get_host_status_note_image("nonotifications.png", _("Notifications are disabled for this host"));
    }

    // Downtime
    if (intval($host->scheduled_downtime_depth) > 0) {
        $host_icons .= get_host_status_note_image("downtime.png", _("This host is in scheduled downtime"));
    }

    // Add graph icon to hosts
    $graph_icon = '';
    if (function_exists('graphexplorer_component_table_icon')) {
        $args = array('host_name' => strval($host->name));
        graphexplorer_component_table_icon('', $args);
        if (isset($args['icons'][0])) {
            $graph_icon = $args['icons'][0];
        }
    }

    // Host icon
    $host_icons .= get_object_icon_html($host->icon_image, $host->icon_image_alt);
 
    if (!isset($class)) {
        $class = '';
    }
    if(is_neptune()) {
        $table .=
            "<tr class='hostrow'>
            <td class='host_name'>
                <a href='../xicore/status.php?show=hostdetail&host={$host->name}'>
                    <span class='status-dot dot-10 " . host_class("$host->current_state") . "'></span>{$host->name}
                </a>
                <div class='hosticons'>" . $host_icons . " " . $graph_icon . "</div>
            </td>
            <td><div class='duration'>{$stateduration}</div></td>
            <td class='statustext'>{$status_info}</td>
        </tr>";
    } else {
        $table .=
            "<tr class='hostrow'>
            <td class='host_name " . host_class("$host->current_state") . "'>
                <a style='color:black' href='../xicore/status.php?show=hostdetail&host={$host->name}'>{$host->name}</a>
                <div class='hosticons'><a href='" . get_host_status_detail_link($host->name) . "'>" . $host_icons . "</a> " . $graph_icon . "</div>
            </td>
            <td><div class='duration'>{$stateduration}</div></td>
            <td class='statustext'>{$status_info}</td>
        </tr>";
    }

}

function noc_service_row(&$table, $host)
{
    $stateduration = get_duration_string(time() - strtotime("$host->last_state_change"));

    // Allow html tags?
    $allow_html = is_null(get_option('allow_status_html')) ? false : get_option('allow_status_html');
    $status_info = ($allow_html == true) ? html_entity_decode(strval($host->status_text)) : strval($host->status_text);

    // Add graph icon to hosts
    $host_graph_icon = '';
    $graph_icon = '';
    if (function_exists('graphexplorer_component_table_icon')) {

        // Host icon
        $args = array('host_name' => strval($host->host_name));
        graphexplorer_component_table_icon('', $args);
        if (isset($args['icons'][0])) {
            $host_graph_icon = $args['icons'][0];
        }

        // Service icon
        $args = array('host_name' => strval($host->host_name), 'service_description' => strval($host->name));
        graphexplorer_component_table_icon('', $args);
        if (isset($args['icons'][0])) {
            $graph_icon = $args['icons'][0];
        }
    }

    if (!isset($class)) {
        $class = '';
    }
    if (host_class("$host->host_current_state") == "hostup") {
        if (is_neptune()) {
            $table .=
            "<tr class='servicerow'>
            <td  class='host_name table-host-width'><a href='../xicore/status.php?show=hostdetail&host={$host->host_name}'><span class='status-dot dot-10 " . host_class("$host->host_current_state") . "'></span>*{$host->host_name}</a> <div class='hosticons'>" . $host_graph_icon . "</div></td>
            <td  class='description table-service-width'><a href='../xicore/status.php?show=servicedetail&host={$host->host_name}&service={$host->name}'><span class='status-dot dot-10 " . service_class("$host->current_state") . "'></span>{$host->name}</a> <div class='hosticons'>" . $graph_icon . "</div></td>
            <td><div class='duration'>{$stateduration}</div></td>
            <td class='statustext'>{$status_info}</td>
        </tr>";
        } else {
            $table .=
            "<tr class='servicerow'>
            <td class='host_name " . host_class("$host->host_current_state") . "'><a style='color:black' href='../xicore/status.php?show=hostdetail&host={$host->host_name}'>*{$host->host_name}</a> <div class='hosticons'>" . $host_graph_icon . "</div></td>
            <td class='description " . service_class("$host->current_state") . "'><a style='color:black' href='../xicore/status.php?show=servicedetail&host={$host->host_name}&service={$host->name}'>{$host->name}</a> <div class='hosticons'>" . $graph_icon . "</div></td>
            <td><div class='duration'>{$stateduration}</div></td>
            <td class='statustext'>{$status_info}</td>
        </tr>";
        }
    }
}

function noc_service_summary_data()
{
    global $request;

    $host = grab_request_var("host", "");
    $service = grab_request_var("service", "");
    $hostgroup = grab_request_var("hostgroup", "");
    $servicegroup = grab_request_var("servicegroup", "");
    $hide_soft = grab_request_var("hide_soft", "");

    // PREP TO GET TOTAL RECORD COUNTS FROM BACKEND...
    $backendargs = array();
    $backendargs["cmd"] = "getservicestatus";
    $backendargs["limitrecords"] = false; // don't limit records
    $backendargs["totals"] = 1; // only get recordcount
    $backendargs["combinedhost"] = true; // get host status too

    $host_ids = array();
    $service_ids = array();

    if ($service != "") {
        $backendargs["host"] = $host;
        $backendargs["service"] = $service;
    }
    else if ($host != "")
        $backendargs["host_name"] = $host;
    else if ($hostgroup != "" || $servicegroup != "") {
        if ($hostgroup != "") {
            $host_ids = get_hostgroup_member_ids($hostgroup);
        } //  limit by servicegroup hosts
        else if ($servicegroup != "") {
            //  limit by servicegroup
            $service_ids = get_servicegroup_member_ids($servicegroup);
        } //  limit by host
        else if ($host != "") {
            $host_ids[] = get_host_id($host);
        }

        $y = 0;
        $service_ids_str = "";
        foreach ($service_ids as $sid) {
            if ($y > 0)
                $service_ids_str .= ",";
            $service_ids_str .= $sid;
            $y++;
        }

        $y = 0;
        $host_ids_str = "";
        foreach ($host_ids as $hid) {
            if ($y > 0)
                $host_ids_str .= ",";
            $host_ids_str .= $hid;
            $y++;
        }

        if ($service_ids_str != "")
            $backendargs["service_id"] = "in:" . $service_ids_str;

        if ($host_ids_str != "")
            $backendargs["host_id"] = "in:" . $host_ids_str;
    }

    $xml = get_xml_service_status($backendargs);
    $total_records = 0;
    if ($xml)
        $total_records = intval($xml->recordcount);

    // get state totals (ok/pending checked later)
    $state_totals = array();
    for ($state_type = 1; $state_type <= 3; $state_type++) {
        $backendargs["current_state"] = $state_type;

        // remove from count if not hard state and hiding soft
        if ($hide_soft) {
            $backendargs["state_type"] = 1;
        }

        $xml = get_xml_service_status($backendargs);
        $state_totals[$state_type] = 0;
        if ($xml)
            $state_totals[$state_type] = intval($xml->recordcount);
    }

    // get ok (non-pending)
    $backendargs["current_state"] = 0;
    $backendargs["has_been_checked"] = 1;
    $xml = get_xml_service_status($backendargs);
    $state_totals[0] = 0;
    if ($xml)
        $state_totals[0] = intval($xml->recordcount);

    // get pending
    $backendargs["current_state"] = 0;
    $backendargs["has_been_checked"] = 0;
    $xml = get_xml_service_status($backendargs);
    $state_totals[4] = 0;
    if ($xml)
        $state_totals[4] = intval($xml->recordcount);

    // total problems
    $total_problems = $state_totals[1] + $state_totals[2] + $state_totals[3];

    // unhandled problems
    $backendargs["current_state"] = "in:1,2,3";
    unset($backendargs["has_been_checked"]);
    $backendargs["problem_acknowledged"] = 0;
    $backendargs["scheduled_downtime_depth"] = 0;
    $backendargs["host_current_state"] = 0; // up state
    $xml = get_xml_service_status($backendargs);
    $unhandled_problems = 0;
    if ($xml)
        $unhandled_problems = intval($xml->recordcount);

    $all = array_sum($state_totals);
    return array($state_totals[0], $state_totals[1], $state_totals[2], $state_totals[3], $state_totals[4], $total_problems, $unhandled_problems, $all);

}

function noc_host_summary_data()
{
    global $request;

    $host = grab_request_var("host", "");
    $service = grab_request_var("service", "");
    $hostgroup = grab_request_var("hostgroup", "");
    $servicegroup = grab_request_var("servicegroup", "");
    $hide_soft = grab_request_var("hide_soft", "");

    // PREP TO GET TOTAL RECORD COUNTS FROM BACKEND...
    $backendargs = array();
    $backendargs["cmd"] = "gethoststatus";
    $backendargs["limitrecords"] = false; // don't limit records
    $backendargs["totals"] = 1; // only get recordcount

    if ($host != "")
        $backendargs["name"] = $host;
    if ($hostgroup != "" || $servicegroup != "") {
        if ($hostgroup != "") {
            $host_ids = get_hostgroup_member_ids($hostgroup);

        } //  limit by hostgroup hosts
        else if ($servicegroup != "") {
            $host_ids = get_servicegroup_host_member_ids($servicegroup);

        } //  limit by host
        else if ($host != "") {
            $host_ids[] = get_host_id($host);
        }
        $y = 0;
        $host_ids_str = "";
        foreach ($host_ids as $hid) {
            if ($y > 0)
                $host_ids_str .= ",";
            $host_ids_str .= $hid;
            $y++;
        }

        if ($host_ids_str != "")
            $backendargs["host_id"] = "in:" . $host_ids_str;
    }

    // get total hosts
    $xml = get_xml_host_status($backendargs);
    $total_records = 0;
    if ($xml) $total_records = intval($xml->recordcount);

    // get host totals (up/pending checked later)
    $state_totals = array();
    for ($current_state = 1; $current_state <= 2; $current_state++) {
        $backendargs["current_state"] = $current_state;

        // remove from count if not hard state and hiding soft
        if ($hide_soft) {
            $backendargs["state_type"] = 1;
            unset($backendargs["totals"]);
        }
        $xml = get_xml_host_status($backendargs);
        if ($hide_soft) {
            $curattempt = intval($xml->hoststatus->current_check_attempt);
            $maxattempt = intval($xml->hoststatus->max_check_attempts);
            if (!empty($curattempt) && !empty($maxattempt)) {
                if ($hide_soft && ($curattempt != $maxattempt)) {
                    $backendargs["state_type"] = 0;
                }
            }
        }

        // back into record count
        $backendargs["totals"] = 1;
        $xml = get_xml_host_status($backendargs);
        $state_totals[$current_state] = 0;
        if ($xml)
            $state_totals[$current_state] = intval($xml->recordcount);
    }

    // get up (non-pending)
    $backendargs["current_state"] = 0;
    $backendargs["has_been_checked"] = 1;
    $xml = get_xml_host_status($backendargs);
    $state_totals[0] = 0;
    if ($xml)
        $state_totals[0] = intval($xml->recordcount);
    // get pending
    $backendargs["current_state"] = 0;
    $backendargs["has_been_checked"] = 0;
    $xml = get_xml_host_status($backendargs);
    $state_totals[3] = 0;
    if ($xml) $state_totals[3] = intval($xml->recordcount);

    // total problems
    $total_problems = $state_totals[1] + $state_totals[2];

    // unhandled problems
    $backendargs["current_state"] = "in:1,2";
    unset($backendargs["has_been_checked"]);
    $backendargs["problem_acknowledged"] = 0;
    $backendargs["scheduled_downtime_depth"] = 0;
    $xml = get_xml_host_status($backendargs);
    $unhandled_problems = 0;
    if ($xml) $unhandled_problems = intval($xml->recordcount);

    $all = array_sum($state_totals);
    return array($state_totals[0], $state_totals[1], $state_totals[2], $state_totals[3], $total_problems, $unhandled_problems, $all);

}


function noc_summary_table()
{
    $host = urlencode(grab_request_var("host", ""));
    $hostgroup = urlencode(grab_request_var("hostgroup", ""));
    $servicegroup = urlencode(grab_request_var("servicegroup", ""));

    // Service data
    list($ok, $warning, $critical, $unknown, $spending, $sproblems, $sunhandled, $sall) = noc_service_summary_data();

    // Host data
    list($up, $down, $unreachable, $hpending, $hproblems, $hunhandled, $hall) = noc_host_summary_data();

    if(is_neptune()) {
        $table = '<div id="summary" class="summary">
    <table class="neptune-operations-center-table" id="summarytable">
      <tr class="strong">
        <th></th>
        <th title="' . _('Hosts in UP state') . '" data-placement="left" class="tt-bind">Up</th>
        <th title="' . _('Host in DOWN state') . '" data-placement="left" class="tt-bind">D</th>
        <th title="' . _('Host in UNREACHABLE state') . '" data-placement="left" class="tt-bind">UR</th>
        <th title="' . _('Hosts with host check PENDING') . '" data-placement="left" class="tt-bind">Pe</th>
        <th title="' . _('Hosts with unhandled problems') . '" data-placement="left" class="tt-bind">UH</th>
        <th title="' . _('Hosts with problems') . '" data-placement="left" class="tt-bind">Pr</th>
        <th title="' . _('Total count of hosts') . '" data-placement="left" class="tt-bind">All</th>
        <th>&nbsp;</th>
        <th title="' . _('Services in OK state')             . '" class="tt-bind" data-placement="left">Ok</th>
        <th title="' . _('Services in WARNING state')        . '" class="tt-bind" data-placement="left">W</th>
        <th title="' . _('Services in CRITICAL state')       . '" class="tt-bind" data-placement="left">Cr</th>
        <th title="' . _('Services in UNKNOWN state')        . '" class="tt-bind" data-placement="left">UK</th>
        <th title="' . _('Services with check PENDING')      . '" class="tt-bind" data-placement="left">Pe</th>
        <th title="' . _('Services with unhandled problems') . '" class="tt-bind" data-placement="left">UH</th>
        <th title="' . _('Services with problems')           . '" class="tt-bind" data-placement="left">Pr</th>
        <th title="' . _('Total count of services')          . '" class="tt-bind" data-placement="left">All</th>
      </tr>

      <tr class="hostrow">
        <td> ' . _("Hosts") . '</td>
        <td class="havehostup"><a href="/nagiosxi/includes/components/xicore/status.php?&show=hosts&hoststatustypes=2&hostgroup='.$hostgroup.'&host='.$host.'"><span class="status-dot dot-10 hostup"></span>'.$up.'</a></td>
        <td class="havehostdown"><a href="/nagiosxi/includes/components/xicore/status.php?&show=hosts&hoststatustypes=4&hostgroup='.$hostgroup.'&host='.$host.'"><span class="status-dot dot-10 hostdown"></span>'.$down.'</a></td>
        <td  style="background-color:transparent !important;" class="serviceunknown"><a href="/nagiosxi/includes/components/xicore/status.php?&show=hosts&hoststatustypes=8&hostgroup='.$hostgroup.'&host='.$host.'"><span style="margin-left:-10px;" class="status-dot dot-10 serviceunknown-dashboard "></span>'.$unreachable.'</a></td>
        <td class="havehostpending"><a href="/nagiosxi/includes/components/xicore/status.php?&show=hosts&hoststatustypes=1&servicestatustypes=0&hostgroup='.$hostgroup.'&host='.$host.'"><span class="status-dot dot-10 hostpending"></span>'.$hpending.'</a></td>
        <td class="haveunhandledhostproblems"><a href="/nagiosxi/includes/components/xicore/status.php?&show=hosts&servicestatustypes=0&hoststatustypes=12&hostattr=10&hostgroup='.$hostgroup.'&host='.$host.'"><span class="status-dot dot-10 hostunhandled"></span>'.$hunhandled.'</a></td>
        <td class="havehostproblems"><a href="/nagiosxi/includes/components/xicore/status.php?&show=hosts&hoststatustypes=12&hostgroup='.$hostgroup.'&host='.$host.'"><span class="status-dot dot-10 hostproblem"></span>'.$hproblems.'</a></td>
        <td class="all"><a href="/nagiosxi/includes/components/xicore/status.php?&show=hosts&hostgroup='.$hostgroup.'&host='.$host.'">'.$hall.'</a></td>';

        if (empty($host)) {
            $table .= '
            <!-- service data -->
            <td class=""> ' . _("Services") . '</td>
            <td class="haveserviceok"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=2&servicegroup='.$servicegroup.'"><span class="status-dot dot-10 serviceok"></span>'.$ok.'</a></td>
            <td class="haveservicewarning"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=4&servicegroup='.$servicegroup.'"><span class="status-dot dot-10 servicewarning-dashboard"></span>'.$warning.'</a></td>
            <td class="haveservicecritical"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=16&servicegroup='.$servicegroup.'"><span class="status-dot dot-10 servicecritical-dashboard"></span>'.$critical.'</a></td>
            <td class="haveserviceunknown" style="background-color:transparent !important;"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=8&servicegroup='.$servicegroup.'"><span class="status-dot dot-10 serviceunknown-dashboard"></span>'.$unknown.'</a></td>
            <td class="haveservicepending"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=1&servicegroup='.$servicegroup.'"><span class="status-dot dot-10 servicepending-dashboard"></span>'.$spending.'</a></td>
            <td class="haveunhandledserviceproblems"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=28&serviceattr=10&servicegroup='.$servicegroup.'"><span class="status-dot dot-10 serviceunhandled"></span>'.$sunhandled.'</a></td>
            <td class="haveserviceproblems"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=28&servicegroup='.$servicegroup.'"><span class="status-dot dot-10 serviceproblem"></span>'.$sproblems.'</a></td>
            <td class="all"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicegroup='.$servicegroup.'">'.$sall.'</a></td>';
        } else {
            $table .= '
            <!-- service data -->
            <td class=""> ' . _("Services") . '</td>
            <td class="haveserviceok"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=2&host='.$host.'"><span class="status-dot dot-10 serviceok"></span>'.$ok.'</a></td>
            <td class="haveservicewarning"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=4&host='.$host.'"><span class="status-dot dot-10 servicewarning-dashboard"></span>'.$warning.'</a></td>
            <td class="haveservicecritical"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=16&host='.$host.'"><span class="status-dot dot-10 servicecritical-dashboard"></span>'.$critical.'</a></td>
            <td class="haveserviceunknown" style="background-color:transparent !important;"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=8&host='.$host.'"><span class="status-dot dot-10 serviceunknown-dashboard"></span>'.$unknown.'</a></td>
            <td class="haveservicepending"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=1&host='.$host.'"><span class="status-dot dot-10 servicepending-dashboard"></span>'.$spending.'</a></td>
            <td class="haveunhandledserviceproblems"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=28&serviceattr=10&host='.$host.'"><span class="status-dot dot-10 serviceunhandled"></span>'.$sunhandled.'</a></td>
            <td class="haveserviceproblems"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=28&host='.$host.'"><span class="status-dot dot-10 serviceproblem"></span>'.$sproblems.'</a></td>
            <td class="all"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&host='.$host.'">'.$sall.'</a></td>';
        }

    $table .= '</tr> <!-- end services row -->
    </table>  <!-- end summary table -->
    </div> <!-- end summmary div -->';



    } else {

        $table = '<div id="summary" class="summary">
    <table class="standardtable service_status_summary_dashlet host_status_summary_dashlet table table-condensed table-striped table-bordered" id="summarytable">
      <tr class="strong">
        <th></th>
        <th title="' . _('Hosts in UP state') . '" data-placement="left" class="tt-bind">Up</th>
        <th title="' . _('Host in DOWN state') . '" data-placement="left" class="tt-bind">D</th>
        <th title="' . _('Host in UNREACHABLE state') . '" data-placement="left" class="tt-bind">UR</th>
        <th title="' . _('Hosts with host check PENDING') . '" data-placement="left" class="tt-bind">Pe</th>
        <th title="' . _('Hosts with unhandled problems') . '" data-placement="left" class="tt-bind">UH</th>
        <th title="' . _('Hosts with problems') . '" data-placement="left" class="tt-bind">Pr</th>
        <th title="' . _('Total count of hosts') . '" data-placement="left" class="tt-bind">All</th>
        <th>&nbsp;</th>
        <th title="' . _('Services in OK state')             . '" class="tt-bind" data-placement="left">Ok</th>
        <th title="' . _('Services in WARNING state')        . '" class="tt-bind" data-placement="left">W</th>
        <th title="' . _('Services in CRITICAL state')       . '" class="tt-bind" data-placement="left">Cr</th>
        <th title="' . _('Services in UNKNOWN state')        . '" class="tt-bind" data-placement="left">UK</th>
        <th title="' . _('Services with check PENDING')      . '" class="tt-bind" data-placement="left">Pe</th>
        <th title="' . _('Services with unhandled problems') . '" class="tt-bind" data-placement="left">UH</th>
        <th title="' . _('Services with problems')           . '" class="tt-bind" data-placement="left">Pr</th>
        <th title="' . _('Total count of services')          . '" class="tt-bind" data-placement="left">All</th>
      </tr>

      <tr class="hostrow">
        <td class="strong"> ' . _("Hosts") . ': </td>
        <td class="hostup havehostup"><a href="/nagiosxi/includes/components/xicore/status.php?&show=hosts&hoststatustypes=2&hostgroup='.$hostgroup.'&host='.$host.'">'.$up.'</a></td>
        <td class="hostdown havehostdown"><a href="/nagiosxi/includes/components/xicore/status.php?&show=hosts&hoststatustypes=4&hostgroup='.$hostgroup.'&host='.$host.'">'.$down.'</a></td>
        <td class="serviceunknown haveserviceunknown"><a href="/nagiosxi/includes/components/xicore/status.php?&show=hosts&hoststatustypes=8&hostgroup='.$hostgroup.'&host='.$host.'">'.$unreachable.'</a></td>
        <td class="hostpending havehostpending"><a href="/nagiosxi/includes/components/xicore/status.php?&show=hosts&hoststatustypes=1&servicestatustypes=0&hostgroup='.$hostgroup.'&host='.$host.'">'.$hpending.'</a></td>
        <td class="unhandledhostproblems haveunhandledhostproblems"><a href="/nagiosxi/includes/components/xicore/status.php?&show=hosts&servicestatustypes=0&hoststatustypes=12&hostattr=10&hostgroup='.$hostgroup.'&host='.$host.'">'.$hunhandled.'</a></td>
        <td class="hostproblems havehostproblems"><a href="/nagiosxi/includes/components/xicore/status.php?&show=hosts&hoststatustypes=12&hostgroup='.$hostgroup.'&host='.$host.'">'.$hproblems.'</a></td>
        <td class="all"><a href="/nagiosxi/includes/components/xicore/status.php?&show=hosts&hostgroup='.$hostgroup.'&host='.$host.'">'.$hall.'</a></td>';

        if (empty($host)) {
            $table .= '
            <!-- service data -->
            <td class="strong"> ' . _("Services") . ': </td>
            <td class="serviceok haveserviceok"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=2&servicegroup='.$servicegroup.'">'.$ok.'</a></td>
            <td class="servicewarning haveservicewarning"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=4&servicegroup='.$servicegroup.'">'.$warning.'</a></td>
            <td class="servicecritical haveservicecritical"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=16&servicegroup='.$servicegroup.'">'.$critical.'</a></td>
            <td class="serviceunknown haveserviceunknown"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=8&servicegroup='.$servicegroup.'">'.$unknown.'</a></td>
            <td class="servicepending haveservicepending"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=1&servicegroup='.$servicegroup.'">'.$spending.'</a></td>
            <td class="unhandledserviceproblems haveunhandledserviceproblems"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=28&serviceattr=10&servicegroup='.$servicegroup.'">'.$sunhandled.'</a></td>
            <td class="serviceproblems haveserviceproblems"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=28&servicegroup='.$servicegroup.'">'.$sproblems.'</a></td>
            <td class="all"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicegroup='.$servicegroup.'">'.$sall.'</a></td>';
        } else {
            $table .= '
            <!-- service data -->
            <td class="strong"> ' . _("Services") . ': </td>
            <td class="serviceok haveserviceok"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=2&host='.$host.'">'.$ok.'</a></td>
            <td class="servicewarning haveservicewarning"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=4&host='.$host.'">'.$warning.'</a></td>
            <td class="servicecritical haveservicecritical"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=16&host='.$host.'">'.$critical.'</a></td>
            <td class="serviceunknown haveserviceunknown"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=8&host='.$host.'">'.$unknown.'</a></td>
            <td class="servicepending haveservicepending"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=1&host='.$host.'">'.$spending.'</a></td>
            <td class="unhandledserviceproblems haveunhandledserviceproblems"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=28&serviceattr=10&host='.$host.'">'.$sunhandled.'</a></td>
            <td class="serviceproblems haveserviceproblems"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&servicestatustypes=28&host='.$host.'">'.$sproblems.'</a></td>
            <td class="all"><a href="/nagiosxi/includes/components/xicore/status.php?&show=services&host='.$host.'">'.$sall.'</a></td>';
        }
    $table .= '</tr> <!-- end services row -->
    </table>  <!-- end summary table -->
    </div> <!-- end summmary div -->';
    }

    print $table;
}
