<?php
//
// Nagios Network Analyzer Config Wizard
// Copyright (c) 2014-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

nna_configwizard_init();

function nna_configwizard_init()
{
    $name = "nna";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.2",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor a source, view, or sourcegroup on a Nagios Network Analyzer server."),
        CONFIGWIZARD_DISPLAYTITLE => _("Nagios Network Analyzer"),
        CONFIGWIZARD_FUNCTION => "nna_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "networkanalyzer.png",
        CONFIGWIZARD_FILTER_GROUPS => array('nagios', 'network'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function nna_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "nna";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            // Get variables that were passed to us
            $nna_server = grab_array_var($inargs, "nna_server", "||");
            $object_type = grab_array_var($inargs, "object_type", "");
            $object_id = grab_array_var($inargs, "object_id", 0);
            $hostname = grab_array_var($inargs, "hostname", "");

            // Grab the view
            $use_view = grab_array_var($inargs, "use_view", 0);
            $view_id = grab_array_var($inargs, "view_id", 0);
            $view_name = grab_array_var($inargs, "view_name", "");

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // get variables that were passed to us
            $nna_server = grab_array_var($inargs, "nna_server", "||");
            $object_type = grab_array_var($inargs, "object_type", "");
            $object_id = grab_array_var($inargs, "object_id", 0);
            $hostname = grab_array_var($inargs, "hostname", "");

            // grab the view..
            $use_view = grab_array_var($inargs, "use_view", 0);
            $view_id = grab_array_var($inargs, "view_id", 0);
            $view_name = grab_array_var($inargs, "view_name", "");

            // Grab server vars
            list($address, $secure, $token) = explode("|", $nna_server);

            // check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($address) == false) {
                $errmsg[$errors++] = "No address specified.";
            }

            if (have_value($token) == false) {
                $errmsg[$errors++] = "No API key specified.";
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;


        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // get variables that were passed to us
            $nna_server = grab_array_var($inargs, "nna_server", "||");
            $object_type = grab_array_var($inargs, "object_type", "");
            $object_id = grab_array_var($inargs, "object_id", 0);
            $hostname = grab_array_var($inargs, "hostname", "");

            // grab the view..
            $use_view = grab_array_var($inargs, "use_view", 0);
            $view_id = grab_array_var($inargs, "view_id", 0);
            $view_name = grab_array_var($inargs, "view_name", "");

            // Grab server vars
            list($address, $secure, $token) = explode("|", $nna_server);

            // if ($object_type == 'sources') {
            //     $object_js_type = "vars.sid = " . $object_id . ";";
            // } else if ($object_type == 'groups') {
            //     $object_js_type = "vars.gid = " . $object_id . ";";
            // }

            // if ($use_view) {
            //     $object_js_view = "vars.vid = " . $view_id . ";";
            // } else {
            //     $object_js_view = "";
            // }

            // Grab graph data for the source
            $s = "http";

            if ($secure) {
                $s = "https";
            }

            $nna_api_url = $s . "://" . $address . "/nagiosna/index.php/api/graphs/execute";

            // Grab the services from the session
            $services = grab_array_var($_SESSION, "nna_services", array());

            if (empty($services)) {
                $services['bytes']['monitor'] = 'on';
                $services['flows']['monitor'] = 'on';
                $services['packets']['monitor'] = 'on';
                $services['behavior']['monitor'] = 'on';
                $services['bytes']['warning'] = '';
                $services['flows']['warning'] = '';
                $services['packets']['warning'] = '';
                $services['behavior']['warning'] = '';
                $services['bytes']['critical'] = '';
                $services['flows']['critical'] = '';
                $services['packets']['critical'] = '';
                $services['behavior']['critical'] = '';
            }

            // Create the new hostname if we are using a view
            if ($use_view && strpos($hostname, $view_name) === FALSE) {
                $hostname .= " - " . $view_name;
            }

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // Get variables that were passed
            $nna_server = grab_array_var($inargs, "nna_server", "||");
            $hostname = grab_array_var($inargs, "hostname", "");
            $services = grab_array_var($inargs, "services", array());
            $object_type = grab_array_var($inargs, "object_type", "");
            $object_id = grab_array_var($inargs, "object_id", 0);

            // grab the view..
            $use_view = grab_array_var($inargs, "use_view", 0);
            $view_id = grab_array_var($inargs, "view_id", 0);
            $view_name = grab_array_var($inargs, "view_name", "");

            // Checking for errors...
            $errors = 0;
            $errmsg = array();

            // If no services are going to be created
            if (count($services) == 0) {
                $_SESSION['nna_services'] = array();
                $errmsg[$errors++] = "You must add services to be defined.";
            }

            // Check to make sure there are warning and critical values for abnormal behavior
            if ($services['bytes']['monitor']) {
                if ($services['bytes']['warning'] == "" || $services['bytes']['critical'] == "") {
                    $errmsg[$errors++] = "You must set a warning and critical value if you are going to monitor bytes.";
                }
            }

            if ($services['flows']['monitor']) {
                if ($services['flows']['warning'] == "" || $services['flows']['critical'] == "") {
                    $errmsg[$errors++] = "You must set a warning and critical value if you are going to monitor flows.";
                }
            }

            if ($services['packets']['monitor']) {
                if ($services['packets']['warning'] == "" || $services['packets']['critical'] == "") {
                    $errmsg[$errors++] = "You must set a warning and critical value if you are going to monitor packets.";
                }
            }

            // Make sure hostname is valid
            if (is_valid_host_name($hostname) == false)
                $errmsg[$errors++] = "Invalid host name.";

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // get variables that were passed to us
            $nna_server = grab_array_var($inargs, 'nna_server', '||');
            $hostname = grab_array_var($inargs, 'hostname');
            $services = grab_array_var($inargs, "services", array());
            $object_type = grab_array_var($inargs, "object_type", "");
            $object_id = grab_array_var($inargs, "object_id", 0);

            // grab the view..
            $use_view = grab_array_var($inargs, "use_view", 0);
            $view_id = grab_array_var($inargs, "view_id", 0);
            $view_name = grab_array_var($inargs, "view_name", "");

            // Store the services in session for later
            $_SESSION['nna_services'] = $services;

            $output = "
            <input type='hidden' name='hostname' value='" . encode_form_val($hostname) . "' />
            <input type='hidden' name='nna_server' value='" . $nna_server . "' />
            <input type='hidden' name='object_type' value='" . $object_type . "' />
            <input type='hidden' name='object_id' value='" . $object_id . "' />
            <input type='hidden' name='use_view' value='" . $use_view . "' />
            <input type='hidden' name='view_id' value='" . $view_id . "' />
            <input type='hidden' name='view_name' value='" . $view_name . "' />";

            // print_r($services);
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:


            $output = '
            ';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            // Grab request vars
            $hostname = grab_array_var($inargs, "hostname", "");
            $nna_server = grab_array_var($inargs, "nna_server", "||");
            $services = grab_array_var($inargs, "services", array());
            $object_type = grab_array_var($inargs, "object_type", "");
            $object_id = grab_array_var($inargs, "object_id", 0);

            // grab the view..
            $use_view = grab_array_var($inargs, "use_view", 0);
            $view_id = grab_array_var($inargs, "view_id", 0);
            $view_name = grab_array_var($inargs, "view_name", "");

            // Grab stuff from NNA servers
            list($address, $use_https, $token, $invalid_cert) = explode("|", $nna_server);
            $hostaddress = $address;

            // Grab the services out of the session (and clear session for a new run)
            $services = grab_array_var($_SESSION, "nna_services", array());
            $_SESSION['nna_services'] = array();

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["address"] = $address;
            $meta_arr["token"] = $token;
            $meta_arr["services"] = $services;
            $meta_arr["https"] = $use_https;
            $meta_arr["object_type"] = $object_type;
            $meta_arr["object_id"] = $object_id;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = array();

            if (!host_exists($hostname)) {

                $exists_command = "";

                if ($object_type == "sources") {
                    $exists_command = "-S";
                } else if ($object_type == "groups") {
                    $exists_command = "-G";
                }

                $exists_command .= " " . $object_id . " --exists";

                // Check if secure
                if ($use_https) {
                    $exists_command .= " --secure";
                }

                if ($invalid_cert) {
                    $exists_command .= " --ignorecert";
                }

                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_nna_host",
                    "host_name" => $hostname,
                    "address" => $hostaddress,
                    "icon_image" => "networkanalyzer.png",
                    "statusmap_image" => "networkanalyzer.png",
                    "check_command" => "check_xi_nna!" . $token . "!" . $exists_command,
                    "_xiwizard" => $wizard_name
                );
            }

            // Loop through all services and add them
            foreach ($services as $type => $service) {

                if (!$service['monitor']) {
                    continue;
                }

                $type_args = " -m " . $type;

                // Do for abnormal behavior services
                if ($type == 'behavior') {

                    $service_name = "Abnormal Behavior";
                    $check_command = "-S " . $object_id . $type_args;

                } // Do for normal services
                else {

                    if ($object_type == 'sources') {
                        $arg_type = "-S ";
                    } else if ($object_type == 'groups') {
                        $arg_type = "-G ";
                    }

                    $service_name = ucfirst($type);
                    $check_command = $arg_type . $object_id . $type_args . " -w " . $service['warning'] . " -c " . $service['critical'];
                }

                // Check if secure
                if ($use_https) {
                    $check_command .= " --secure";
                }

                // Check if we are using a view
                if ($use_view) {
                    $check_command .= " -v " . $view_id;
                }

                $objs[] = array(
                    "type" => OBJECTTYPE_SERVICE,
                    "host_name" => $hostname,
                    "service_description" => $service_name,
                    "use" => "xiwizard_nna_service",
                    "check_command" => "check_xi_nna!" . $token . "!" . $check_command,
                    "_xiwizard" => $wizard_name);

            }

            //~ echo "OBJECTS:<BR>";
            //~ print_r($objs);
            //~ exit();

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}