<?php
//
// Website Config Wizard
// Copyright (c) 2008-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

website_configwizard_init();

function website_configwizard_init()
{
    $name = "website";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.2",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor a website."),
        CONFIGWIZARD_DISPLAYTITLE => _("Website"),
        CONFIGWIZARD_FUNCTION => "website_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "www_server.png",
        CONFIGWIZARD_FILTER_GROUPS => array('website'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param        $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function website_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "website";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

             // Remove any existing session variables if this is the first time on the page
            if (isset($_POST['backButton']) == false) {
                unset($_SESSION['website_wizard_url']);
                unset($_SESSION['website_wizard_hostname']);
                unset($_SESSION['website_wizard_port']);
                unset($_SESSION['website_wizard_username']);
                unset($_SESSION['website_wizard_serviceargs']);
                $_SESSION['website_wizard_services'] = array('0' => 'http', '1' => 'ping', '2'=> 'dns', '3'=> 'dnsip');
            }

            $url = grab_array_var($inargs, "url", "http://");

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $url = grab_array_var($inargs, "url");
            $url = nagiosccm_replace_user_macros($url);

            if (array_key_exists('website_wizard_url', $_SESSION) && $url == "") {
                $url = $_SESSION['website_wizard_url'];
            }

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($url) == false) {
                $errmsg[$errors++] = _("No URL specified.");
            } else if (!valid_url($url)) {
                $errmsg[$errors++] = _("Invalid URL.");
            }

            if ($url) {
                $_SESSION['website_wizard_url'] = $url;
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $url = grab_array_var($inargs, "url");

            $urlparts = parse_url($url);

            $hostname = grab_array_var($urlparts, "host");
            $urlscheme = grab_array_var($urlparts, "scheme");
            $port = grab_array_var($urlparts, "port");
            $username = grab_array_var($urlparts, "user");
            $password = grab_array_var($urlparts, "pass");
            $services = grab_array_var($inargs, "services", array());
            $serviceargs = grab_array_var($inargs, "serviceargs", array());

            if (array_key_exists('website_wizard_url', $_SESSION) && $url == "") {
                $url = $_SESSION['website_wizard_url'];
            }
            if (array_key_exists('website_wizard_hostname', $_SESSION) && $hostname == "") {
                $hostname = $_SESSION['website_wizard_hostname'];
            }
            if (array_key_exists('website_wizard_port', $_SESSION) && $port == "") {
                $port = $_SESSION['website_wizard_port'];
            }
            if (array_key_exists('website_wizard_username', $_SESSION) && $username == "") {
                $username = $_SESSION['website_wizard_username'];
            }
            if (array_key_exists('website_wizard_services', $_SESSION) && !$services) {
                $services = $_SESSION['website_wizard_services'];
            }
            if (array_key_exists('website_wizard_serviceargs', $_SESSION) && !$serviceargs) {
                $serviceargs = $_SESSION['website_wizard_serviceargs'];
            }

            if ($urlscheme == "https") {
                $ssl = "on";
            } else {
                $ssl = "off";
            }

            if ($port == "") {
                if ($ssl == "on") {
                    $port = 443;
                } else {
                    $port = 80;
                }
            }

            $basicauth = "";

            if ($username != "") {
                $basicauth = "on";
            }

            $ip = gethostbyname($hostname);

            $hostname = grab_array_var($inargs, "hostname", $hostname);
            $port = grab_array_var($inargs, "port", $port);
            $ssl = grab_array_var($inargs, "ssl", $ssl);
            $basicauth = grab_array_var($inargs, "basicauth", $basicauth);
            $username = grab_array_var($inargs, "username", $username);
            $password = grab_array_var($inargs, "password", $password);
            $httpcontentstr = grab_array_var($inargs, "httpcontentstr", "");
            $httpregexstr = grab_array_var($inargs, "httpregexstr", "");
            $sslcertdays = grab_array_var($inargs, "sslcertdays", 30);
            $onredirect = grab_array_var($inargs, "onredirect", "ok");

            if ($url) {
                $_SESSION['website_wizard_url'] = $url;
            }
            if ($hostname) {
                $_SESSION['website_wizard_hostname'] = $hostname;
            }
            if ($port) {
                $_SESSION['website_wizard_port'] = $port;
            }
            if ($username) {
                $_SESSION['website_wizard_username'] = $username;
            }
            if ($services) {
                $_SESSION['website_wizard_services'] = $services;
            }
            if ($serviceargs) {
                $_SESSION['website_wizard_serviceargs'] = $serviceargs;
            }

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // get variables that were passed to us
            $url = grab_array_var($inargs, "url", "");
            $hostname = grab_array_var($inargs, "hostname", "");
            $port = grab_array_var($inargs, "port", "");
            $username = grab_array_var($inargs, "username", "");
            $services = grab_array_var($inargs, "services", array());
            $serviceargs = grab_array_var($inargs, "serviceargs", array());

            if (isset($_POST['submitButton2']) && empty($services)) {
                unset($_SESSION['website_wizard_services']);
            }

            if (array_key_exists('website_wizard_url', $_SESSION) && $url == "") {
                $url = $_SESSION['website_wizard_url'];
            }
            if (array_key_exists('website_wizard_hostname', $_SESSION) && $hostname == "") {
                $hostname = $_SESSION['website_wizard_hostname'];
            }
            if (array_key_exists('website_wizard_port', $_SESSION) && $port == "") {
                $port = $_SESSION['website_wizard_port'];
            }
            if (array_key_exists('website_wizard_username', $_SESSION) && $username == "") {
                $username = $_SESSION['website_wizard_username'];
            }
            if (array_key_exists('website_wizard_services', $_SESSION) && !$services) {
                $services = $_SESSION['website_wizard_services'];
            }
            if (array_key_exists('website_wizard_serviceargs', $_SESSION) && !$serviceargs) {
                $serviceargs = $_SESSION['website_wizard_serviceargs'];
            }

            if (!empty($hostname)) {
                $hostname = nagiosccm_replace_user_macros($hostname);
            }
            if (!empty($port)) {
                $port = nagiosccm_replace_user_macros($port);
            }
            if (!empty($username)) {
                $username = nagiosccm_replace_user_macros($username);
            }
            if (!empty($password)) {
                $password = nagiosccm_replace_user_macros($password);
            }

            // check for errors
            $errors = 0;
            $errmsg = array();

            if (is_valid_host_name($hostname) == false)
                $errmsg[$errors++] = _("Invalid host name.");

            if (have_value($url) == false)
                $errmsg[$errors++] = _("No URL specified.");
            else if (!valid_url($url))
                $errmsg[$errors++] = _("Invalid URL.");

            if (empty($services) == true) {
                $errmsg[$errors++] = _("No services specified.");
            }

            if (in_array("httpcontent", $services)) {
                if (array_key_exists("httpcontentstr", $serviceargs)) {
                    if (have_value($serviceargs["httpcontentstr"]) == false)
                        $errmsg[$errors++] = _("You must specify a string to expect in the web page content.");
                }
            }

            if (in_array("httpregex", $services)) {
                if (array_key_exists("httpregexstr", $serviceargs)) {
                    if (have_value($serviceargs["httpregexstr"]) == false)
                        $errmsg[$errors++] = _("You must specify a regular expression to expect in the web page content.");
                }
            }

            if (array_key_exists("sslcert", $services)) {
                if (array_key_exists("sslcertdays", $serviceargs)) {
                    $n = intval($serviceargs["sslcertdays"]);
                    if ($n <= 0)
                        $errmsg[$errors++] = _("Invalid number of days for SSL certificate expiration.");
                } else
                    $errmsg[$errors++] = _("You must specify the number of days until SSL certificate expiration.");
            }

            if ($url) {
                $_SESSION['website_wizard_url'] = $url;
            }
            if ($hostname) {
                $_SESSION['website_wizard_hostname'] = $hostname;
            }
            if ($port) {
                $_SESSION['website_wizard_port'] = $port;
            }
            if ($username) {
                $_SESSION['website_wizard_username'] = $username;
            }
            if ($services) {
                $_SESSION['website_wizard_services'] = $services;
            }
            if ($serviceargs) {
                $_SESSION['website_wizard_serviceargs'] = $serviceargs;
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // get variables that were passed to us
            $url = grab_array_var($inargs, "url");
            $hostname = grab_array_var($inargs, "hostname");
            $ip = grab_array_var($inargs, "ip");
            $ssl = grab_array_var($inargs, "ssl");
            $port = grab_array_var($inargs, "port");
            $username = grab_array_var($inargs, "username");
            $password = grab_array_var($inargs, "password");
            $onredirect = grab_array_var($inargs, "onredirect");

            $services = grab_array_var($inargs, "services");
            $serviceargs = grab_array_var($inargs, "serviceargs");

            if (array_key_exists('website_wizard_url', $_SESSION) && $url == "") {
                $url = $_SESSION['website_wizard_url'];
            }
            if (array_key_exists('website_wizard_hostname', $_SESSION) && $hostname == "") {
                $hostname = $_SESSION['website_wizard_hostname'];
            }
            if (array_key_exists('website_wizard_port', $_SESSION) && $port == "") {
                $port = $_SESSION['website_wizard_port'];
            }
            if (array_key_exists('website_wizard_username', $_SESSION) && $username == "") {
                $username = $_SESSION['website_wizard_username'];
            }
            if (array_key_exists('website_wizard_services', $_SESSION) && !$services) {
                $services = $_SESSION['website_wizard_services'];
            }
            if (array_key_exists('website_wizard_serviceargs', $_SESSION) && !$serviceargs) {
                $serviceargs = $_SESSION['website_wizard_serviceargs'];
            }

            $services_serial = grab_array_var($inargs, "services_serial", base64_encode(json_encode($services)));
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", base64_encode(json_encode($serviceargs)));

            if ($url) {
                $_SESSION['website_wizard_url'] = $url;
            }
            if ($hostname) {
                $_SESSION['website_wizard_hostname'] = $hostname;
            }
            if ($port) {
                $_SESSION['website_wizard_port'] = $port;
            }
            if ($username) {
                $_SESSION['website_wizard_username'] = $username;
            }
            if ($services) {
                $_SESSION['website_wizard_services'] = $services;
            }
            if ($serviceargs) {
                $_SESSION['website_wizard_serviceargs'] = $serviceargs;
            }

            $output = '

        <input type="hidden" name="url" value="' . encode_form_val($url) . '">
        <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '">
        <input type="hidden" name="ip" value="' . encode_form_val($ip) . '">
        <input type="hidden" name="ssl" value="' . encode_form_val($ssl) . '">
        <input type="hidden" name="port" value="' . encode_form_val($port) . '">
        <input type="hidden" name="username" value="' . encode_form_val($username) . '">
        <input type="hidden" name="password" value="' . encode_form_val($password) . '">
        <input type="hidden" name="onredirect" value="' . $onredirect . '">
        <input type="hidden" name="services_serial" value="' . $services_serial . '">
        <input type="hidden" name="serviceargs_serial" value="' . $serviceargs_serial . '">
';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            $output = '

            ';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $hostname = trim(grab_array_var($inargs, "hostname", ""));
            $ip = grab_array_var($inargs, "ip", "");
            $url = grab_array_var($inargs, "url", "");
            $ssl = grab_array_var($inargs, "ssl");
            $port = grab_array_var($inargs, "port");
            $username = grab_array_var($inargs, "username");
            $password = grab_array_var($inargs, "password");
            $onredirect = grab_array_var($inargs, "onredirect");

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            $urlparts = parse_url($url);
            $hostaddress = $urlparts["host"];

            if (array_key_exists('website_wizard_url', $_SESSION) && $url == "") {
                $url = $_SESSION['website_wizard_url'];
            }
            if (array_key_exists('website_wizard_hostname', $_SESSION) && $hostname == "") {
                $hostname = $_SESSION['website_wizard_hostname'];
            }
            if (array_key_exists('website_wizard_port', $_SESSION) && $port == "") {
                $port = $_SESSION['website_wizard_port'];
            }
            if (array_key_exists('website_wizard_username', $_SESSION) && $username == "") {
                $username = $_SESSION['website_wizard_username'];
            }
            if (array_key_exists('website_wizard_services', $_SESSION) && !$services) {
                $services = $_SESSION['website_wizard_services'];
            }

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["ip"] = $ip;
            $meta_arr["url"] = $url;
            $meta_arr["ssl"] = $ssl;
            $meta_arr["port"] = $port;
            $meta_arr["username"] = $username;
            $meta_arr["password"] = $password;
            $meta_arr["onredirect"] = $onredirect;
            $meta_arr["services"] = $services;
            $meta_arr["serivceargs"] = $serviceargs;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);


            $objs = array();

            if (!host_exists($hostname)) {

                // Choose the port for check_tcp
                $tmp_port = $port;
                if ($port == "") {
                    $tmp_port = "80";
                    if ($ssl == "on") {
                        $tmp_port = "443";
                    }
                }

                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_website_host",
                    "host_name" => $hostname,
                    "address" => $hostaddress,
                    "icon_image" => "www_server.png",
                    "statusmap_image" => "www_server.png",
                    "check_command" => "check_tcp!$tmp_port!",
                    "_xiwizard" => $wizard_name,
                );
            }

            $pluginopts = "";

            $vhost = $urlparts["host"];

            $pluginopts .= " -f " . $onredirect; // on redirect, follow (OK status)
            if (!empty($ip)) {
                $pluginopts .= " -I " . $ip; // ip address
            }

            $urlpath = grab_array_var($urlparts, "path", "/");
            $pluginopts .= " -u \"" . $urlpath . "\"";

            if ($ssl == "on")
                $pluginopts .= " -S --sni";
            if ($port != "")
                $pluginopts .= " -p " . $port;
            if ($username != "")
                $pluginopts .= " -a \"" . $username . ":" . $password . "\"";

            if (count($objs) > 0) {
                if ($ssl == "on") {
                    $objs[0]["check_command"] = "check_xi_service_http!" . $pluginopts;
                } else if (!empty($port)) {
                    $objs[0]["check_command"] = "check_xi_host_http!-p " . intval($port);
                }
            }

            // see which services we should monitor
            foreach ($services as $service) {

                switch ($service) {

                    case "http":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "HTTP",
                            "use" => "xiwizard_website_http_service",
                            "check_command" => "check_xi_service_http!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "httpcontent":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Web Page Content",
                            "use" => "xiwizard_website_http_content_service",
                            //"check_command" => "check_xi_service_http_content!".$serviceargs["httpcontentstr"],
                            "check_command" => "check_xi_service_http!-s \"" . $serviceargs["httpcontentstr"] . "\" " . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "httpregex":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Web Page Regex Match",
                            "use" => "xiwizard_website_http_content_service",
                            "check_command" => "check_xi_service_http!-r \"" . $serviceargs["httpregexstr"] . "\" " . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "sslcert":
                        $cert_port = '';
                        if (!empty($port)) {
                            $cert_port = ' -p ' . intval($port);
                        }
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "SSL Certificate",
                            "use" => "xiwizard_website_http_cert_service",
                            "check_command" => " check_xi_service_http_cert!" . $serviceargs["sslcertdays"] . $cert_port . " -S --sni",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "ping":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Ping",
                            "use" => "xiwizard_website_ping_service",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "dns":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "DNS Resolution",
                            "use" => "xiwizard_website_dns_service",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "dnsip":
                        if (empty($ip)) {
                            $ip = gethostbyname($hostname);
                        }
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "DNS IP Match",
                            "use" => "xiwizard_website_dnsip_service",
                            "check_command" => "check_xi_service_dns!-a " . $ip . "",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    default:
                        break;
                }
            }

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}