<?php
//
// WatchGuard Config Wizard
// Copyright (c) 2008-2024 Nagios Enterprises, LLC. All rights reserved.
//
// TODOS:
// * Smarter MRTG file update
//     Current implementation is naive in that it only looks for a single existing address/port match
//     Make it smarter by determining missing ports in the MRTG file and only adding those...

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

watchguard_configwizard_init();

function watchguard_configwizard_init()
{
    $name = "watchguard";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.2",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor a WatchGuard device."),
        CONFIGWIZARD_DISPLAYTITLE => _("WatchGuard"),
        CONFIGWIZARD_FUNCTION => "watchguard_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "watchguard.png",
        CONFIGWIZARD_FILTER_GROUPS => array('network'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );

    register_configwizard($name, $args);
}

/**
 *
 * Generates the instructions to be passed to the command line to generate an MRTG configuration.
 *
 * @param $snmpopts     Array - Given by the getstage1html
 * @param $address      String - Address of the network device
 * @param $snmpversion  String - Must be either 1, 2, 3
 * @param $defaultspeed String - If nothing is returned by ifSpeed, use this value.
 *
 * @return $cmd String - String to be executed
 */
function watchguard_configwizard_get_cfgmaker_cmd($snmpopts, $address, $port, $snmpversion = "1", $defaultspeed = "100000000")
{
    $cmd = "/usr/bin/cfgmaker ";
    $args[] = "--show-op-down";
    $args[] = "--noreversedns";
    $args[] = "--zero-speed";
    $args[] = escapeshellarg($defaultspeed);

    if (empty($snmpopts['v3_username'])) {
        // Run SNMPv1, SNMPv2, SNMPv2c code here
        $username = $snmpopts['snmpcommunity'];
        $delimitors = ":::::";

        if (!empty($port)) { $delimitors = ":".intval($port)."::::"; }

        $args[] = escapeshellarg("{$username}@{$address}{$delimitors}".(int)$snmpversion);

    } else {
        // Run SNMPv3 code here
        $args[] = "--enablesnmpv3";
        $args[] = "--snmp-options=:::::3";

        if (!empty($snmpopts['v3_username'])) {
            $args[] = "--username=" . escapeshellarg($snmpopts['v3_username']);
        }

        if (!empty($snmpopts['v3_auth_password'])) {
            $args[] = "--authprotocol=" . escapeshellarg(strtolower($snmpopts['v3_auth_proto']));
            $args[] = "--authpassword=" . escapeshellarg($snmpopts['v3_auth_password']);
        }

        if (!empty($snmpopts['v3_priv_password'])) {
            $args[] = "--privprotocol=" . escapeshellarg(strtolower($snmpopts['v3_priv_proto']));
            $args[] = "--privpassword=" . escapeshellarg($snmpopts['v3_priv_password']);
        }

        $args[] = "--contextengineid=0";
        $args[] = escapeshellarg($address);
    }

    $cmd .= implode(' ', $args);

    // Run the cfgmaker through the user macros
    $cmd = nagiosccm_replace_user_macros($cmd);

    return $cmd;
}


/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function watchguard_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "watchguard";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {

        case CONFIGWIZARD_MODE_GETSTAGE1HTML:
            $address = grab_array_var($inargs, "ip_address", "");
            $address_port = grab_array_var($inargs, "port", 161);

            $snmpversion = grab_array_var($inargs, "snmpversion", "2c");
            $snmpopts_serial = grab_array_var($inargs, "snmpopts_serial", "");

            if (!empty($snmpopts_serial)) {
                $snmpopts = json_decode(base64_decode($snmpopts_serial), true);

            } else {
                // Set the defaults if nothing is set yet
                $snmpopts_default = array(
                    "snmpcommunity" => "public",
                    "v3_security_level" => "",
                    "v3_username" => "",
                    "v3_auth_password" => "",
                    "v3_priv_password" => "",
                    "v3_auth_proto" => "MD5",
                    "v3_priv_proto" => "DES"
                );

                $snmpopts = grab_array_var($inargs, "snmpopts", $snmpopts_default);
            }

            $portnames = grab_array_var($inargs, "portnames", "number");
            $scaninterfaces = grab_array_var($inargs, "scaninterfaces", "on");
            $default_port_speed = grab_array_var($inargs, "default_port_speed", 100000000);
            $warn_speed_in_percent = grab_array_var($inargs, "warn_speed_in_percent", 50);
            $warn_speed_out_percent = grab_array_var($inargs, "warn_speed_out_percent", 50);
            $crit_speed_in_percent = grab_array_var($inargs, "crit_speed_in_percent", 80);
            $crit_speed_out_percent = grab_array_var($inargs, "crit_speed_out_percent", 80);

            if (!watchguard_configwizard_checkversion()) {
                $output = "<br/><strong>" . _("Error: This wizard requires Nagios XI 2014 or later.") . "</strong>";
            } else {

                # Get the existing host/node configurations.
                # TODO: Include passwords/secrets?
                $nodes = get_configwizard_hosts($wizard_name);

                ########################################################################################
                # Load the html
                # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
                #   to load the PHP from the Step1 file into the $output string.
                ########################################################################################
                ob_start();
                include __DIR__.'/steps/step1.php';
                $output = ob_get_clean();

            } // end else

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $address_port = grab_array_var($inargs, "port", 161);
            $address = nagiosccm_replace_user_macros($address);
            $address_port = nagiosccm_replace_user_macros($address_port);

            $snmpopts_serial = grab_array_var($inargs, "snmpopts_serial", "");
            if (!empty($snmpopts_serial)) {
                $snmpopts = json_decode(base64_decode($snmpopts_serial), true);

            } else {
                // Set the defaults if nothing is set yet
                $snmpopts_default = array(
                    "snmpcommunity" => "public",
                    "v3_security_level" => "",
                    "v3_username" => "",
                    "v3_auth_password" => "",
                    "v3_priv_password" => "",
                    "v3_auth_proto" => "MD5",
                    "v3_priv_proto" => "DES"
                );

                $snmpopts = grab_array_var($inargs, "snmpopts", $snmpopts_default);
            }

            $scaninterfaces = grab_array_var($inargs, "scaninterfaces");
            $snmpversion = grab_array_var($inargs, "snmpversion", "2c");
            $default_port_speed = grab_array_var($inargs, "default_port_speed", 100000000);
            $errors = 0;
            $errmsg = array();

            // Do error checking
            if (have_value($address) == false) {
                $errmsg[$errors++] = _("No address specified.");
            } else if (!valid_ip($address)) {
                $errmsg[$errors++] = _("Invalid IP address.");
            } else if (empty($snmpopts['snmpcommunity']) && empty($snmpopts['v3_username'])) {
                $errmsg[$errors++] = _("Must give either community or username.");
            }

            // Error results
            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            // If user wants to scan interfaces, immediately launch the command and start working on it....
            if ($scaninterfaces == "on") {

                // Set temp scan directory... normally "/usr/local/nagioxi/tmp"
                $tmp_dir = get_tmp_dir();
                $outfile = $tmp_dir . "/mrtgscan-" . str_replace(array('/', '..'), '', $address);
                $donefile = $outfile . ".done";

                // Get rid of old "done" file if it exists
                if (file_exists($donefile)) {
                    unlink($donefile);
                }

                // Run MRTG's cfgmaker command in the background
                // TODO - see if data already exists in mrtg.cfg and skip this step....
                $cfgmaker_cmd = watchguard_configwizard_get_cfgmaker_cmd($snmpopts, $address, $address_port, $snmpversion, $default_port_speed);
                $cmd = $cfgmaker_cmd . " > " . escapeshellarg($outfile) . " ; touch " . escapeshellarg($donefile) . " > /dev/null &";
                exec($cmd);
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $address_port = grab_array_var($inargs, "port", 161);
            $snmpopts_serial = grab_array_var($inargs, "snmpopts_serial", "");

            if (!empty($snmpopts_serial)) {
                $snmpopts = json_decode(base64_decode($snmpopts_serial), true);

            } else {
                $snmpopts_default = array(
                    "snmpcommunity" => "public",
                    "v3_security_level" => "",
                    "v3_username" => "",
                    "v3_auth_password" => "",
                    "v3_priv_password" => "",
                    "v3_auth_proto" => "md5",
                    "v3_priv_proto" => "des"
                );

                $snmpopts = grab_array_var($inargs, "snmpopts", $snmpopts_default);
                $snmpopts_serial = base64_encode(json_encode($snmpopts));
            }

            $vendor = grab_array_var($inargs, "vendor", "");
            $portnames = grab_array_var($inargs, "portnames");
            $scaninterfaces = grab_array_var($inargs, "scaninterfaces");
            $snmpversion = grab_array_var($inargs, "snmpversion", "2c");
            $default_port_speed = grab_array_var($inargs, "default_port_speed", 100000000);
            $warn_speed_in_percent = grab_array_var($inargs, "warn_speed_in_percent", 50);
            $warn_speed_out_percent = grab_array_var($inargs, "warn_speed_out_percent", 50);
            $crit_speed_in_percent = grab_array_var($inargs, "crit_speed_in_percent", 80);
            $crit_speed_out_percent = grab_array_var($inargs, "crit_speed_out_percent", 80);
            $hostname = @gethostbyaddr($address);

            $services = "";
            $services_default = array(
                "cpu_usage" => "on",
                "ping" => "on",
                "active_connections" => "on",
                "total_sent_packets" => "on",
                "total_received_packets" => "on",
                "stream_requests_total" => "on",
                "stream_requests_drop" => "on",
                "total_sent_bytes" => "on",
                "total_received_bytes" => "on",
            );

            /* Check to see if there is any information in the $services_serial array. This is in case someone used
            a back form button. We use this data to populate the displayed forms. */
            $services_serial = grab_array_var($inargs, "services_serial");

            if ($services_serial != "") {
                $services = json_decode(base64_decode($services_serial), true);
            }

            if (!is_array($services)) {
                $services = grab_array_var($inargs, "services", $services_default);
            }

            $serviceargs = "";
            $serviceargs_default = array(
                "cpu_usage_warning" => "20",
                "cpu_usage_critical" => "40",
                "ping_warning" => "20",
                "ping_critical" => "40",
                "active_connections_warning" => "300",
                "active_connections_critical" => "500",
                "total_sent_bytes_warning" => "1000",
                "total_sent_bytes_critical" => "2000",
                "total_received_bytes_warning" => "1000",
                "total_received_bytes_critical" => "2000",
                "total_sent_packets_warning" => "1000",
                "total_sent_packets_critical" => "2000",
                "total_received_packets_warning" => "1000",
                "total_received_packets_critical" => "2000",
                "stream_requests_total_warning" => "300",
                "stream_requests_total_critical" => "500",
                "stream_requests_drop_warning" => "300",
                "stream_requests_drop_critical" => "500",
            );

            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial");

            if ($serviceargs_serial != "") {
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            }

            if (!is_array($serviceargs)) {
                $serviceargs = grab_array_var($inargs, "serviceargs", $serviceargs_default);
            }

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // get variables that were passed to us
            $hostname = grab_array_var($inargs, "hostname");
            $address = grab_array_var($inargs, "ip_address");
            $portnames = grab_array_var($inargs, "portnames");
            $snmpopts_serial = grab_array_var($inargs, "snmpopts_serial", "");
            if (!empty($snmpopts_serial)) {
                $snmpopts = json_decode(base64_decode($snmpopts_serial), true);
            } else {
                $snmpopts_default = array(
                    "snmpcommunity" => "public",
                    "v3_security_level" => "",
                    "v3_username" => "",
                    "v3_auth_password" => "",
                    "v3_priv_password" => "",
                    "v3_auth_proto" => "md5",
                    "v3_priv_proto" => "des"
                );
                $snmpopts = grab_array_var($inargs, "snmpopts", $snmpopts_default);
                $snmpopts_serial = base64_encode(json_encode($snmpopts));
            }
            $snmpversion = grab_array_var($inargs, "snmpversion", "1");
            $vendor = grab_array_var($inargs, "vendor");
            $scaninterfaces = grab_array_var($inargs, "scaninterfaces");
            $warn_speed_in_percent = grab_array_var($inargs, "warn_speed_in_percent", 50);
            $warn_speed_out_percent = grab_array_var($inargs, "warn_speed_out_percent", 50);
            $crit_speed_in_percent = grab_array_var($inargs, "crit_speed_in_percent", 80);
            $crit_speed_out_percent = grab_array_var($inargs, "crit_speed_out_percent", 80);

            // Check for errors
            $errors = 0;
            $errmsg = array();
            if (is_valid_host_name($hostname) == false) {
                $errmsg[$errors++] = _("Invalid host name.");
            }

            // TODO - check rate in/out warning and critical thresholds

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }
            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $vendor = grab_array_var($inargs, "vendor");
            $portnames = grab_array_var($inargs, "portnames");
            $snmpopts_serial = grab_array_var($inargs, "snmpopts_serial", "");
            if ($snmpopts_serial != "") {
                $snmpopts = json_decode(base64_decode($snmpopts_serial), true);
            } else {
                $snmpopts_default = array(
                    "snmpcommunity" => "public",
                    "v3_security_level" => "",
                    "v3_username" => "",
                    "v3_auth_password" => "",
                    "v3_priv_password" => "",
                    "v3_auth_proto" => "md5",
                    "v3_priv_proto" => "des",
                );
                $snmpopts = grab_array_var($inargs, "snmpopts", $snmpopts_default);
                $snmpopts_serial = base64_encode(json_encode($snmpopts));
            }
            $snmpversion = grab_array_var($inargs, "snmpversion", "1");
            $scaninterfaces = grab_array_var($inargs, "scaninterfaces");
            $warn_speed_in_percent = grab_array_var($inargs, "warn_speed_in_percent", 50);
            $warn_speed_out_percent = grab_array_var($inargs, "warn_speed_out_percent", 50);
            $crit_speed_in_percent = grab_array_var($inargs, "crit_speed_in_percent", 80);
            $crit_speed_out_percent = grab_array_var($inargs, "crit_speed_out_percent", 80);

            $services = grab_array_var($inargs, "services");

            if($service_serial == ""){
                $services_serial =  base64_encode(json_encode($services));
            } else {
                $services_serial = grab_array_var($inargs, "services_serial", base64_encode(json_encode($services)));
            }

            $serviceargs = grab_array_var($inargs, "serviceargs");

            if($serviceargs_serial == ""){
                $serviceargs_serial =  base64_encode(json_encode($serviceargs));
            } else {
                $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", base64_encode(json_encode($serviceargs)));
            }

            $output = '
                <input type="hidden" name="ip_address" value="' . encode_form_val($address) . '" />
                <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '" />
                <input type="hidden" name="snmpopts_serial" value="' . $snmpopts_serial . '" />
                <input type="hidden" name="snmpversion" value="' . encode_form_val($snmpversion) . '" />
                <input type="hidden" name="vendor" value="' . encode_form_val($vendor) . '" />
                <input type="hidden" name="portnames" value="' . encode_form_val($portnames) . '" />
                <input type="hidden" name="scaninterfaces" value="' . encode_form_val($scaninterfaces) . '" />
                <input type="hidden" name="warn_speed_in_percent" value="' . encode_form_val($warn_speed_in_percent) . '" />
                <input type="hidden" name="crit_speed_in_percent" value="' . encode_form_val($crit_speed_in_percent) . '" />
                <input type="hidden" name="warn_speed_out_percent" value="' . encode_form_val($warn_speed_out_percent) . '" />
                <input type="hidden" name="crit_speed_out_percent" value="' . encode_form_val($crit_speed_out_percent) . '" />
                <input type="hidden" name="services_serial" value="' . $services_serial . '" />
                <input type="hidden" name="serviceargs_serial" value="' . $serviceargs_serial . '" />
            ';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            $output = '

            ';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $hostname = grab_array_var($inargs, "hostname", "");
            $address = grab_array_var($inargs, "ip_address", "");
            $snmpopts_serial = grab_array_var($inargs, "snmpopts_serial", "");
            if ($snmpopts_serial != "") {
                $snmpopts = json_decode(base64_decode($snmpopts_serial), true);
            } else {
                $snmpopts_default = array(
                    "snmpcommunity" => "public",
                    "v3_security_level" => "",
                    "v3_username" => "",
                    "v3_auth_password" => "",
                    "v3_priv_password" => "",
                    "v3_auth_proto" => "md5",
                    "v3_priv_proto" => "des",
                );
                $snmpopts = grab_array_var($inargs, "snmpopts", $snmpopts_default);
                $snmpopts_serial = base64_encode(json_encode($snmpopts));
            }
            $snmpversion = grab_array_var($inargs, "snmpversion", "1");
            $portnames = grab_array_var($inargs, "portnames");
            $scaninterfaces = grab_array_var($inargs, "scaninterfaces");
            $vendor = grab_array_var($inargs, "vendor");

            $hostaddress = $address;

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["ip_address"] = $address;
            $meta_arr["snmpopts_serial"] = $snmpopts_serial;
            $meta_arr["snmpversion"] = $snmpversion;
            $meta_arr["portnames"] = $portnames;
            $meta_arr["scaninterfaces"] = $scaninterfaces;
            $meta_arr["vendor"] = $vendor;
            $meta_arr["services"] = $services;
            $meta_arr["serviceargs"] = $serviceargs;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            // ABSTRACTION FOR OIDS
            $CPU_oid = "1.3.6.1.4.1.3097.6.3.78.0";
            $AConns_oid = "1.3.6.1.4.1.3097.6.3.80.0";
            $Sent_Packets_oid = "1.3.6.1.4.1.3097.6.3.10.0";
            $Recv_Packets_oid = "1.3.6.1.4.1.3097.6.3.11.0";
            $Strm_Requests_oid = "1.3.6.1.4.1.3097.6.3.30.0";
            $Strm_Dropped_oid = "1.3.6.1.4.1.3097.6.3.34.0";
            $Sent_Bytes_oid = "1.3.6.1.4.1.3097.6.3.8.0";
            $Recv_Bytes_oid = "1.3.6.1.4.1.3097.6.3.9.0";

            $objs = array();

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_watchguard_host",
                    "host_name" => $hostname,
                    "address" => $hostaddress,
                    "icon_image" => "watchguard.png",
                    "statusmap_image" => "watchguard.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            $have_bandwidth = false;

            // see which services we should monitor
            foreach ($services as $svc => $svcstate) {

                //echo "PROCESSING: $svc -> $svcstate<BR>\n";
                if ($svc != "ping" && $svc != "port") {
                    $warn = $serviceargs["{$svc}_warning"];
                    $crit = $serviceargs["{$svc}_critical"];
                }

                //check for SNMP v3 and if true add credentials to service command(s)
                $snmpv3_credentials = "";
                if ($snmpversion == 3) {
                    $priv_password_and_proto = "";
                    if (!empty($snmpopts['v3_priv_password'])) {
                        $priv_password_and_proto = "-x {$snmpopts['v3_priv_proto']}";
                        $priv_password_and_proto .= " -X {$snmpopts['v3_priv_password']}";
                    }

                    $snmpv3_credentials = "-u {$snmpopts['v3_username']} -A {$snmpopts['v3_auth_password']} {$priv_password_and_proto} -a {$snmpopts["v3_auth_proto"]} -l {$snmpopts['v3_security_level']}";
                }

                switch ($svc) {

                    case "ping":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Ping",
                            "use" => "xiwizard_switch_ping_service",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "cpu_usage":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "CPU Usage",
                            "use" => "xiwizard_watchguard_service",
                            "check_command" => "check_xi_service_snmp_watchguard!-N '-c " . $snmpopts['snmpcommunity'] . " -v " . $snmpversion . " " . $snmpv3_credentials . "' {$CPU_oid} -l CPU_Usage -u '%' -d 'CPU Usage' -w $warn -c $crit -D 100",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "active_connections":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Active Connections",
                            "use" => "xiwizard_watchguard_service",
                            "check_command" => "check_xi_service_snmp_watchguard!-N '-c " . $snmpopts['snmpcommunity'] . " -v " . $snmpversion . " " . $snmpv3_credentials . "' {$AConns_oid} -l Active_Connections -d 'Active Connections' -w $warn -c $crit",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "total_sent_packets":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Sent Packets",
                            "use" => "xiwizard_watchguard_service",
                            "check_command" => "check_xi_service_snmp_watchguard!-N '-c " . $snmpopts['snmpcommunity'] . " -v " . $snmpversion . " " . $snmpv3_credentials . "' {$Sent_Packets_oid} -l Total_Sent_Packets -d 'Total Sent Packets/Sec' -w $warn -c $crit -e",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "total_received_packets":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Received Packets",
                            "use" => "xiwizard_watchguard_service",
                            "check_command" => "check_xi_service_snmp_watchguard!-N '-c " . $snmpopts['snmpcommunity'] . " -v " . $snmpversion . " " . $snmpv3_credentials . "' {$Recv_Packets_oid} -l Total_Received_Packets -d 'Total Received Packets/Sec' -w $warn -c $crit -e",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "stream_requests_total":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Stream Requests",
                            "use" => "xiwizard_watchguard_service",
                            "check_command" => "check_xi_service_snmp_watchguard!-N '-c " . $snmpopts['snmpcommunity'] . " -v " . $snmpversion . " " . $snmpv3_credentials . "' {$Strm_Requests_oid} -l Total_Stream_Requests -d 'Total Stream Requests/Sec' -w $warn -c $crit -e",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "stream_requests_drop":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Stream Drops",
                            "use" => "xiwizard_watchguard_service",
                            "check_command" => "check_xi_service_snmp_watchguard!-N '-c " . $snmpopts['snmpcommunity'] . " -v " . $snmpversion . " " . $snmpv3_credentials . "' {$Strm_Dropped_oid} -l Total_Stream_Requests_Dropped -d 'Total Stream Requests Dropped/Sec' -w $warn -c $crit -e",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "total_sent_bytes":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Total Sent Bytes",
                            "use" => "xiwizard_watchguard_service",
                            "check_command" => "check_xi_service_snmp_watchguard!-N '-c " . $snmpopts['snmpcommunity'] . " -v " . $snmpversion . " " . $snmpv3_credentials . "' {$Sent_Bytes_oid} -l Sent_Bytes -u KB -d 'Total Sent Bytes/Sec' -w $warn -c $crit -e -D 1024",
                            "_xiwizard" => $wizard_name,
                        );

                    case "total_received_bytes":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Total Received Bytes",
                            "use" => "xiwizard_watchguard_service",
                            "check_command" => "check_xi_service_snmp_watchguard!-N '-c " . $snmpopts['snmpcommunity'] . " -v " . $snmpversion . " " . $snmpv3_credentials . "' {$Recv_Bytes_oid} -l Received_Bytes -u KB -d 'Total Received Bytes/Sec' -w $warn -c $crit -e -D 1024",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "port":

                        foreach ($svcstate as $portnum => $portstate) {
                            //echo "HAVE PORT $portnum<BR>\n";

                            $portname = _("Port ") . $portnum;
                            if (array_key_exists("portname", $serviceargs)) {
                                if (array_key_exists($portnum, $serviceargs["portname"])) {
                                    $portname = $serviceargs["portname"][$portnum];
                                    $badchars = explode(" ", "` ~ ! $ % ^ & * | ' \" < > ? , ( ) = \\ { } [ ]");
                                    $portname = str_replace($badchars, " ", $portname);
                                }
                            }

                            // monitor bandwidth
                            if (array_key_exists("bandwidth", $serviceargs)) {
                                if (array_key_exists($portnum, $serviceargs["bandwidth"])) {
                                    //echo "MONITOR BANDWIDTH ON $portnum<BR>\n";

                                    $have_bandwidth = true;

                                    $warn_pair = $serviceargs["bandwidth_warning_input_value"][$portnum] . "," . $serviceargs["bandwidth_warning_output_value"][$portnum];
                                    $crit_pair = $serviceargs["bandwidth_critical_input_value"][$portnum] . "," . $serviceargs["bandwidth_critical_output_value"][$portnum];

                                    switch ($serviceargs["bandwidth_speed_label"][$portnum]) {
                                        case "Gbps":
                                            $label = "G";
                                            break;
                                        case "Mbps":
                                            $label = "M";
                                            break;
                                        case "Kbps":
                                            $label = "K";
                                            break;
                                        default:
                                            $label = "B";
                                            break;
                                    }

                                    $objs[] = array(
                                        "type" => OBJECTTYPE_SERVICE,
                                        "host_name" => $hostname,
                                        "service_description" => $portname . " Bandwidth",
                                        "use" => "xiwizard_switch_port_bandwidth_service",
                                        "check_command" => "check_xi_service_mrtgtraf!" . $hostaddress . "_" . $portnum . ".rrd!" . $warn_pair . "!" . $crit_pair . "!" . $label,
                                        "_xiwizard" => $wizard_name,
                                    );
                                }
                            }

                            // monitor port status
                            if (array_key_exists("portstatus", $serviceargs)) {
                                if (array_key_exists($portnum, $serviceargs["portstatus"])) {
                                    //echo "MONITOR PORT STATUS ON $portnum<BR>\n";
                                    if ($snmpversion != 3) {
                                        $objs[] = array(
                                            "type" => OBJECTTYPE_SERVICE,
                                            "host_name" => $hostname,
                                            "service_description" => $portname . " Status",
                                            "use" => "xiwizard_switch_port_status_service",
                                            "check_command" => "check_xi_service_ifoperstatus!" . $snmpopts['snmpcommunity'] . "!" . $portnum . "!-v " . (int)$snmpversion,
                                            "_xiwizard" => $wizard_name,
                                        );
                                    } else {

                                        // If privlidge password
                                        $priv_password_and_proto = "";
                                        if (!empty($snmpopts['v3_priv_password'])) {
                                            $priv_password_and_proto = "-x {$snmpopts['v3_priv_proto']}";
                                            $priv_password_and_proto .= " -X {$snmpopts['v3_priv_password']}";
                                        }

                                        $objs[] = array(
                                            "type" => OBJECTTYPE_SERVICE,
                                            "host_name" => $hostname,
                                            "service_description" => $portname . " Status",
                                            "use" => "xiwizard_switch_port_status_service",
                                            "check_command" => "check_xi_service_ifoperstatusnag!{$portnum}!-v{$snmpversion} -u {$snmpopts['v3_username']} -A {$snmpopts['v3_auth_password']} {$priv_password_and_proto} -a {$snmpopts["v3_auth_proto"]} -l {$snmpopts['v3_security_level']}",
                                            "_xiwizard" => $wizard_name,
                                        );
                                    }
                                }
                            }
                        }
                        break;

                    default:
                        break;
                }
            }

            //echo "OBJECTS:<BR>";
            //print_r($objs);
            //exit();

            // tell MRTG to start monitoring the watchguard device
            if ($have_bandwidth == true) {
                $tmp_dir = get_tmp_dir();
                $outfile = $tmp_dir . "/mrtgscan-" . $address;
                watchguard_configwizard_add_walk_file_to_mrtg($outfile, $address);
                //echo "ADDED WALK FILE TO MRTG...";
            }
            //else
            //  echo "WE DON'T HAVE BANDWIDTH...";

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}

/**
 * @param $f
 * @param $address
 *
 * @return array
 */
function watchguard_configwizard_read_walk_file($f, $address)
{

    $output = array();

    // open the walk file for reading
    $fi = fopen($f, "r");
    if ($fi) {

        while (!feof($fi)) {

            $buf = fgets($fi);

            // skip comments
            $pos = strpos($buf, "#");
            if ($pos !== false)
                continue;

            // found the target line (contains port number)
            if (preg_match('/Target\[' . $address . '_([0-9\.]+)\]/', $buf, $matches)) {
                $port_number = $matches[1];
                //echo "FOUND PORT $port_number<BR>\n";
                if (!array_key_exists($port_number, $output)) {
                    $output[$port_number] = array(
                        "port_number" => $port_number,
                        "port_description" => _("Port ") . $port_number,
                        "max_bytes" => 0,
                        "port_long_description" => _("Port ") . $port_number,
                    );
                }
            }
            // we have the port speed
            if (preg_match('/MaxBytes\[' . $address . '_([0-9\.]+)\]: ([0-9\.]+)/', $buf, $matches)) {
                $port_number = $matches[1];
                $max_bytes = $matches[2];
                //echo "PORT $port_number SPEED IS $max_bytes<BR>\n";
                //$output[$port_number]=$max_bytes;

                if (!array_key_exists($port_number, $output)) {
                    $output[$port_number] = array(
                        "port_number" => $port_number,
                        "port_description" => _("Port ") . $port_number,
                        "max_bytes" => $max_bytes,
                        "port_long_description" => _("Port ") . $port_number,
                    );
                } else
                    $output[$port_number]["max_bytes"] = $max_bytes;

            }
            // we found the description
            //modified so that the short description will replace port number if found
            if (preg_match('/MRTG_INT_DESCR=/', $buf, $matches)) {
                //key string
                $key = 'MRTG_INT_DESCR="';
                //find position of value and grab substring
                $position = strpos($buf, $key) + strlen($key);
                $short_descrip = substr($buf, $position, (strlen($buf)));
                //strip quotes and spaces
                $short_descrip = trim(str_replace('"', NULL, $short_descrip));
                //echo $short_descrip.'<br />';
                // save the description
                $output[$port_number]["port_description"] = $short_descrip;
                $output[$port_number]["port_long_description"] = $short_descrip;
                $longKey = "<td>" . $short_descrip;
            }

            //check for user defined description
            if (isset($longKey) && strpos($buf, $longKey)) {
                $position = strpos($buf, $longKey) + strlen($longKey);
                $long_descrip = substr($buf, $position, (strlen($buf)));
                //strip td tag and spaces
                $long_descrip = trim(str_replace("</td>", NULL, $long_descrip));
                // save the description
                if ($long_descrip != '') $output[$port_number]["port_long_description"] = $long_descrip;

            }

        }
        //end IF FILE is open


        fclose($fi);
    }

    //print_r($output);

    return $output;
}

/**
 * @param $f
 * @param $address
 *
 * @return bool
 */
function watchguard_configwizard_add_walk_file_to_mrtg($f, $address)
{
    $debug = false;

    $mrtg_cfg = "/etc/mrtg/mrtg.cfg";
    $mrtg_conf_dir = "/etc/mrtg/conf.d";
    $address_config = "{$address}.cfg";

    if ($debug) {
        echo "Checking for existing config for $address at {$address}.cfg.";

        $dir_contents = scandir($mrtg_conf_dir);
        if (in_array($address_config, $dir_contents) === true) {
            echo "Config exists, it will be overwritten.";
        } else {
            echo "Config does not exist, it will be created.";
        }
    }

    // Open the devices config file for writing.
    $absolute_address_config = "{$mrtg_conf_dir}/{$address_config}";
    $fo = fopen($absolute_address_config, "w");

    if (!$fo) {

        if ($debug) {
            echo "UNABLE TO OPEN {$mrtg_cfg} FOR APPENDING!<br>";
        }
        return false;
    }

    // open the walk file for reading.
    $fi = fopen($f, "r");
    if ($fi) {

        fprintf($fo, "\n\n#### ADDED BY NAGIOSXI (USER: %s, DATE: %s) ####\n", get_user_attr(0, "username"), get_datetime_string(time()));

        while (!feof($fi)) {
            $buf = fgets($fi);
            fputs($fo, $buf);
        }
    }

    fclose($fo);

    // immediately tell mrtg to generate data from the new walk file
    // if we didn't do this, nagios might send alerts about missing rrd files!
    $cmd = "mrtg {$f} &";
    exec($cmd);

    return true;
}

/**
 * @param $max_bytes
 * @param $speed
 * @param $label
 *
 * @return string
 */
function watchguard_configwizard_get_readable_port_line_speed($max_bytes, &$speed, &$label)
{

    //$bps=$max_bytes/8;
    $bps = $max_bytes * 8;

    $kbps = $bps / 1e3;
    $mbps = $bps / 1e6;
    $gbps = $bps / 1e9;

    if ($gbps >= 1) {
        $speed = $gbps;
        $label = "Gbps";
    } else if ($mbps >= 1) {
        $speed = $mbps;
        $label = "Mbps";
    } else if ($kbps >= 1) {
        $speed = $kbps;
        $label = "Kbps";
    } else {
        $speed = $bps . " bps";
        $label = "bps";
    }

    $output = number_format($speed, 2) . " " . $label;

    return $output;
}


/**
 * @param $warn_in_speed
 * @param $warn_out_speed
 * @param $crit_in_speed
 * @param $crit_out_speed
 * @param $label
 */
function watchguard_configwizard_recalculate_speeds(&$warn_in_speed, &$warn_out_speed, &$crit_in_speed, &$crit_out_speed, &$label)
{

    while (1) {

        if ($label == "bps")
            break;

        $maxval = max($warn_in_speed, $warn_out_speed, $crit_in_speed, $crit_out_speed);

        if ($maxval < 1) {

            switch ($label) {
                case "Gbps":
                    //echo "GBPS=$warn_in_speed<BR>";
                    $label = "Mbps";
                    break;
                case "Mbps":
                    $label = "Kbps";
                    break;
                case "Kbps":
                    $label = "bps";
                    break;
                default:
                    break;
            }

            // bump down a level
            $warn_in_speed *= 1000;
            $warn_out_speed *= 1000;
            $crit_in_speed *= 1000;
            $crit_out_speed *= 1000;

        } else
            break;
    }

}

/**
 * @return bool
 */
function watchguard_configwizard_checkversion()
{
    if (!function_exists('get_product_release'))
        return false;
    if (get_product_release() < 399)
        return false;

    return true;
}

// Check if we should define nagiosccm functionality
// THIS SHOULD BE PHASED OUT ONCE NEW RELEASES OF XI COME OUT!
if (!function_exists('nagiosccm_replace_user_macros')) {
    /**
     * @param $str
     *
     * @return mixed
     */
    function nagiosccm_replace_user_macros($str)
    {
        if (empty($str)) {
            return "";
        }

        // Grab the resource.cfg and read it
        $lines = file('/usr/local/nagios/etc/resource.cfg', FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        $user_macros = array();
        $user_macro_values = array();

        foreach ($lines as $k => $line) {
            if ($line[0] != "#") {
                list($macro, $value) = explode("=", $line);
                $user_macros[] = trim($macro);
                $user_macro_values[] = trim($value);
            }
        }

        // Replace macros in the string given
        $newstr = str_replace($user_macros, $user_macro_values, $str);
        return $newstr;
    }
}
