#!/usr/bin/env python3
import winrm
import argparse
import getpass
import os
import json
from sys import argv

class NoDiskMetrics(Exception):
    pass

try:
    # Parse command-line arguments
    parser = argparse.ArgumentParser(description='winrm_check_disk.py uses WinRM to obtain metrics of a disk on a remote, Windows-based system.')
    parser.add_argument('-H', '--host', required=True, type=str, help='IP address or host name of the Windows system.')
    parser.add_argument('-u', '--user', required=True, type=str, help='Username for connecting to the Windows system.')
    parser.add_argument('-a', '--auth', required=True, type=str, choices=['basic-http', 'basic-https'], help='Authentication mechanism for the Windows system. Strongly recommended you avoid basic-http.')
    parser.add_argument('-p', '--password', required=True, type=str, help='Password for connecting to the Windows system.')
    parser.add_argument('-C', '--certvalidation', required=False, action="count", default=0)
    parser.add_argument('-d', '--diskname', required=True, type=str, help='Name of the disk in which you want metrics on. (Case Sensitive!)')
    parser.add_argument('-w', '--warning', required=False, type=float, help='Percent usage at which a warning alert with be sent.')
    parser.add_argument('-c', '--critical', required=False, type=float, help='Percent usage at which a critical alert with be sent')
    parser.add_argument('-f', '--format', required=False, type=str, help='Format you want your metrics measured in. (TB, GB, MB, KB)', default="GB")

    args = parser.parse_args(argv[1:])

except argparse.ArgumentError as e:
    print(f"\nArgument error: {str(e)}")
    exit(1)
except Exception as e:
    print(f"\nAn error occurred during argument parsing: {str(e)}")
    exit(1)

try:
    authentication = None

    if args.auth == 'basic-http':
        # Use basic-http Authentication
        authentication = 'basic'

    elif args.auth == 'basic-https':
        # Use basic-https Authentication
        authentication = 'ssl'

    elif args.auth == 'cert':
        # Use Certificate Authentication (TLS transport with SSL enabled)
        authentication = winrm.transport.TlsTransport(ssl=True)


    # disk metrics powershell script that returns as stringified json
    ps_disk = f"""
    $strComputer = $Host
    Clear

    $disk_drives = Get-PSDrive -PSProvider FileSystem -Name {args.diskname}
    
    @{{used=$disk_drives.used;free=$disk_drives.free}} | ConvertTo-Json -Compress
    """

    # Create a WinRM session with the provided host, user, password, and authentication method
    if args.certvalidation > 0:
        winrmsession = winrm.Session(args.host, auth=(args.user, args.password), transport=authentication)
    else:
        winrmsession = winrm.Session(args.host, auth=(args.user, args.password), transport=authentication, server_cert_validation='ignore')


    # if there are no metrics for a drive, powershell returns them as null
    # since python cannot handle this null response, it is defined below so we can cleanly exit
    null=None

    # run the ps script and decode the output
    r = winrmsession.run_ps(ps_disk)
    final_dictionary = eval(r.std_out.decode('utf-8'))

    if final_dictionary['free'] is None:
        raise NoDiskMetrics

    byte_format = {
        "TB": 1099511627776,
        "GB": 1073741824,
        "MB": 1048576,
        "KB": 1024
    }

    # calculate the disk stats
    disk_used = round(final_dictionary['used'] / byte_format[args.format], 2)
    disk_free = round(final_dictionary['free'] / byte_format[args.format], 2)
    disk_total = disk_used + disk_free
    disk_used_percent = round((disk_used/disk_total) * 100, 2)


    # check if the disk usage is within bounds of the warning/critical variables
    status_dict = {
        0: "OK",
        1: "WARNING",
        2: "CRITICAL",
        3: "UNKNOWN"
    }

    status_code = 0
    if args.warning and disk_used_percent >= args.warning:
        status_code = 1
    if args.critical and disk_used_percent >= args.critical:
        status_code = 2
    
    # build and return the status message/perf data
    disk_warning = args.warning if args.warning else ''
    disk_critical = args.critical if args.critical else ''
    message = f"{str(status_dict[status_code])}: Used disk space is {str(disk_used_percent)}% (Used: {str(disk_used)} {str(args.format)}, Free: {str(disk_free)} {str(args.format)}, Total: {str(disk_total)} {str(args.format)}) | 'disk_usage'={str(disk_used_percent)}%;{str(disk_warning)};{str(disk_critical)};"
    
    print(message)
    exit(status_code)

except winrm.exceptions.WinRMTransportError as e:
    print(f"WinRM transport error: {str(e)}")
    exit(4)
except NoDiskMetrics as e:
    print(f"UNKNOWN: No disk metrics found for disk '{str(args.diskname)}'")
    exit(4)
except Exception as e:
    print(f"An error occurred during WinRM session setup or command execution: {str(e)}")
    exit(4)#