#!/usr/bin/env python3
import winrm
import argparse
import getpass
import os
from sys import argv

try:
    # Parse command-line arguments
    parser = argparse.ArgumentParser(description='winrm_check_cpu.py uses WinRM to obtain CPU statistics on a remote, Windows-based system.')
    parser.add_argument('-H', '--host', required=True, type=str, help='IP address or host name of the Windows system.')
    parser.add_argument('-u', '--user', required=True, type=str, help='Username for connecting to the Windows system.')
    parser.add_argument('-a', '--auth', required=True, type=str, choices=['basic-http', 'basic-https'], help='Authentication mechanism for the Windows system. Strongly recommended you avoid basic-http.')
    parser.add_argument('-p', '--password', required=True, type=str, help='Password for connecting to the Windows system.')
    parser.add_argument('-C', '--certvalidation', required=False, action="count", default=0)

    parser.add_argument('-n', '--name', required=True, type=str, help='Name of the service you are checking.')
    parser.add_argument('-s', '--status', required=True, type=str, help='Expected status of the service you are checking.')

    args = parser.parse_args(argv[1:])

except argparse.ArgumentError as e:
    print(f"\nArgument error: {str(e)}")
    exit(1)
except Exception as e:
    print(f"\nAn error occurred during argument parsing: {str(e)}")
    exit(1)

try:
    authentication = None

    if args.auth == 'basic-http':
        # Use basic-http Authentication
        authentication = 'basic'

    elif args.auth == 'basic-https':
        # Use basic-https Authentication
        authentication = 'ssl'

    elif args.auth == 'cert':
        # Use Certificate Authentication (TLS transport with SSL enabled)
        authentication = winrm.transport.TlsTransport(ssl=True)


    # service status powershell script
    ps_service = f"""
    $strComputer = $Host
    Clear

    (Get-Service -Name {args.name}).Status
    """


    # Create a WinRM session with the provided host, user, password, and authentication method
    if args.certvalidation > 0:
        winrmsession = winrm.Session(args.host, auth=(args.user, args.password), transport=authentication)
    else:
        winrmsession = winrm.Session(args.host, auth=(args.user, args.password), transport=authentication, server_cert_validation='ignore')


    # run the ps script and decode the output
    r = winrmsession.run_ps(ps_service)
    status_output = r.std_out.decode('utf-8').rstrip()

    status_dict = {
        0: "OK",
        1: "WARNING",
        2: "CRITICAL",
        3: "UNKNOWN"
    }
    status_message = {
        0: f"is {str(args.status)} as it should!",
        2: f"is {str(status_output)}, it should be {str(args.status)}!"
    }

    status_code = 0
    if args.status.lower() !=  status_output.lower():
        status_code = 2
    
    # build and return the status message
    message = f"{str(status_dict[status_code])}: {str(args.name)} {(status_message[status_code])}"
    
    print(message)
    exit(status_code)

except winrm.exceptions.WinRMTransportError as e:
    print(f"WinRM transport error: {str(e)}")
    exit(4)
except Exception as e:
    print(f"An error occurred during WinRM session setup or command execution: {str(e)}")
    exit(4)