#!/usr/bin/env python3

import winrm
import argparse
from sys import argv
import getpass
import os

try:
    # Parse command-line arguments
    parser = argparse.ArgumentParser(description='winrm_check_cpu.py uses WinRM to obtain CPU statistics on a remote, Windows-based system.')
    parser.add_argument('-H', '--host', required=True, type=str, help='IP address or host name of the Windows system.')
    parser.add_argument('-u', '--user', required=True, type=str, help='Username for connecting to the Windows system.')
    # choices=['cert', 'kerberos', 'credssp', 'basic-http', 'basic-https'], all supported auth methods to potentially add bellow*
    parser.add_argument('-a', '--auth', required=True, type=str, choices=['basic-http', 'basic-https'], help='Authentication mechanism for the Windows system. Strongly recommended you avoid basic-http.')
    parser.add_argument('-p', '--password', required=False, type=str, help='Optionally, you can present the password as a command line argument.')
    parser.add_argument('-C', '--certvalidation', required=False, action="count", default=0)

    args = parser.parse_args(argv[1:])

except argparse.ArgumentError as e:
    print(f"\nArgument error: {str(e)}")
    exit(1)
except Exception as e:
    print(f"\nAn error occurred during argument parsing: {str(e)}")
    exit(1)

password = None

if args.password is not None:
    password = args.password

if not password:
    # Prompt for password input securely
    password = getpass.getpass('Password: ')

if not password:
    # Check if password is provided as an environment variable
    password = os.environ.get('WINRM_PASSWORD')

if not password:
    print("Password not provided. Exiting...")
    exit(1)

try:
    authentication = None

    if args.auth == 'basic-http':
        # Use basic-http Authentication
        authentication = 'basic'

    elif args.auth == 'basic-https':
        # Use basic-https Authentication
        authentication = 'ssl'

    elif args.auth == 'cert':
        # Use Certificate Authentication (TLS transport with SSL enabled)
        authentication = winrm.transport.TlsTransport(ssl=True)


    winrmsession = winrm.Session(args.host, auth=(args.user, password), transport=authentication, server_cert_validation='ignore')
    r = winrmsession.run_ps("Write-Output 'Testing Session Validity!'")

    print("Valid Session!")
    exit(1)

except winrm.exceptions.WinRMTransportError as e:
    print(f"WinRM transport error: {str(e)}")
    exit(4)
except Exception as e:
    print(f"An error occurred during WinRM session setup or command execution: {str(e)}")
    exit(4)