<?php
//
// NRPE Config Wizard
// Copyright (c) 2010-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

nrpe_configwizard_init();

function nrpe_configwizard_init()
{
    $name = "nrpe";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.2",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor a remote Linux/Unix server using NRPE."),
        CONFIGWIZARD_DISPLAYTITLE => "NRPE",
        CONFIGWIZARD_FUNCTION => "nrpe_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "remote.png",
        CONFIGWIZARD_FILTER_GROUPS => array('linux','otheros', 'server'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function nrpe_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "nrpe";

    // Check if we were sent here for some reason... set OS distro to Linux
    $sent = grab_request_var("sent", 0);
    $address = grab_request_var("sentaddress", "");
    $osdistro = "";

    if ($sent) {
        $osdistro = "Linux";
    }

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $address = grab_array_var($inargs, "ip_address", $address);
            $osdistro = grab_array_var($inargs, "osdistro", $osdistro);

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:
            $back = array_key_exists("backButton", $_POST);
            if ($back) break;
            
            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $address = nagiosccm_replace_user_macros($address);

            $osdistro = grab_array_var($inargs, "osdistro", "");

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (empty($address)) {
                $errmsg[$errors++] = _("No address specified.");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $osdistro = grab_array_var($inargs, "osdistro", "");
            $ssl = grab_array_var($inargs, "ssl", "");

            $ha = @gethostbyaddr($address);

            if (empty($ha)) {
                $ha = $address;
            }

            $hostname = grab_array_var($inargs, "hostname", $ha);
            $password = "";

            $services = "";
            $services_serial = grab_array_var($inargs, "services_serial", "");

            if ($services_serial != "") {
                $services = json_decode(base64_decode($services_serial), true);
            }

            if (!is_array($services)) {
                $services_default = array(
                    "ping" => 1,
                    "commands" => array(),
                );
                $services_default["commands"][0] = "on";
                $services_default["commands"][1] = "on";
                $services_default["commands"][2] = "on";
                $services = grab_array_var($inargs, "services", $services_default);
            }

            $serviceargs = "";
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            if ($serviceargs_serial != "") {
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            }

            if (!is_array($serviceargs)) {
                $serviceargs_default = array(
                    "commands" => array(),
                );

                for ($x = 0; $x < 5; $x++) {
                    if ($x == 0) {
                        $serviceargs_default['commands'][$x]['command'] = 'check_users';
                        $serviceargs_default['commands'][$x]['args'] = '';
                        $serviceargs_default['commands'][$x]['args'] = '-w 5 -c 10';
                        $serviceargs_default['commands'][$x]['name'] = 'Current Users';
                    } else if ($x == 1) {
                        $serviceargs_default['commands'][$x]['command'] = 'check_load';
                        $serviceargs_default['commands'][$x]['args'] = '';
                        $serviceargs_default['commands'][$x]['args'] = '-w 5,10,15 -c 10,20,30';
                        $serviceargs_default['commands'][$x]['name'] = 'Current Load';
                    } else if ($x == 2) {
                        $serviceargs_default['commands'][$x]['command'] = 'check_procs';
                        $serviceargs_default['commands'][$x]['args'] = '-w 150 -c 250';
                        $serviceargs_default['commands'][$x]['name'] = 'Total Processes';
                    } else {
                        $serviceargs_default['commands'][$x]['command'] = '';
                        $serviceargs_default['commands'][$x]['args'] = '';
                        $serviceargs_default['commands'][$x]['name'] = '';
                        $services['commands'][$x] = '';
                    }
                }

                $serviceargs = grab_array_var($inargs, "serviceargs", $serviceargs_default);
            }

            // Create a list of all the distros there is an NRPE installation for and get icon
            $install_distros = array("RHEL", "CentOS", "Fedora", "OpenSUSE", "SUSE", "Ubuntu", "Debian");
            $icon = nagioscore_get_ui_url() . "images/logos/" . nrpe_configwizard_get_distro_icon($osdistro);

            $agent_url = "";
            $install_doc_url = "";

            if (in_array($osdistro, $install_distros)) {
                $agent_url = "https://assets.nagios.com/downloads/nagiosxi/agents/linux-nrpe-agent.tar.gz";
                $install_doc_url = "https://assets.nagios.com/downloads/nagiosxi/docs/Installing_The_XI_Linux_Agent.pdf";
            } else if ($osdistro == "AIX") {
                $agent_url = "https://assets.nagios.com/downloads/nagiosxi/agents/aix-nrpe-agent.tar.gz";
                $install_doc_url = "https://assets.nagios.com/downloads/nagiosxi/docs/Installing_The_XI_AIX_Agent.pdf";
                $deprecated = "";
            } else if ($osdistro == "Solaris") {
                $agent_url = "https://assets.nagios.com/downloads/nagiosxi/agents/solaris-nrpe-agent.tar.gz";
                $install_doc_url = "https://assets.nagios.com/downloads/nagiosxi/docs/Installing_The_XI_Solaris_Agent.pdf";
            } else if ($osdistro == "Mac") {
                $agent_url = "https://assets.nagios.com/downloads/nagiosxi/agents/macosx-nrpe-agent.tar.gz";
                $install_doc_url = "https://assets.nagios.com/downloads/nagiosxi/docs/Installing_the_XI_Mac_OSX_Agent.pdf";
            }

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:
            $back = array_key_exists("backButton", $_POST);
            if ($back) break;
            
            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $hostname = nagiosccm_replace_user_macros($hostname);
            $osdistro = grab_array_var($inargs, "osdistro", "");

            // Check for errors
            $errors = 0;
            $errmsg = array();
            if (is_valid_host_name($hostname) == false) {
                $errmsg[$errors++] = "Invalid host name.";
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }
            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $osdistro = grab_array_var($inargs, "osdistro", "");
            $services = grab_array_var($inargs, "services", "");
            $serviceargs = grab_array_var($inargs, "serviceargs", "");
            $ssl = grab_array_var($inargs, "ssl", "on");
            $services_serial = (!empty($services) ? base64_encode(json_encode($services)) : grab_array_var($inargs, "services_serial", ''));
            $serviceargs_serial = (!empty($serviceargs) ? base64_encode(json_encode($serviceargs)) : grab_array_var($inargs, "serviceargs_serial", ''));

            if ($services_serial != "") {
                $services = json_decode(base64_decode($services_serial), true);
            }

            if ($serviceargs_serial != "") {
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            }

            $output = '
            <input type="hidden" name="ip_address" value="' . $address . '">
            <input type="hidden" name="hostname" value="' . $hostname . '">
            <input type="hidden" name="osdistro" value="' . $osdistro . '">
            <input type="hidden" name="ssl" value="' . $ssl . '">
            <input type="hidden" name="services_serial" value="' . base64_encode(json_encode($services)) . '">
            <input type="hidden" name="serviceargs_serial" value="' . base64_encode(json_encode($serviceargs)) . '">';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $hostname = grab_array_var($inargs, "hostname", "");
            $address = grab_array_var($inargs, "ip_address", "");
            $osdistro = grab_array_var($inargs, "osdistro", "");
            $ssl = grab_array_var($inargs, "ssl", "on");
            $hostaddress = $address;

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            // Save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["ip_address"] = $address;
            $meta_arr["osdistro"] = $osdistro;
            $meta_arr["services"] = $services;
            $meta_arr["serviceargs"] = $serviceargs;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = array();
            $icon = nrpe_configwizard_get_distro_icon($osdistro);

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_linuxserver_host",
                    "host_name" => $hostname,
                    "address" => $hostaddress,
                    "icon_image" => $icon,
                    "statusmap_image" => $icon,
                    "_xiwizard" => $wizard_name,
                );
            }

            // Optional non-SSL args to add
            $sslargs = "";
            if ($ssl == "off") {
                $sslargs .= " -n";
            }

            // See which services we should monitor
            foreach ($services as $svc => $svcstate) {

                switch ($svc) {

                    case "ping":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Ping",
                            "use" => "xiwizard_linuxserver_ping_service",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "commands":

                        $enabledcmds = $svcstate;
                        foreach ($enabledcmds as $pid => $pstate) {

                            $pname = $serviceargs["commands"][$pid]["command"];
                            $pargs = $serviceargs["commands"][$pid]["args"];
                            $pdesc = $serviceargs["commands"][$pid]["name"];

                            $checkcommand = "check_nrpe!" . $pname . "!" . $sslargs;
                            if ($pargs != "")
                                $checkcommand .= " -a '" . $pargs . "'";
                            
                            if ($pdesc != "") {
                                $objs[] = array(
                                    "type" => OBJECTTYPE_SERVICE,
                                    "host_name" => $hostname,
                                    "service_description" => $pdesc,
                                    "use" => "generic-service",
                                    "check_command" => $checkcommand,
                                    "_xiwizard" => $wizard_name,
                                );
                            }
                        }
                        break;

                    default:
                        break;
                }
            }

            // Return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}


/**
 * @param $osdistro
 *
 * @return string
 */
function nrpe_configwizard_get_distro_icon($osdistro)
{

    $icon = "linux.png";

    switch ($osdistro) {
        case "Solaris":
            $icon = "solaris.png";
            break;
        case "AIX":
            $icon = "aix.png";
            break;
        case "HP-UX":
            $icon = "hp-ux.png";
            break;
        case "FreeBSD":
            $icon = "freebsd2.png";
            break;
        case "NetBSD":
            $icon = "netbsd.png";
            break;
        case "OpenBSD":
            $icon = "openbsd.png";
            break;
        case "RHEL":
            $icon = "redhat.png";
            break;
        case "Fedora":
            $icon = "fedora.png";
            break;
        case "CentOS":
            $icon = "centos.png";
            break;
        case "Ubuntu":
            $icon = "ubuntu.png";
            break;
        case "Debian":
            $icon = "debian.png";
            break;
        case "OpenSUSE":
            $icon = "opensuse.png";
            break;
        case "SUSE":
            $icon = "suse_enterprise.png";
            break;
        case "Mac":
            $icon = "osx.png";
            break;
        case "Other":
            $icon = "remote.png";
        default:
            break;
    }

    return $icon;
}
