<?php
//
// Microsoft Exchange Config Wizard
// Copyright (c) 2011-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__).'/../configwizardhelper.inc.php');

exchange_configwizard_init();

function exchange_configwizard_init()
{
    $name = "exchange";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "3.0.1",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor a Microsoft&reg; Exchange server."),
        CONFIGWIZARD_DISPLAYTITLE => "Exchange Server",
        CONFIGWIZARD_FUNCTION => "exchange_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "exchange2010.png",
        CONFIGWIZARD_FILTER_GROUPS => array('windows','email', 'server'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function exchange_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "exchange";

    #$agent_url = "http://www.nsclient.org/download/";
    #$agent32_stable_url = "https://assets.nagios.com/downloads/nagiosxi/agents/NSClient++/NSCP-0.4.4.23-Win32.msi";
    #$agent64_stable_url = "https://assets.nagios.com/downloads/nagiosxi/agents/NSClient++/NSCP-0.4.4.23-x64.msi";

    /* Prerequisite software */
    $NCPA_download_url = "https://www.nagios.org/ncpa/#downloads";
    $NCPA_docs_url = "https://www.nagios.org/ncpa/#docs";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $ip_address = grab_array_var($inargs, "ip_address", "");
            $version = grab_array_var($inargs, "version", "2019");
            $port = grab_array_var($inargs, "port", "5693");
            $token = grab_array_var($inargs, "token", "");
            $selectedhostconfig = grab_array_var($inargs, "selectedhostconfig", "");
            $operation = grab_array_var($inargs, "operation", "new");

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $ip_address = grab_array_var($inargs, "ip_address", "");
            $ip_address_replaced = nagiosccm_replace_user_macros($ip_address);

            $version = grab_array_var($inargs, "version", "");

            // NCPA
            $port = grab_array_var($inargs, "port", "5693");
            $token = grab_array_var($inargs, "token", "");

            $port_replaced = nagiosccm_replace_user_macros($port);
            $token_replaced = nagiosccm_replace_user_macros($token);

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($ip_address_replaced) == false) {
                $errmsg[$errors++] = _('No address specified.');
            } else if (!valid_ip($ip_address_replaced)) {
                $errmsg[$errors++] = _('Invalid IP address.');
            }

            if (have_value($version) == false) {
                $errmsg[$errors++] = _('No server version specified.');
            }

            if (have_value($token) == false) {
                $errmsg[$errors++] = _('No NCPA Token specified.');
            }

            // Test the connection if no errors
            if (empty($errors)) {
                // The URL we will use to query the NCPA agent, and do a walk
                // of all monitorable items.
                $query_url = "https://{$ip_address}:{$port}/testconnect/?token=".urlencode($token);
                $query_url_replaced = "https://{$ip_address_replaced}:{$port_replaced}/testconnect/?token=".urlencode($token_replaced);

                // For a simple requeest, make timeout shorter than php default or typical gateway timeout of 60
                $context = array("ssl" => array("verify_peer" => false, "verify_peer_name" => false), "http" => array('timeout' => 10));

                // All we want to do is test if we can hit this URL.
                // Error Control Operator - @ - http://php.net/manual/en/language.operators.errorcontrol.php
                $raw_json = @file_get_contents($query_url_replaced, false, stream_context_create($context));

                if ($raw_json === FALSE || empty($raw_json)) {
                    # FYI, the way this is setup, if the user uses the same url & token,
                    # both will be replaced with "your_token".
                    /* This error message is hiding macro expansion values. */
                    $safe_url = str_replace('token='.urlencode($token_replaced), "token=your_token", $query_url);
                    $errmsg[$errors++] = _("Unable to contact server at")." {$safe_url}.";

                } else {
                    $json = json_decode($raw_json, true);

                    if (!array_key_exists('value', $json)) {
                        $errmsg[$errors++] = _("Bad token for connection.");
                    }
                }
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $ip_address = grab_array_var($inargs, "ip_address");
            $version = grab_array_var($inargs, "version");

            // NCPA
            $port = grab_array_var($inargs, "port");
            $token = grab_array_var($inargs, "token");

            $hn = @gethostbyaddr($ip_address);

            if ($hn == "")  $hn = $ip_address;

            $hostname = grab_array_var($inargs, "hostname", $hn);
            $password = grab_array_var($inargs, "password", "");

            $services = "";
            $services_serial = grab_array_var($inargs, "services_serial", "");

            if ($services_serial != "") {
                $services = json_decode(base64_decode($services_serial), true);
            }

            # Old version of OWA (Outlook Web Access) 5.5, 2000, 2003.
            $owa_url = grab_array_var($inargs, "owa_url", "http://".$hostname."/exchange/");

            $selectedhostconfig = grab_array_var($inargs, "selectedhostconfig", "");
            $operation = grab_array_var($inargs, "operation", "new");

            $use_2003_deprecated_stats = false;
            $use_2007_plus_stats = false;

            switch ($version) {
                case "5.5":
                case "2000":
                case "2003":
                    $use_2003_deprecated_stats = true; # Works with 2003
                    break;
                case "2007":
                case "2010":
                case "2013":
                case "2016":
                case "2019":
                    # New version of OWA (Outlook Web Access)
                    $owa_url = grab_array_var($inargs, "owa_url", "http://".$hostname."/owa/");
                    $use_2007_plus_stats = true;       # Works with 2007 - 2019
                    break;
            }

            if (!is_array($services)) {
                $services_default = array_merge(
                    array(
                        "ping" => 1,
                        "smtp" => 1,
                        "imap" => 1,
                        "pop" => 1,
                        "rbl" => 1,
                        "owa_http" => 1,
                        "owa_https" => 0,
                        "core_services" => 1,
                        "web_services" => 1,
                    ),
                    (($use_2003_deprecated_stats) ?
                        array(
                            "pending_routing" => 1,     # Exchange 2003
                            "remote_queue_length" => 1, # Exchange 2003
                        ) : array()
                    ),
                    (($use_2007_plus_stats) ?
                        array(
                            "messages_queued_for_delivery" => 1,    # Exchange 2007 - 2019
                        ) : array()
                    )
                );

                $services = grab_array_var($inargs, "services", $services_default);
            }

            $serviceargs = "";
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            if ($serviceargs_serial != "") {
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            }

            if (!is_array($serviceargs)) {
                $serviceargs_default = array(
                    "rbl_servers" => "zen.spamhaus.org bl.spamcop.net dnsbl.ahbl.org dnsbl.njabl.org dnsbl.sorbs.net virbl.dnsbl.bit.nl rbl.efnet.org phishing.rbl.msrbl.net 0spam.fusionzero.com list.dsbl.org multihop.dsbl.org unconfirmed.dsbl.org will-spam-for-food.eu.org blacklist.spambag.org blackholes.brainerd.net blackholes.uceb.org spamsources.dnsbl.info map.spam-rbl.com ns1.unsubscore.com psbl.surriel.com l2.spews.dnsbl.sorbs.net bl.csma.biz sbl.csma.biz dynablock.njabl.org no-more-funn.moensted.dk  ubl.unsubscore.com dnsbl-1.uceprotect.net dnsbl-2.uceprotect.net dnsbl-3.uceprotect.net spamguard.leadmon.net opm.blitzed.org bl.spamcannibal.org rbl.schulte.org dnsbl.ahbl.org virbl.dnsbl.bit.nl combined.rbl.msrbl.net",
                    "core_service_names" => "",
                    "web_service_names" => "W3SVC",
                    "pending_routing_warning" => 25,
                    "pending_routing_critical" => 100,
                    "remote_queue_length_warning" => 25,
                    "remote_queue_length_critical" => 50,
                    "messages_queued_for_delivery_warning" => 25,
                    "messages_queued_for_delivery_critical" => 50,
                );

                switch ($version) {
                    case "2019":
                        // assumed for 2019 - Exchange Services List (one additional to 2016 list)
                        // Verified against: https://docs.microsoft.com/en-us/exchange/plan-and-deploy/deployment-ref/services-overview?view=exchserver-2019
                        $serviceargs_default["core_service_names"] = "HostControllerService,MSComplianceAudit,MSExchangeADTopology,MSExchangeCompliance,MSExchangeDagMgmt,MSExchangeDelivery,MSExchangeDiagnostics,MSExchangeFastSearch,MSExchangeFrontEndTransport,MSExchangeHM,MSExchangeHMRecovery,MSExchangeIS,MSExchangeMailboxAssistants,MSExchangeMailboxReplication,MSExchangeNotificationsBroker,MSExchangeRepl,MSExchangeRPC,MSExchangeServiceHost,MSExchangeSubmission,MSExchangeThrottling,MSExchangeTransport,SearchExchangeTracing,";
                        // Optional, but setup as automatic during default install.
                        $serviceargs_default["core_service_names"] .= "MSExchangeAntispamUpdate,MSExchangeEdgeSync,MSExchangeTransportLogSearch,MSExchangeUM,MSExchangeUMCR";
                            // These are manual services in 2019 and cause a CRITICAL return if included: MSExchangeImap4,MSExchangeIMAP4BE,MSExchangePop3,MSExchangePOP3BE,WSBExchange
                        break;
                    case "2016":
                        // assumed for 2016 - Exchange Services List
                        // Verified against: https://docs.microsoft.com/en-us/exchange/plan-and-deploy/deployment-ref/services-overview?view=exchserver-2016
                        // sc query type= service state= all > allserviceslist.txt ==> search for "Microsoft Exchange" to find the services.
                        $serviceargs_default["core_service_names"] = "HostControllerService,MSComplianceAudit,MSExchangeADTopology,MSExchangeCompliance,MSExchangeDagMgmt,MSExchangeDelivery,MSExchangeDiagnostics,MSExchangeFastSearch,MSExchangeFrontEndTransport,MSExchangeHM,MSExchangeHMRecovery,MSExchangeIS,MSExchangeMailboxAssistants,MSExchangeMailboxReplication,MSExchangeRepl,MSExchangeRPC,MSExchangeServiceHost,MSExchangeSubmission,MSExchangeThrottling,MSExchangeTransport,SearchExchangeTracing,";
                        // Optional, but setup as automatic during default install.
                        $serviceargs_default["core_service_names"] .= "MSExchangeAntispamUpdate,MSExchangeEdgeSync,MSExchangeTransportLogSearch,MSExchangeUM,MSExchangeUMCR";
                            // These are manual services in 2016 and cause a CRITICAL return if included: MSExchangeImap4,MSExchangeIMAP4BE,MSExchangePop3,MSExchangePOP3BE,MSExchangeNotificationsBroker,WSBExchange
                        break;
                    case "2013":
                        // assumed for 2013 - nagioswiki.com
                        $serviceargs_default["core_service_names"] = "MSExchangeADTopology,MSExchangeAntispamUpdate,MSExchangeDagMgmt,MSExchangeDelivery,MSExchangeDiagnostics,MSExchangeEdgeSync,MSExchangeFastSearch,MSExchangeFrontEndTransport,MSExchangeHM,MSExchangeImap4,MSExchangeIMAP4BE,MSExchangeIS,MSExchangeMailboxAssistants,MSExchangeMailboxReplication,MSExchangePop3,MSExchangePOP3BE,MSExchangeRepl,MSExchangeRPC,MSExchangeServiceHost,MSExchangeSubmission,MSExchangeThrottling,MSExchangeTransport,MSExchangeTransportLogSearch,MSExchangeUM,MSExchangeUMCR";
                        break;
                    case "2010":
                        // assumed for 2010 - nagioswiki.com
                        $serviceargs_default["core_service_names"] = "MSExchangeADTopology,MSExchangeAntispamUpdate,MSExchangeEdgeSync,MSExchangeFDS,MSExchangeImap4,MSExchangeIS,MSExchangeMailboxAssistants,MSExchangeMailSubmission,MSExchangeMonangePop3,MSExchangeRepl,MSExchangeSA,MSExchangeSearch,MSExchangeServiceHost,MSExchangeTransport,MSExchangeTransportLogSearch,msftesql-Exchange";
                        break;
                    case "2007":
                        // known for 2007 - nagioswiki.com
                        $serviceargs_default["core_service_names"] = "MSExchangeADTopology,MSExchangeAntispamUpdate,MSExchangeEdgeSync,MSExchangeFDS,MSExchangeImap4,MSExchangeIS,MSExchangeMailboxAssistants,MSExchangeMailSubmission,MSExchangeMonangePop3,MSExchangeRepl,MSExchangeSA,MSExchangeSearch,MSExchangeServiceHost,MSExchangeTransport,MSExchangeTransportLogSearch,msftesql-Exchange";
                        break;
                    case "2003":
                        // assumed for 2003 - nagioswiki.com
                        $serviceargs_default["core_service_names"] = "MSExchangeIS,MSExchangeMTA,SMTPSVC,RESvc";
                    case "2000":
                        // known for 2000 - nagioswiki.com
                        $serviceargs_default["core_service_names"] = "MSExchangeIS,MSExchangeMTA,SMTPSVC,RESvc";
                        break;
                    case "5.5":
                    default:
                        // known for 5.5 - Nagios Exchange (http://exchange.nagios.org/directory/Plugins/Email-and-Groupware/Microsoft-Exchange/Check-MS-Exchange-Server-Health/details)
                        $serviceargs_default["core_service_names"] = "MSExchangeDS,MSExchangeES,MSExchangeIMC,MSExchangeIS,MSExchangeMTA,MSExchangeSA ";
                        break;
                }

                $serviceargs = grab_array_var($inargs, "serviceargs", $serviceargs_default);
            }

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // Get variables that were passed to us
            $ip_address = grab_array_var($inargs, "ip_address");
            $version = grab_array_var($inargs, "version", "");
            $hostname = grab_array_var($inargs, "hostname");
            $hostname_replaced = nagiosccm_replace_user_macros($hostname);

            $password = grab_array_var($inargs, "password");
            $password_replaced = nagiosccm_replace_user_macros($password);

            $owa_url = grab_array_var($inargs, "owa_url", "");
            $owa_url_replaced = nagiosccm_replace_user_macros($owa_url);

            // NCPA
            $port = grab_array_var($inargs, "port");
            $token = grab_array_var($inargs, "token");

            $port_replaced = nagiosccm_replace_user_macros($port);
            $token_replaced = nagiosccm_replace_user_macros($token);

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (is_valid_host_name($hostname_replaced) == false) {
                $errmsg[$errors++] = "Invalid host name.";
            }

            # password is not used with NCPA version
            #if (preg_match('/[^a-zA-Z0-9 .\:_-@]/', $password_replaced)) {
            #    $errmsg[$errors++] = _("Password contains invalid characters.");
            #}

            if (valid_url($owa_url_replaced) == false) {
                $errmsg[$errors++] = _("Invalid URL.");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // Get variables that were passed to us
            $ip_address = grab_array_var($inargs, "ip_address");
            $version = grab_array_var($inargs, "version", "");
            $hostname = grab_array_var($inargs, "hostname");
            $password = grab_array_var($inargs, "password");
            $owa_url = grab_array_var($inargs, "owa_url", "");

            // NCPA
            $port = grab_array_var($inargs, "port");
            $token = grab_array_var($inargs, "token");

            $services = "";
            $services_serial = grab_array_var($inargs, "services_serial");

            if ($services_serial != "") {
                $services = json_decode(base64_decode($services_serial), true);

            } else {
                $services = grab_array_var($inargs, "services");
            }

            $serviceargs = "";
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial");

            if ($serviceargs_serial != "") {
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            } else {
                $serviceargs = grab_array_var($inargs, "serviceargs");
            }

            $output = '

        <input type="hidden" name="ip_address" value="'.encode_form_val($ip_address).'">
        <input type="hidden" name="version" value="'.encode_form_val($version).'">
        <input type="hidden" name="port" value="'.encode_form_val($port).'">
        <input type="hidden" name="token" value="'.encode_form_val($token).'">
        <input type="hidden" name="owa_url" value="'.encode_form_val($owa_url).'">
        <input type="hidden" name="hostname" value="'.encode_form_val($hostname).'">
        <input type="hidden" name="password" value="'.encode_form_val($password).'">
        <input type="hidden" name="services_serial" value="'.base64_encode(json_encode($services)).'">
        <input type="hidden" name="serviceargs_serial" value="'.base64_encode(json_encode($serviceargs)).'">
 ';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $hostname = grab_array_var($inargs, "hostname", "");
            $version = grab_array_var($inargs, "version", "");
            $ip_address = grab_array_var($inargs, "ip_address", "");
            $password = grab_array_var($inargs, "password", "");
            $hostaddress = $ip_address;
            $owa_url = grab_array_var($inargs, "owa_url", "");

            // NCPA
            $port = grab_array_var($inargs, "port", "");
            $token = grab_array_var($inargs, "token", "");

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            // Escape values for check_command line
            if (function_exists('nagiosccm_replace_command_line')) {
                $token = nagiosccm_replace_command_line($token);
            } else {
                $token = str_replace('!', '\!', $token);
            }

            /*
            echo "SERVICES<BR>";
            print_r($services);
            echo "<BR>";
            echo "SERVICEARGS<BR>";
            print_r($serviceargs);
            echo "<BR>";
            */

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["ip_address"] = $ip_address;
            $meta_arr["password"] = $password;
            $meta_arr["services"] = $services;
            $meta_arr["serivceargs"] = $serviceargs;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = array();

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_exchange_host",
                    "host_name" => $hostname,
                    "address" => $hostaddress,
                    "icon_image" => "exchange2010.png",
                    "statusmap_image" => "exchange2010.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            $commonopts = "-t '$token' -P '$port'";

            // see which services we should monitor
            foreach ($services as $svc => $svcstate) {

                //echo "PROCESSING: $svc -> $svcstate<BR>\n";

                switch ($svc) {

                    case "ping":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Ping",
                            "use" => "xiwizard_exchange_ping_service",
                            "_xiwizard" => $wizard_name,
                        );

                        break;

                    case "imap":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "IMAP",
                            "use" => "xiwizard_imap_service",
                            "check_command" => "check_xi_service_imap!-j",
                            "_xiwizard" => $wizard_name,
                        );

                        break;

                    case "pop":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "POP",
                            "use" => "xiwizard_pop_service",
                            "check_command" => "check_xi_service_pop!-j",
                            "_xiwizard" => $wizard_name,
                        );

                        break;

                    case "smtp":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "SMTP",
                            "use" => "xiwizard_smtp_service",
                            "_xiwizard" => $wizard_name,
                        );

                        break;

                    case "rbl":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Blacklist Status",
                            "use" => "xiwizard_exchange_service",
                            "check_command" => "check_exchange_rbl!-B ".$serviceargs["rbl_servers"],
                            "_xiwizard" => $wizard_name,
                        );

                        break;

                    case "owa_http":
                    case "owa_https":

                        $pluginopts = "";

                        # Pull apart the OWA url, so we can use it for an http_check.
                        # We are using the $owa_url, because it may be different than the IP/address, initially setup in the wizard.
                        $owa_url_parts = parse_url($owa_url);
                        $vhost = $owa_url_parts["host"];

                        if ($vhost == "") {
                            $vhost = $ip_address;
                        }

                        #
                        # If $vhost != $ip_address, this will add a second -H <address>, which will override the default IP/DNS address
                        # created using the ip_address from CONFIGWIZARD_MODE_GETSTAGE1HTML, e.g.,
                        #
                        #                   $USER1$/check_http -H $HOSTADDRESS$  $ARG1$.................................................
                        # /usr/local/nagios/libexec/check_http -H 192.168.145.90 -H owa.nagios.com -f ok -I 192.168.145.90 -u "/owa/" -S
                        #
                        # -H, --hostname=ADDRESS
                        #    Host name argument for servers using host headers (virtual host)
                        #    Append a port to include it in the header (eg: example.com:5000)
                        #
                        # --sni
                        #    Enable SSL/TLS hostname extension support (SNI)
                        if (strcasecmp($vhost, $ip_address) != 0) {
                            $pluginopts .= " -H ".$vhost." --sni";
                        }

                        #
                        # -f, --onredirect=<ok|warning|critical|follow|sticky|stickyport>
                        #    How to handle redirected pages. sticky is like follow but stick to the
                        #    specified IP address. stickyport also ensures port stays the same.
                        #$pluginopts .= " -f ok"; // on redirect, return OK status
                        $pluginopts .= " -f follow"; // on redirect, follow and return the status
                        $pluginopts .= " -I ".$ip_address; // ip address

                        $owa_url_path = $owa_url_parts["path"];

                        if ($owa_url_path == "") {
                            $owa_url_path = "/";
                        }

                        # -u, --url=PATH URL to GET or POST (default: /)
                        $pluginopts .= " -u \"".$owa_url_path."\"";

                        #
                        # -S, --ssl=VERSION[+]
                        #    Connect via SSL. Port defaults to 443. VERSION is optional, and prevents
                        #    auto-negotiation (2 = SSLv2, 3 = SSLv3, 1 = TLSv1, 1.1 = TLSv1.1,
                        #    1.2 = TLSv1.2). With a '+' suffix, newer versions are also accepted.
                        if ($svc == "owa_https") {
                            $pluginopts .= " -S";
                        }

                        #if (!empty($port)) {   # Incorrect.  Use port from $owa_url_parts.
                        if (array_key_exists("port", $owa_url_parts) && !empty($owa_url_parts["port"])) {
                            $pluginopts .= " -p ".$owa_url_parts["port"];
                        }

                        # NOT USED
                        # -a, --authorization=AUTH_PAIR
                        #    Username:password on sites with basic authentication
                        #if (!empty($username)) {
                        #    $pluginopts .= " -a \"".$username.":".$password."\"";
                        #}

                        $sname = "OWA HTTP";

                        if ($svc == "owa_https") {
                            $sname .= "S";
                        }

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => $sname,
                            "use" => "xiwizard_exchange_service",
                            "check_command" => "check_xi_service_http!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );

                        break;

                    case "core_services":

                        $coreServices = explode(",", $serviceargs["core_service_names"]);
                        $coreServiceArgs = "";

                        foreach ($coreServices as $i => $service) {
                            $coreServiceArgs .= "service=$service".(($i + 1 < count($coreServices)) ? "," : "");
                        }

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Exchange Core Services",
                            #"use" => "xiwizard_exchange_service",
                            "use" => "xiwizard_ncpa_service",
                            #"check_command" => "check_xi_service_nsclient!".$password."!SERVICESTATE!-l ".$serviceargs["core_service_names"]." -d SHOWALL",
                            "check_command" => "check_xi_ncpa!".$commonopts." -M 'services' -q '".$coreServiceArgs.",status=running'",
                            "_xiwizard" => $wizard_name);

                        break;

                    case "web_services":

                        $webServices = explode(",", $serviceargs["web_service_names"]);
                        $webServiceArgs = "";

                        foreach ($webServices as $i => $service) {
                            $webServiceArgs .= "service=$service".(($i + 1 < count($webServices)) ? "," : "");
                        }

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Exchange Web Services for: ".$service,
                            #"use" => "xiwizard_exchange_service",
                            "use" => "xiwizard_ncpa_service",
                            #"check_command" => "check_xi_service_nsclient!".$password."!SERVICESTATE!-l ".$serviceargs["web_service_names"]." -d SHOWALL",
                            "check_command" => "check_xi_ncpa!".$commonopts." -M 'services' -q '".$webServiceArgs.",status=running'",
                            "_xiwizard" => $wizard_name);

                        break;

                    # 2003 Exchange
                    case "pending_routing":

                        #$checkcommand = "check_xi_service_nsclient!".$password."!COUNTER!-l \"\\\\SMTP Server(_Total)\\\\Messages Pending Routing\"";
                        $checkcommand = "check_xi_ncpa!".$commonopts." -M 'windowscounters/SMTP Server(_Total)/Messages Pending Routing'";

                        if ($serviceargs["pending_routing_warning"] != "") {
                            $checkcommand .= " -w ".$serviceargs["pending_routing_warning"];
                        }

                        if ($serviceargs["pending_routing_critical"] != "") {
                            $checkcommand .= " -c ".$serviceargs["pending_routing_critical"];
                        }

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Messages Pending Routing",
                            #"use" => "xiwizard_windowsserver_nsclient_service",
                            "use" => "xiwizard_ncpa_service",
                            "check_command" => $checkcommand,
                            "_xiwizard" => $wizard_name,
                        );

                        break;

                    # 2003 Exchange
                    case "remote_queue_length":

                        #$checkcommand = "check_xi_service_nsclient!".$password."!COUNTER!-l \"\\\\SMTP Server(_Total)\\\\Remote Queue Length\"";
                        $checkcommand = "check_xi_ncpa!".$commonopts." -M 'windowscounters/SMTP Server(_Total)/Remote Queue Length'";

                        if ($serviceargs["remote_queue_length_warning"] != "") {
                            $checkcommand .= " -w ".$serviceargs["remote_queue_length_warning"];
                        }

                        if ($serviceargs["remote_queue_length_critical"] != "") {
                            $checkcommand .= " -c ".$serviceargs["remote_queue_length_critical"];
                        }

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Remote Queue Length",
                            #"use" => "xiwizard_windowsserver_nsclient_service",
                            "use" => "xiwizard_ncpa_service",
                            "check_command" => $checkcommand,
                            "_xiwizard" => $wizard_name,
                        );

                        break;

                    # 2007 - 2019 Exchange
                    case "messages_queued_for_delivery":

                        $checkcommand = "check_xi_ncpa!".$commonopts." -M 'windowscounters/MSExchangeTransport Queues(_Total)/Messages Queued for Delivery'";

                        if ($serviceargs["messages_queued_for_delivery_warning"] != "") {
                            $checkcommand .= " -w ".$serviceargs["messages_queued_for_delivery_warning"];
                        }

                        if ($serviceargs["messages_queued_for_delivery_critical"] != "") {
                            $checkcommand .= " -c ".$serviceargs["messages_queued_for_delivery_critical"];
                        }

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Messages Queued for Delivery",
                            #"use" => "xiwizard_windowsserver_nsclient_service",
                            "use" => "xiwizard_ncpa_service",
                            "check_command" => $checkcommand,
                            "_xiwizard" => $wizard_name,
                        );

                        break;

                    default:
                        break;
                }
            }

            //echo "OBJECTS:<BR>";
            //print_r($objs);
            //exit();

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}

?>

