<?php
//
// SNMP Config Wizard
// Copyright (c) 2008-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

snmp_configwizard_init();

function snmp_configwizard_init()
{
    $name = "snmp";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.2",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor a device, service, or application using SNMP."),
        CONFIGWIZARD_DISPLAYTITLE => "SNMP",
        CONFIGWIZARD_FUNCTION => "snmp_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "snmp.png",
        CONFIGWIZARD_FILTER_GROUPS => array('network', 'server'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );

    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function snmp_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    global $cfg;
    $wizard_name = "snmp";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            // Remove any existing session variables if this is the first time on the page
            if (isset($_POST['backButton']) == false) {
                unset($_SESSION['snmp_wizard_address']);
                unset($_SESSION['snmp_wizard_hostname']);
                unset($_SESSION['snmp_wizard_port']);
                unset($_SESSION['snmp_wizard_snmpcommunity']);
                unset($_SESSION['snmp_wizard_snmpversion']);
                unset($_SESSION['snmp_wizard_services']);
                unset($_SESSION['snmp_wizard_serviceargs']);
            }

            $address = grab_array_var($inargs, "ip_address", "");

            $selectedhostconfig = grab_array_var($inargs, "selectedhostconfig", "");
            $config_serial = grab_array_var($inargs, "selectedhostconfig", "");
            $operation = grab_array_var($inargs, "operation", "");

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $address = nagiosccm_replace_user_macros($address);

            if (array_key_exists('snmp_wizard_address', $_SESSION) && $address == "") {
                $address = $_SESSION['snmp_wizard_address'];
            }

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($address) == false) {
                $errmsg[$errors++] = _("No address specified.");
            }

            if ($address) {
                $_SESSION['snmp_wizard_address'] = $address;
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $port = grab_array_var($inargs, "port", 161);
            $ha = @gethostbyaddr($address);
            $hostname = grab_array_var($inargs, "hostname", $ha);
            $snmpcommunity = grab_array_var($inargs, "snmpcommunity", "public");
            $snmpversion = grab_array_var($inargs, "snmpversion", "2c");
            $services = grab_array_var($inargs, "services", array());
            $serviceargs = grab_array_var($inargs, "serviceargs", array());

            if ($ha == "") {
                $ha = $address;
            }

            if (array_key_exists('snmp_wizard_address', $_SESSION) && $address == "") {
                $address = $_SESSION['snmp_wizard_address'];
            }

            if (array_key_exists('snmp_wizard_hostname', $_SESSION) && $hostname == "") {
                $hostname = $_SESSION['snmp_wizard_hostname'];
            }

            if (array_key_exists('snmp_wizard_snmpcommunity', $_SESSION) && $snmpcommunity == "") {
                $snmpcommunity = $_SESSION['snmp_wizard_snmpcommunity'];
            }

            if (array_key_exists('snmp_wizard_snmpversion', $_SESSION) && $snmpversion == "") {
                $snmpversion = $_SESSION['snmp_wizard_snmpversion'];
            }

            if (array_key_exists('snmp_wizard_port', $_SESSION) && $port == "") {
                $port = $_SESSION['snmp_wizard_port'];
            }

            if (array_key_exists('snmp_wizard_services', $_SESSION) && !$services) {
                $services = $_SESSION['snmp_wizard_services'];
            }

            if (array_key_exists('snmp_wizard_serviceargs', $_SESSION) && !$serviceargs) {
                $serviceargs = $_SESSION['snmp_wizard_serviceargs'];
            }

            $selectedhostconfig = grab_array_var($inargs, "selectedhostconfig", "");
            $config_serial = grab_array_var($inargs, "selectedhostconfig", "");
            $operation = grab_array_var($inargs, "operation", "");

            // Use encoded data (if user came back from future screen)
            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            if (!array_key_exists("v3_security_level", $serviceargs)) $serviceargs["v3_security_level"] = "authPriv";
            if (!array_key_exists("v3_username", $serviceargs)) $serviceargs["v3_username"] = "";
            if (!array_key_exists("v3_privacy_password", $serviceargs)) $serviceargs["v3_privacy_password"] = "";
            if (!array_key_exists("v3_auth_password", $serviceargs)) $serviceargs["v3_auth_password"] = "";
            if (!array_key_exists("v3_auth_proto", $serviceargs)) $serviceargs["v3_auth_proto"] = "md5";
            if (!array_key_exists("v3_priv_proto", $serviceargs)) $serviceargs["v3_priv_proto"] = "des";
            if (!array_key_exists("v3_auth_proto", $serviceargs)) $serviceargs["v3_auth_proto"] = "";
            if (!array_key_exists("v3_priv_proto", $serviceargs)) $serviceargs["v3_priv_proto"] = "";

            // Initialize or fill in missing array variables
            if (!array_key_exists("oid", $services)) {
                $services["oid"] = array();
            }

            if (!array_key_exists("oid", $serviceargs)) {
                $serviceargs["oid"] = array();
            }

            for ($x = 0; $x < 6; $x++) {
                if (!array_key_exists($x, $services["oid"])) {
                    $services["oid"][$x] = "";
                }

                if (!array_key_exists($x, $serviceargs["oid"]) && isset($_POST['submitButton2'])) {
                    $oid = "";
                    $name = "";
                    $label = "";
                    $units = "";
                    $matchtype = "";
                    $warning = "";
                    $critical = "";
                    $string = "";
                    $mib = "";

                    if ($x == 0) {
                        $oid = "sysUpTime.0";
                        $name = "Uptime";
                        $matchtype = "none";
                    }

                    if ($x == 1) {
                        $oid = "ifOperStatus.1";
                        $name = "Port 1 Status";
                        $string = "1";
                        $matchtype = "string";
                        $mib = "RFC1213-MIB";
                    }

                    if ($x == 2) {
                        $oid = ".1.3.6.1.4.1.2.3.51.1.2.1.5.1.0";
                        $name = "IBM RSA II Adapter Temperature";
                        $label = "Ambient Temp";
                        $units = "C";
                        $matchtype = "numeric";
                        $warning = "29";
                        $critical = "35";
                    }

                    if ($x == 3) {
                        $oid = "1.3.6.1.4.1.3076.2.1.2.17.1.7.0,1.3.6.1.4.1.3076.2.1.2.17.1.9.0";
                        $name = "Cisco VPN Sessions";
                        $label = "Active Sessions";
                        $matchtype = "numeric";
                        $warning = ":70,:8";
                        $critical = ":75,:10";
                    }

                    $serviceargs["oid"][$x] = array(
                        "oid" => $oid,
                        "name" => $name,
                        "label" => $label,
                        "units" => $units,
                        "matchtype" => $matchtype,
                        "warning" => $warning,
                        "critical" => $critical,
                        "string" => $string,
                        "mib" => $mib,
                    );
                }
            }

            if ($address) {
                $_SESSION['snmp_wizard_address'] = $address;
            }

            if ($hostname) {
                $_SESSION['snmp_wizard_hostname'] = $hostname;
            }

            if ($port) {
                $_SESSION['snmp_wizard_port'] = $port;
            }

            if ($snmpcommunity) {
                $_SESSION['snmp_wizard_snmpcommunity'] = $snmpcommunity;
            }

            if ($snmpversion) {
                $_SESSION['snmp_wizard_snmpversion'] = $snmpversion;
            }

            if ($services) {
                $_SESSION['snmp_wizard_services'] = $services;
            }

            if ($serviceargs) {
                $_SESSION['snmp_wizard_serviceargs'] = $serviceargs;
            }

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $port = grab_array_var($inargs, "port");

            if (array_key_exists('snmp_wizard_address', $_SESSION) && $address == "") {
                $address = $_SESSION['snmp_wizard_address'];
            }

            if (array_key_exists('snmp_wizard_hostname', $_SESSION) && $hostname == "") {
                $hostname = $_SESSION['snmp_wizard_hostname'];
            }

            if (array_key_exists('snmp_wizard_port', $_SESSION) && $port == "") {
                $port = $_SESSION['snmp_wizard_port'];
            }

            if (array_key_exists('snmp_wizard_port', $_SESSION) && $port == "") {
                $port = $_SESSION['snmp_wizard_port'];
            }

            $hostname = nagiosccm_replace_user_macros($hostname);

            $snmpcommunity = grab_array_var($inargs, "snmpcommunity", "");
            $snmpversion = grab_array_var($inargs, "snmpversion", "");
            $services = grab_array_var($inargs, "services", array());
            $serviceargs = grab_array_var($inargs, "serviceargs", array());

            // use encoded data (if user came back from future screen)
            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            if (isset($_POST['submitButton2']) && count($services) == 0) {
                unset($_SESSION['snmp_wizard_services']);
            }

            if (array_key_exists('snmp_wizard_snmpcommunity', $_SESSION) && $snmpcommunity == "") {
                $snmpcommunity = $_SESSION['snmp_wizard_snmpcommunity'];
            }

            if (array_key_exists('snmp_wizard_snmpversion', $_SESSION) && $snmpversion == "") {
                $snmpversion = $_SESSION['snmp_wizard_snmpversion'];
            }

            if (array_key_exists('snmp_wizard_services', $_SESSION) && !$services) {
                $services = $_SESSION['snmp_wizard_services'];
            }

            if (array_key_exists('snmp_wizard_serviceargs', $_SESSION) && !$serviceargs) {
                $serviceargs = $_SESSION['snmp_wizard_serviceargs'];
            }

            if ($serviceargs_serial != "") {
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            }

            // check for errors
            $errors = 0;
            $errmsg = array();

            // Prune empty entries
            if (array_key_exists("oid", $services)) {
                foreach ($services['oid'] as $idx => $service) {
                    if (empty($serviceargs["oid"][$idx]["oid"])) {
                        unset($services['oid'][$idx]);
                        unset($serviceargs['oid'][$idx]);
                    }
                }
            }

            if (is_valid_host_name($hostname) == false) {
                $errmsg[$errors++] = _("Invalid host name.");
            }

            if (is_valid_host_name($hostname) == false) {
                $errmsg[$errors++] = _("Invalid host name.");
            }

            if (!array_key_exists("oid", $services) || count($services["oid"]) == 0) {
                $errmsg[$errors++] = _("You have not selected any OIDs to monitor.");
            } else foreach ($services["oid"] as $index => $indexval) {
                // get oid
                $oid = $serviceargs["oid"][$index]["oid"];

                // skip empty oids
                if ($oid == "") {
                    continue;
                }

                // test match arguments
                switch ($serviceargs["oid"][$index]["matchtype"]) {
                    case "numeric":
                        if ($serviceargs["oid"][$index]["warning"] == "")
                            $errmsg[$errors++] = _("Invalid warning numeric range for OID ") . encode_form_val($oid);
                        if ($serviceargs["oid"][$index]["critical"] == "")
                            $errmsg[$errors++] = _("Invalid critical numeric range for OID ") . encode_form_val($oid);
                        break;
                    case "string":
                        if ($serviceargs["oid"][$index]["string"] == "")
                            $errmsg[$errors++] = _("Invalid string match for OID ") . encode_form_val($oid);
                        break;
                    default:
                        break;
                }
            }

            if ($address) {
                $_SESSION['snmp_wizard_address'] = $address;
            }

            if ($hostname) {
                $_SESSION['snmp_wizard_hostname'] = $hostname;
            }

            if ($port) {
                $_SESSION['snmp_wizard_port'] = $port;
            }

            if ($snmpcommunity) {
                $_SESSION['snmp_wizard_snmpcommunity'] = $snmpcommunity;
            }

            if ($snmpversion) {
                $_SESSION['snmp_wizard_snmpversion'] = $snmpversion;
            }

            if ($services) {
                $_SESSION['snmp_wizard_services'] = $services;
            }

            if ($serviceargs) {
                $_SESSION['snmp_wizard_serviceargs'] = $serviceargs;
            }

            // Test SNMP connectivity before reporting other errors
            $TEST_SNMP_TIMEOUT = 5000000; // microseconds
            $test_oid = "iso.3.6.1.2.1.1.1.0"; // Simple, generic OID that should be very easy to respond to
            
            // Add port to address if we need to
            $runaddress = $address;
            if (!empty($port) && $port != 161) {
                $runaddress = $address.":".$port;
            }

            if ($snmpversion == "3") {
                $securitylevel = grab_array_var($serviceargs, "v3_security_level", "");
                $username = grab_array_var($serviceargs, "v3_username", "");
                $authproto = grab_array_var($serviceargs, "v3_auth_proto", "");
                $privproto = grab_array_var($serviceargs, "v3_priv_proto", "");
                $authpassword = grab_array_var($serviceargs, "v3_auth_password", "");
                $privacypassword = grab_array_var($serviceargs, "v3_privacy_password", "");

                // Run through user macros to account for illegal characters
                $username_replaced = nagiosccm_replace_user_macros($username);
                $authpassword_replaced = nagiosccm_replace_user_macros($authpassword);
                $privacypassword_replaced = nagiosccm_replace_user_macros($privacypassword);

                if (!$username) {
                    $errmsg[$errors++] = "Username required for v3";
                }

                if ($securitylevel != 'noAuthNoPriv' && !$authpassword) {
                    $errmsg[$errors++] = "Authentication password required for v3 authNoPriv and authPriv";
                }

                if ($securitylevel == 'authPriv' && !$privacypassword) {
                    $errmsg[$errors++] = "Privacy password required for v3 authPriv";
                }
                
                if ($errors == 0) {
                    $test_response = snmp3_real_walk($runaddress, $username_replaced, $securitylevel, $authproto, $authpassword_replaced, $privproto, $privacypassword_replaced, $test_oid, $TEST_SNMP_TIMEOUT);
                }

                } else if ($snmpversion == "2c") {
                    $test_response = snmp2_real_walk($runaddress, $snmpcommunity, $test_oid, $TEST_SNMP_TIMEOUT);

                } else {
                    $test_response = snmprealwalk($runaddress, $snmpcommunity, $test_oid, $TEST_SNMP_TIMEOUT);
                }

                if ($errors == 0 && !$test_response) {
                    $errmsg[$errors++] = "This server does not respond to SNMP requests (not snmp, or possibly wrong information).";
                }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $port = grab_array_var($inargs, "port", "");
            $snmpcommunity = grab_array_var($inargs, "snmpcommunity");
            $snmpversion = grab_array_var($inargs, "snmpversion");
            $services = grab_array_var($inargs, "services");
            $serviceargs = grab_array_var($inargs, "serviceargs");

            $services_serial = (!empty($services) ? base64_encode(json_encode($services)) : grab_array_var($inargs, "services_serial", ''));
            $serviceargs_serial = (!empty($serviceargs) ? base64_encode(json_encode($serviceargs)) : grab_array_var($inargs, "serviceargs_serial", ''));

            if (array_key_exists('snmp_wizard_services', $_SESSION) && !$services) {
                $services = $_SESSION['snmp_wizard_services'];
            }

            if (array_key_exists('snmp_wizard_snmpcommunity', $_SESSION) && $snmpcommunity == "") {
                $snmpcommunity = $_SESSION['snmp_wizard_snmpcommunity'];
            }

            if (array_key_exists('snmp_wizard_hostname', $_SESSION) && $hostname == "") {
                $hostname = $_SESSION['snmp_wizard_hostname'];
            }

            if (array_key_exists('snmp_wizard_port', $_SESSION) && $port == "") {
                $port = $_SESSION['snmp_wizard_port'];
            }

            if ($port) {
                $_SESSION['snmp_wizard_port'] = $port;
            }

            if ($services) {
                $_SESSION['snmp_wizard_services'] = $services;
            }

            if ($snmpcommunity) {
                $_SESSION['snmp_wizard_snmpcommunity'] = $snmpcommunity;
            }

            $output = '
            <input type="hidden" name="ip_address" value="' . encode_form_val($address) . '">
            <input type="hidden" name="port" value="' . intval($port) . '">
            <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '">
            <input type="hidden" name="snmpcommunity" value="' . encode_form_val($snmpcommunity) . '">
            <input type="hidden" name="snmpversion" value="' . encode_form_val($snmpversion) . '">
            <input type="hidden" name="services_serial" value="' . $services_serial . '">
            <input type="hidden" name="serviceargs_serial" value="' . $serviceargs_serial . '">
            ';

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            $output = '

            ';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:
            $hostname = grab_array_var($inargs, "hostname", "");
            $address = grab_array_var($inargs, "ip_address", "");
            $port = grab_array_var($inargs, "port", "");
            $hostaddress = $address;

            $snmpcommunity = grab_array_var($inargs, "snmpcommunity");
            $snmpversion = grab_array_var($inargs, "snmpversion", "");

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            if (array_key_exists('snmp_wizard_address', $_SESSION) && $address == "") {
                $address = $_SESSION['snmp_wizard_address'];
            }

            if (array_key_exists('snmp_wizard_hostname', $_SESSION) && $hostname == "") {
                $hostname = $_SESSION['snmp_wizard_hostname'];
            }

            if (array_key_exists('snmp_wizard_port', $_SESSION) && $port == "") {
                $port = $_SESSION['snmp_wizard_port'];
            }

            if (array_key_exists('snmp_wizard_snmpcommunity', $_SESSION) && $snmpcommunity == "") {
                $snmpcommunity = $_SESSION['snmp_wizard_snmpcommunity'];
            }

            if (array_key_exists('snmp_wizard_snmpversion', $_SESSION) && $snmpversion == "") {
                $snmpversion = $_SESSION['snmp_wizard_snmpversion'];
            }

            if (array_key_exists('snmp_wizard_services', $_SESSION) && !$services) {
                $services = $_SESSION['snmp_wizard_services'];
            }

            if (array_key_exists('snmp_wizard_serviceargs', $_SESSION) && !$serviceargs) {
                $serviceargs = $_SESSION['snmp_wizard_serviceargs'];
            }

            if ($snmpcommunity) {
                $_SESSION['printer_wizard_snmpcommunity'] = $snmpcommunity;
            }

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["ip_address"] = $address;
            $meta_arr["snmpcommunity"] = $snmpcommunity;
            $meta_arr["snmpversion"] = $snmpversion;
            $meta_arr["services"] = $services;
            $meta_arr["serivceargs"] = $serviceargs;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = array();

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_genericnetdevice_host",
                    "host_name" => $hostname,
                    "address" => $hostaddress,
                    "icon_image" => "snmp.png",
                    "statusmap_image" => "snmp.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            // see which services we should monitor
            foreach ($services as $svc => $svcstate) {
                switch ($svc) {

                    case "oid":

                        $enabledservices = $svcstate;

                        foreach ($enabledservices as $sid => $sstate) {

                            $oid = $serviceargs["oid"][$sid]["oid"];
                            $name = $serviceargs["oid"][$sid]["name"];
                            $label = $serviceargs["oid"][$sid]["label"];
                            $units = str_replace(' ', '', $serviceargs["oid"][$sid]["units"]);
                            $matchtype = $serviceargs["oid"][$sid]["matchtype"];
                            $warning = $serviceargs["oid"][$sid]["warning"];
                            $critical = $serviceargs["oid"][$sid]["critical"];
                            $string = $serviceargs["oid"][$sid]["string"];
                            $mib = $serviceargs["oid"][$sid]["mib"];

                            $sdesc = $name;

                            $cmdargs = "";

                            // add port if we need to
                            if (!empty($port) && $port != 161) {
                                $cmdargs .= " -p " . intval($port);
                            }

                            // oid
                            if ($oid != "") {
                                $cmdargs .= " -o " . escapeshellarg($oid);
                            }

                            // snmp community
                            if ($snmpcommunity != "" && $snmpversion != "3") {
                                $escaped_snmpcommunity = escapeshellarg_nowrap($snmpcommunity);
                                $cmdargs .= " -C " . $escaped_snmpcommunity;
                            }

                            // snmp version
                            if ($snmpversion != "") {
                                $cmdargs .= " -P " . $snmpversion;
                            }

                            // snmp v3 stuff
                            if ($snmpversion == "3") {

                                $securitylevel = grab_array_var($serviceargs, "v3_security_level");
                                $username = grab_array_var($serviceargs, "v3_username");
                                $authproto = grab_array_var($serviceargs, "v3_auth_proto");
                                $privproto = grab_array_var($serviceargs, "v3_priv_proto");
                                $authpassword = grab_array_var($serviceargs, "v3_auth_password");
                                $privacypassword = grab_array_var($serviceargs, "v3_privacy_password");

                                if ($securitylevel != "")
                                    $cmdargs .= " --seclevel=" . $securitylevel;
                                if ($username != "")
                                    $cmdargs .= " --secname=" . escapeshellarg($username);
                                if ($authproto != "")
                                    $cmdargs .= " --authproto=" . $authproto;
                                if ($authpassword != "")
                                    $cmdargs .= " --authpasswd=" . escapeshellarg_nowrap($authpassword);
                                if ($privacypassword != "")
                                    $cmdargs .= " --privpasswd=" . escapeshellarg_nowrap($privacypassword);
                                if ($privproto != "")
                                    $cmdargs .= " --privproto=" . $privproto;
                            }

                            // label
                            if ($label != "") {
                                $cmdargs .= " -l " . escapeshellarg($label);
                            }

                            // units
                            if ($units != "") {
                                $cmdargs .= " -u " . escapeshellarg($units);
                            }

                            // mib
                            if ($mib != "") {
                                $cmdargs .= " -m " . $mib;
                            }

                            // match type...
                            switch ($matchtype) {
                                case "numeric":
                                    if ($warning != "")
                                        $cmdargs .= " -w " . $warning;
                                    if ($critical != "")
                                        $cmdargs .= " -c " . $critical;
                                    break;
                                case "string":
                                    if ($string != "")
                                        $cmdargs .= " -r " . escapeshellarg($string);
                                    break;
                                default:
                                    break;
                            }

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => $sdesc,
                                "use" => "xiwizard_snmp_service",
                                "check_command" => "check_xi_service_snmp!" . $cmdargs,
                                "_xiwizard" => $wizard_name,
                            );
                        }

                        break;

                    default:
                        break;
                }
            }

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}