
// todo: all of the mysql bindings need to be gone through and compared against the schema to ensure doubles aren't being cast as ints (won't error, but the data won't be right either)

int ndo_neb_handle_process(int type, void * d)
{
    return ndo_handle_process(main_thread_context, type, d);
}

int ndo_handle_process(ndo_query_context * q_ctx, int type, void * d)
{
    int ndo_return = NDO_OK;
    if (ndo_failed_load || q_ctx->connection_severed) { return NDO_OK; }
    //ndo_log("handling process(int type, void * d)");
    trace_func_handler(process);

    nebstruct_process_data * data = d;
    char * program_version = get_program_version();
    char * program_mod_date = get_program_modification_date();
    int program_pid = (int) getpid();

    switch (data->type) {

    case NEBTYPE_PROCESS_PRELAUNCH:

        if (ndo_startup_skip_writing_objects != TRUE) {
            ndo_clear_tables(q_ctx);
            ndo_set_all_objects_inactive(q_ctx);
        }

        break;


    case NEBTYPE_PROCESS_START:

        if (ndo_startup_skip_writing_objects != TRUE) {
            pthread_create(&startup_thread, NULL, ndo_startup_thread, NULL);
            pthread_detach(startup_thread);
        }

        break;


    case NEBTYPE_PROCESS_EVENTLOOPSTART:

        if (ndo_startup_skip_writing_objects != TRUE) {
            // scheduling_info is a global from Core
            ndo_write_runtime_variables(q_ctx, scheduling_info);
        }

        break;


    case NEBTYPE_PROCESS_SHUTDOWN:
    case NEBTYPE_PROCESS_RESTART:

        if (ndo_process_options & NDO_PROCESS_PROCESS) {

            if (q_ctx->connection_severed) {
                trace_return_ok();
            }
            MYSQL_RESET_BIND(HANDLE_PROCESS_SHUTDOWN);

            MYSQL_BIND_INT(HANDLE_PROCESS_SHUTDOWN, data->timestamp.tv_sec);

            MYSQL_BIND(HANDLE_PROCESS_SHUTDOWN);
            MYSQL_EXECUTE(HANDLE_PROCESS_SHUTDOWN);
        }

        break;
    }

    if (ndo_process_options & NDO_PROCESS_PROCESS) {

        if (q_ctx->connection_severed) {
            trace_return_ok();
        }
        MYSQL_RESET_BIND(HANDLE_PROCESS);

        MYSQL_BIND_INT(HANDLE_PROCESS, data->type);
        MYSQL_BIND_INT(HANDLE_PROCESS, data->timestamp.tv_sec);
        MYSQL_BIND_INT(HANDLE_PROCESS, data->timestamp.tv_usec);
        MYSQL_BIND_INT(HANDLE_PROCESS, program_pid);
        MYSQL_BIND_STR(HANDLE_PROCESS, program_version);
        MYSQL_BIND_STR(HANDLE_PROCESS, program_mod_date);

        MYSQL_BIND(HANDLE_PROCESS);
        MYSQL_EXECUTE(HANDLE_PROCESS);
    }

    trace_return_ok();
}


int ndo_neb_handle_timed_event(int type, void * d)
{
    return ndo_handle_timed_event(main_thread_context, type, d);
}

int ndo_handle_timed_event(ndo_query_context * q_ctx, int type, void * d)
{
    int ndo_return = NDO_OK;
    if (ndo_failed_load || q_ctx->connection_severed) { return NDO_OK; }
    //ndo_log("handling timed_event(int type, void * d)");
    trace_func_handler(timed_event);


    nebstruct_timed_event_data * data = d;

    int object_id = 0;

    if (data->type == NEBTYPE_TIMEDEVENT_SLEEP || data->type == NEBTYPE_TIMEDEVENT_END) {
        trace_return_ok_cond("data->type == NEBTYPE_TIMEDEVENT_SLEEP || NEBTYPE_TIMEDEVENT_END");
    }

    if (data->event_type == EVENT_SERVICE_CHECK) {

        service * svc = data->event_data;
        object_id = ndo_get_object_id_name2(q_ctx, FALSE, NDO_OBJECTTYPE_SERVICE, svc->host_name, svc->description);
    }
    else if (data->event_type == EVENT_HOST_CHECK) {

        host * hst = data->event_data;
        object_id = ndo_get_object_id_name1(q_ctx, FALSE, NDO_OBJECTTYPE_HOST, hst->name);
    }
    else if (data->event_type == EVENT_SCHEDULED_DOWNTIME) {

        /* At the time of this comment, nagios core brokers scheduled downtime events in 
         * broker_downtime_data (events.c: 1308, etc) as well as as broker_timed_event (events.c: 1381, etc). 
         * Not certain about all paths, but at least one frees-and-nulls the event_data pointer.
         */
        if (data->event_data == NULL) {
            return NDO_OK;
        }

        scheduled_downtime * dwn = find_downtime(ANY_DOWNTIME, *(unsigned long *) data->event_data);

        if (dwn == NULL) {
            /* NOTE: This path always causes incorrect behavior, but is needed due to Nagios Core 
               sometimes providing the wrong downtime_id. Find downtime.c line 352, remove_event 
               for host/service downtimes causes segfault and crashes when setting the two vars 
               in the else branch below */

            // Not logging this for now since it will be relatively common on customer systems, and we can't "resolve" it.
            // char logthis[255] = {};
            // sprintf(logthis, "Warning: could not find downtime with ID %d. Some data may be inaccurate.", *(unsigned long *) data->event_data);
            // ndo_log(logthis, 2);
            object_id = 0;
        }
        else {
            char * host_name = dwn->host_name;
            char * service_description = dwn->service_description;

            if (service_description != NULL && strlen(service_description) > 0) {
                object_id = ndo_get_object_id_name2(q_ctx, FALSE, NDO_OBJECTTYPE_SERVICE, host_name, service_description);
            }
            else {
                object_id = ndo_get_object_id_name1(q_ctx, FALSE, NDO_OBJECTTYPE_HOST, host_name);
            }
        }
    }

    if (object_id == NDO_ERROR) {
        trace_return_error_cond("Could not get object ID for downtime's parent object.");
    }

    if (data->type == NEBTYPE_TIMEDEVENT_ADD) {

        if (q_ctx->connection_severed) {
        trace_return_ok();
    }
    MYSQL_RESET_BIND(HANDLE_TIMEDEVENT_ADD);

        MYSQL_BIND_INT(HANDLE_TIMEDEVENT_ADD, data->event_type);
        MYSQL_BIND_INT(HANDLE_TIMEDEVENT_ADD, data->timestamp.tv_sec);
        MYSQL_BIND_INT(HANDLE_TIMEDEVENT_ADD, data->timestamp.tv_usec);
        MYSQL_BIND_INT(HANDLE_TIMEDEVENT_ADD, data->run_time);
        MYSQL_BIND_INT(HANDLE_TIMEDEVENT_ADD, data->recurring);
        MYSQL_BIND_INT(HANDLE_TIMEDEVENT_ADD, object_id);

        MYSQL_BIND(HANDLE_TIMEDEVENT_ADD);
        MYSQL_EXECUTE(HANDLE_TIMEDEVENT_ADD);
    }

    else if (data->type == NEBTYPE_TIMEDEVENT_REMOVE || data->type == NEBTYPE_TIMEDEVENT_EXECUTE) {

        if (q_ctx->connection_severed) {
        trace_return_ok();
    }
    MYSQL_RESET_BIND(HANDLE_TIMEDEVENT_REMOVE);

        MYSQL_BIND_INT(HANDLE_TIMEDEVENT_REMOVE, data->event_type);
        MYSQL_BIND_INT(HANDLE_TIMEDEVENT_REMOVE, data->timestamp.tv_sec);
        MYSQL_BIND_INT(HANDLE_TIMEDEVENT_REMOVE, data->recurring);
        MYSQL_BIND_INT(HANDLE_TIMEDEVENT_REMOVE, object_id);

        MYSQL_BIND(HANDLE_TIMEDEVENT_REMOVE);
        MYSQL_EXECUTE(HANDLE_TIMEDEVENT_REMOVE);
    }

    if (data->type == NEBTYPE_TIMEDEVENT_EXECUTE && (data->event_type == EVENT_SERVICE_CHECK || data->event_type == EVENT_HOST_CHECK)) {

        if (q_ctx->connection_severed) {
        trace_return_ok();
    }
    MYSQL_RESET_BIND(HANDLE_TIMEDEVENT_EXECUTE);

        MYSQL_BIND_INT(HANDLE_TIMEDEVENT_EXECUTE, data->run_time);

        MYSQL_BIND(HANDLE_TIMEDEVENT_EXECUTE);
        MYSQL_EXECUTE(HANDLE_TIMEDEVENT_EXECUTE);
    }

    trace_return_ok();
}


/* This is a default mutex, even though you could argue
 * for a PTHREAD_MUTEX_RECURSIVE. In this case, the child function
 * will only call this recursively if it can't run a query for some
 * reason. Rather than cause an infinite loop, it seems better here
 * to immediately release and exit early.
 */
int ndo_neb_handle_log(int type, void * d)
{
    static int ndo_inside_handle_log = 0;
    int ndo_own_pid = getpid();

    int ret = NDO_OK;
    if (ndo_inside_handle_log) {
        /* If a log handler query fails and we log it, don't try to run another query as a result. */
        return ret;
    }

    if (ndo_own_pid != ndo_logging_pid) {
        /* An example of why mixing threads and forks can be bad.
         * If this condition is met, we're actually running from the daemonizing process,
         * which can't coordinate via mutex locking. It will still appear to be the same "thread"
         * as the main thread in the master process as well.
         */
        return ret;
    }

    int lock_error = pthread_mutex_lock(&logging_context_mtx);
    if (lock_error == 0) {
        ndo_inside_handle_log += 1;
        ret = ndo_handle_log(logging_context, type, d);
        ndo_inside_handle_log -= 1;

        pthread_mutex_unlock(&logging_context_mtx);
    }
    return ret;
}

int ndo_handle_log(ndo_query_context * q_ctx, int type, void * d)
{
    int ndo_return = NDO_OK;
    if (ndo_is_logging) { return NDO_OK; }
    if (ndo_failed_load || q_ctx->connection_severed) { return NDO_OK; }
    /*
    trace_func_handler();
    */

    nebstruct_log_data * data = d;

    /* this particular function is a bit weird because it starts passing
       logs to the neb module before the database initialization has occured,
       so if the db hasn't been initialized, we just return */
    if (q_ctx->conn == NULL || q_ctx->connected != TRUE) {
        return NDO_OK;
    }

    if (data->data == NULL || strlen(data->data) == 0) {
        trace_return_ok_cond("data->data == NULL || strlen(data->data) == 0");
        /*
        return NDO_OK;
        */
    }

    if (q_ctx->connection_severed) {
        trace_return_ok();
    }
    MYSQL_RESET_BIND(HANDLE_LOG_DATA);

    MYSQL_BIND_INT(HANDLE_LOG_DATA, data->entry_time);
    MYSQL_BIND_INT(HANDLE_LOG_DATA, data->timestamp.tv_sec);
    MYSQL_BIND_INT(HANDLE_LOG_DATA, data->timestamp.tv_usec);
    MYSQL_BIND_INT(HANDLE_LOG_DATA, data->data_type);
    MYSQL_BIND_STR(HANDLE_LOG_DATA, data->data);

    /* ndo_handle_log uses special macros for binding and execution to allow for special error handling.
     * these macros never execute a code path where ndo_log is called recursively.
     */
    MYSQL_BIND_LOG(HANDLE_LOG_DATA);
    MYSQL_EXECUTE_LOG(HANDLE_LOG_DATA);

    /* 
    trace_return_ok();
    */
    return NDO_OK;
}


int ndo_neb_handle_system_command(int type, void * d)
{
    return ndo_handle_system_command(main_thread_context, type, d);
}

int ndo_handle_system_command(ndo_query_context * q_ctx, int type, void * d)
{
    int ndo_return = NDO_OK;
    if (ndo_failed_load || q_ctx->connection_severed) { return NDO_OK; }
    //ndo_log("handling system_command(int type, void * d)");
    trace_func_handler(system_command);

    nebstruct_system_command_data * data = d;

    if (q_ctx->connection_severed) {
        trace_return_ok();
    }
    MYSQL_RESET_BIND(HANDLE_SYSTEM_COMMAND);

    MYSQL_BIND_INT(HANDLE_SYSTEM_COMMAND, data->start_time.tv_sec);
    MYSQL_BIND_INT(HANDLE_SYSTEM_COMMAND, data->start_time.tv_usec);
    MYSQL_BIND_INT(HANDLE_SYSTEM_COMMAND, data->end_time.tv_sec);
    MYSQL_BIND_INT(HANDLE_SYSTEM_COMMAND, data->end_time.tv_usec);
    MYSQL_BIND_STR(HANDLE_SYSTEM_COMMAND, data->command_line);
    MYSQL_BIND_INT(HANDLE_SYSTEM_COMMAND, data->timeout);
    MYSQL_BIND_INT(HANDLE_SYSTEM_COMMAND, data->early_timeout);
    MYSQL_BIND_INT(HANDLE_SYSTEM_COMMAND, data->execution_time);
    MYSQL_BIND_INT(HANDLE_SYSTEM_COMMAND, data->return_code);
    MYSQL_BIND_STR(HANDLE_SYSTEM_COMMAND, data->output);
    MYSQL_BIND_STR(HANDLE_SYSTEM_COMMAND, data->output);

    MYSQL_BIND(HANDLE_SYSTEM_COMMAND);
    MYSQL_EXECUTE(HANDLE_SYSTEM_COMMAND);

    trace_return_ok();
}


int ndo_neb_handle_event_handler(int type, void * d)
{
    return ndo_handle_event_handler(main_thread_context, type, d);
}

int ndo_handle_event_handler(ndo_query_context * q_ctx, int type, void * d)
{
    int ndo_return = NDO_OK;
    if (ndo_failed_load || q_ctx->connection_severed) { return NDO_OK; }
    //ndo_log("handling event_handler(int type, void * d)");
    trace_func_handler(event_handler);

    nebstruct_event_handler_data * data = d;
    int object_id = 0;

    int command_object_id = 0;

    if (data->eventhandler_type == SERVICE_EVENTHANDLER || data->eventhandler_type == GLOBAL_SERVICE_EVENTHANDLER) {
        object_id = ndo_get_object_id_name2(q_ctx, FALSE, NDO_OBJECTTYPE_SERVICE, data->host_name, data->service_description);
    }
    else {
        object_id = ndo_get_object_id_name1(q_ctx, FALSE, NDO_OBJECTTYPE_HOST, data->host_name);
    }

    command_object_id = ndo_get_object_id_name1(q_ctx, FALSE, NDO_OBJECTTYPE_COMMAND, data->command_name);

    if (object_id == NDO_ERROR) {
        trace_return_error_cond("Could not get object ID for event handler's parent object.");
    }

    if (command_object_id == NDO_ERROR) {
        trace_return_error_cond("Could not get object ID for event handler's command");
    }

    if (q_ctx->connection_severed) {
        trace_return_ok();
    }
    MYSQL_RESET_BIND(HANDLE_EVENT_HANDLER);

    MYSQL_BIND_INT(HANDLE_EVENT_HANDLER, data->start_time.tv_sec);
    MYSQL_BIND_INT(HANDLE_EVENT_HANDLER, data->start_time.tv_usec);
    MYSQL_BIND_INT(HANDLE_EVENT_HANDLER, data->end_time.tv_sec);
    MYSQL_BIND_INT(HANDLE_EVENT_HANDLER, data->end_time.tv_usec);
    MYSQL_BIND_INT(HANDLE_EVENT_HANDLER, data->eventhandler_type);
    MYSQL_BIND_INT(HANDLE_EVENT_HANDLER, object_id);
    MYSQL_BIND_INT(HANDLE_EVENT_HANDLER, data->state);
    MYSQL_BIND_INT(HANDLE_EVENT_HANDLER, data->state_type);
    MYSQL_BIND_INT(HANDLE_EVENT_HANDLER, command_object_id);
    MYSQL_BIND_STR(HANDLE_EVENT_HANDLER, data->command_args);
    MYSQL_BIND_STR(HANDLE_EVENT_HANDLER, data->command_line);
    MYSQL_BIND_INT(HANDLE_EVENT_HANDLER, data->timeout);
    MYSQL_BIND_INT(HANDLE_EVENT_HANDLER, data->early_timeout);
    MYSQL_BIND_INT(HANDLE_EVENT_HANDLER, data->execution_time);
    MYSQL_BIND_INT(HANDLE_EVENT_HANDLER, data->return_code);
    MYSQL_BIND_STR(HANDLE_EVENT_HANDLER, data->output);
    MYSQL_BIND_STR(HANDLE_EVENT_HANDLER, data->output);

    MYSQL_BIND(HANDLE_EVENT_HANDLER);
    MYSQL_EXECUTE(HANDLE_EVENT_HANDLER);

    trace_return_ok();
}


int ndo_neb_handle_host_check(int type, void * d)
{
    return ndo_handle_host_check(main_thread_context, type, d);
}

int ndo_handle_host_check(ndo_query_context * q_ctx, int type, void * d)
{
    int ndo_return = NDO_OK;
    if (ndo_failed_load || q_ctx->connection_severed) { return NDO_OK; }
    //ndo_log("handling host_check(int type, void * d)");
    trace_func_handler(host_check);

    nebstruct_host_check_data * data = d;
    int object_id = 0;

    int command_object_id = 0;

    /* this is the only data we care about / need */
    if (data->type != NEBTYPE_HOSTCHECK_PROCESSED) {
        trace_return_ok_cond("data->type != NEBTYPE_HOSTCHECK_PROCESSED");
    }

    object_id = ndo_get_object_id_name1(q_ctx, FALSE, NDO_OBJECTTYPE_HOST, data->host_name);

    command_object_id = ndo_get_object_id_name1(q_ctx, FALSE, NDO_OBJECTTYPE_COMMAND, data->command_name);

    if (object_id == NDO_ERROR) {
        trace_return_error_cond("Could not get object ID for host check's parent host.");
    }

    if (command_object_id == NDO_ERROR) {
        trace_return_error_cond("Could not get object ID for host check's command");
    }


    if (q_ctx->connection_severed) {
        trace_return_ok();
    }
    MYSQL_RESET_BIND(HANDLE_HOST_CHECK);

    MYSQL_BIND_INT(HANDLE_HOST_CHECK, data->start_time.tv_sec);
    MYSQL_BIND_INT(HANDLE_HOST_CHECK, data->start_time.tv_usec);
    MYSQL_BIND_INT(HANDLE_HOST_CHECK, data->end_time.tv_sec);
    MYSQL_BIND_INT(HANDLE_HOST_CHECK, data->end_time.tv_usec);
    MYSQL_BIND_INT(HANDLE_HOST_CHECK, object_id);
    MYSQL_BIND_INT(HANDLE_HOST_CHECK, data->check_type);
    MYSQL_BIND_INT(HANDLE_HOST_CHECK, data->current_attempt);
    MYSQL_BIND_INT(HANDLE_HOST_CHECK, data->max_attempts);
    MYSQL_BIND_INT(HANDLE_HOST_CHECK, data->state);
    MYSQL_BIND_INT(HANDLE_HOST_CHECK, data->state_type);
    MYSQL_BIND_INT(HANDLE_HOST_CHECK, data->timeout);
    MYSQL_BIND_INT(HANDLE_HOST_CHECK, data->early_timeout);
    MYSQL_BIND_DOUBLE(HANDLE_HOST_CHECK, data->execution_time);
    MYSQL_BIND_DOUBLE(HANDLE_HOST_CHECK, data->latency);
    MYSQL_BIND_INT(HANDLE_HOST_CHECK, data->return_code);
    MYSQL_BIND_STR(HANDLE_HOST_CHECK, data->output);
    MYSQL_BIND_STR(HANDLE_HOST_CHECK, data->long_output);
    MYSQL_BIND_STR(HANDLE_HOST_CHECK, data->perf_data);
    MYSQL_BIND_INT(HANDLE_HOST_CHECK, command_object_id);
    MYSQL_BIND_STR(HANDLE_HOST_CHECK, data->command_args);
    MYSQL_BIND_STR(HANDLE_HOST_CHECK, data->command_line);

    MYSQL_BIND(HANDLE_HOST_CHECK);
    MYSQL_EXECUTE(HANDLE_HOST_CHECK);

    trace_return_ok();
}

int ndo_neb_handle_service_check(int type, void * d)
{
    return ndo_handle_service_check(main_thread_context, type, d);
}

int ndo_handle_service_check(ndo_query_context * q_ctx, int type, void * d)
{
    int ndo_return = NDO_OK;
    if (ndo_failed_load || q_ctx->connection_severed) { return NDO_OK; }
    //ndo_log("handling service_check(int type, void * d)");
    trace_func_handler(service_check);

    nebstruct_service_check_data * data = d;
    int object_id = 0;

    int command_object_id = 0;

    /* this is the only data we care about / need */
    if (data->type != NEBTYPE_SERVICECHECK_PROCESSED) {
        trace_return_ok_cond("data->type != NEBTYPE_SERVICECHECK_PROCESSED");
    }

    object_id = ndo_get_object_id_name2(q_ctx, FALSE, NDO_OBJECTTYPE_SERVICE, data->host_name, data->service_description);

    if (object_id == NDO_ERROR) {
        trace_return_error_cond("Could not get object ID for service check's parent service.");
    }

    if (data->command_name != NULL) {
        /* Nagios Core appears to always pass NULL for its command arguments when brokering NEBTYPE_SERVICECHECK_PROCESSED.
         * It's not clear why this was done, so we're working around it here for now.
         */
        command_object_id = ndo_get_object_id_name1(q_ctx, FALSE, NDO_OBJECTTYPE_COMMAND, data->command_name);

        if (command_object_id == NDO_ERROR) {
            trace_return_error_cond("Could not get object ID for service check's command");
        }
    }

    if (q_ctx->connection_severed) {
        trace_return_ok();
    }
    MYSQL_RESET_BIND(HANDLE_SERVICE_CHECK);

    MYSQL_BIND_INT(HANDLE_SERVICE_CHECK, data->start_time.tv_sec);
    MYSQL_BIND_INT(HANDLE_SERVICE_CHECK, data->start_time.tv_usec);
    MYSQL_BIND_INT(HANDLE_SERVICE_CHECK, data->end_time.tv_sec);
    MYSQL_BIND_INT(HANDLE_SERVICE_CHECK, data->end_time.tv_usec);
    MYSQL_BIND_INT(HANDLE_SERVICE_CHECK, object_id);
    MYSQL_BIND_INT(HANDLE_SERVICE_CHECK, data->check_type);
    MYSQL_BIND_INT(HANDLE_SERVICE_CHECK, data->current_attempt);
    MYSQL_BIND_INT(HANDLE_SERVICE_CHECK, data->max_attempts);
    MYSQL_BIND_INT(HANDLE_SERVICE_CHECK, data->state);
    MYSQL_BIND_INT(HANDLE_SERVICE_CHECK, data->state_type);
    MYSQL_BIND_INT(HANDLE_SERVICE_CHECK, data->timeout);
    MYSQL_BIND_INT(HANDLE_SERVICE_CHECK, data->early_timeout);
    MYSQL_BIND_DOUBLE(HANDLE_SERVICE_CHECK, data->execution_time);
    MYSQL_BIND_DOUBLE(HANDLE_SERVICE_CHECK, data->latency);
    MYSQL_BIND_INT(HANDLE_SERVICE_CHECK, data->return_code);
    MYSQL_BIND_STR(HANDLE_SERVICE_CHECK, data->output);
    MYSQL_BIND_STR(HANDLE_SERVICE_CHECK, data->long_output);
    MYSQL_BIND_STR(HANDLE_SERVICE_CHECK, data->perf_data);
    MYSQL_BIND_INT(HANDLE_SERVICE_CHECK, command_object_id);
    MYSQL_BIND_STR(HANDLE_SERVICE_CHECK, data->command_args);
    MYSQL_BIND_STR(HANDLE_SERVICE_CHECK, data->command_line);

    MYSQL_BIND(HANDLE_SERVICE_CHECK);
    MYSQL_EXECUTE(HANDLE_SERVICE_CHECK);

    trace_return_ok();
}


int ndo_neb_handle_comment(int type, void * d)
{
    return ndo_handle_comment(main_thread_context, type, d);
}

int ndo_handle_comment(ndo_query_context * q_ctx, int type, void * d)
{
    int ndo_return = NDO_OK;
    if (ndo_failed_load || q_ctx->connection_severed) { return NDO_OK; }
    //ndo_log("handling comment(int type, void * d)");
    trace_func_handler(comment);

    nebstruct_comment_data * data = d;
    int object_id = 0;

    // NOTE: Under the current startup model (see ndo_thread_comment), 
    // hosts and services will always be written to the DB before comments are handled.
    // host/service object IDs should not need to be inserted if they are missing.
    if (data->comment_type == SERVICE_COMMENT) {
        object_id = ndo_get_object_id_name2(q_ctx, FALSE, NDO_OBJECTTYPE_SERVICE, data->host_name, data->service_description);
    }
    else {
        object_id = ndo_get_object_id_name1(q_ctx, FALSE, NDO_OBJECTTYPE_HOST, data->host_name);
    }

    if (object_id == NDO_ERROR) {
        trace_return_error_cond("Could not get object ID for comment's parent object.");
    }

    if (data->type == NEBTYPE_COMMENT_ADD || data->type == NEBTYPE_COMMENT_LOAD) {

        if (q_ctx->connection_severed) {
            trace_return_ok();
        }
        MYSQL_RESET_BIND(HANDLE_COMMENT_ADD);

        MYSQL_BIND_INT(HANDLE_COMMENT_ADD, data->comment_type);
        MYSQL_BIND_INT(HANDLE_COMMENT_ADD, data->entry_type);
        MYSQL_BIND_INT(HANDLE_COMMENT_ADD, object_id);
        MYSQL_BIND_INT(HANDLE_COMMENT_ADD, data->entry_time);
        MYSQL_BIND_INT(HANDLE_COMMENT_ADD, data->comment_id);
        MYSQL_BIND_STR(HANDLE_COMMENT_ADD, data->author_name);
        MYSQL_BIND_STR(HANDLE_COMMENT_ADD, data->comment_data);
        MYSQL_BIND_INT(HANDLE_COMMENT_ADD, data->persistent);
        MYSQL_BIND_INT(HANDLE_COMMENT_ADD, data->source);
        MYSQL_BIND_INT(HANDLE_COMMENT_ADD, data->expires);
        MYSQL_BIND_INT(HANDLE_COMMENT_ADD, data->expire_time);
        MYSQL_BIND_INT(HANDLE_COMMENT_ADD, data->timestamp.tv_sec);
        MYSQL_BIND_INT(HANDLE_COMMENT_ADD, data->timestamp.tv_usec);

        MYSQL_BIND(HANDLE_COMMENT_ADD);
        MYSQL_EXECUTE(HANDLE_COMMENT_ADD);

        if (q_ctx->connection_severed) {
            trace_return_ok();
        }
        if ((ndo_process_options & NDO_PROCESS_COMMENT_HISTORY) == 0) {
            trace_return_ok();
        }
        MYSQL_RESET_BIND(HANDLE_COMMENT_HISTORY_ADD);

        MYSQL_BIND_INT(HANDLE_COMMENT_HISTORY_ADD, data->comment_type);
        MYSQL_BIND_INT(HANDLE_COMMENT_HISTORY_ADD, data->entry_type);
        MYSQL_BIND_INT(HANDLE_COMMENT_HISTORY_ADD, object_id);
        MYSQL_BIND_INT(HANDLE_COMMENT_HISTORY_ADD, data->entry_time);
        MYSQL_BIND_INT(HANDLE_COMMENT_HISTORY_ADD, data->comment_id);
        MYSQL_BIND_STR(HANDLE_COMMENT_HISTORY_ADD, data->author_name);
        MYSQL_BIND_STR(HANDLE_COMMENT_HISTORY_ADD, data->comment_data);
        MYSQL_BIND_INT(HANDLE_COMMENT_HISTORY_ADD, data->persistent);
        MYSQL_BIND_INT(HANDLE_COMMENT_HISTORY_ADD, data->source);
        MYSQL_BIND_INT(HANDLE_COMMENT_HISTORY_ADD, data->expires);
        MYSQL_BIND_INT(HANDLE_COMMENT_HISTORY_ADD, data->expire_time);
        MYSQL_BIND_INT(HANDLE_COMMENT_HISTORY_ADD, data->timestamp.tv_sec);
        MYSQL_BIND_INT(HANDLE_COMMENT_HISTORY_ADD, data->timestamp.tv_usec);

        MYSQL_BIND(HANDLE_COMMENT_HISTORY_ADD);
        MYSQL_EXECUTE(HANDLE_COMMENT_HISTORY_ADD);
    }

    if (data->type == NEBTYPE_COMMENT_DELETE) {

        if (q_ctx->connection_severed) {
            trace_return_ok();
        }

        /**
         * This is logically somewhat different from the other checks against
         * NDO_PROCESS_COMMENT_HISTORY and NDO_PROCESS_DOWNTIME_HISTORY. I'm not sure why Bryan
         * originally put this history handling query before the "normal" handler when every 
         * other block has it after, and I don't know the downstream effects of that, so we're 
         * just putting it inside the conditional block instead ¯\_(ツ)_/¯
         *  -swolf 2023-06-20
         */
        if (ndo_process_options & NDO_PROCESS_COMMENT_HISTORY) {
            MYSQL_RESET_BIND(HANDLE_COMMENT_HISTORY_DELETE);

            MYSQL_BIND_INT(HANDLE_COMMENT_HISTORY_DELETE, data->timestamp.tv_sec);
            MYSQL_BIND_INT(HANDLE_COMMENT_HISTORY_DELETE, data->timestamp.tv_usec);
            MYSQL_BIND_INT(HANDLE_COMMENT_HISTORY_DELETE, data->entry_time);
            MYSQL_BIND_INT(HANDLE_COMMENT_HISTORY_DELETE, data->comment_id);

            MYSQL_BIND(HANDLE_COMMENT_HISTORY_DELETE);
            MYSQL_EXECUTE(HANDLE_COMMENT_HISTORY_DELETE);
        }

        if (q_ctx->connection_severed) {
            trace_return_ok();
        }
        MYSQL_RESET_BIND(HANDLE_COMMENT_DELETE);

        MYSQL_BIND_INT(HANDLE_COMMENT_DELETE, data->entry_time);
        MYSQL_BIND_INT(HANDLE_COMMENT_DELETE, data->comment_id);

        MYSQL_BIND(HANDLE_COMMENT_DELETE);
        MYSQL_EXECUTE(HANDLE_COMMENT_DELETE);
    }

    trace_return_ok();
}


int ndo_neb_handle_downtime(int type, void * d)
{
    return ndo_handle_downtime(main_thread_context, type, d);
}

int ndo_handle_downtime(ndo_query_context * q_ctx, int type, void * d)
{
    int ndo_return = NDO_OK;
    if (ndo_failed_load || q_ctx->connection_severed) { return NDO_OK; }
    //ndo_log("handling downtime(int type, void * d)");
    trace_func_handler(downtime);

    nebstruct_downtime_data * data = d;
    int object_id = 0;

    time_t entry_time = ndo_truncate_time(data->entry_time);
    time_t start_time = ndo_truncate_time(data->start_time);
    time_t end_time = ndo_truncate_time(data->end_time);

    if (data->downtime_type == SERVICE_DOWNTIME) {
        object_id = ndo_get_object_id_name2(q_ctx, FALSE, NDO_OBJECTTYPE_SERVICE, data->host_name, data->service_description);
    }
    else {
        object_id = ndo_get_object_id_name1(q_ctx, FALSE, NDO_OBJECTTYPE_HOST, data->host_name);
    }

    if (object_id == NDO_ERROR) {
        trace_return_error_cond("Could not get object ID for comment's parent object.");
    }

    if (data->type == NEBTYPE_DOWNTIME_ADD || data->type == NEBTYPE_DOWNTIME_LOAD) {

        if (q_ctx->connection_severed) {
            trace_return_ok();
        }
        MYSQL_RESET_BIND(HANDLE_DOWNTIME_ADD);

        MYSQL_BIND_INT(HANDLE_DOWNTIME_ADD, data->downtime_type);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_ADD, object_id);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_ADD, entry_time);
        MYSQL_BIND_STR(HANDLE_DOWNTIME_ADD, data->author_name);
        MYSQL_BIND_STR(HANDLE_DOWNTIME_ADD, data->comment_data);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_ADD, data->downtime_id);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_ADD, data->triggered_by);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_ADD, data->fixed);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_ADD, data->duration);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_ADD, start_time);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_ADD, end_time);

        MYSQL_BIND(HANDLE_DOWNTIME_ADD);
        MYSQL_EXECUTE(HANDLE_DOWNTIME_ADD);


        if (q_ctx->connection_severed) {
            trace_return_ok();
        }
        if ((ndo_process_options & NDO_PROCESS_DOWNTIME_HISTORY) == 0) {
            trace_return_ok();
        }
        MYSQL_RESET_BIND(HANDLE_DOWNTIME_HISTORY_ADD);

        MYSQL_BIND_INT(HANDLE_DOWNTIME_HISTORY_ADD, data->downtime_type);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_HISTORY_ADD, object_id);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_HISTORY_ADD, entry_time);
        MYSQL_BIND_STR(HANDLE_DOWNTIME_HISTORY_ADD, data->author_name);
        MYSQL_BIND_STR(HANDLE_DOWNTIME_HISTORY_ADD, data->comment_data);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_HISTORY_ADD, data->downtime_id);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_HISTORY_ADD, data->triggered_by);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_HISTORY_ADD, data->fixed);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_HISTORY_ADD, data->duration);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_HISTORY_ADD, start_time);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_HISTORY_ADD, end_time);

        MYSQL_BIND(HANDLE_DOWNTIME_HISTORY_ADD);
        MYSQL_EXECUTE(HANDLE_DOWNTIME_HISTORY_ADD);
    }

    if (data->type == NEBTYPE_DOWNTIME_START) {

        if (q_ctx->connection_severed) {
            trace_return_ok();
        }
        MYSQL_RESET_BIND(HANDLE_DOWNTIME_START);

        MYSQL_BIND_INT(HANDLE_DOWNTIME_START, data->timestamp.tv_sec);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_START, data->timestamp.tv_usec);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_START, object_id);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_START, data->downtime_type);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_START, entry_time);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_START, start_time);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_START, end_time);

        MYSQL_BIND(HANDLE_DOWNTIME_START);
        MYSQL_EXECUTE(HANDLE_DOWNTIME_START);


        if (q_ctx->connection_severed) {
            trace_return_ok();
        }
        if ((ndo_process_options & NDO_PROCESS_DOWNTIME_HISTORY) == 0) {
            trace_return_ok();
        }
        MYSQL_RESET_BIND(HANDLE_DOWNTIME_HISTORY_START);

        MYSQL_BIND_INT(HANDLE_DOWNTIME_HISTORY_START, data->timestamp.tv_sec);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_HISTORY_START, data->timestamp.tv_usec);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_HISTORY_START, object_id);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_HISTORY_START, data->downtime_type);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_HISTORY_START, entry_time);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_HISTORY_START, start_time);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_HISTORY_START, end_time);

        MYSQL_BIND(HANDLE_DOWNTIME_HISTORY_START);
        MYSQL_EXECUTE(HANDLE_DOWNTIME_HISTORY_START);
    }

    if (data->type == NEBTYPE_DOWNTIME_STOP) {

        int cancelled = 0;

        if (data->attr == NEBATTR_DOWNTIME_STOP_CANCELLED) {
            cancelled = 1;
        }

        if (q_ctx->connection_severed) {
            trace_return_ok();
        }
        if ((ndo_process_options & NDO_PROCESS_DOWNTIME_HISTORY) == 0) {
            trace_return_ok();
        }
        MYSQL_RESET_BIND(HANDLE_DOWNTIME_HISTORY_STOP);

        MYSQL_BIND_INT(HANDLE_DOWNTIME_HISTORY_STOP, data->timestamp.tv_sec);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_HISTORY_STOP, data->timestamp.tv_usec);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_HISTORY_STOP, cancelled);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_HISTORY_STOP, object_id);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_HISTORY_STOP, data->downtime_type);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_HISTORY_STOP, entry_time);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_HISTORY_STOP, start_time);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_HISTORY_STOP, end_time);

        MYSQL_BIND(HANDLE_DOWNTIME_HISTORY_STOP);
        MYSQL_EXECUTE(HANDLE_DOWNTIME_HISTORY_STOP);
    }

    if (data->type == NEBTYPE_DOWNTIME_STOP || data->type == NEBTYPE_DOWNTIME_DELETE) {

        if (q_ctx->connection_severed) {
            trace_return_ok();
        }
        MYSQL_RESET_BIND(HANDLE_DOWNTIME_STOP);

        MYSQL_BIND_INT(HANDLE_DOWNTIME_STOP, data->downtime_type);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_STOP, object_id);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_STOP, entry_time);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_STOP, start_time);
        MYSQL_BIND_INT(HANDLE_DOWNTIME_STOP, end_time);

        MYSQL_BIND(HANDLE_DOWNTIME_STOP);
        MYSQL_EXECUTE(HANDLE_DOWNTIME_STOP);
    }

    trace_return_ok();
}


int ndo_neb_handle_flapping(int type, void * d)
{
    return ndo_handle_flapping(main_thread_context, type, d);
}

int ndo_handle_flapping(ndo_query_context * q_ctx, int type, void * d)
{
    int ndo_return = NDO_OK;
    if (ndo_failed_load || q_ctx->connection_severed) { return NDO_OK; }
    //ndo_log("handling flapping(int type, void * d)");
    trace_func_handler(flapping);

    nebstruct_flapping_data * data = d;
    int object_id = 0;
    nagios_comment * comment = NULL;
    time_t comment_time = 0;

    if (data->flapping_type == SERVICE_FLAPPING) {
        object_id = ndo_get_object_id_name2(q_ctx, FALSE, NDO_OBJECTTYPE_SERVICE, data->host_name, data->service_description);
        comment = find_service_comment(data->comment_id);
    }
    else {
        object_id = ndo_get_object_id_name1(q_ctx, FALSE, NDO_OBJECTTYPE_HOST, data->host_name);
        comment = find_host_comment(data->comment_id);
    }

    if (object_id == NDO_ERROR) {
        trace_return_error_cond("Could not get object ID for flapping event's parent object");
    }

    if (data->comment_id != 0L && comment != NULL) {
        comment_time = comment->entry_time;
    }

    if (q_ctx->connection_severed) {
        trace_return_ok();
    }
    MYSQL_RESET_BIND(HANDLE_FLAPPING);

    MYSQL_BIND_INT(HANDLE_FLAPPING, data->timestamp.tv_sec);
    MYSQL_BIND_INT(HANDLE_FLAPPING, data->timestamp.tv_usec);
    MYSQL_BIND_INT(HANDLE_FLAPPING, data->type);
    MYSQL_BIND_INT(HANDLE_FLAPPING, data->attr);
    MYSQL_BIND_INT(HANDLE_FLAPPING, data->flapping_type);
    MYSQL_BIND_INT(HANDLE_FLAPPING, object_id);
    MYSQL_BIND_DOUBLE(HANDLE_FLAPPING, data->percent_change);
    MYSQL_BIND_DOUBLE(HANDLE_FLAPPING, data->low_threshold);
    MYSQL_BIND_DOUBLE(HANDLE_FLAPPING, data->high_threshold);
    MYSQL_BIND_INT(HANDLE_FLAPPING, comment_time);
    MYSQL_BIND_INT(HANDLE_FLAPPING, data->comment_id);

    MYSQL_BIND(HANDLE_FLAPPING);
    MYSQL_EXECUTE(HANDLE_FLAPPING);

    trace_return_ok();
}


int ndo_neb_handle_program_status(int type, void * d)
{
    return ndo_handle_program_status(main_thread_context, type, d);
}

int ndo_handle_program_status(ndo_query_context * q_ctx, int type, void * d)
{
    int ndo_return = NDO_OK;
    if (ndo_failed_load || q_ctx->connection_severed) { return NDO_OK; }
    //ndo_log("handling program_status(int type, void * d)");
    trace_func_handler(program_status);

    nebstruct_program_status_data * data = d;

    if (q_ctx->connection_severed) {
        trace_return_ok();
    }
    MYSQL_RESET_BIND(HANDLE_PROGRAM_STATUS);

    MYSQL_BIND_INT(HANDLE_PROGRAM_STATUS, data->timestamp.tv_sec);
    MYSQL_BIND_INT(HANDLE_PROGRAM_STATUS, data->program_start);
    MYSQL_BIND_INT(HANDLE_PROGRAM_STATUS, data->pid);
    MYSQL_BIND_INT(HANDLE_PROGRAM_STATUS, data->daemon_mode);
    MYSQL_BIND_INT(HANDLE_PROGRAM_STATUS, data->last_log_rotation);
    MYSQL_BIND_INT(HANDLE_PROGRAM_STATUS, data->notifications_enabled);
    MYSQL_BIND_INT(HANDLE_PROGRAM_STATUS, data->active_service_checks_enabled);
    MYSQL_BIND_INT(HANDLE_PROGRAM_STATUS, data->passive_service_checks_enabled);
    MYSQL_BIND_INT(HANDLE_PROGRAM_STATUS, data->active_host_checks_enabled);
    MYSQL_BIND_INT(HANDLE_PROGRAM_STATUS, data->passive_host_checks_enabled);
    MYSQL_BIND_INT(HANDLE_PROGRAM_STATUS, data->event_handlers_enabled);
    MYSQL_BIND_INT(HANDLE_PROGRAM_STATUS, data->flap_detection_enabled);
    MYSQL_BIND_INT(HANDLE_PROGRAM_STATUS, data->process_performance_data);
    MYSQL_BIND_INT(HANDLE_PROGRAM_STATUS, data->obsess_over_hosts);
    MYSQL_BIND_INT(HANDLE_PROGRAM_STATUS, data->obsess_over_services);
    MYSQL_BIND_INT(HANDLE_PROGRAM_STATUS, data->modified_host_attributes);
    MYSQL_BIND_INT(HANDLE_PROGRAM_STATUS, data->modified_service_attributes);
    MYSQL_BIND_STR(HANDLE_PROGRAM_STATUS, data->global_host_event_handler);
    MYSQL_BIND_STR(HANDLE_PROGRAM_STATUS, data->global_service_event_handler);

    MYSQL_BIND(HANDLE_PROGRAM_STATUS);
    MYSQL_EXECUTE(HANDLE_PROGRAM_STATUS);

    trace_return_ok();
}

/* This function intentionally left as internal-only */
int ndo_handle_customvariables_change(ndo_query_context *q_ctx, time_t time, customvariablesmember *custom_variables, int object_id)
{
    int ndo_return = NDO_OK;
    if (ndo_failed_load || q_ctx->connection_severed) { return NDO_OK; }
    customvariablesmember *iter = NULL;
    for (iter = custom_variables; iter != NULL; iter = iter->next) {
        if (!(iter->has_been_modified)) {
            continue;
        }

        /* Execute the query that adds an entry to nagios_customvariablestatus */
        if (q_ctx->connection_severed) {
            trace_return_ok();
        }
        MYSQL_RESET_BIND(HANDLE_CUSTOMVARS_CHANGE);
        MYSQL_BIND_INT(HANDLE_CUSTOMVARS_CHANGE, object_id);
        MYSQL_BIND_INT(HANDLE_CUSTOMVARS_CHANGE, time);
        MYSQL_BIND_STR(HANDLE_CUSTOMVARS_CHANGE, iter->variable_name);
        MYSQL_BIND_STR(HANDLE_CUSTOMVARS_CHANGE, iter->variable_value);
        MYSQL_BIND(HANDLE_CUSTOMVARS_CHANGE);
        MYSQL_EXECUTE(HANDLE_CUSTOMVARS_CHANGE);
    }

    return NDO_OK;
}


int ndo_neb_handle_host_status(int type, void * d)
{
    return ndo_handle_host_status(main_thread_context, type, d);
}

int ndo_handle_host_status(ndo_query_context * q_ctx, int type, void * d)
{
    int ndo_return = NDO_OK;
    if (ndo_failed_load || q_ctx->connection_severed) { return NDO_OK; }
    //ndo_log("handling host_status(int type, void * d)");
    trace_func_handler(host_status);

    nebstruct_host_status_data * data = d;
    int host_object_id = 0;
    int timeperiod_object_id = 0;
    host * hst = NULL;
    timeperiod * tm = NULL;

    if (data->object_ptr == NULL) {
        NDO_REPORT_ERROR("Broker data pointer(s) is/are null");
        trace_return_ok_cond("data->object_ptr == NULL");
    }

    hst = data->object_ptr;
    tm = hst->check_period_ptr;

    host_object_id = ndo_get_object_id_name1(q_ctx, FALSE, NDO_OBJECTTYPE_HOST, hst->name);
    if (tm != NULL && tm->name != NULL) {
        timeperiod_object_id = ndo_get_object_id_name1(q_ctx, FALSE, NDO_OBJECTTYPE_TIMEPERIOD, tm->name);
    }

    if (host_object_id == NDO_ERROR) {
        trace_return_error_cond("Could not get object ID for host status's parent host.");
    }
    if (timeperiod_object_id == NDO_ERROR) {
        trace_return_error_cond("Could not get object ID for host status's timeperiod.");
    }

    if (hst->modified_attributes & MODATTR_CUSTOM_VARIABLE) {
        /* Update the custom variables table as well */
        ndo_handle_customvariables_change(q_ctx, data->timestamp.tv_sec, hst->custom_variables, host_object_id);
    }

    if (q_ctx->connection_severed) {
        trace_return_ok();
    }
    MYSQL_RESET_BIND(HANDLE_HOST_STATUS);

    MYSQL_BIND_INT(HANDLE_HOST_STATUS, database_instance_id);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, host_object_id);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, data->timestamp.tv_sec);
    MYSQL_BIND_STR(HANDLE_HOST_STATUS, hst->plugin_output);
    MYSQL_BIND_STR(HANDLE_HOST_STATUS, hst->long_plugin_output);
    MYSQL_BIND_STR(HANDLE_HOST_STATUS, hst->perf_data);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->current_state);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->has_been_checked);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->should_be_scheduled);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->current_attempt);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->max_attempts);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->last_check);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->next_check);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->check_type);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->check_options);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->last_state_change);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->last_hard_state_change);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->last_hard_state);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->last_time_up);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->last_time_down);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->last_time_unreachable);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->state_type);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->last_notification);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->next_notification);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->no_more_notifications);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->notifications_enabled);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->problem_has_been_acknowledged);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->acknowledgement_type);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->current_notification_number);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->accept_passive_checks);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->checks_enabled);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->event_handler_enabled);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->flap_detection_enabled);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->is_flapping);
    MYSQL_BIND_DOUBLE(HANDLE_HOST_STATUS, hst->percent_state_change);
    MYSQL_BIND_DOUBLE(HANDLE_HOST_STATUS, hst->latency);
    MYSQL_BIND_DOUBLE(HANDLE_HOST_STATUS, hst->execution_time);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->scheduled_downtime_depth);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->process_performance_data);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->obsess);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, hst->modified_attributes);
    MYSQL_BIND_STR(HANDLE_HOST_STATUS, hst->event_handler);
    MYSQL_BIND_STR(HANDLE_HOST_STATUS, hst->check_command);
    MYSQL_BIND_DOUBLE(HANDLE_HOST_STATUS, hst->check_interval);
    MYSQL_BIND_DOUBLE(HANDLE_HOST_STATUS, hst->retry_interval);
    MYSQL_BIND_INT(HANDLE_HOST_STATUS, timeperiod_object_id);

    MYSQL_BIND(HANDLE_HOST_STATUS);
    MYSQL_EXECUTE(HANDLE_HOST_STATUS);

    trace_return_ok();
}


int ndo_neb_handle_service_status(int type, void * d)
{
    return ndo_handle_service_status(main_thread_context, type, d);
}

int ndo_handle_service_status(ndo_query_context * q_ctx, int type, void * d)
{
    int ndo_return = NDO_OK;
    if (ndo_failed_load || q_ctx->connection_severed) { return NDO_OK; }
    //ndo_log("handling service_status(int type, void * d)");
    trace_func_handler(service_status);

    nebstruct_service_status_data * data = d;

    int service_object_id = 0;
    int timeperiod_object_id = 0;
    service * svc = NULL;
    timeperiod * tm = NULL;

    if (data->object_ptr == NULL) {
        NDO_REPORT_ERROR("Broker data pointer(s) is/are null");
        trace_return_ok_cond("data->object_ptr == NULL");
    }

    svc = data->object_ptr;
    tm = svc->check_period_ptr;

    if (svc->host_name == NULL || svc->description == NULL) {
        ndo_log("host_name or service_description was NULL when handling service status", NSLOG_RUNTIME_WARNING);
        return NDO_OK;
    }

    if (strcmp(svc->host_name, "") == 0 || strcmp(svc->description, "") == 0) {
        ndo_log("host_name or service_description was empty when handling service status", NSLOG_RUNTIME_WARNING);
        return NDO_OK;
    }

    service_object_id = ndo_get_object_id_name2(q_ctx, FALSE, NDO_OBJECTTYPE_SERVICE, svc->host_name, svc->description);

    if (service_object_id == 0) {
        ndo_log("service_object_id was 0 when handling service status", NSLOG_RUNTIME_WARNING);
        return NDO_OK;
    }
    if (tm != NULL && tm->name != NULL) {
        timeperiod_object_id = ndo_get_object_id_name1(q_ctx, FALSE, NDO_OBJECTTYPE_TIMEPERIOD, tm->name);
    }


    if (service_object_id == NDO_ERROR) {
        trace_return_error_cond("Could not get object ID for service status's parent service.");
    }

    if (timeperiod_object_id == NDO_ERROR) {
        trace_return_error_cond("Could not get object ID for service status's timeperiod.");
    }

    if (svc->modified_attributes & MODATTR_CUSTOM_VARIABLE) {
        /* Update the custom variables table as well */
        ndo_handle_customvariables_change(q_ctx, data->timestamp.tv_sec, svc->custom_variables, service_object_id);
    }

    if (q_ctx->connection_severed) {
        trace_return_ok();
    }
    MYSQL_RESET_BIND(HANDLE_SERVICE_STATUS);

    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, service_object_id);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, data->timestamp.tv_sec);
    MYSQL_BIND_STR(HANDLE_SERVICE_STATUS, svc->plugin_output);
    MYSQL_BIND_STR(HANDLE_SERVICE_STATUS, svc->long_plugin_output);
    MYSQL_BIND_STR(HANDLE_SERVICE_STATUS, svc->perf_data);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->current_state);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->has_been_checked);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->should_be_scheduled);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->current_attempt);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->max_attempts);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->last_check);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->next_check);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->check_type);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->check_options);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->last_state_change);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->last_hard_state_change);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->last_hard_state);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->last_time_ok);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->last_time_warning);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->last_time_unknown);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->last_time_critical);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->state_type);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->last_notification);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->next_notification);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->no_more_notifications);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->notifications_enabled);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->problem_has_been_acknowledged);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->acknowledgement_type);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->current_notification_number);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->accept_passive_checks);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->checks_enabled);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->event_handler_enabled);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->flap_detection_enabled);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->is_flapping);
    MYSQL_BIND_DOUBLE(HANDLE_SERVICE_STATUS, svc->percent_state_change);
    MYSQL_BIND_DOUBLE(HANDLE_SERVICE_STATUS, svc->latency);
    MYSQL_BIND_DOUBLE(HANDLE_SERVICE_STATUS, svc->execution_time);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->scheduled_downtime_depth);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->process_performance_data);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->obsess);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, svc->modified_attributes);
    MYSQL_BIND_STR(HANDLE_SERVICE_STATUS, svc->event_handler);
    MYSQL_BIND_STR(HANDLE_SERVICE_STATUS, svc->check_command);
    MYSQL_BIND_DOUBLE(HANDLE_SERVICE_STATUS, svc->check_interval);
    MYSQL_BIND_DOUBLE(HANDLE_SERVICE_STATUS, svc->retry_interval);
    MYSQL_BIND_INT(HANDLE_SERVICE_STATUS, timeperiod_object_id);

    MYSQL_BIND(HANDLE_SERVICE_STATUS);
    MYSQL_EXECUTE(HANDLE_SERVICE_STATUS);

    trace_return_ok();
}


int ndo_neb_handle_contact_status(int type, void * d)
{
    return ndo_handle_contact_status(main_thread_context, type, d);
}

int ndo_handle_contact_status(ndo_query_context * q_ctx, int type, void * d)
{
    int ndo_return = NDO_OK;
    if (ndo_failed_load || q_ctx->connection_severed) { return NDO_OK; }
    //ndo_log("handling contact_status(int type, void * d)");
    trace_func_handler(contact_status);

    nebstruct_contact_status_data * data = d;

    int contact_object_id = 0;
    contact * cnt = NULL;

    if (data->object_ptr == NULL) {
        NDO_REPORT_ERROR("Broker data pointer(s) is/are null");
        trace_return_ok_cond("data->object_ptr == NULL");
    }

    cnt = data->object_ptr;

    contact_object_id = ndo_get_object_id_name1(q_ctx, FALSE, NDO_OBJECTTYPE_CONTACT, cnt->name);

    if (contact_object_id == NDO_ERROR) {
        trace_return_error_cond("Could not get object ID for contact status's parent contact.");
    }

    if (cnt->modified_attributes & MODATTR_CUSTOM_VARIABLE) {
        /* Update the custom variables table as well */
        ndo_handle_customvariables_change(q_ctx, data->timestamp.tv_sec, cnt->custom_variables, contact_object_id);
    }

    if (q_ctx->connection_severed) {
        trace_return_ok();
    }
    MYSQL_RESET_BIND(HANDLE_CONTACT_STATUS);

    MYSQL_BIND_INT(HANDLE_CONTACT_STATUS, contact_object_id);
    MYSQL_BIND_INT(HANDLE_CONTACT_STATUS, data->timestamp.tv_sec);
    MYSQL_BIND_INT(HANDLE_CONTACT_STATUS, cnt->host_notifications_enabled);
    MYSQL_BIND_INT(HANDLE_CONTACT_STATUS, cnt->service_notifications_enabled);
    MYSQL_BIND_INT(HANDLE_CONTACT_STATUS, cnt->last_host_notification);
    MYSQL_BIND_INT(HANDLE_CONTACT_STATUS, cnt->last_service_notification);
    MYSQL_BIND_INT(HANDLE_CONTACT_STATUS, cnt->modified_attributes);
    MYSQL_BIND_INT(HANDLE_CONTACT_STATUS, cnt->modified_host_attributes);
    MYSQL_BIND_INT(HANDLE_CONTACT_STATUS, cnt->modified_service_attributes);

    MYSQL_BIND(HANDLE_CONTACT_STATUS);
    MYSQL_EXECUTE(HANDLE_CONTACT_STATUS);

    trace_return_ok();
}


int ndo_neb_handle_notification(int type, void * d)
{
    return ndo_handle_notification(main_thread_context, type, d);
}

int ndo_handle_notification(ndo_query_context * q_ctx, int type, void * d)
{
    int ndo_return = NDO_OK;
    if (ndo_failed_load || q_ctx->connection_severed) { return NDO_OK; }
    //ndo_log("handling notification(int type, void * d)");
    trace_func_handler(notification);

    nebstruct_notification_data * data = d;
    int object_id = 0;

    if (data->notification_type == SERVICE_NOTIFICATION) {
        object_id = ndo_get_object_id_name2(q_ctx, FALSE, NDO_OBJECTTYPE_SERVICE, data->host_name, data->service_description);
    }
    else {
        object_id = ndo_get_object_id_name1(q_ctx, FALSE, NDO_OBJECTTYPE_HOST, data->host_name);
    }

    if (object_id == NDO_ERROR) {
        trace_return_error_cond("Could not get object ID for notification's parent object.");
    }

    if (q_ctx->connection_severed) {
        trace_return_ok();
    }
    MYSQL_RESET_BIND(HANDLE_NOTIFICATION);

    MYSQL_BIND_INT(HANDLE_NOTIFICATION, data->start_time.tv_sec);
    MYSQL_BIND_INT(HANDLE_NOTIFICATION, data->start_time.tv_usec);
    MYSQL_BIND_INT(HANDLE_NOTIFICATION, data->end_time.tv_sec);
    MYSQL_BIND_INT(HANDLE_NOTIFICATION, data->end_time.tv_usec);
    MYSQL_BIND_INT(HANDLE_NOTIFICATION, data->notification_type);
    MYSQL_BIND_INT(HANDLE_NOTIFICATION, data->reason_type);
    MYSQL_BIND_INT(HANDLE_NOTIFICATION, object_id);
    MYSQL_BIND_INT(HANDLE_NOTIFICATION, data->state);
    MYSQL_BIND_STR(HANDLE_NOTIFICATION, data->output);
    MYSQL_BIND_STR(HANDLE_NOTIFICATION, data->output);
    MYSQL_BIND_INT(HANDLE_NOTIFICATION, data->escalated);
    MYSQL_BIND_INT(HANDLE_NOTIFICATION, data->contacts_notified);

    MYSQL_BIND(HANDLE_NOTIFICATION);
    MYSQL_EXECUTE(HANDLE_NOTIFICATION);

    /* because of the way we've changed ndo, we can no longer keep passing this
       around inside the database object.

       we can be sure that this will be the appropriate id, as the brokered
       event data for notification happens directly before each contact is
       then notified (and that data also brokered) */
    ndo_last_notification_id = mysql_insert_id(q_ctx->conn);

    trace_return_ok();
}


int ndo_neb_handle_contact_notification(int type, void * d)
{
    return ndo_handle_contact_notification(main_thread_context, type, d);
}

int ndo_handle_contact_notification(ndo_query_context * q_ctx, int type, void * d)
{
    int ndo_return = NDO_OK;
    if (ndo_failed_load || q_ctx->connection_severed) { return NDO_OK; }
    //ndo_log("handling contact_notification(int type, void * d)");
    trace_func_handler(contact_notification);

    nebstruct_contact_notification_data * data = d;

    int contact_object_id = 0;
    contact * cnt = NULL;

    if (data->contact_ptr == NULL) {
        NDO_REPORT_ERROR("Broker data pointer is null");
        trace_return_ok_cond("data->contact_ptr == NULL");
    }

    cnt = data->contact_ptr;

    contact_object_id = ndo_get_object_id_name1(q_ctx, FALSE, NDO_OBJECTTYPE_CONTACT, cnt->name);

    if (contact_object_id == NDO_ERROR) {
        trace_return_error_cond("Could not get object ID for contact notification's parent contact.");
    }
    if (q_ctx->connection_severed) {
        trace_return_ok();
    }
    MYSQL_RESET_BIND(HANDLE_CONTACT_NOTIFICATION);

    MYSQL_BIND_INT(HANDLE_CONTACT_NOTIFICATION, ndo_last_notification_id);
    MYSQL_BIND_INT(HANDLE_CONTACT_NOTIFICATION, data->start_time.tv_sec);
    MYSQL_BIND_INT(HANDLE_CONTACT_NOTIFICATION, data->start_time.tv_usec);
    MYSQL_BIND_INT(HANDLE_CONTACT_NOTIFICATION, data->end_time.tv_sec);
    MYSQL_BIND_INT(HANDLE_CONTACT_NOTIFICATION, data->end_time.tv_usec);
    MYSQL_BIND_INT(HANDLE_CONTACT_NOTIFICATION, contact_object_id);

    MYSQL_BIND(HANDLE_CONTACT_NOTIFICATION);
    MYSQL_EXECUTE(HANDLE_CONTACT_NOTIFICATION);

    /* see notes regarding `ndo_last_notification_id` - those apply here too */
    ndo_last_contact_notification_id = mysql_insert_id(q_ctx->conn);

    trace_return_ok();
}


int ndo_neb_handle_contact_notification_method(int type, void * d)
{
    return ndo_handle_contact_notification_method(main_thread_context, type, d);
}

int ndo_handle_contact_notification_method(ndo_query_context * q_ctx, int type, void * d)
{
    int ndo_return = NDO_OK;
    if (ndo_failed_load || q_ctx->connection_severed) { return NDO_OK; }
    //ndo_log("handling contact_notification_method(int type, void * d)");
    trace_func_handler(contact_notification_method);

    nebstruct_contact_notification_method_data * data = d;

    int command_object_id = ndo_get_object_id_name1(q_ctx, FALSE, NDO_OBJECTTYPE_COMMAND, data->command_name);

    if (command_object_id == NDO_ERROR) {
        trace_return_error_cond("Could not get object ID for contact notification method's parent command.");
    }

    if (q_ctx->connection_severed) {
        trace_return_ok();
    }
    MYSQL_RESET_BIND(HANDLE_CONTACT_NOTIFICATION_METHOD);

    MYSQL_BIND_INT(HANDLE_CONTACT_NOTIFICATION_METHOD, ndo_last_contact_notification_id);
    MYSQL_BIND_INT(HANDLE_CONTACT_NOTIFICATION_METHOD, data->start_time.tv_sec);
    MYSQL_BIND_INT(HANDLE_CONTACT_NOTIFICATION_METHOD, data->start_time.tv_usec);
    MYSQL_BIND_INT(HANDLE_CONTACT_NOTIFICATION_METHOD, data->end_time.tv_sec);
    MYSQL_BIND_INT(HANDLE_CONTACT_NOTIFICATION_METHOD, data->end_time.tv_usec);
    MYSQL_BIND_INT(HANDLE_CONTACT_NOTIFICATION_METHOD, command_object_id);
    MYSQL_BIND_STR(HANDLE_CONTACT_NOTIFICATION_METHOD, data->command_args);

    MYSQL_BIND(HANDLE_CONTACT_NOTIFICATION_METHOD);
    MYSQL_EXECUTE(HANDLE_CONTACT_NOTIFICATION_METHOD);

    trace_return_ok();
}


int ndo_neb_handle_external_command(int type, void * d)
{
    return ndo_handle_external_command(main_thread_context, type, d);
}

int ndo_handle_external_command(ndo_query_context * q_ctx, int type, void * d)
{
    int ndo_return = NDO_OK;
    if (ndo_failed_load || q_ctx->connection_severed) { return NDO_OK; }
    //ndo_log("handling external_command(int type, void * d)");
    trace_func_handler(external_command);

    nebstruct_external_command_data * data = d;

    if (data->type != NEBTYPE_EXTERNALCOMMAND_START) {
        trace_return_ok_cond("data->type != NEBTYPE_EXTERNALCOMMAND_START");
    }

    if (q_ctx->connection_severed) {
        trace_return_ok();
    }
    MYSQL_RESET_BIND(HANDLE_EXTERNAL_COMMAND);

    MYSQL_BIND_INT(HANDLE_EXTERNAL_COMMAND, data->command_type);
    MYSQL_BIND_INT(HANDLE_EXTERNAL_COMMAND, data->entry_time);
    MYSQL_BIND_STR(HANDLE_EXTERNAL_COMMAND, data->command_string);
    MYSQL_BIND_STR(HANDLE_EXTERNAL_COMMAND, data->command_args);

    MYSQL_BIND(HANDLE_EXTERNAL_COMMAND);
    MYSQL_EXECUTE(HANDLE_EXTERNAL_COMMAND);

    trace_return_ok();
}


int ndo_neb_handle_acknowledgement(int type, void * d)
{
    return ndo_handle_acknowledgement(main_thread_context, type, d);
}

int ndo_handle_acknowledgement(ndo_query_context * q_ctx, int type, void * d)
{
    int ndo_return = NDO_OK;
    if (ndo_failed_load || q_ctx->connection_severed) { return NDO_OK; }
    //ndo_log("handling acknowledgement(int type, void * d)");
    trace_func_handler(acknowledgement);

    nebstruct_acknowledgement_data * data = d;

    int object_id = 0;

    if (data->acknowledgement_type == SERVICE_ACKNOWLEDGEMENT) {
        object_id = ndo_get_object_id_name2(q_ctx, FALSE, NDO_OBJECTTYPE_SERVICE, data->host_name, data->service_description);
    }
    else {
        object_id = ndo_get_object_id_name1(q_ctx, FALSE, NDO_OBJECTTYPE_HOST, data->host_name);
    }

    if (object_id == NDO_ERROR) {
        trace_return_error_cond("Could not get object ID for acknowledgement's parent object.");
    }

    if (q_ctx->connection_severed) {
        trace_return_ok();
    }
    MYSQL_RESET_BIND(HANDLE_ACKNOWLEDGEMENT);

    MYSQL_BIND_INT(HANDLE_ACKNOWLEDGEMENT, data->timestamp.tv_sec);
    MYSQL_BIND_INT(HANDLE_ACKNOWLEDGEMENT, data->timestamp.tv_usec);
    MYSQL_BIND_INT(HANDLE_ACKNOWLEDGEMENT, data->acknowledgement_type);
    MYSQL_BIND_INT(HANDLE_ACKNOWLEDGEMENT, object_id);
    MYSQL_BIND_INT(HANDLE_ACKNOWLEDGEMENT, data->state);
    MYSQL_BIND_STR(HANDLE_ACKNOWLEDGEMENT, data->author_name);
    MYSQL_BIND_STR(HANDLE_ACKNOWLEDGEMENT, data->comment_data);
    MYSQL_BIND_INT(HANDLE_ACKNOWLEDGEMENT, data->is_sticky);
    MYSQL_BIND_INT(HANDLE_ACKNOWLEDGEMENT, data->persistent_comment);
    MYSQL_BIND_INT(HANDLE_ACKNOWLEDGEMENT, data->notify_contacts);

    MYSQL_BIND(HANDLE_ACKNOWLEDGEMENT);
    MYSQL_EXECUTE(HANDLE_ACKNOWLEDGEMENT);

    trace_return_ok();
}


int ndo_neb_handle_statechange(int type, void * d)
{
    return ndo_handle_statechange(main_thread_context, type, d);
}

int ndo_handle_statechange(ndo_query_context * q_ctx, int type, void * d)
{
    int ndo_return = NDO_OK;
    if (ndo_failed_load || q_ctx->connection_severed) { return NDO_OK; }
    //ndo_log("handling statechange(int type, void * d)");
    trace_func_handler(statechange);

    nebstruct_statechange_data * data = d;

    int object_id = 0;
    int last_state = 0;
    int last_hard_state = 0;

    if (data->type != NEBTYPE_STATECHANGE_END) {
        trace_return_ok_cond("data->type != NEBTYPE_STATECHANGE_END");
    }

    if (data->statechange_type == SERVICE_STATECHANGE) {

        service * svc = data->object_ptr;

        object_id = ndo_get_object_id_name2(q_ctx, FALSE, NDO_OBJECTTYPE_SERVICE, svc->host_name, svc->description);

        last_state = svc->last_state;
        last_hard_state = svc->last_hard_state;
    }
    else {

        host * hst = data->object_ptr;

        object_id = ndo_get_object_id_name1(q_ctx, FALSE, NDO_OBJECTTYPE_HOST, hst->name);

        last_state = hst->last_state;
        last_hard_state = hst->last_hard_state;
    }

    if (object_id == NDO_ERROR) {
        trace_return_error_cond("Could not get object ID for state change's parent object.");
    }

    if (q_ctx->connection_severed) {
        trace_return_ok();
    }
    MYSQL_RESET_BIND(HANDLE_STATE_CHANGE);

    MYSQL_BIND_INT(HANDLE_STATE_CHANGE, data->timestamp.tv_sec);
    MYSQL_BIND_INT(HANDLE_STATE_CHANGE, data->timestamp.tv_usec);
    MYSQL_BIND_INT(HANDLE_STATE_CHANGE, object_id);
    MYSQL_BIND_INT(HANDLE_STATE_CHANGE, data->state);
    MYSQL_BIND_INT(HANDLE_STATE_CHANGE, data->state_type);
    MYSQL_BIND_INT(HANDLE_STATE_CHANGE, data->current_attempt);
    MYSQL_BIND_INT(HANDLE_STATE_CHANGE, data->max_attempts);
    MYSQL_BIND_INT(HANDLE_STATE_CHANGE, last_state);
    MYSQL_BIND_INT(HANDLE_STATE_CHANGE, last_hard_state);
    MYSQL_BIND_STR(HANDLE_STATE_CHANGE, data->output);
    MYSQL_BIND_STR(HANDLE_STATE_CHANGE, data->longoutput);

    MYSQL_BIND(HANDLE_STATE_CHANGE);
    MYSQL_EXECUTE(HANDLE_STATE_CHANGE);

    trace_return_ok();
}


int ndo_neb_handle_retention(int type, void * d)
{
    return ndo_handle_retention(type, d);
}

int ndo_handle_retention(int type, void * d)
{
    int ndo_return = NDO_OK;
    if (ndo_failed_load) { return NDO_OK; }
    //ndo_log("handling retention(int type, void * d)");
    trace_func_handler(retention);

    nebstruct_retention_data * data = d;

    if (data->type == NEBTYPE_RETENTIONDATA_ENDLOAD) {
        ndo_write_config(NDO_CONFIG_DUMP_RETAINED);
    }

    trace_return_ok();
}
