<?php
//
// SNMP Trap Sender Component
// Copyright (c) 2010-2021 Nagios Enterprises, LLC. All rights reserved.
//

require_once(dirname(__FILE__) . '/../componenthelper.inc.php');

$snmptrapsender_component_name = "snmptrapsender";
snmptrapsender_component_init();

////////////////////////////////////////////////////////////////////////
// COMPONENT INIT FUNCTIONS
////////////////////////////////////////////////////////////////////////

function snmptrapsender_component_init()
{
    global $snmptrapsender_component_name;
    $desc = "";

    // Check XI version
    $versionok = snmptrapsender_component_checkversion();
    if (!$versionok) {
        $desc .= "<b>" . _("Error: This component requires Nagios XI 2024R1.1.3 or later.") . "</b>  ";
    }

    // Check required component installation
    $installok = snmptrapsender_component_checkinstallation();
    if (!$installok) {
        $desc .= "<b>" . _("Installation Required!") . "</b>
        " . _("You must login to the server as the root user and run the following commands to complete the installation of this component") . ":<br>
        <i>cd /usr/local/nagiosxi/html/includes/components/" . $snmptrapsender_component_name . "/</i><br>
        <i>chmod +x install.sh</i><br>
        <i>./install.sh</i><br>";
    }

    $args = array(
        COMPONENT_NAME => $snmptrapsender_component_name,
        COMPONENT_AUTHOR => "Nagios Enterprises, LLC",
        COMPONENT_DESCRIPTION => _("Allows Nagios XI to send SNMP traps to other network management systems when host and service alerts occur.  ") . $desc,
        COMPONENT_TITLE => "SNMP Trap Sender",
        COMPONENT_VERSION => '1.7.1',
        COMPONENT_CONFIGFUNCTION => "snmptrapsender_component_config_func",
    );

    register_component($snmptrapsender_component_name, $args);
}


///////////////////////////////////////////////////////////////////////////////////////////
//CONFIG FUNCTIONS
///////////////////////////////////////////////////////////////////////////////////////////

function snmptrapsender_component_config_func($mode, $inargs, &$outargs, &$result)
{
    global $snmptrapsender_component_name;

    // Initialize return code and output
    $result = 0;
    $output = "";

    switch ($mode) {
    case COMPONENT_CONFIGMODE_GETSETTINGSHTML:

        // defaults
        $trap_hosts = array();
        for ($x = 0; $x <= 4; $x++) {
            $trap_hosts[$x] = array(
                "address" => "",
                //set port to empty
                "port" => "",
                "tcp" => 0,
                "community" => "public",
                "downtime" => 0,
            );
        }

        $settings_raw = get_option("snmptrapsender_component_options");
        if ($settings_raw == "") {
            $settings = array(
                "enabled" => 0
            );
        } else {
            $settings = unserialize($settings_raw);
        }

        // initial values
        $enabled = grab_array_var($settings, "enabled", "");
        $debug = grab_array_var($settings, "debug", "");
        $trap_hosts = grab_array_var($settings, "trap_hosts", $trap_hosts);
        $snmpversion = grab_array_var($inargs, "snmpversion", "2c");

        // trim empty lines
        foreach ($trap_hosts as $x => $sa) {
            if ($sa["address"] == "") {
                unset($trap_hosts[$x]);
            } else {
                $trap_hosts[$x]["v3_username"] = decrypt_data($sa['v3_username']);
                $trap_hosts[$x]["v3_auth_password"] = decrypt_data($sa['v3_auth_password']);
                $trap_hosts[$x]["v3_privacy_password"] = decrypt_data($sa['v3_privacy_password']);
            }
        }

        // Add an empty row at the end ...
        $trap_hosts[] = array(
                "address" => "",
                "port" => "",
                "tcp" => 0,
                "community" => "public",
                "downtime" => 0,
        );
        
        $trap_hosts_count = count($trap_hosts);

        // fix missing values
        for ($x = 0; $x < $trap_hosts_count; $x++) {
            if (is_array($trap_hosts[$x]) && !array_key_exists("hoststateid", $trap_hosts[$x])) {
                $trap_hosts[$x]["hoststateid"] = "0";
            }
            if (is_array($trap_hosts[$x]) && !array_key_exists("servicestateid", $trap_hosts[$x])) {
                $trap_hosts[$x]["servicestateid"] = "0";
            }
            if (is_array($trap_hosts[$x]) && !array_key_exists("statetype", $trap_hosts[$x])) {
                $trap_hosts[$x]["statetype"] = "BOTH";
            }
            if (is_array($trap_hosts[$x]) && !array_key_exists("port", $trap_hosts[$x])) {
                $trap_hosts[$x]["port"] = "";
            }
            if (is_array($trap_hosts[$x]) && !array_key_exists("tcp", $trap_hosts[$x])) {
                $trap_hosts[$x]["tcp"] = "";
            }
            if (is_array($trap_hosts[$x]) && !array_key_exists("downtime", $trap_hosts[$x])) {
                $trap_hosts[$x]["downtime"] = "";
            }
        }

        // Use current request data if available
        if ($services == "") {
            $services = grab_array_var($inargs, "services", array());
        }
        if ($serviceargs == "") {
            $serviceargs = grab_array_var($inargs, "serviceargs", array());
        }
        if (!array_key_exists("v3_security_level", $serviceargs)) {
            $serviceargs["v3_security_level"] = "authPriv";
        }
        if (!array_key_exists("v3_username", $serviceargs)) {
            $serviceargs["v3_username"] = "";
        }
        if (!array_key_exists("v3_privacy_password", $serviceargs)) {
            $serviceargs["v3_privacy_password"] = "";
        }
        if (!array_key_exists("v3_auth_password", $serviceargs)) {
            $serviceargs["v3_auth_password"] = "";
        }
        if (!array_key_exists("v3_auth_proto", $serviceargs)) {
            $serviceargs["v3_auth_proto"] = "";
        }
        if (!array_key_exists("v3_priv_proto", $serviceargs)) {
            $serviceargs["v3_priv_proto"] = "";     
        }

        $trap_hosts_count = count($trap_hosts);        
        $component_url = get_component_url_base($snmptrapsender_component_name);
        $output = '';
        $eventhandlersok = snmptrapsender_component_checkeventhandlers();

        if (!$eventhandlersok) {
            $output .= "<font color='red'><b>WARNING:</b> " . _("Event handlers are currently disabled.  This will prevent the SNMP trap sender from working!") . "</font>";
        }

        $output .= '

<script type="text/javascript">

$(document).ready( function() 
{
    $(".snmpversion").change( change_attr ).trigger("change");     
});
';

if (is_neptune()) {
    $output.= '
        function change_attr() 
        {    
            var version_val = $(this).val();
            var parent_row = $(this).closest("tr");
        
            if (version_val == "3") {
                parent_row.find(".neptune-snmp-sender-2c").hide()
                parent_row.find(".neptune-snmp-sender-3").show()
        
                if (parent_row.find(".host-address").val()) {
                    parent_row.find(".username").attr("required", true);
                    parent_row.find(".priv-pass").attr("required", true);
                    parent_row.find(".auth-pass").attr("required", true);
                }
                
            } else if (version_val == "1" || "2c") {
                parent_row.find(".neptune-snmp-sender-2c").show()
                parent_row.find(".neptune-snmp-sender-3").hide()
                
                if (parent_row.find(".host-address").val()) {
                    parent_row.find(".username").attr("required", false);
                    parent_row.find(".priv-pass").attr("required", false);
                    parent_row.find(".auth-pass").attr("required", false);
                }
            }
        }
    ';
} else {
    $output.= '
    function change_attr() 
    {    
        var version_val = $(this).val();
        var parent_row = $(this).closest("tr");
    
        if (version_val == "3") {
            parent_row.find(".community").attr("disabled", true);            
            parent_row.find(".username").attr("disabled", false);
            parent_row.find(".priv-pass").attr("disabled", false);
            parent_row.find(".auth-pass").attr("disabled", false);
            parent_row.find(".engineid").attr("disabled", false);
            parent_row.find(".sec-level").attr("disabled", false);
            parent_row.find(".auth-proto").attr("disabled", false);
            parent_row.find(".priv-proto").attr("disabled", false);
    
            if (parent_row.find(".host-address").val()) {
                parent_row.find(".username").attr("required", true);
                parent_row.find(".priv-pass").attr("required", true);
                parent_row.find(".auth-pass").attr("required", true);
            }
            
        } else if (version_val == "1" || "2c") {  
            parent_row.find(".community").attr("disabled", false);
            parent_row.find(".username").attr("disabled", true);
            parent_row.find(".priv-pass").attr("disabled", true);
            parent_row.find(".auth-pass").attr("disabled", true);
            parent_row.find(".engineid").attr("disabled", true);
            parent_row.find(".sec-level").attr("disabled", true);
            parent_row.find(".auth-proto").attr("disabled", true);
            parent_row.find(".priv-proto").attr("disabled", true);
        }
    }
';
}

$output .= '
</script>

<h5 class="ul">' . _('Integration Settings') . '</h5>

<table class="table table-condensed table-no-border table-auto-width">
    <tr>
        <td></td>
        <td class="checkbox">
            <label>
                <input type="checkbox" class="checkbox" id="enabled" name="enabled" ' . is_checked($enabled, 1) . '>
                ' . _('Enable SNMP trap sender integration') . '
            </label>
        </td>
    </tr>
    <tr>
        <td></td>
        <td class="checkbox">
            <label>
                <input type="checkbox" class="checkbox" id="debug" name="debug" ' . is_checked($debug, "on") . '>
                ' . _('Enable Debug Logging') . '<br><div class="subtext">' . _('This will log traps sent to /usr/local/nagiosxi/var/components/snmptrapsender.log') . '</div></b>
            </label>
        </td>
    </tr>
</table>

<h5 class="ul">' . _('Trap Hosts') . '</h5>
    
<p>
    ' . _('Specify the addresses of the hosts that SNMP traps should be sent to.  If you want to prevent traps from being sent during downtime check the checkbox for each host.') . '<br>
    ' . _('If you leave the Port field blank it will use the default port 162 and UDP protocol.  Select the checkbox to use the TCP protocol.') . '
</p>';

if (is_neptune()) {
    $output .= '
<table class="table table-condensed table-bordered table-striped neptune-ts-table">
    <thead>
        <tr>
            <th class="">' . _('Host Address') . '</th>
            <th class=""">' . _('General Options') . '</th>
            <th class="">' . _('Security Options') . '</th>
        </tr>
    </thead>

    <tbody id="table_body"> ';         

            for ($x = 0; $x < $trap_hosts_count; $x++) {                

                $output .= '

        <tr class="">
            <td style="vertical-align: top; width: 33%;" class="">
                <input type="text" size="15" name="trap_hosts[' . $x . '][address]" value="' . htmlentities($trap_hosts[$x]["address"]) . '" class="form-control host-address neptune-ts-input">
            </td>
            <td style="vertical-align: top; width: 33%;"">
                <div class="neptune-ts-input-row">
                    <label>Port</label>
                    <input type="text" size="5" name="trap_hosts[' . $x . '][port]" value="' . htmlentities($trap_hosts[$x]["port"]) . '" class="form-control neptune-ts-input">
                </div>
                
                <div class="neptune-ts-input-row">
                    <label>Use TCP</label>
                    <input type="checkbox" id="tcp" name="trap_hosts[' . $x . '][tcp]" value="1"' . is_checked($trap_hosts[$x]['tcp'], 1) . '><br />
                </div>
                
                <div class="neptune-ts-input-row">
                    <label>Hosts</label>
                    <select name="trap_hosts[' . $x . '][hoststateid]" class="form-control neptune-ts-input">
                        <option value="0" ' . is_selected($trap_hosts[$x]['hoststateid'], "0") . '>ALL</option>
                        <option value="1" ' . is_selected($trap_hosts[$x]['hoststateid'], "1") . '>DOWN</option>
                        <option value="-1" ' . is_selected($trap_hosts[$x]['hoststateid'], "-1") . '>NONE</option>
                    </select>
                </div>
                
                <div class="neptune-ts-input-row">
                    <label>Services</label>
                    <select name="trap_hosts[' . $x . '][servicestateid]" class="form-control neptune-ts-input">
                        <option value="0" ' . is_selected($trap_hosts[$x]['servicestateid'], "0") . '>ALL</option>
                        <option value="2" ' . is_selected($trap_hosts[$x]['servicestateid'], "2") . '>CRITICAL</option>
                        <option value="1" ' . is_selected($trap_hosts[$x]['servicestateid'], "1") . '>WARNING</option>
                        <option value="-1" ' . is_selected($trap_hosts[$x]['servicestateid'], "-1") . '>NONE</option>
                    </select>
                </div>
                
                <div class="neptune-ts-input-row">
                    <label>State Type</label>
                    <select name="trap_hosts[' . $x . '][statetype]" class="form-control neptune-ts-input">
                        <option value="BOTH" ' . is_selected($trap_hosts[$x]['statetype'], "BOTH") . '>BOTH</option>
                        <option value="HARD" ' . is_selected($trap_hosts[$x]['statetype'], "HARD") . '>HARD</option>
                        <option value="SOFT" ' . is_selected($trap_hosts[$x]['statetype'], "SOFT") . '>SOFT</option>
                    </select>
                </div>
                
                <div class="neptune-ts-input-row">
                    <label>Don\'t Send During Downtime</label>
                    <input type="checkbox" id="downtime" name="trap_hosts[' . $x . '][downtime]" value="1"' . is_checked($trap_hosts[$x]['downtime'], 1) . '>
                </div>
                
            </td>
            
            <td style="vertical-align: top; width: 33%;"">
                <div class="neptune-ts-input-row">
                    <label>SNMP Version</label>
                    <select class="snmpversion form-control neptune-ts-input" name="trap_hosts[' . $x . '][snmpversion]">
                        <option value="2c" ' . is_selected($trap_hosts[$x]['snmpversion'], "2c") . '>2c</option>
                        <option value="3" ' . is_selected($trap_hosts[$x]['snmpversion'], "3") . '>3</option>
                    </select>
                </div>
                
                <div class="neptune-snmp-sender-2c">
                    <div class="neptune-ts-input-row">
                        <label>Community</label>
                        <input type="text" size="10" name="trap_hosts[' . $x . '][community]" value="' . htmlentities($trap_hosts[$x]["community"]) . '" class="form-control community neptune-ts-input">
                    </div>
                </div>
                
                <div class="neptune-snmp-sender-3">
                    <div class="neptune-ts-input-row">
                        <label>Security Level</label>
                        <select name="trap_hosts[' . $x . '][v3_security_level]" class="form-control sec-level neptune-ts-input">
                            <option value="noAuthNoPriv" ' . is_selected($trap_hosts[$x]["v3_security_level"], "noAuthNoPriv") . '>noAuthNoPriv</option>
                            <option value="authNoPriv" ' . is_selected($trap_hosts[$x]["v3_security_level"], "authNoPriv") . '>authNoPriv</option>
                            <option value="authPriv" ' . is_selected($trap_hosts[$x]["v3_security_level"], "authPriv") . '>authPriv</option>
                        </select>
                    </div>
                    
                    <div class="neptune-ts-input-row">
                        <label>Username</label>
                        <input type="text" size="10" name="trap_hosts[' . $x . '][v3_username]" value="' . htmlentities($trap_hosts[$x]["v3_username"]) . '" class="form-control username neptune-ts-input">
                    </div>
                    
                    <div class="neptune-ts-input-row">
                        <label>Auth Password</label>
                        <input type="text" size="10" name="trap_hosts[' . $x . '][v3_auth_password]" value="' . htmlentities($trap_hosts[$x]["v3_auth_password"]) . '" class="form-control auth-pass neptune-ts-input" >
                    </div>
                    
                    <div class="neptune-ts-input-row">
                        <label>Privacy Password</label>
                        <input type="text" size="10" name="trap_hosts[' . $x . '][v3_privacy_password]" value="' . htmlentities($trap_hosts[$x]["v3_privacy_password"]) . '" class="form-control priv-pass neptune-ts-input" >
                    </div>
                    
                    <div class="neptune-ts-input-row">
                        <label>Engine ID</label>
                        <input type="text" size="10" name="trap_hosts[' . $x . '][engine_id]" value="' . htmlentities($trap_hosts[$x]["engine_id"]) . '" class="form-control engineid neptune-ts-input" >
                    </div>
                    
                    <div class="neptune-ts-input-row">
                        <label>Auth Protocol</label>
                        <select name="trap_hosts[' . $x . '][v3_auth_proto]" class="form-control auth-proto neptune-ts-input">
                            <option value="" ' . is_selected($trap_hosts[$x]["v3_auth_proto"], "") . '>None</option>
                            <option value="md5" ' . is_selected($trap_hosts[$x]["v3_auth_proto"], "md5") . '>MD5</option>
                            <option value="sha" ' . is_selected($trap_hosts[$x]["v3_auth_proto"], "sha") . '>SHA</option>
                        </select>
                    </div>
                    
                    <div class="neptune-ts-input-row">
                        <label>Priv Protocol</label>
                        <select name="trap_hosts[' . $x . '][v3_priv_proto]" class="form-control priv-proto neptune-ts-input">
                            <option value="" ' . is_selected($trap_hosts[$x]["v3_priv_proto"], "") . '>None</option>
                            <option value="des" ' . is_selected($trap_hosts[$x]["v3_priv_proto"], "des") . '>DES</option>
                            <option value="aes" ' . is_selected($trap_hosts[$x]["v3_priv_proto"], "aes") . '>AES</option>
                        </select>
                    </div>
                </div>
            </td>
        </tr>';
            }

            $output .= '
    </tbody>
</table>';
} else {
    $output .= '
<table class="table table-condensed table-bordered table-striped table-auto-width">
    <thead>
        <tr>
            <th class="center"><nobr>' . _('Host Address') . '</nobr></th>
            <th class="center"><nobr>' . _('Port') . '</nobr></th>
            <th class="center"><nobr>' . _('Use TCP') . '</nobr></th>
            <th class="center"><nobr>' . _('Hosts') . '</nobr></th>
            <th class="center"><nobr>' . _('Services') . '</nobr></th>
            <th class="center"><nobr>' . _('State Type') . '</nobr></th>
            <th class="center">' . _('Don\'t Send During Downtime') . '</th>
            <th class="center"><nobr>' . _('SNMP Version') . '</nobr></th>
            <th class="center"><nobr>' . _('Community') . '</nobr></th>
            <th class="center"><nobr>' . _('Security Level') . '</nobr></th>
            <th class="center"><nobr>' . _('Username') . '</nobr></th>
            <th class="center"><nobr>' . _('Auth Password') . '</nobr></th>
            <th class="center"><nobr>' . _('Privacy Password') . '</nobr></th>
            <th class="center"><nobr>' . _('Engine ID') . '</nobr></th>
            <th class="center"><nobr>' . _('Auth Protocol') . '</nobr></th>
            <th class="center"><nobr>' . _('Priv Protocol') . '</nobr></th>           
        </tr>
    </thead>

    <tbody id="table_body"> ';         

            for ($x = 0; $x < $trap_hosts_count; $x++) {                

                $output .= '

        <tr class="center">
            <td>
                <input type="text" size="15" name="trap_hosts[' . $x . '][address]" value="' . htmlentities($trap_hosts[$x]["address"]) . '" class="form-control host-address">
            </td>
            <td>
                <input type="text" size="5" name="trap_hosts[' . $x . '][port]" value="' . htmlentities($trap_hosts[$x]["port"]) . '" class="form-control">
            </td>
            <td>
                <input type="checkbox" id="tcp" name="trap_hosts[' . $x . '][tcp]" value="1"' . is_checked($trap_hosts[$x]['tcp'], 1) . '>
            </td>
            <td>
                <select name="trap_hosts[' . $x . '][hoststateid]" class="form-control">
                    <option value="0" ' . is_selected($trap_hosts[$x]['hoststateid'], "0") . '>ALL</option>
                    <option value="1" ' . is_selected($trap_hosts[$x]['hoststateid'], "1") . '>DOWN</option>
                    <option value="-1" ' . is_selected($trap_hosts[$x]['hoststateid'], "-1") . '>NONE</option>
                </select>
            </td>
            <td>
                <select name="trap_hosts[' . $x . '][servicestateid]" class="form-control">
                    <option value="0" ' . is_selected($trap_hosts[$x]['servicestateid'], "0") . '>ALL</option>
                    <option value="2" ' . is_selected($trap_hosts[$x]['servicestateid'], "2") . '>CRITICAL</option>
                    <option value="1" ' . is_selected($trap_hosts[$x]['servicestateid'], "1") . '>WARNING</option>
                    <option value="-1" ' . is_selected($trap_hosts[$x]['servicestateid'], "-1") . '>NONE</option>
                </select>
            </td>
            <td>
                <select name="trap_hosts[' . $x . '][statetype]" class="form-control">
                    <option value="BOTH" ' . is_selected($trap_hosts[$x]['statetype'], "BOTH") . '>BOTH</option>
                    <option value="HARD" ' . is_selected($trap_hosts[$x]['statetype'], "HARD") . '>HARD</option>
                    <option value="SOFT" ' . is_selected($trap_hosts[$x]['statetype'], "SOFT") . '>SOFT</option>
                </select>
            </td>
            <td>
                <input type="checkbox" id="downtime" name="trap_hosts[' . $x . '][downtime]" value="1"' . is_checked($trap_hosts[$x]['downtime'], 1) . '>
            </td>
            <td>
                <select class="snmpversion form-control" name="trap_hosts[' . $x . '][snmpversion]">
                    <option value="2c" ' . is_selected($trap_hosts[$x]['snmpversion'], "2c") . '>2c</option>
                    <option value="3" ' . is_selected($trap_hosts[$x]['snmpversion'], "3") . '>3</option>
                </select>
            </td>
            <td>
                <input type="text" size="10" name="trap_hosts[' . $x . '][community]" value="' . htmlentities($trap_hosts[$x]["community"]) . '" class="form-control community">
            </td>
            <td>
                <select name="trap_hosts[' . $x . '][v3_security_level]" class="form-control sec-level">
                    <option value="noAuthNoPriv" ' . is_selected($trap_hosts[$x]["v3_security_level"], "noAuthNoPriv") . '>noAuthNoPriv</option>
                    <option value="authNoPriv" ' . is_selected($trap_hosts[$x]["v3_security_level"], "authNoPriv") . '>authNoPriv</option>
                    <option value="authPriv" ' . is_selected($trap_hosts[$x]["v3_security_level"], "authPriv") . '>authPriv</option>
                </select>
            </td>
            <td>
                <input type="text" size="10" name="trap_hosts[' . $x . '][v3_username]" value="' . htmlentities($trap_hosts[$x]["v3_username"]) . '" class="form-control username">
            </td>
            <td>
                <input type="text" size="10" name="trap_hosts[' . $x . '][v3_auth_password]" value="' . htmlentities($trap_hosts[$x]["v3_auth_password"]) . '" class="form-control auth-pass" >
            </td>
            <td>
                <input type="text" size="10" name="trap_hosts[' . $x . '][v3_privacy_password]" value="' . htmlentities($trap_hosts[$x]["v3_privacy_password"]) . '" class="form-control priv-pass" >
            </td>
            <td>
                <input type="text" size="10" name="trap_hosts[' . $x . '][engine_id]" value="' . htmlentities($trap_hosts[$x]["engine_id"]) . '" class="form-control engineid" >
            </td>
            <td>
                <select name="trap_hosts[' . $x . '][v3_auth_proto]" class="form-control auth-proto">
                    <option value="" ' . is_selected($trap_hosts[$x]["v3_auth_proto"], "") . '>None</option>
                    <option value="md5" ' . is_selected($trap_hosts[$x]["v3_auth_proto"], "md5") . '>MD5</option>
                    <option value="sha" ' . is_selected($trap_hosts[$x]["v3_auth_proto"], "sha") . '>SHA</option>
                </select>
            </td>
            <td>
                <select name="trap_hosts[' . $x . '][v3_priv_proto]" class="form-control priv-proto">
                    <option value="" ' . is_selected($trap_hosts[$x]["v3_priv_proto"], "") . '>None</option>
                    <option value="des" ' . is_selected($trap_hosts[$x]["v3_priv_proto"], "des") . '>DES</option>
                    <option value="aes" ' . is_selected($trap_hosts[$x]["v3_priv_proto"], "aes") . '>AES</option>
                </select>
            </td>
        </tr>';
            }

            $output .= '
    </tbody>
</table>';
}


$output .= '
<h5 class="ul">MIBs</h5>
    
<p>' . _('You should install the following MIBs on the trap management hosts') . ':</p>
<p>
    <a href="' . $component_url . '/mibs/NAGIOS-NOTIFY-MIB.txt">NAGIOS-NOTIFY-MIB.txt</a><br>
    <a href="' . $component_url . '/mibs/NAGIOS-ROOT-MIB.txt">NAGIOS-ROOT-MIB.txt</a><br>
</p>';

    break;

    case COMPONENT_CONFIGMODE_SAVESETTINGS:

        // get variables
        $enabled = checkbox_binary(grab_array_var($inargs, "enabled", ""));
        $trap_hosts = grab_array_var($inargs, "trap_hosts", "");
        $debug = grab_array_var($inargs, "debug", false);
            
        $log_file = '/usr/local/nagiosxi/var/components/snmptrapsender.log';

        // Renumber items & add a UID for each item
        $settings_new = array();
        $index = 0;
        foreach ($trap_hosts as $trap_host) {
            if(!empty($trap_host["address"])) {
                $settings_new[$index] = $trap_host;
                if(array_key_exists("v3_username", $trap_host)) {
                    $settings_new[$index]["v3_username"] = encrypt_data($trap_host["v3_username"]);
                }
                if(array_key_exists("v3_auth_password", $trap_host)) {
                    $settings_new[$index]["v3_auth_password"] = encrypt_data($trap_host["v3_auth_password"]);
                }
                if(array_key_exists("v3_privacy_password", $trap_host)) {
                    $settings_new[$index]["v3_privacy_password"] = encrypt_data($trap_host["v3_privacy_password"]);
                }
                $index += 1;
            }
        }
        $trap_hosts = $settings_new;

        
        // validate variables
        $errors = 0;
        $errmsg = array();
        if ($enabled == 1) {
        }

        // handle errors
        if ($errors > 0) {
            $outargs[COMPONENT_ERROR_MESSAGES] = $errmsg;
            $result = 1;
            return '';
        }

        // save settings
        $settings = array(
            "enabled" => $enabled,
            "trap_hosts" => $trap_hosts,
            "debug" => $debug,
            "log_file" => $log_file,

        );
        
        set_option("snmptrapsender_component_options", serialize($settings));

        // info messages
        $okmsg = array();
        $okmsg[] = "Settings updated.";
        $outargs[COMPONENT_INFO_MESSAGES] = $okmsg;

    break;

    default:
    break;

    }

    return $output;
}


////////////////////////////////////////////////////////////////////////
// EVENT HANDLER AND NOTIFICATION FUNCTIONS
////////////////////////////////////////////////////////////////////////

register_callback(CALLBACK_EVENT_PROCESSED, 'snmptrapsender_component_eventhandler');


function snmptrapsender_component_eventhandler($cbtype, $args)
{

    echo "*** GLOBAL HANDLER (snmptrapsender)...\n";
    print_r($args);

    switch ($args["event_type"]) {
    case EVENTTYPE_STATECHANGE:
        snmptrapsender_component_handle_statechange_event($args);
        break;
    default:
        break;
    }
}


function snmptrapsender_component_handle_statechange_event($args)
{

    // the commands we run
    $service_trap_command = "/usr/bin/snmptrap -v 2c -c public 192.168.5.4 '' NAGIOS-NOTIFY-MIB::nSvcEvent nSvcHostname s \"%host%\" nSvcDesc s \"%service%\" nSvcStateID i %servicestateid% nSvcOutput s \"%serviceoutput%\"";
    $host_trap_command = "/usr/bin/snmptrap -v 2c -c public 192.168.5.4 '' NAGIOS-NOTIFY-MIB::nHostEvent nHostname s \"%host%\" nHostStateID i %hoststateid% nHostOutput s \"%hostoutput%\"";
    $meta = grab_array_var($args, "event_meta", array());
    $handler_type = grab_array_var($meta, "handler-type", "");

    // load settings
    $settings_raw = get_option("snmptrapsender_component_options");
    if ($settings_raw == "") {
        // settings have not been configured yet...
        echo "SNMP TRAP SENDER NOT CONFIGURED!\n";
        return;
    } else {
        $settings = unserialize($settings_raw);
    }

    // are we enabled?
    $enabled = grab_array_var($settings, "enabled", "");
    if ($enabled != 1) {
        echo "SNMP TRAP SENDER NOT ENABLED! VALUE='$enabled'\n";
        return;
    }

    switch ($handler_type) {
    case "host":
        if (array_key_exists("trap_hosts", $settings)) {

            // loop through all trap hosts
            foreach ($settings["trap_hosts"] as $th) {
                echo "PROCESSING:\n";

                // get address, community and port
                $address = grab_array_var($th, "address");
                $community = grab_array_var($th, "community");
                $port = grab_array_var($th, "port");
                $snmpversion = grab_array_var($th, "snmpversion");    
                $securitylevel = grab_array_var($th, "v3_security_level");
                $username = grab_array_var($th, "v3_username");
                $authproto = grab_array_var($th, "v3_auth_proto");
                $privproto = grab_array_var($th, "v3_priv_proto");
                $authpassword = grab_array_var($th, "v3_auth_password");
                $privacypassword = grab_array_var($th, "v3_privacy_password");
                $engineid = grab_array_var($th, "engine_id");

                // This data needs to be decrypted
                $username = decrypt_data($username);
                $authpassword = decrypt_data($authpassword);
                $privacypassword = decrypt_data($privacypassword);

                // only send to hosts that have address and community defined
                if ($address != "") {
                    echo "PROCESSING:\n";
                    print_r($th);

                    // filters
                    if (isset($th['hoststateid']) && $th['hoststateid'] != 0) {
                        if ($meta['hoststateid'] < $th['hoststateid'] || $th['hoststateid'] == -1) {
                            echo "Host matched state filter, skipping... TRAPHANDLER STATE SETTING=" . $th['hoststateid'] . " EVENT STATE=" . $meta['hoststateid'] . "\n";
                            continue;
                        }
                    }
                    if (isset($th['statetype']) && $th['statetype'] != "BOTH") {
                        if ($th['statetype'] != $meta['hoststatetype']) {
                            echo "Host matched type filter, skipping... TRAPHANDLER STATETYPE SETTING=" . $th['statetype'] . " EVENT STATETYPE=" . $meta['hoststatetype'] . "\n";
                            continue;
                        }
                    }
                    if (snmptrapsender_component_retrievedowntime($meta) && grab_array_var($th, 'downtime', 0) == 1) {
                        echo "Host is in scheduled downtime... EVENT DOWNTIME=1 \n";
                        continue;
                    }
                    if ($th['port'] != "") {
                        $port = ":" . $port;
                    } else {
                        $port = '';
                    }                        
                    if (grab_array_var($th, 'tcp', 0) == 1) {
                        $tcp = 'tcp:';
                        echo "Using TCP protocol..\n";
                    } else {
                        $tcp = '';
                    }

                    if ($snmpversion == "3") {
                        if (empty($engineid))
                            $engineid_string = "";
                        else
                            $engineid_string = "-e " . $engineid;
                        
                        $trap_command = "/usr/bin/snmptrap -v 3 $engineid_string -u $username -a $authproto -A $authpassword -x $privproto -X $privacypassword $tcp$address$port '' NAGIOS-NOTIFY-MIB::nHostEvent nHostname s \"%host%\" nHostStateID i %hoststateid% nHostOutput s \"%hostoutput%\" nHostAlias s \"%hostalias%\"";
                    } else {
                        $trap_command = "/usr/bin/snmptrap -v 2c -c $community $tcp$address$port '' NAGIOS-NOTIFY-MIB::nHostEvent nHostname s \"%host%\" nHostStateID i %hoststateid% nHostOutput s \"%hostoutput%\" nHostAlias s \"%hostalias%\"";
                    }

                    snmptrapsender_component_sendtrap($address, $community, $trap_command, $meta, $settings);
                }
            }
        }
    break;

    case "service":
        if (array_key_exists("trap_hosts", $settings)) {

            // loop through all trap hosts
            foreach ($settings["trap_hosts"] as $th) {

                // get address, community and port
                $address = grab_array_var($th, "address");
                $community = grab_array_var($th, "community");
                $port = grab_array_var($th, "port");
                $snmpversion = grab_array_var($th, "snmpversion");    
                $securitylevel = grab_array_var($th, "v3_security_level");
                $username = grab_array_var($th, "v3_username");
                $authproto = grab_array_var($th, "v3_auth_proto");
                $privproto = grab_array_var($th, "v3_priv_proto");
                $authpassword = grab_array_var($th, "v3_auth_password");
                $privacypassword = grab_array_var($th, "v3_privacy_password");
                $engineid = grab_array_var($th, "engine_id");
                
                // This data needs to be decrypted
                $username = decrypt_data($username);
                $authpassword = decrypt_data($authpassword);
                $privacypassword = decrypt_data($privacypassword);

                // only send to hosts that have an address
                if ($address != "") {
                    echo "PROCESSING:\n";
                    print_r($th);

                    // filters
                    if (isset($th['servicestateid']) && $th['servicestateid'] != 0) {
                        if ($meta['servicestateid'] < $th['servicestateid'] || $th['servicestateid'] == -1) {
                            echo "Service matched state filter, skipping... TRAPHANDLER STATE SETTING=" . $th['servicestateid'] . " EVENT STATE=" . $meta['servicestateid'] . "\n";
                            continue;
                        }
                    }
                    if (isset($th['statetype']) && $th['statetype'] != "BOTH") {
                        if ($th['statetype'] != $meta['servicestatetype']) {
                            echo "Service matched type filter, skipping... TRAPHANDLER STATETYPE SETTING=" . $th['statetype'] . " EVENT STATETYPE=" . $meta['servicestatetype'] . "\n";
                            continue;
                        }
                    }
                    if (snmptrapsender_component_retrievedowntime($meta) && grab_array_var($th, 'downtime', 0) == 1) {
                        echo "Service is in scheduled downtime... EVENT DOWNTIME=1 \n";
                        continue;
                    }

                    if ($th['port'] != "") {
                        $port = ":" . $port;
                    } else {
                        $port = '';
                    }

                    if (grab_array_var($th, 'tcp', 0) == 1) {
                        $tcp = 'tcp:';
                        echo "Using TCP protocol..\n";
                    } else {
                        $tcp = '';
                    }

                    if ($snmpversion == "3") {
                        
                        if (empty($engineid))
                            $engineid_string = "";
                        else
                            $engineid_string = "-e " . $engineid;
                        
                        $trap_command = "/usr/bin/snmptrap -v 3 $engineid_string -u $username -a $authproto -A $authpassword -x $privproto -X $privacypassword $tcp$address$port '' NAGIOS-NOTIFY-MIB::nSvcEvent nSvcHostname s \"%host%\" nSvcDesc s \"%service%\" nSvcStateID i %servicestateid% nSvcOutput s \"%serviceoutput%\" nHostAlias s \"%hostalias%\"";
                    } else {
                        if (empty($community)) {
                            continue;
                        }
                        $trap_command = "/usr/bin/snmptrap -v 2c -c $community $tcp$address$port '' NAGIOS-NOTIFY-MIB::nSvcEvent nSvcHostname s \"%host%\" nSvcDesc s \"%service%\" nSvcStateID i %servicestateid% nSvcOutput s \"%serviceoutput%\" nHostAlias s \"%hostalias%\"";
                    }

                    snmptrapsender_component_sendtrap($address, $community, $trap_command, $meta, $settings);                    
                }
            }
        }
    break;

    default;
    echo "Invalid Handle Type\n";
    echo "Args passed in:\n";
    print_r($args);
    break;
    }
}

function snmptrapsender_component_sendtrap($host, $community, $command, $meta, $settings)
{

    // pre-process command for variables
    $processed_command = process_notification_text($command, $meta);

    echo "RUNNING COMMAND: $processed_command\n";
    
    if (!empty($settings['debug'])){
        $t = date("Y-m-d H:i:s");
        file_put_contents($settings['log_file'], $t . " - " .$processed_command."\n", FILE_APPEND);
    }

    // run the command
    exec($processed_command);
}


///////////////////////////////////////////////////////////////////////////////////////////
// MISC FUNCTIONS
///////////////////////////////////////////////////////////////////////////////////////////

function snmptrapsender_component_checkversion()
{
    if (!function_exists('get_product_release'))
        return false;
    //requires greater than 2024R1.1.3
    if (get_product_release() < 60103)
        return false;

    return true;
}

function snmptrapsender_component_checkinstallation()
{
    global $snmptrapsender_component_name;

    $f = "/usr/local/nagiosxi/html/includes/components/" . $snmptrapsender_component_name . "/installed.ok";
    $f2 = "/usr/local/nagiosxi/html/includes/components/" . $snmptrapsender_component_name . "/installed";
    $logfile = "/usr/local/nagiosxi/var/components/snmptrapsender.log";

    // Install file doesn't exist
    // Lets also check if the log file exists. Installed file won't exist of RPM's
    if (!file_exists($f) && !file_exists($f2) && !file_exists($logfile)) {
        return false;
    }

    return true;
}

function snmptrapsender_component_checkeventhandlers()
{
    $xml = get_xml_program_status();
    if ($xml) {
        $v = intval($xml->programstatus->event_handlers_enabled);
        if ($v == 1) {
            return true;
        }
    }

    return false;
}

function snmptrapsender_component_retrievedowntime($meta)
{
    $handler_type = grab_array_var($meta, "handler-type", "");

    if (!empty($handler_type)) {
        if ($handler_type == 'host') {
            $req = array("host_name" => $meta['host']);
            $obj = simplexml_load_string(get_host_status_xml_output($req));
            $dt = intval($obj ->hoststatus ->scheduled_downtime_depth);

            if ($dt > 0) {
                return true;
            }
            return false;
        } else if ($handler_type == 'service') {
            $req = array("name" => $meta['service'], "host_name" => $meta['host']);
            $obj = simplexml_load_string(get_service_status_xml_output($req));
            $dt = intval($obj ->servicestatus ->scheduled_downtime_depth);

            if ($dt > 0) {
                return true;
            }

            return false;
        }
    }
}
