<?php
//
// NOC screen ajax
// Copyright (c) 2018-2021 Nagios Enterprises, LLC. All rights reserved.
//

error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once(dirname(__FILE__) . '/../../common.inc.php');

// Initialization stuff
pre_init();
init_session();

// Grab GET or POST variables and check prereqs
grab_request_vars();
check_prereqs();
check_authentication(false);

route_request();

function route_request()
{
    $mode = grab_request_var("mode", "");
    switch ($mode) {
        case "audio-upload":
            do_upload_audio();
            break;
        case "load-audio":
            do_load_audio();
            break;
        case "delete-audio":
            do_delete_audio();
            break;
        case 'save-audio-prefs':
            do_save_audio_prefs();
            break;
        case 'get-audio-prefs':
            do_get_audio_prefs();
        default:
            break;
    }
}

function do_upload_audio() {

    foreach($_FILES['audio']['error'] as $error) {
        if ($error != 0) {
            print json_encode(array(
                'error' => true,
                'errmsg' => _('Could not transfer file to server.')
            ));
            return;
        }
    }

    $tmp_loc = $_FILES['audio']['tmp_name'][0];
    $type = $_FILES['audio']['type'][0];
    $file_name = basename($_FILES['audio']['name'][0]);
    $target_path = get_alarm_file_path($file_name);

    // Verify file type
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $ext = finfo_file($finfo, $tmp_loc);
    finfo_close($finfo);
    if (strpos($ext, 'audio') === false || strpos($type, 'audio') === false) {
        $ret = array('error' => true, 'errmsg' => _('File is not a valid audio file.'));
        print json_encode($ret);
        return;
    }

    // log it
    send_to_audit_log(_("User uploaded audio '") . $file_name . "'", AUDITLOGTYPE_CHANGE);

    $ret = array('status' => 'success');

    if (@move_uploaded_file($tmp_loc, $target_path)) {

        // fix perms
        chmod($target_path, 0640);
        chgrp($target_path, "nagios");
    } else {

        // error
        $ret = array('error' => true, 'errmsg' => sprintf(_('File uploaded but could not be moved to the appropriate directory. Apache user or nagios group must have write permissions on directory %s.'), get_base_dir() . '/sounds/alarm'));
    }

    print json_encode($ret);

    return;
}

// Retrieve the NOC screen alarms from utils-sounds.inc.php
// return as JSON data.
function do_load_audio() {
    print json_encode(get_alarm_sounds());
}

// Requires a basename-able file + admin permissions
// Deletes the audio file from the server.
function do_delete_audio() {
    $fname = grab_request_var('file');
    if (empty($fname)) {
        print json_encode(array(
            'error' => true,
            'errmsg' => _('No file provided')
        ));
    }

    $fname = basename($fname);

    if (!is_admin()) {
        print json_encode(array(
            'error' => true,
            'errmsg' => _('Insufficient permission to delete audio clips')
        ));
        return;
    }

    delete_alarm_sound($fname);
    print json_encode(array('status' => 'success'));
}

function do_save_audio_prefs() {
    $events = grab_request_var('events', array());
    $sound = grab_request_var('sound', '');

    if (empty($events)) {
        print json_encode(array(
            'error' => true,
            'errmsg' => _('Please select at least one event from the list')
        ));
        return;
    }

    if (empty($sound)) {
        print json_encode(array(
            'error' => true,
            'errmsg' => _('Please select a sound from the list')
        ));
        return;
    }

    if($sound === 'clear_sound_prefs'){
        $sound = '';
    }

    $prefs = get_user_meta(0, 'noc_sound_preferences', base64_encode(serialize(array())));
    $prefs = unserialize(base64_decode($prefs));

    for ($i = 0; $i < count($events); $i++) {
        $prefs[$events[$i]] = $sound;
    }

    $prefs = base64_encode(serialize($prefs));
    $user_meta_data = set_user_meta(0, 'noc_sound_preferences', $prefs);
    if(!empty($user_meta_data)){
        print json_encode(array(
            'success' => 'true',
            'successmsg' => _('Settings have succesfully been updated'),
            'status' => 'success'
        ));
    } else {
        print json_encode(array(
            'error' => true,
            'errmsg' => _('Failed to set user data')
        ));
    }
}

function do_get_audio_prefs() {
    $prefs = get_user_meta(0, 'noc_sound_preferences', base64_encode(serialize(array())));
    $prefs = unserialize(base64_decode($prefs));
    print json_encode($prefs);
}