<?php
//
// Linux Server Config Wizard
// Copyright (c) 2014-2025 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');
require_once(dirname(__FILE__) . '/../../../config/deployment/includes/utils-deployment.inc.php');

linux_server_configwizard_init();

function linux_server_configwizard_init() {

    $name = "linux-server";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "3.0.3",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor a remote Linux server, with NCPA."),
        CONFIGWIZARD_DISPLAYTITLE => _("Linux Server"),
        CONFIGWIZARD_FUNCTION => "linux_server_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "linux-server.png",
        CONFIGWIZARD_FILTER_GROUPS => array('linux', 'server'),
        CONFIGWIZARD_REQUIRES_VERSION => 70000
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function linux_server_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null, $extra = array())
{
    // Grab core wizard info
    $wizard_name = grab_array_var($inargs, "wizard");
    $wiz_args = get_configwizard_by_name($wizard_name);
    $wizard_title = isset($wiz_args[CONFIGWIZARD_DISPLAYTITLE]) ? $wiz_args[CONFIGWIZARD_DISPLAYTITLE] : '';

    /* Prerequisite software */
    $NCPA_download_url = "https://www.nagios.org/ncpa/#downloads";
    $NCPA_docs_url = "https://www.nagios.org/ncpa/#docs";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {

        case CONFIGWIZARD_MODE_GETSTAGE1HTML:
            $ip_address = grab_array_var($inargs, "ip_address", "");
            $port = grab_array_var($inargs, "port", "5693");
            $token = grab_array_var($inargs, "token", "");
            $no_ssl_verify = grab_array_var($inargs, "no_ssl_verify", 1);
            $os = grab_array_var($inargs, "os", grab_array_var($extra, "os", ""));

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:
            // Get variables that were passed to us
            $ip_address = grab_array_var($inargs, "ip_address", "");
            $port = grab_array_var($inargs, "port", "5693");
            $token = grab_array_var($inargs, "token", "");
            $no_ssl_verify = grab_array_var($inargs, "no_ssl_verify", 1);

            // For multi-host ncpa run (from deployment)
            $hosts = grab_request_var("hosts", array());
            $job_id = grab_request_var("job_id", 0);

            // Grab deploy id if it exists
            $deploy_id = grab_request_var('deploy_id', 0);

            if (!empty($deploy_id)) {
                $agent = deploy_get_agent($deploy_id);
                $metadata = json_decode(base64_decode($agent['metadata']), true);
                $token = decrypt_data($metadata['token']);
                $ip_address = $agent['address'];
            }

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($ip_address) == false && empty($hosts)) {
                $errmsg[$errors++] = _("No address specified.");
            }

            if (have_value($port) == false) {
                $errmsg[$errors++] = _("No port number specified.");
            }

            // Test the connection if no errors
            if (empty($errors) && empty($hosts)) {
                $ip_address_replaced = nagiosccm_replace_user_macros($ip_address);
                $port_replaced = nagiosccm_replace_user_macros($port);
                $token_replaced = nagiosccm_replace_user_macros($token);

                // The URL we will use to query the NCPA agent, and do a walk
                // of all monitorable items.
                $query_url = "https://{$ip_address}:{$port}/testconnect/?token=".urlencode($token);
                $query_url_replaced = "https://{$ip_address_replaced}:{$port_replaced}/testconnect/?token=".urlencode($token_replaced);

                // Remove SSL verification or not
                $context = array("ssl" => array("verify_peer" => true, "verify_peer_name" => true));

                if ($no_ssl_verify) {
                    $context['ssl']['verify_peer'] = false;
                    $context['ssl']['verify_peer_name'] = false;
                }

                // All we want to do is test if we can hit this URL.
                // Error Control Operator - @ - http://php.net/manual/en/language.operators.errorcontrol.php

                // For a simple requeest, make timeout shorter than php default or typical gateway timeout of 60
                ini_set("default_socket_timeout", 10);
                $raw_json = @file_get_contents($query_url_replaced, false, stream_context_create($context));

                if ($raw_json === FALSE || empty($raw_json)) {
                    $safe_url = str_replace(urlencode($token_replaced), "your_token", $query_url);
                    $errmsg[$errors++] = _("Unable to contact server at") . " {$safe_url}.";

                } else {
                    $json = json_decode($raw_json, true);

                    if (!array_key_exists('value', $json)) {
                        $errmsg[$errors++] = _("Bad token for connection.");
                    }
                }
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:
            // Get variables that were passed to us
            $ip_address = grab_array_var($inargs, "ip_address", "");
            $port = grab_array_var($inargs, "port", "5693");
            $token = grab_array_var($inargs, "token", "");
            $no_ssl_verify = grab_array_var($inargs, "no_ssl_verify", 1);
            $hostname = grab_array_var($inargs, 'hostname', @gethostbyaddr($ip_address));
            $default_mem_units = grab_array_var($inargs, 'default_mem_units', 'Gi');

            // Set os to extra os
            $os = grab_array_var($inargs, "os", grab_array_var($extra, "os", ""));

            // For multi-host ncpa run (from deployment)
            $hosts = grab_request_var("hosts", array());
            $job_id = grab_request_var("job_id", 0);

            if (!empty($job_id)) {
                if (count($hosts) == 1) {
                    $job = deploy_get_job($job_id);
                    $token = decrypt_data($job['ncpa_token']);
                    $ip_address = $hosts[0];
                    $hostname = gethostbyaddr($ip_address);
                    $hosts = array();
                    $os = $job['os'];
                } else {
                    // Don't set the normal things because it's multi-host...
                }
            }

            // Grab deploy id if it exists
            $deploy_id = grab_request_var('deploy_id', 0);

            if (!empty($deploy_id)) {
                $agent = deploy_get_agent($deploy_id);
                $metadata = json_decode(base64_decode($agent['metadata']), true);
                $token = decrypt_data($metadata['token']);
                $ip_address = $agent['address'];
                $hostname = $agent['hostname'];
                $os = $agent['os'];
            }

            $rp_address = nagiosccm_replace_user_macros($ip_address);
            $rp_port = nagiosccm_replace_user_macros($port);
            $rp_token = nagiosccm_replace_user_macros($token);

            $services = grab_array_var($inargs, "services", array());
            $services_serial = grab_array_var($inargs, "services_serial", "");

            if ($services_serial) {
                $services = json_decode(base64_decode($services_serial), true);
            }

            $categories = array();
            $root = array();

            // If we are running the wizard on a single host, get the proper data
            // otherwise we need to use generic data
            if (empty($hosts)) {
                $api_url = "https://{$rp_address}:{$rp_port}/api";
                $interface_data = ncpa_get_api_data('interface', $api_url, $rp_token, $no_ssl_verify);
                $disk_data = ncpa_get_api_data('disk', $api_url, $rp_token, $no_ssl_verify);
                $service_data = ncpa_get_api_data('services', $api_url, $rp_token, $no_ssl_verify);
                $process_data = ncpa_get_api_data('processes', $api_url, $rp_token, $no_ssl_verify);
                $plugin_data = ncpa_get_api_data('plugins', $api_url, $rp_token, $no_ssl_verify);

                $cpu_data = ncpa_get_api_data('cpu/percent', $api_url, $rp_token, $no_ssl_verify, array('aggregate' => 'avg'));
                $user_data = ncpa_get_api_data('user/count', $api_url, $rp_token, $no_ssl_verify);
                $mem_data = ncpa_get_api_data('memory/virtual/percent', $api_url, $rp_token, $no_ssl_verify);
                $swap_data = ncpa_get_api_data('memory/swap/percent', $api_url, $rp_token, $no_ssl_verify);

                # Find the process count.
                end($process_data);
                $key = key($process_data);
                reset($process_data);
                $totalprocs_data = $key + 1;
            } else {
                $disk_data = array('' => '');
                $interface_data = array('' => '');
                $service_data = array();
                $process_data = array();
                $plugin_data = array();
            }

            $root['disk'] = $disk_data;
            $root['interface'] = $interface_data;
            $root['services'] = $service_data;

            // Set up processes
            $tmp = array();

            foreach ($process_data as $p) {
                if (array_key_exists($p['name'], $tmp)) {
                    $tmp[$p['name']]++;
                } else {
                    $tmp[$p['name']] = 1;
                }
            }

            $root['processes'] = $tmp;

            // Sort plugin data
            if (!empty($plugin_data)) {
                sort($plugin_data);
            }

            $root['plugins'] = $plugin_data;
            $icon = _("Generic");

            if (!empty($os)) {
                $icon = '<img src="' . nagioscore_get_ui_url() . "images/logos/" . get_os_icon($os) . '" style="">';
            }

            $defaults = grab_array_var($extra, "defaults", "");

            // Non-NCPA
            $default_services['ping']['monitor'] = 'on';

            // Additional NCPA checks for Linux
            $default_services['totalprocs']['monitor'] = 'on';
            $default_services['totalprocs']['warning'] = 150;
            $default_services['totalprocs']['critical'] = 250;

            // PLACEHOLDER FOR NEW NCPA checks for Linux
            $default_services['yum']['monitor'] = 'on';
            $default_services['apt']['monitor'] = 'on';
            $default_services['load']['monitor'] = "on";
            $default_services['load']['warning'] = "15,10,5";
            $default_services['load']['critical'] = "30,20,10";
            $default_services['openfiles']['monitor'] = "on";
            $default_services['openfiles']['warning'] = 30;
            $default_services['openfiles']['critical'] = 50;

            // Standard NCPA
            // Set defaults for services
            $default_services['cpu_usage']['monitor'] = 'on';
            $default_services['cpu_usage']['warning'] = 20;
            $default_services['cpu_usage']['critical'] = 40;
            $default_services['cpu_usage']['average'] = 1;
            $default_services['users']['monitor'] = 'on';
            $default_services['users']['warning'] = 2;
            $default_services['users']['critical'] = 4;
            $default_services['memory_usage']['monitor'] = 'on';
            $default_services['memory_usage']['warning'] = 50;
            $default_services['memory_usage']['critical'] = 80;
            $default_services['swap_usage']['monitor'] = 'on';
            $default_services['swap_usage']['warning'] = 5;
            $default_services['swap_usage']['critical'] = 10;

            if (isset($root['disk']['logical'])) {
                $id = 0;

                foreach ($root['disk']['logical'] as $title => $value) {
                    if (empty($value['device_name'])) {
                        continue;
                    }

                    $default_services['disk'][$id]['monitor'] = 'on';
                    $default_services['disk'][$id]['warning'] = 70;
                    $default_services['disk'][$id]['critical'] = 90;
                    $default_services['disk'][$id]['name'] = $value['device_name'];

                    if (is_array($value['device_name'])) {
                        $default_services['disk'][$id]['name'] = $value['device_name'][0];
                    }

                    $default_services['disk'][$id]['title'] = $title;
                    $id++;
                }
            } else if (!empty($hosts)) {
                $default_services['disk'][0]['monitor'] = 'on';
                $default_services['disk'][0]['warning'] = 70;
                $default_services['disk'][0]['critical'] = 90;
                $default_services['disk'][0]['name'] = "/";
            }

            ksort($root['interface']);
            $id = 0;

            foreach ($root['interface'] as $title => $value) {
                if (stripos($title, "Local Area Connection") !== false || stripos($title, "eth") !== false || stripos($title, "ens") !== false || stripos($title, "Wireless") !== false) {
                    $default_services['interface'][$id]['monitor'] = 'on';
                } else {
                    $default_services['interface'][$id]['monitor'] = 'off';
                }

                $default_services['interface'][$id]['warning'] = 10;
                $default_services['interface'][$id]['critical'] = 100;
                $default_services['interface'][$id]['name'] = $title;

                $id++;
            }

            // Create defaults for processes
            if ($defaults == "desktop") {

                // Add explorer
                $default_services['process'][] = array(
                    'monitor' => 'off',
                    'name' => 'explorer.exe',
                    'desc' => 'Explorer',
                    'warning' => '',
                    'critical' => '1:'
                );
            }

            $default_services['process'][] = array(
                'monitor' => 'off',
                'name' => '',
                'desc' => '',
                'warning' => '',
                'critical' => ''
            );

            $default_services['services'][] = array(
                'monitor' => 'off',
                'name' => '',
                'desc' => '',
                'state' => 'running'
            );

            // Create only one plugin if no plugins saved
            $default_services['plugins'][0]['monitor'] = 'off';
            $default_services['plugins'][0]['name'] = '';
            $default_services['plugins'][0]['desc'] = '';
            $default_services['plugins'][0]['args'] = '';

            if (empty($services)) {
                $services = $default_services;
            }

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:
            // Get variables that were passed to us
            $ip_address = grab_array_var($inargs, 'ip_address');
            $hostname = grab_array_var($inargs, 'hostname');
            $port = grab_array_var($inargs, 'port');
            $token = grab_array_var($inargs, 'token');
            $default_mem_units = grab_array_var($inargs, 'default_mem_units');

            $hosts = grab_array_var($inargs,  "hosts", array());
            $job_id = grab_array_var($inargs, "job_id", 0);

            // Check for errors
            $errors = 0;
            $errmsg = array();
            if (empty($hosts) && is_valid_host_name($hostname) == false) {
                $errmsg[$errors++] = "Invalid host name.";
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }
            break;

        case CONFIGWIZARD_MODE_GETSTAGE3HTML:
            // Get variables that were passed to us
            $ip_address = grab_array_var($inargs, 'ip_address');
            $hostname = grab_array_var($inargs, 'hostname');
            $port = grab_array_var($inargs, 'port');
            $token = grab_array_var($inargs, 'token');
            $services = grab_array_var($inargs, 'services', array());
            if (empty($services)) {
                $services_serial = grab_array_var($inargs, "services_serial", "");
                $services = json_decode(base64_decode($services_serial), true);
            }

            $default_mem_units = grab_array_var($inargs, 'default_mem_units');
            $os = grab_array_var($inargs, 'os', '');

            $job_id = grab_array_var($inargs, 'job_id', 0);
            $hosts = grab_array_var($inargs, 'hosts', array());

            $output = '
            <input type="hidden" name="ip_address" value="' . encode_form_val($ip_address) . '" />
            <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '" />
            <input type="hidden" name="port" value="' . encode_form_val($port) . '" />
            <input type="hidden" name="token" value="' . encode_form_val($token) . '" />
            <input type="hidden" name="os" value="' . encode_form_val($os) . '" />
            <input type="hidden" name="default_mem_units" value="' . encode_form_val($default_mem_units) . '" />
            <input type="hidden" name="services_serial" value="' . base64_encode(json_encode($services)) . '" />
            <input type="hidden" name="job_id" value="' . intval($job_id) . '" />';

            if (!empty($hosts)) {
                foreach ($hosts as $h) {
                    $output .= '<input type="hidden" name="hosts[]" value="' . $h . '" />';
                }
            }

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:
            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:
            // Get variables that were passed to us
            $ip_address = grab_array_var($inargs, 'ip_address');
            $hostname = grab_array_var($inargs, 'hostname');
            $port = grab_array_var($inargs, 'port');
            $token = grab_array_var($inargs, 'token');
            $services = grab_array_var($inargs, 'services', array());
            if (empty($services)) {
                $services_serial = grab_array_var($inargs, "services_serial", "");
                $services = json_decode(base64_decode($services_serial), true);
            }
            $default_mem_units = grab_array_var($inargs, 'default_mem_units');
            $os = grab_array_var($inargs, 'os', '');

            $job_id = grab_array_var($inargs, 'job_id', 0);
            $hosts = grab_array_var($inargs, 'hosts', array());

            $output = '
            <input type="hidden" name="ip_address" value="' . encode_form_val($ip_address) . '" />
            <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '" />
            <input type="hidden" name="port" value="' . encode_form_val($port) . '" />
            <input type="hidden" name="token" value="' . encode_form_val($token) . '" />
            <input type="hidden" name="os" value="' . encode_form_val($os) . '" />
            <input type="hidden" name="default_mem_units" value="' . encode_form_val($default_mem_units) . '" />
            <input type="hidden" name="services_serial" value="' . base64_encode(json_encode($services)) . '" />
            <input type="hidden" name="job_id" value="' . intval($job_id) . '" />';

            if (!empty($hosts)) {
                foreach ($hosts as $h) {
                    $output .= '<input type="hidden" name="hosts[]" value="' . $h . '" />';
                }
            }
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:
            $hostname = grab_array_var($inargs, "hostname", "");
            $ip_address = grab_array_var($inargs, "ip_address", "");
            $port = grab_array_var($inargs, "port", "");
            $token = grab_array_var($inargs, "token", "");
            $default_mem_units = grab_array_var($inargs, 'default_mem_units');
            $services_serial = grab_array_var($inargs, "services_serial", "");
            $services = json_decode(base64_decode($services_serial), true);
            $os = grab_array_var($inargs, "os", "");

            $hosts = grab_array_var($inargs,  "hosts", array());
            $job_id = grab_array_var($inargs, "job_id", 0);

            // Get job info and token
            if (!empty($job_id) && !empty($hosts)) {
                $job = deploy_get_job($job_id);
                $token = decrypt_data($job['ncpa_token']);
            }

            // Save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["ip_address"] = $ip_address;
            $meta_arr["port"] = $port;
            $meta_arr["token"] = $token;
            $meta_arr["services"] = $services;
            $meta_arr["job_id"] = $job_id;
            $meta_arr["hosts"] = $hosts;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            // Escape values for check_command line
            if (function_exists('nagiosccm_replace_command_line')) {
                $token = nagiosccm_replace_command_line($token, '', '!');
            }

            $objs = array();

            if (empty($hosts)) {

                if (!host_exists($hostname)) {
                    $objs[] = array(
                        "type" => OBJECTTYPE_HOST,
                        "use" => "xiwizard_linuxserver_host",
                        "host_name" => $hostname,
                        "address" => $ip_address,
                        "icon_image" => get_os_icon($os),
                        "statusmap_image" => get_os_icon($os),
                        "_xiwizard" => $wizard_name);
                }

            } else {

                foreach ($hosts as $h) {
                    if (!host_exists($h)) {
                        $objs[] = array(
                            "type" => OBJECTTYPE_HOST,
                            "use" => "xiwizard_linuxserver_host",
                            "host_name" => $h,
                            "address" => $h,
                            "_xiwizard" => $wizard_name);
                    }
                }

            }

            // Common plugin opts
            $commonopts = "-t " . escapeshellarg_nowrap($token)." ";
            if ($port) {
                $commonopts .= "-P ".intval($port)." ";
            }

            // If we don't already have an array of hosts, make it
            $hostnames = array($hostname);
            if (!empty($hosts)) {
                $hostnames = $hosts;
            }

            foreach ($hostnames as $hostname) {
                foreach ($services as $type => $args) {
                    $pluginopts = "";
                    $pluginopts .= $commonopts;

                    switch ($type) {

                        case "ping":
                         $objs[] = array(
                             "type" => OBJECTTYPE_SERVICE,
                             "host_name" => $hostname,
                             "service_description" => "Ping",
                             "use" => "xiwizard_linuxserver_ping_service",
                             "_xiwizard" => $wizard_name,
                         );
                         break;

                        case "totalprocs":
                            if (!array_key_exists('monitor', $args)) break;

                            $pluginopts .= "-M processes";

                            if (!empty($args['warning'])) {
                                $pluginopts .= " -w " . escapeshellarg($args["warning"]);
                            }
                            if (!empty($args['critical'])) {
                                $pluginopts .= " -c " . escapeshellarg($args["critical"]);
                            }

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => "Total Processes",
                                "use" => "xiwizard_ncpa_service",
                                "check_command" => "check_xi_ncpa!" . $pluginopts,
                                "_xiwizard" => $wizard_name
                            );
                            break;
                            if (!array_key_exists('monitor', $args)) break;

                            $pluginopts .= "-M user/count";

                            if (!empty($args['warning'])) {
                                $pluginopts .= " -w " . escapeshellarg($args["warning"]);
                            }
                            if (!empty($args['critical'])) {
                                $pluginopts .= " -c " . escapeshellarg($args["critical"]);
                            }

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => "User Count",
                                "use" => "xiwizard_ncpa_service",
                                "check_command" => "check_xi_ncpa!" . $pluginopts,
                                "_xiwizard" => $wizard_name
                            );
                            break;


                        ### PLACEHOLDER Implement in the next version.  Currently, plugins would have to be manually installed.
                        ### START
                        case "yum":
                            if (!$RHEL) break;
                            if (!array_key_exists('monitor', $args)) break;

                            $pluginopts .= "-M yum";

                            if (!empty($args['warning'])) {
                                $pluginopts .= " -w " . escapeshellarg($args["warning"]);
                            }
                            if (!empty($args['critical'])) {
                                $pluginopts .= " -c " . escapeshellarg($args["critical"]);
                            }

                            if (!empty($args['average'])) {
                                $pluginopts .= " -q 'aggregate=avg'";
                            }

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => "Yum Updates",
                                "use" => "xiwizard_ncpa_service",
                                "check_command" => "check_xi_ncpa!" . $pluginopts,
                                "_xiwizard" => $wizard_name
                            );
                            break;

                        case "apt":
                            if (!$DEB) break;
                            if (!array_key_exists('monitor', $args)) break;

                            $pluginopts .= "-M apt";

                            if (!empty($args['warning'])) {
                                $pluginopts .= " -w " . escapeshellarg($args["warning"]);
                            }
                            if (!empty($args['critical'])) {
                                $pluginopts .= " -c " . escapeshellarg($args["critical"]);
                            }

                            if (!empty($args['average'])) {
                                $pluginopts .= " -q 'aggregate=avg'";
                            }

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => "APT Updates",
                                "use" => "xiwizard_ncpa_service",
                                "check_command" => "check_xi_ncpa!" . $pluginopts,
                                "_xiwizard" => $wizard_name
                            );
                            break;

                        case "load":
                            if (!array_key_exists('monitor', $args)) break;

                            $pluginopts .= "-M load";

                            if (!empty($args['warning'])) {
                                $pluginopts .= " -w " . escapeshellarg($args["warning"]);
                            }
                            if (!empty($args['critical'])) {
                                $pluginopts .= " -c " . escapeshellarg($args["critical"]);
                            }

                            if (!empty($args['average'])) {
                                $pluginopts .= " -q 'aggregate=avg'";
                            }

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => "Load",
                                "use" => "xiwizard_ncpa_service",
                                "check_command" => "check_xi_ncpa!" . $pluginopts,
                                "_xiwizard" => $wizard_name
                            );
                            break;

                        case "openfiles":
                            if (!array_key_exists('monitor', $args)) break;

                            $pluginopts .= "-M files/open";

                            if (!empty($args['warning'])) {
                                $pluginopts .= " -w " . escapeshellarg($args["warning"]);
                            }
                            if (!empty($args['critical'])) {
                                $pluginopts .= " -c " . escapeshellarg($args["critical"]);
                            }

                            if (!empty($args['average'])) {
                                $pluginopts .= " -q 'aggregate=avg'";
                            }

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => "Open Files",
                                "use" => "xiwizard_ncpa_service",
                                "check_command" => "check_xi_ncpa!" . $pluginopts,
                                "_xiwizard" => $wizard_name
                            );
                            break;

                        ### END Manual pLugins
                        ### END PLACEHOLDER

                        case "cpu_usage":
                            if (!array_key_exists('monitor', $args)) break;

                            $pluginopts .= "-M cpu/percent";

                            if (!empty($args['warning'])) {
                                $pluginopts .= " -w " . escapeshellarg($args["warning"]);
                            }
                            if (!empty($args['critical'])) {
                                $pluginopts .= " -c " . escapeshellarg($args["critical"]);
                            }

                            if (!empty($args['average'])) {
                                $pluginopts .= " -q 'aggregate=avg'";
                            }

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => "CPU Usage",
                                "use" => "xiwizard_ncpa_service",
                                "check_command" => "check_xi_ncpa!" . $pluginopts,
                                "_xiwizard" => $wizard_name
                            );
                            break;

                        case "users":
                            if (!array_key_exists('monitor', $args)) {
                                break;
                            }
                            $pluginopts .= "-M user/count";

                            if (!empty($args['warning'])) {
                                $pluginopts .= " -w " . escapeshellarg($args["warning"]);
                            }
                            if (!empty($args['critical'])) {
                                $pluginopts .= " -c " . escapeshellarg($args["critical"]);
                            }

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => "User Count",
                                "use" => "xiwizard_ncpa_service",
                                "check_command" => "check_xi_ncpa!" . $pluginopts,
                                "_xiwizard" => $wizard_name
                            );
                            break;

                        case "memory_usage":
                            if (!array_key_exists('monitor', $args)) {
                                break;
                            }
                            $pluginopts .= "-M memory/virtual";

                            if (!empty($default_mem_units)) {
                                $pluginopts .= " -u " . escapeshellarg($default_mem_units);
                            }
                            if (!empty($args['warning'])) {
                                $pluginopts .= " -w " . escapeshellarg($args["warning"]);
                            }
                            if (!empty($args['critical'])) {
                                $pluginopts .= " -c " . escapeshellarg($args["critical"]);
                            }

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => "Memory Usage",
                                "use" => "xiwizard_ncpa_service",
                                "check_command" => "check_xi_ncpa!" . $pluginopts,
                                "_xiwizard" => $wizard_name
                            );
                            break;

                        case "swap_usage":
                            if (!array_key_exists('monitor', $args)) {
                                break;
                            }
                            $pluginopts .= "-M memory/swap";

                            if (!empty($default_mem_units)) {
                                $pluginopts .= " -u " . escapeshellarg($default_mem_units);
                            }
                            if (!empty($args['warning'])) {
                                $pluginopts .= " -w " . escapeshellarg($args["warning"]);
                            }
                            if (!empty($args['critical'])) {
                                $pluginopts .= " -c " . escapeshellarg($args["critical"]);
                            }

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => "Swap Usage",
                                "use" => "xiwizard_ncpa_service",
                                "check_command" => "check_xi_ncpa!" . $pluginopts,
                                "_xiwizard" => $wizard_name
                            );
                            break;

                        case "disk":
                            foreach ($args as $id => $metrics) {
                                if (!array_key_exists('monitor', $metrics)) {
                                    continue;
                                }

                                // Ensure we are using the right name/title for the disk API (and verify it has | instead of / or \)
                                $title = $metrics['name'];
                                if (isset($metrics['title'])) {
                                    $title = $metrics['title'];
                                }
                                $title = str_replace(array('\\', '/'), '|', $title);

                                $disk = "disk/logical/{$title}";
                                $theseopts = "{$pluginopts} -M " . escapeshellarg($disk);

                                if (!empty($metrics["warning"])) {
                                    $theseopts .= " -w " . escapeshellarg($metrics["warning"]);
                                }
                                if (!empty($metrics["critical"])) {
                                    $theseopts .= " -c " . escapeshellarg($metrics["critical"]);
                                }

                                // Make sure back slash doesn't escape service description line
                                $description = str_replace(array('\\', ','), array('/', ' '), $metrics['name']);

                                $objs[] = array(
                                    "type" => OBJECTTYPE_SERVICE,
                                    "host_name" => $hostname,
                                    "service_description" => "Disk Usage on " . $description,
                                    "use" => "xiwizard_ncpa_service",
                                    "check_command" => "check_xi_ncpa!" . $theseopts,
                                    "_xiwizard" => $wizard_name
                                );
                            }
                            break;

                        case "interface":
                            foreach ($args as $id => $metrics) {
                                if (!array_key_exists('monitor', $metrics) || empty($metrics['name'])) {
                                    continue;
                                }

                                $title = $metrics['name'];

                                $theseopts = "{$pluginopts} -M 'interface/{$title}/bytes_sent' -d -u M";

                                if (!empty($metrics["warning"])) {
                                    $theseopts .= " -w " . escapeshellarg($metrics["warning"]);
                                }
                                if (!empty($metrics["critical"])) {
                                    $theseopts .= " -c " . escapeshellarg($metrics["critical"]);
                                }

                                $objs[] = array(
                                    "type" => OBJECTTYPE_SERVICE,
                                    "host_name" => $hostname,
                                    "service_description" => "{$title} Bandwidth - Outbound",
                                    "use" => "xiwizard_ncpa_service",
                                    "check_command" => "check_xi_ncpa!" . $theseopts,
                                    "_xiwizard" => $wizard_name);

                                $theseopts = "{$pluginopts} -M 'interface/{$title}/bytes_recv' -d -u M";

                                if (!empty($metrics["warning"])) {
                                    $theseopts .= " -w " . escapeshellarg($metrics["warning"]);
                                }
                                if (!empty($metrics["critical"])) {
                                    $theseopts .= " -c " . escapeshellarg($metrics["critical"]);
                                }

                                $objs[] = array(
                                    "type" => OBJECTTYPE_SERVICE,
                                    "host_name" => $hostname,
                                    "service_description" => "{$title} Bandwidth - Inbound",
                                    "use" => "xiwizard_ncpa_service",
                                    "check_command" => "check_xi_ncpa!" . $theseopts,
                                    "_xiwizard" => $wizard_name
                                );
                            }
                            break;

                        case "services":
                            foreach ($args as $i => $service) {
                                if (!array_key_exists('monitor', $service) || empty($service['name']) || empty($service['desc'])) {
                                    continue;
                                }
                                $query = "service=" . str_replace('$', '$$', $service["name"]) . ",status=" . $service["state"];
                                $theseopts = "{$pluginopts} -M 'services' -q " . escapeshellarg($query);
                                $description = str_replace(array('\\', ','), array('/', ' '), $service['desc']);

                                $objs[] = array(
                                    "type" => OBJECTTYPE_SERVICE,
                                    "host_name" => $hostname,
                                    "service_description" => $description,
                                    "use" => "xiwizard_ncpa_service",
                                    "check_command" => "check_xi_ncpa!" . $theseopts,
                                    "_xiwizard" => $wizard_name
                                );
                            }
                            break;

                        case "process":
                            foreach ($args as $i => $metrics) {
                                if (!array_key_exists('monitor', $metrics) || empty($metrics['desc'])) {
                                    continue;
                                }
                                $proc_name = $metrics['name'];
                                $theseopts = "{$pluginopts} -M 'processes' -q 'name={$proc_name}'";

                                if (!empty($metrics["warning"])) {
                                    $theseopts .= " -w " . escapeshellarg($metrics["warning"]);
                                }
                                if (!empty($metrics["critical"])) {
                                    $theseopts .= " -c " . escapeshellarg($metrics["critical"]);
                                }

                                $description = str_replace(array('\\', ','), array('/', ' '), $metrics['desc']);

                                $objs[] = array(
                                    "type" => OBJECTTYPE_SERVICE,
                                    "host_name" => $hostname,
                                    "service_description" => $description,
                                    "use" => "xiwizard_ncpa_service",
                                    "check_command" => "check_xi_ncpa!" . $theseopts,
                                    "_xiwizard" => $wizard_name
                                );
                            }
                            break;

                        case "plugins":
                            foreach ($args as $i => $plugin) {
                                if (!array_key_exists('monitor', $plugin) || empty($plugin['desc'])) {
                                    continue;
                                }
                                $theseopts = "{$pluginopts} -M " . escapeshellarg("plugins/".$plugin['name']);

                                if (!empty($plugin['args'])) {
                                    $theseopts .= " -q " . escapeshellarg("args=".$plugin['args']);
                                }

                                $description = str_replace(array('\\', ','), array('/', ' '), $plugin['desc']);

                                $objs[] = array(
                                    "type" => OBJECTTYPE_SERVICE,
                                    "host_name" => $hostname,
                                    "service_description" => $description,
                                    "use" => "xiwizard_ncpa_service",
                                    "check_command" => "check_xi_ncpa!" . $theseopts,
                                    "_xiwizard" => $wizard_name
                                );
                            }
                            break;

                        default:
                            break;
                    }
                }
            }

            // Return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;
            break;

        default:
            break;
    }

    return $output;
}

/**
 * Get operating system icon based on OS name
 *
 * @param   string  $os     Operating system
 * @return  string          Icon file
 */
function get_os_icon($os) {
    $icon = "linux-server.png";
    $os = strtolower($os);

    switch ($os) {
        case "aix":
            $icon = "aix.png";
            break;
        case "centos":
            $icon = "centos.png";
            break;
        case "debian":
            $icon = "debian.png";
            break;
        case "fedora":
            $icon = "fedora.png";
            break;
        case "opensuse":
            $icon = "opensuse.png";
            break;
        case "oracle":
            $icon = "oracle-linux.gif";
            break;
        case "osx":
            $icon = "osx.png";
            break;
        case "rhel":
            $icon = "redhat.png";
            break;
        case "sol":
            $icon = "solaris.png";
            break;
        case "suse":
            $icon = "suse_enterprise.png";
            break;
        case "ubuntu":
            $icon = "ubuntu.png";
            break;
    }

    return $icon;
}


/**
 * Send a request to the NCPA API and get the returned JSON as an array
 *
 * @param   string  $endpoint       API endpoint
 * @param   string  $api_url        API URL
 * @param   string  $token          Security token
 * @param   boolean $no_ssl_verify  If true, do not verify SSL certificate
 * @param   array   $params         Optional HTTP parameters passed via the URL
 * @return  array
 */
function ncpa_get_api_data($endpoint, $api_url, $token, $no_ssl_verify, $params = array()) {
    // Remove SSL verification or not
    $context = array("ssl" => array("verify_peer" => true, "verify_peer_name" => true));
    if ($no_ssl_verify) {
        $context['ssl']['verify_peer'] = false;
        $context['ssl']['verify_peer_name'] = false;
    }

    // Setup API url
    $params['token'] = $token;
    $full_api_url = $api_url."/".$endpoint."?".http_build_query($params);

    // Do connection and get data
    $data = @file_get_contents($full_api_url, false, stream_context_create($context));

    if ($data === false) {
        return array();
    }

    $data = json_decode($data, true);

    if (is_array($data) && array_key_exists('value', $data)) {
        $data = array_pop($data['value']);
    } elseif (is_array($data)) {
        $data = array_pop($data);
    }

    return $data;
}

