#!/usr/bin/php
<?php

##########################################################################################################################################
#
# check_mssql_server - A Nagios plugin to check Microsoft SQL Server
#
# Copyright (c) 2022 Nagios Enterprises
# Version 3.1.0 Copyright (c) 2022 Nagios Enterprises, LLC (Laura Gute <lgute@nagios.com>)
#
#   This plugin will check the general health of an MS SQL Server, Server.
#   It allows you to set warning and critical thresholds.
#
#   Requires:
#       pdo
#       pdo_dblib or pdo_odbc
#       simplexml
#
# License Information:
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
##########################################################################################################################################

$logger = new Logger("check_msql_server");

$overrideErrors = false;    # Used when we are running multiple tests, so all are executed.
$requestedTest;             # Keeps a record of the original test requested.  Used when we are running multiple tests.
$customArgs;                # Arguments to create custom tests.

# Default to the most common, which supports 2008+ and most of 2005.
$perfType = "default";
$TABLES = array("deprecated" => array("perf"   => "sys.sysperfinfo",                            # SQL Server 2000
                                      "memory" => "!!Not Available before SQLServer 2008!!"),
                "default"    => array("perf"   => "sys.dm_os_performance_counters",             # SQL Server 2008+
                                      "memory" => "sys.dm_os_sys_memory"),
                "azuresqldb" => array("perf"   => "sys.dm_os_performance_counters",             # Azure SQL DB
                                      "memory" => "sys.dm_db_resource_stats"),
                "sqldw"      => array("perf"   => "sys.dm_pdw_nodes_os_performance_counters",   # Azure Synapse Analytics (SQL DW)
                                      "memory" => "sys.dm_pdw_nodes_os_sys_memory"),
                "pdw"        => array("perf"   => "sys.dm_pdw_nodes_os_performance_counters",   # Parallel Data Warehouse 
                                      "memory" => "sys.dm_pdw_nodes_os_sys_memory")
            );

define("PROGRAM", "check_mssql_server.php");
define("VERSION", "3.1.0");
define("STATUS_OK", 0);
define("STATUS_WARNING", 1);
define("STATUS_CRITICAL", 2);
define("STATUS_UNKNOWN", 3);
define("DEBUG", false);

/*****************************************************************************************************************************************
 * MS SQL Server Performance Metrics
 *
 * sys.sysperfinfo/sys.dm_os_performance_counters/sys.dm_pdw_nodes_os_performance
 *
 * There are 5 types of counters (counter_type) and 4 categories.
 *
 *    Counter              (counter_type) (Category)
 *
 * 1) PERF_COUNTER_LARGE_RAWCOUNT (65792) (Point in Time) – Returns the last observed value for the counter.
 * 2) PERF_COUNTER_BULK_COUNT (272696576) (Delta) – Average # of operations completed during each second of the sample interval.
 * 3) PERF_LARGE_RAW_BASE    (1073939712) (NA)    – Base value found in the calculation of PERF_AVERAGE_BULK.
 * 4) PERF_AVERAGE_BULK      (1073874176) (Delta Ratio) and
 * 5) PERF_LARGE_RAw_FRACTION (537003264) (Ratio)
 *      - # of items processed, on average, during an operation.
 *      - These counter types display a ratio of the items processed (such as bytes sent) to the number of operations
 *        completed, and requires a base property with PERF_LARGE_RAW_BASE as the counter type.
 *
 * Categories       Calculation
 * =============    ====================================================================================================================
 * Point in Time    None.  Many of these Counters use instance_name, which may be the name of a database, or something else.
 *
 * Delta            The formula for the current metric value is (A2-A1)/(T2-T1) 
 *
 *                  Where A1 and A2 are the values of the monitored PERF_COUNTER_BULK_COUNT counter, taken at sample times T1 and T2
 *                  T1 and T2 are the times when the sample values are taken.
 *
 *                  Page lookups/sec = (854,521 – 852,433)/(621,366,686-621,303,043) 
 *                                   = 2,088 / 63,643 ms 
 *                                   = 2,088/63 sec = 32.1 /sec
 *
 * Ratio            The formula is A1/B1 * 100.  Where A1 is type 537003264 and B1 is type 1073939712.
 * 
 *                  Buffer Cache Hit Ratio % = 100 * Buffer Cache Hit Ratio / Buffer Cache Hit Ratio Base
 *                                          = 100 * 2,135 / 3,573
 *                                          = 59.75%
 * 
 * Delta Ratio      The formula is (A2 – A1)/(B2 – B1) / Interval - where A1 and A2 are type 1073874176 and B1 & B2 are type 1073939712,
 *                  A1 & B1, A2 & B2 are collected at the same time and Interval is the time between the two.
 *
 *                  Average Wait Time (ms) for the interval between these two measurements is:
 *                      = (A2 – A1)/(B2 – B1) / Interval
 *                      = (53736 ms -52939 ms)/(23-18) = 797 ms / 5 = 159.4 ms
 *
 * NOTE: Whatever user is used to connect to the SQL Server, needs to have GRANT VIEW SERVER STATE, at least, in order to access table(s).
 *
 *****************************************************************************************************************************************/
/*
 * NOTE:  sys.sysperfinfo is from SQL SERVER 2000 and is included for backward compatibility.
 *        sys.sysperfinfo is deprecated and, according to Microsoft, will be removed in a future version.
 *
 * The default is sys.sysperfinfo.  You can specify which version you want to use (see TABLES)
 */
# Same as BASE_QUERY, as long as instance_name is provided in the MODE entry.
#$INST_QUERY  = "SELECT cntr_value as value, DATEDIFF(SECOND, '1970-01-01', GETUTCDATE()) as utctimestamp ".
#               "FROM @table WHERE counter_name='@counter_name' AND instance_name='@instance_name';";
# Same as BASE_QUERY, as long as instance_name is provided in the MODE entry, as ''.
#$OBJE_QUERY  = "SELECT cntr_value as value, DATEDIFF(SECOND, '1970-01-01', GETUTCDATE()) as utctimestamp ".
#               "FROM @table WHERE counter_name='@counter_name';";
$BASE_QUERY = "SELECT cntr_value as value, DATEDIFF(SECOND, '1970-01-01', GETUTCDATE()) as utctimestamp ".
              "FROM @table WHERE counter_name='@counter_name' AND instance_name='@instance_name';";
$RATIO_QUERY = "SELECT cntr_value as value, DATEDIFF(SECOND, '1970-01-01', GETUTCDATE()) as utctimestamp ".
               "FROM @table WHERE counter_name LIKE '@counter_name%' AND instance_name='@instance_name';";
$CON_QUERY   = "SELECT count(*) as value, DATEDIFF(SECOND, '1970-01-01', GETUTCDATE()) as utctimestamp  FROM sys.sysprocesses;";

$MEM_QUERY;
$DEFAULT_MEM_QUERY  = "SELECT 100*(1.0-(available_physical_memory_kb/(total_physical_memory_kb*1.0))) as value, ".
                      "       DATEDIFF(SECOND, '1970-01-01', GETUTCDATE()) as utctimestamp  FROM @table;";
$AZURE_MEM_QUERY    = "SELECT MAX(avg_memory_usage_percent) AS 'value', ".
                      "       DATEDIFF(SECOND, '1970-01-01', GETUTCDATE()) as utctimestamp  FROM (SELECT TOP 4 * FROM @table) t;";

# Azure SQL DB Memory - FUTURE - Check if this applies to PDW and Azure Synapse Analytics
# https://docs.microsoft.com/en-us/azure/azure-sql/database/monitoring-with-dmvs / NewRelic.sql (Average or Max of the records for the last minute.)
#SELECT
#    AVG(avg_cpu_percent) AS [AvgCpuPercent], 
#    MAX(avg_cpu_percent) AS [MaxCpuPercent],
#    AVG(avg_data_io_percent) AS [AvgDataIoPercent], 
#    MAX(avg_data_io_percent) AS [MaxDataIoPercent],
#    AVG(avg_log_write_percent) AS [AvgLogWritePercent], 
#    MAX(avg_log_write_percent) AS [MaxLogWritePercent],
#    AVG(avg_memory_usage_percent) AS [AvgMemoryUsagePercent], 
#    MAX(avg_memory_usage_percent) AS [MaxMemoryUsagePercent]
#FROM (SELECT TOP 4 * FROM sys.dm_db_resource_stats) t;

# Gets the cntr_type of the requested counter_name, so we know which custom query and calculations are needed.
$CALC_QUERY = "SELECT cntr_type FROM @table WHERE counter_name='@counter_name' AND instance_name='@instance_name';";

# We only look at the first record, so why return all 256???
# Now returns the most recent record.
#
# PHP 5.3 pdo dblib truncates xml.  CAST to TEXT fixes the issue.  If it is still getting truncated, try setting the textsize.
#     $connection->query("SET TEXTSIZE 2147483647;");

$RING_QUERY  = "SELECT TOP 1 [timestamp], CAST(record as TEXT) as [value], DATEDIFF(SECOND, '1970-01-01', GETUTCDATE()) as [utctimestamp] ".
               "FROM sys.dm_os_ring_buffers WITH ( NOLOCK ) ".
               "WHERE ring_buffer_type=N'RING_BUFFER_SCHEDULER_MONITOR' ".
#               "  AND record LIKE N'%<SystemHealth>%' ".    # Not necessary. FYI: this table is not supported by MS.
               "ORDER BY [timestamp] DESC;";

# pdo_dblib is having issues with the CONVERT(XML, record) portion of the original query.  We now simply get the string of XML
# and process it in PHP.
#
#             "SET ANSI_PADDING ON; ".
#             "SELECT record.value('(./Record/SchedulerMonitorEvent/SystemHealth/ProcessUtilization)[1]', 'int') AS [value], ".
#             "       DATEDIFF(SECOND, '1970-01-01', GETUTCDATE()) as [utctimestamp] ".
#             "FROM (SELECT [timestamp], CONVERT(XML, record) AS [record] ".
#                   "FROM sys.dm_os_ring_buffers WITH ( NOLOCK ) ".
#                   "WHERE ring_buffer_type=N'RING_BUFFER_SCHEDULER_MONITOR' ".
#                   "AND record LIKE N'%<SystemHealth>%' ".
#                   "ORDER BY [timestamp] DESC) as x;";

$MODES;

# NOTE: 'help' not used???
$DATABASE_MODES = array(
        'activetrans' => array(
            'help' => 'Active Transactions',
            'stdout' => 'Active Transactions is @result',
            'label' => 'active_trans',
            'query' => $BASE_QUERY,
            'counter_name' => 'Active Transactions',
            'instance_name' => 'master',    # May be overriden on command line by --instancename.
            'type' => 'standard',
        ),
        'datasize' => array(
            'help' => 'Database Size',
            'stdout' => 'Database size is @result KB',
            'label' => 'data_file_size',
            'query' => $BASE_QUERY,
            'counter_name' => 'Data File(s) Size (KB)',
            'instance_name' => 'master',
            'type' => 'standard',
            'unit' => 'KB',
        ),
        'logbytesflushed' => array(
            'help' => 'Log Bytes Flushed Per Second',
            'stdout' => 'Log Bytes Flushed/sec is @result/sec',
            'label' => 'log_bytes_flushed',
            'query' => $BASE_QUERY,
            'counter_name' => 'Log Bytes Flushed/sec',
            'instance_name' => 'master',
            'type' => 'delta'
        ),
        'logcachehit' => array(
            'help' => 'Log Cache Hit Ratio',
            'stdout' => 'Log Cache Hit Ratio is @result%',
            'label' => 'log_cache_hit_ratio',
            'query' => $RATIO_QUERY,
            'counter_name' => 'Log Cache Hit Ratio',
            'instance_name' => 'master',    # May be overriden on command line by --instancename.
            'type' => 'ratio',
            'modifier' => 100,
            'unit' => '%',
        ),
        'logfilessize' => array(
            'help' => 'Log File(s) Size (KB)',
            'stdout' => 'Log File(s) Size is @result KB',
            'label' => 'log_files_size',
            'query' => $BASE_QUERY,
            'counter_name' => 'Log File(s) Size (KB)',
            'instance_name' => 'master',
            'type' => 'standard',
            'unit' => 'KB',
        ),
        'logfilesused' => array(
            'help' => 'Log File(s) Used Size (KB)',
            'stdout' => 'Log File(s) Used Size @result KB',
            'label' => 'log_files_used_size',
            'query' => $BASE_QUERY,
            'counter_name' => 'Log File(s) Used Size (KB)',
            'instance_name' => 'master',
            'type' => 'standard',
            'unit' => 'KB',
        ),
        'logfileusage' => array(
            'help' => 'Log File Usage',
            'stdout' => 'Log File Usage is @result%',
            'label' => 'log_file_usage',
            'query' => $BASE_QUERY,
            'counter_name' => 'Percent Log Used',
            'instance_name' => 'master',
            'type' => 'standard',
            'unit' => '%',
        ),
        'logflushes' => array(
            'help' => 'Log Flushes Per Second',
            'stdout' => 'Log Flushes/sec is @result/sec',
            'label' => 'log_flushes_per_sec',
            'query' => $BASE_QUERY,
            'counter_name' => 'Log Flushes/sec',
            'instance_name' => 'master',
            'type' => 'delta'
        ),
        'logflushwaits' => array(
            'help' => 'Log Flush Waits Per Second',
            'stdout' => 'Log Flush Waits/sec is @result/sec',
            'label' => 'log_flush_waits',
            'query' => $BASE_QUERY,
            'counter_name' => 'Log Flush Waits/sec',
            'instance_name' => 'master',
            'type' => 'delta'
        ),
        'loggrowths' => array(
            'help' => 'Log Growths',
            'stdout' => 'Log Growths is @result',
            'label' => 'log_growths',
            'query' => $BASE_QUERY,
            'counter_name' => 'Log Growths',
            'instance_name' => 'master',
            'type' => 'standard'
        ),
        'logshrinks' => array(
            'help' => 'Log Shrinks',
            'stdout' => 'Log Shrinks is @result',
            'label' => 'log_shrinks',
            'query' => $BASE_QUERY,
            'counter_name' => 'Log Shrinks',
            'instance_name' => 'master',
            'type' => 'standard'
        ),
        'logtruncs' => array(
            'help' => 'Log Truncations',
            'stdout' => 'Log Truncations is @result',
            'label' => 'log_truncations',
            'query' => $BASE_QUERY,
            'counter_name' => 'Log Truncations',
            'instance_name' => 'master',
            'type' => 'standard'
        ),
        'logwait' => array(
            'help' => 'Log Flush Wait Time',
            'stdout' => 'Log Flush Wait Time is @result ms',
            'label' => 'log_wait_time',
            'query' => $BASE_QUERY,
            'counter_name' => 'Log Flush Wait Time',
            'instance_name' => 'master',
            'type' => 'standard',
            'unit' => 'ms',
        ),
        'transpsec' => array(
            'help' => 'Transactions Per Second',
            'stdout' => 'Transactions/sec is @result/sec',
            'label' => 'transactions_per_sec',
            'query' => $BASE_QUERY,
            'counter_name' => 'Transactions/sec',
            'instance_name' => 'master',
            'type' => 'delta'
        ),

        'custom' => array(
            'help' => 'Run a custom test, against the database.'
        ),

        'time2connect' => array(
            'help' => 'Time to connect to the database.',
            'stdout' => 'Time to connect was @result s',
            'label' => 'time',
            'unit' => 's',
        ),
        'test' => array(
            'help' => 'Run tests of all queries against the database.'
        ),
        'runall' => array(
            'help' => 'Run tests of all queries against the database, with status and performance data.'
        )
);

$SERVER_MODES = array(
        'autoparamattempts' => array(
            'help' => 'Auto-Param Attempts/sec',
            'stdout' => 'Auto-Param Attempts/sec is @result/sec',
            'label' => 'autoparam_attempts',
            'query' => $BASE_QUERY,
            'counter_name' => 'Auto-Param Attempts/sec',
            'instance_name' => '',
            'type' => 'delta',
        ),
        'avglatchwait' => array(
            'help' => 'Average Latch Wait Time (ms)',
            'stdout' => 'Average Latch Wait Time is @result ms',
            'label' => 'averagewait',
            'query' => $RATIO_QUERY,
            'counter_name' => 'Average Latch Wait Time', # (ms)
            'instance_name' => '',
            'type' => 'deltaratio',
            'unit' => 'ms',
        ),
        'averagewait' => array(
            'help' => 'Average Wait Time (ms)',
            'stdout' => 'Average Wait Time is @result ms',
            'label' => 'averagewait',
            'query' => $RATIO_QUERY,
            'counter_name' => 'Average Wait Time',
            'instance_name' => '_Total',
            'type' => 'deltaratio',
            'unit' => 'ms',
        ),
        'batchreq' => array(
            'help' => 'Batch Requests/Sec',
            'stdout' => 'Batch Requests/Sec is @result/sec',
            'label' => 'batch_requests',
            'query' => $BASE_QUERY,
            'counter_name' => 'Batch Requests/sec',
            'instance_name' => '',
            'type' => 'delta',
        ),
        'bufferhitratio' => array(
            'help' => 'Buffer Cache Hit Ratio',
            'stdout' => 'Buffer Cache Hit Ratio is @result%',
            'label' => 'buffer_cache_hit_ratio',
            'query' => $RATIO_QUERY,
            'counter_name' => 'Buffer cache hit ratio',
            'type' => 'ratio',
            'modifier' => 100,
            'unit' => '%',
        ),
        'cachehit' => array(
            'help' => 'Cache Hit Ratio',
            'stdout' => 'Cache Hit Ratio is @result%',
            'label' => 'cache_hit_ratio',
            'query' => $RATIO_QUERY,
            'counter_name' => 'Cache Hit Ratio',
            'instance_name' => '_Total',
            'type' => 'ratio',
            'modifier' => 100,
            'unit' => '%',
        ),
        'cacheobjcounts' => array(
            'help' => 'Cache Object Counts',
            'stdout' => 'Cache Object Counts is @result',
            'label' => 'cache_obj_counts',
            'query' => $BASE_QUERY,
            'counter_name' => 'Cache Object Counts',
            'instance_name' => '_Total',
            'type' => 'standard',
        ),
        'cacheobjsinuse' => array(
            'help' => 'Cache Objects in use',
            'stdout' => 'Cache Objects in use is @result',
            'label' => 'cache_objs_inuse',
            'query' => $BASE_QUERY,
            'counter_name' => 'Cache Objects in use',
            'instance_name' => '_Total',
            'type' => 'standard',
        ),
        'cachepages' => array(
            'help' => 'Cache Pages',
            'stdout' => 'Cache Pages is @result',
            'label' => 'cache_pages',
            'query' => $BASE_QUERY,
            'counter_name' => 'Cache Pages',
            'instance_name' => '_Total',
            'type' => 'standard',
        ),
        'checkpoints' => array(
            'help' => 'Checkpoint Pages/sec',
            'stdout' => 'Checkpoint Pages/sec is @result/sec',
            'label' => 'checkpoint_pages',
            'query' => $BASE_QUERY,
            'counter_name' => 'Checkpoint pages/Sec',
            'instance_name' => '',
            'type' => 'delta'
        ),
        'connectionreset' => array(
            'help' => 'Connection reset/sec',
            'stdout' => 'Connection reset/sec is @result/sec',
            'label' => 'connection_reset',
            'query' => $BASE_QUERY,
            'counter_name' => 'Connection reset/sec',
            'instance_name' => '',
            'type' => 'delta'
        ),
        'connections' => array(
            'help' => 'Number of open connections',
            'stdout' => 'Number of open connections is @result',
            'label' => 'connections',
            'type' => 'standard',
            'query' => $CON_QUERY,
        ),
        'cpu' => array(
            'help' => 'Server CPU utilization',
            'stdout' => 'Current CPU utilization is @result%',
            'label' => 'cpu',
            'type' => 'xml',
            'xpath'  => '/Record/SchedulerMonitorEvent/SystemHealth/ProcessUtilization[1]',
            'xpath2' => '/Record/SchedluerMonitorEvent/SystemHealth/ProcessUtilization[1]',
            'ring_buffer_type' => "N'RING_BUFFER_SCHEDULER_MONITOR'",
            'query' => $RING_QUERY,
            'unit' => '%',
        ),
        'databasepages' => array(
            'help' => 'Database Pages',
            'stdout' => 'Database pages are @result',
            'label' => 'database_pages',
            'type' => 'standard',
            'query' => $BASE_QUERY,
            'counter_name' => 'Database pages',
            'instance_name' => '',
        ),
        'deadlocks' => array(
            'help' => 'Deadlocks/sec',
            'stdout' => 'Deadlocks/sec is @result/sec',
            'label' => 'deadlocks',
            'query' => $BASE_QUERY,
            'counter_name' => 'Number of Deadlocks/sec',
            'instance_name' => '_Total',
            'type' => 'delta',
        ),
        'failedautoparams' => array(
            'help' => 'Failed Auto-Params/sec',
            'stdout' => 'Failed Auto-Params/sec is @result/sec',
            'label' => 'failed_autoparams',
            'query' => $BASE_QUERY,
            'counter_name' => 'Failed Auto-Params/sec',
            'instance_name' => '',
            'type' => 'delta'
        ),
        'forwardedrecords' => array(
            'help' => 'Forwarded Records/sec',
            'stdout' => 'Forwarded Records/sec is @result/sec',
            'label' => 'forwarded_records',
            'query' => $BASE_QUERY,
            'counter_name' => 'Forwarded Records/sec',
            'instance_name' => '',
            'type' => 'delta'
        ),
        'freeliststalls' => array(
            'help' => 'Free List Stalls/sec',
            'stdout' => 'Free List Stalls/sec is @result/sec',
            'label' => 'free_list_stalls',
            'query' => $BASE_QUERY,
            'counter_name' => 'Free List Stalls/sec',
            'instance_name' => '',
            'type' => 'delta'
        ),
        'freepages' => array(
            'help' => 'Free Pages (Cumulative)',
            'stdout' => 'Free pages is @result',
            'label' => 'free_pages',
            'type' => 'standard',
            'query' => $BASE_QUERY,
            'counter_name' => 'Free pages',
            'instance_name' => '',
        ),
        'fullscans' => array(
            'help' => 'Full Scans/sec',
            'stdout' => 'Full Scans/sec is @result/sec',
            'label' => 'full_scans',
            'query' => $BASE_QUERY,
            'counter_name' => 'Full Scans/sec',
            'instance_name' => '',
            'type' => 'delta',
        ),
        'grantedwsmem' => array(
            'help' => 'Granted Workspace Memory (KB)',
            'stdout' => 'Granted Workspace Memory is @result KB',
            'label' => 'granted_ws_mem',
            'query' => $BASE_QUERY,
            'counter_name' => 'Granted Workspace Memory (KB)',
            'instance_name' => '',
            'type' => 'standard',
            'unit' => 'KB',
        ),
        'indexsearches' => array(
            'help' => 'Index Searches/sec',
            'stdout' => 'Index Searches/sec is @result/sec',
            'label' => 'index_searches',
            'query' => $BASE_QUERY,
            'counter_name' => 'Index Searches/sec',
            'instance_name' => '',
            'type' => 'delta'
        ),
        'latchwaits' => array(
            'help' => 'Latch Waits/sec',
            'stdout' => 'Latch Waits/sec is @result/sec',
            'label' => 'latch_waits',
            'query' => $BASE_QUERY,
            'counter_name' => 'Latch Waits/sec',
            'instance_name' => '',
            'type' => 'delta'
        ),
        'lazywrites' => array(
            'help' => 'Lazy Writes/sec',
            'stdout' => 'Lazy Writes/sec is @result/sec',
            'label' => 'lazy_writes',
            'query' => $BASE_QUERY,
            'counter_name' => 'Lazy writes/sec',
            'instance_name' => '',
            'type' => 'delta'
        ),
        'lockrequests' => array(
            'help' => 'Lock Requests/sec',
            'stdout' => 'Lock Requests/sec is @result/sec',
            'label' => 'lock_requests',
            'query' => $BASE_QUERY,
            'counter_name' => 'Lock requests/sec',
            'instance_name' => '_Total',
            'type' => 'delta',
        ),
        'locktimeouts' => array(
            'help' => 'Lock Timeouts/sec',
            'stdout' => 'Lock Timeouts/sec is @result/sec',
            'label' => 'lock_timeouts',
            'query' => $BASE_QUERY,
            'counter_name' => 'Lock timeouts/sec',
            'instance_name' => '_Total',
            'type' => 'delta',
        ),
        'lockwait' => array(
            'help' => 'Lock Wait Time (ms)',
            'stdout' => 'Lock Wait Time is @result ms',
            'label' => 'lockwait',
            'query' => $BASE_QUERY,
            'counter_name' => 'Lock Wait Time (ms)',
            'instance_name' => '_Total',
            #'type' => 'standard',   # This counter_name's cntr_type specifies delta.
            'type' => 'delta',
            'unit' => 'ms',
        ),
        'lockwaits' => array(
            'help' => 'Lockwaits/sec',
            'stdout' => 'Lockwaits/sec is @result/sec',
            'label' => 'lockwaits',
            'query' => $BASE_QUERY,
            'counter_name' => 'Lock Waits/sec',
            'instance_name' => '_Total',
            'type' => 'delta',
        ),
        'logins' => array(
            'help' => 'Logins/sec',
            'stdout' => 'Logins/sec is @result/sec',
            'label' => 'logins',
            'query' => $BASE_QUERY,
            'counter_name' => 'Logins/sec',
            'instance_name' => '',
            'type' => 'delta'
        ),
        'logouts' => array(
            'help' => 'Logouts/sec',
            'stdout' => 'Logouts/sec is @result/sec',
            'label' => 'logouts',
            'query' => $BASE_QUERY,
            'counter_name' => 'Logouts/sec',
            'instance_name' => '',
            'type' => 'delta'
        ),
        'longesttrans' => array(
            'help' => 'Longest Transaction Running Time',
            'stdout' => 'Longest Transaction Running Time is @result s',
            'label' => 'longest_trans',
            'query' => $BASE_QUERY,
            'counter_name' => 'Longest Transaction Running Time',
            'instance_name' => '',
            'type' => 'standard',
            'unit' => 's',
        ),
        'maxwsmem' => array(
            'help' => 'Maximum Workspace Memory (KB)',
            'stdout' => 'Maximum Workspace Memory is @result KB',
            'label' => 'max_ws_mem',
            'query' => $BASE_QUERY,
            'counter_name' => 'Maximum Workspace Memory (KB)',
            'instance_name' => '',
            'type' => 'standard',
            'unit' => 'KB',
        ),
        'memgrantsoutstand' => array(
            'help' => 'Memory Grants Outstanding',
            'stdout' => 'Memory Grants Outstanding is @result',
            'label' => 'mem_grants_outstand',
            'query' => $BASE_QUERY,
            'counter_name' => 'Memory Grants Outstanding',
            'instance_name' => '',
            'type' => 'standard'
        ),
        'memgrantspend' => array(
            'help' => 'Memory Grants Pending',
            'stdout' => 'Memory Grants Pending is @result',
            'label' => 'mem_grants_pend',
            'query' => $BASE_QUERY,
            'counter_name' => 'Memory Grants Pending',
            'instance_name' => '',
            'type' => 'standard'
        ),
        'memory' => array(
            'help' => 'Used server memory',
            'stdout' => 'Server using @result% of memory',
            'label' => 'memory',
            'type' => 'standard',
        ),
        'numsuperlatches' => array(
            'help' => 'Number of SuperLatches',
            'stdout' => 'Number of SuperLatches is @result',
            'label' => 'num_superlatches',
            'query' => $BASE_QUERY,
            'counter_name' => 'Number of SuperLatches',
            'instance_name' => '',
            'type' => 'standard'
        ),
        'pagelife' => array(
            'help' => 'Page Life Expectancy',
            'stdout' => 'Page Life Expectancy is @result s',
            'label' => 'page_life_expectancy',
            'query' => $BASE_QUERY,
            'counter_name' => 'Page life expectancy',
            'instance_name' => '',
            'type' => 'standard',
            'unit' => 's'
        ),
        'pagelooks' => array(
            'help' => 'Page Lookups/sec',
            'stdout' => 'Page Lookups/sec is @result/sec',
            'label' => 'page_lookups',
            'query' => $BASE_QUERY,
            'counter_name' => 'Page lookups/sec',
            'instance_name' => '',
            'type' => 'delta'
        ),
        'pagereads' => array(
            'help' => 'Page Reads/sec',
            'stdout' => 'Page Reads/sec is @result/sec',
            'label' => 'page_reads',
            'query' => $BASE_QUERY,
            'counter_name' => 'Page reads/sec',
            'instance_name' => '',
            'type' => 'delta'
        ),
        'pagesplits' => array(
            'help' => 'Page Splits/sec',
            'stdout' => 'Page Splits/sec is @result/sec',
            'label' => 'page_splits',
            'query' => $BASE_QUERY,
            'counter_name' => 'Page Splits/sec',
            'instance_name' => '',
            'type' => 'delta',
        ),
        'pagewrites' => array(
            'help' => 'Page Writes/sec',
            'stdout' => 'Page Writes/sec is @result/sec',
            'label' => 'page_writes',
            'query' => $BASE_QUERY,
            'counter_name' => 'Page writes/sec',
            'instance_name' => '',
            'type' => 'delta',
        ),
        'processesblocked' => array(
            'help' => 'Processes blocked',
            'stdout' => 'Processes blocked is @result KB',
            'label' => 'processes_blocked',
            'query' => $BASE_QUERY,
            'counter_name' => 'Processes blocked',
            'instance_name' => '',
            'type' => 'standard'
        ),
        'readahead' => array(
            'help' => 'Readahead Pages/sec',
            'stdout' => 'Readahead Pages/sec is @result/sec',
            'label' => 'readaheads',
            'query' => $BASE_QUERY,
            'counter_name' => 'Readahead pages/sec',
            'instance_name' => '',
            'type' => 'delta',
        ),
        'safeautoparams' => array(
            'help' => 'Safe Auto-Params/sec',
            'stdout' => 'Safe Auto-Params/sec is @result/sec',
            'label' => 'safe_auto_params',
            'query' => $BASE_QUERY,
            'counter_name' => 'Safe Auto-Params/sec',
            'instance_name' => '',
            'type' => 'delta'
        ),
        'sqlattentionrate' => array(
            'help' => 'SQL Attention rate',
            'stdout' => 'SQL Attention rate is @result',
            'label' => 'sql_attention_rate',
            'query' => $BASE_QUERY,
            'counter_name' => 'SQL Attention rate',
            'instance_name' => '',
            'type' => 'delta'
        ),
        'sqlcompilations' => array(
            'help' => 'SQL Compilations/sec',
            'stdout' => 'SQL Compilations/sec is @result/sec',
            'label' => 'sql_compilations',
            'query' => $BASE_QUERY,
            'counter_name' => 'SQL Compilations/sec',
            'instance_name' => '',
            'type' => 'delta',
        ),
        'sqlrecompilations' => array(
            'help' => 'SQL Re-Compilations/sec',
            'stdout' => 'SQL Re-Compilations/sec is @result/sec',
            'label' => 'sql_recompilations',
            'query' => $BASE_QUERY,
            'counter_name' => 'SQL Re-Compilations/sec',
            'instance_name' => '',
            'type' => 'delta'
        ),
        'stolenpages' => array(
            'help' => 'Stolen Pages',
            'stdout' => 'Stolen pages are @result',
            'label' => 'stolen_pages',
            'type' => 'standard',
            'query' => $BASE_QUERY,
            'counter_name' => 'Stolen pages',
            'instance_name' => '',
        ),
        'superlatchdemotes' => array(
            'help' => 'SuperLatch Demotions/sec',
            'stdout' => 'SuperLatch Demotions/sec is @result/sec',
            'label' => 'superlatch_demotions',
            'query' => $BASE_QUERY,
            'counter_name' => 'SuperLatch Demotions/sec',
            'instance_name' => '',
            'type' => 'delta'
        ),
        'superlatchpromotes' => array(
            'help' => 'SuperLatch Promotions/sec',
            'stdout' => 'SuperLatch Promotions/sec is @result/sec',
            'label' => 'superlatch_promotions',
            'query' => $BASE_QUERY,
            'counter_name' => 'SuperLatch Promotions/sec',
            'instance_name' => '',
            'type' => 'delta'
        ),
        'tablelockescalate' => array(
            'help' => 'Table Lock Escalations/sec',
            'stdout' => 'Table Lock Escalations/sec is @result/sec',
            'label' => 'table_lock_escalations',
            'query' => $BASE_QUERY,
            'counter_name' => 'Table Lock Escalations/sec',
            'instance_name' => '',
            'type' => 'delta'
        ),
        'targetpages' => array(
            'help' => 'Target Pages',
            'stdout' => 'Target pages are @result',
            'label' => 'target_pages',
            'type' => 'standard',
            'query' => $BASE_QUERY,
            'counter_name' => 'Target pages',
            'instance_name' => '',
        ),
        'targetsrvmem' => array(
            'help' => 'Target Server Memory (KB)',
            'stdout' => 'Target Server Memory is @result KB',
            'label' => 'target_srv_mem',
            'query' => $BASE_QUERY,
            'counter_name' => 'Target Server Memory (KB)',
            'instance_name' => '',
            'type' => 'standard',
            'unit' => 'KB',
        ),
        'totallatchwait' => array(
            'help' => 'Total Latch Wait Time (ms)',
            'stdout' => 'Total Latch Wait Time is @result ms',
            'label' => 'total_latch_wait',
            'query' => $BASE_QUERY,
            'counter_name' => 'Total Latch Wait Time (ms)',
            'instance_name' => '',
            'type' => 'delta',
            'unit' => 'ms'
        ),
        'totalpages' => array(
            'help' => 'Total Pages (Cumulative)',
            'stdout' => 'Total pages is @result',
            'label' => 'totalpages',
            'type' => 'standard',
            'query' => $BASE_QUERY,
            'counter_name' => 'Total pages',
            'instance_name' => '',
        ),
        'totalsrvmem' => array(
            'help' => 'Total Server Memory (KB)',
            'stdout' => 'Total Server Memory is @result KB',
            'label' => 'total_srv_mem',
            'query' => $BASE_QUERY,
            'counter_name' => 'Total Server Memory (KB)',
            'instance_name' => '',
            'type' => 'standard',
            'unit' => 'KB',
        ),
        'unsafeautoparams' => array(
            'help' => 'Unsafe Auto-Params/sec',
            'stdout' => 'Unsafe Auto-Params/sec is @result/sec',
            'label' => 'unsafe_autoparams',
            'query' => $BASE_QUERY,
            'counter_name' => 'Unsafe Auto-Params/sec',
            'instance_name' => '',
            'type' => 'delta'
        ),
        'usercons' => array(
            'help' => 'User Connections',
            'stdout' => 'User Connections is @result',
            'label' => 'user_connections',
            'query' => $BASE_QUERY,
            'counter_name' => 'User Connections',
            'instance_name' => '',
            'type' => 'standard',
        ),
        'workfilescreated' => array(
            'help' => 'Workfiles Created/sec',
            'stdout' => 'Workfiles Created/sec is @result/sec',
            'label' => 'workfiles_created',
            'query' => $BASE_QUERY,
            'counter_name' => 'Workfiles Created/sec',
            'instance_name' => '',
            'type' => 'delta'
        ),
        'worktablescacheratio' => array(
            'help' => 'Worktables From Cache Ratio',
            'stdout' => 'Worktables From Cache Ratio is @result%',
            'label' => 'worktables_cache_ratio',
            'query' => $RATIO_QUERY,
            'counter_name' => 'Worktables From Cache',  # Ratio
            'instance_name' => '',
            'type' => 'ratio',
            'unit' => '%',
            'modifier' => 100,
        ),
        'worktablescreated' => array(
            'help' => 'Worktables Created/sec',
            'stdout' => 'Worktables Created/sec is @result/sec',
            'label' => 'worktables_created',
            'query' => $BASE_QUERY,
            'counter_name' => 'Worktables Created/sec',
            'instance_name' => '',
            'type' => 'delta'
        ),

        # General tests and debugging tests.
        'time2connect' => array(
            'help' => 'Time to connect to the database.',
            'stdout' => 'Time to connect was @result s',
            'label' => 'time',
            'unit' => 's',
        ),
        'custom' => array(
            'help' => 'Run a custom test, against the database.'
        ),
        'test' => array(
            'help' => 'Run tests of all queries against the database.'
        ),
        'runall' => array(
            'help' => 'Run tests of all queries against the database, with status and performance data.'
        )
);

function parse_args() {
    global $logger;
    $logger->debug("", __METHOD__, __LINE__);

    $argSpecs = array(
        // Required arguments
        // Individual specifies a flag, rather than a key/value pair.
        array('short' => '',
            'long' => 'checktype',
            'default' => '',
            'required' => true,
            'individual' => false,
            'help' => 'Specify database or server checks'),
        array('short' => 'H',
            'long' => 'hostname',
            'default' => '',
            'required' => true,
            'individual' => false,
            'help' => 'Specify MS SQL Server Address'),
        array('short' => 'U',
            'long' => 'username',
            'default' => '',
            'required' => true,
            'individual' => false,
            'help' => 'Specify User\'s Name'),
        array('short' => 'P',
            'long' => 'password',
            'default' => '',
            'required' => true,
            'individual' => false,
            'help' => 'Specify User\'s Password'),
        // This is used for instance_name, in the perf info table, not to set db, in the connection.
        // Required for Database checks, not required for Server checks and the default is "master".
        array('short' => 'i',
            'long' => 'instancename',
            'default' => 'master',
            'required' => false,
            'individual' => false,
            'help' => 'Specify the database to check'),
        // Optional Connection Information
        array('short' => 'I',
            'long' => 'instance',
            'default' => '',
            'required' => false,
            'individual' => false,
            'help' => 'Specify Instance'),
        array('short' => 'p',
            'long' => 'port',
            'default' => '1433',
            'required' => false,
            'individual' => false,
            'help' => 'Specify Port'),
        // Optional Nagios Plugin Information
        array('short' => 'w',
            'long' => 'warning',
            'default' => '',
            'required' => true,
            'individual' => false,
            'help' => 'Specify warning range'),
        array('short' => 'c',
            'long' => 'critical',
            'default' => '',
            'required' => true,
            'individual' => false,
            'help' => 'Specify critical range'),
        // Help
        array('short' => 'h',
            'long' => 'help',
            'default' => '',
            'required' => false,
            'individual' => true,
            'help' => 'This usage message'),
        array('short' => 'V',
            'long' => 'version',
            'default' => '',
            'required' => false,
            'individual' => true,
            'help' => 'The version of this plugin'),
        array('short' => 'v',
            'long' => 'verbose',
            'default' => '',
            'required' => false,
            'individual' => true,
            'help' => 'Increase verbosity up to 3 times, e.g., -vvv'),
        array('short' => '',
            'long' => 'complex',
            'default' => '',
            'required' => false,
            'individual' => true,
            'help' => 'Show traditional/complex log file entries, for use with verbose.'),
        array('short' => '',
            'long' => 'perftype',
            'default' => 'default',
            'required' => false,
            'individual' => false,
            'help' => 'Specify the performance type of the server. Default is sys.sysperfinfo.'),
        array('short' => '',
            'long' => 'tdsversion',
            'default' => '7.4',
            'required' => false,
            'individual' => false,
            'help' => 'Specify the TDS version to use when connecting to the MS SQL server. e.g., 7.0. Default is 7.4.'),

        // Test to run
        array('short' => '',
            'long' => 'mode',
            'default' => 'False',
            'required' => true,
            'individual' => false,
            'help' => 'Must choose one and only one Test'),
        // Arguments for Custom Tests
        array('short' => '',
            'long' => 'custom',
            'default' => 'False',
            'required' => false,
            'individual' => false,
            'help' => 'Provide a JSON string of the custom test arguments.  \'("counter_name":"Page life expectancy",instance_name:"","unit":"","modifier":"","ring_buffer_type":"","xpath":"")\''),
    );
    
    $options = parse_specs($argSpecs);

    return $options;
}

function parse_specs($argSpecs) {
    global $MODES;
    global $DATABASE_MODES;
    global $SERVER_MODES;
    global $MEM_QUERY;
    global $DEFAULT_MEM_QUERY;
    global $AZURE_MEM_QUERY;
    global $TABLES;
    global $USAGE_STRING1;
    global $VERSION;
    global $perfType;
    global $logger;
    global $customArgs;
    global $requestedTest;
    $logger->debug("", __METHOD__, __LINE__);
    
    $shortOptions = '';
    $longOptions = array();
    $options = array();
    
    /**************************************************************************************
     * Create the array that will be passed to getopt, which accepts an array of arrays.
     * Each internal array has three entries, short option, long option and required.
     **************************************************************************************/
    foreach($argSpecs as $argSpec) {

        if (!empty($argSpec['short'])) {
            # getopt() - Optional ::, isn't working, so just make everything, that isn't a flag, required, for getopt().
            #$shortOptions .= "{$argSpec['short']}".(!empty($argSpec['required']) ? ':' : (empty($argSpec['individual']) ? '::' : ''));
            $shortOptions .= "{$argSpec['short']}".(!empty($argSpec['required']) ? ':' : (empty($argSpec['individual']) ? ':' : ''));
        }

        if (!empty($argSpec['long'])) {
            # getopt() - Optional ::, isn't working, so just make everything, that isn't a flag, required, for getopt().
            #$longOptions[] = "{$argSpec['long']}".(!empty($argSpec['required']) ? ':' : (empty($argSpec['individual']) ? '::' : ''));
            $longOptions[] = "{$argSpec['long']}".(!empty($argSpec['required']) ? ':' : (empty($argSpec['individual']) ? ':' : ''));
        }
    }

    /**************************************************************************************
     * Parse the command line args, with the builtin getopt function
     **************************************************************************************/
    $parsedArgs = getopt($shortOptions, $longOptions);

    /**************************************************************************************
     * This version of getopt() stops when it hits arguments that are not in the short
     * and/or long options.  Which means, if there is a "bad/unrecognized" argument, but
     * there were enough to run the script, anything after the "bad" argument is ignored.
     * So, if you add -vvv, or anything else to the end of the line, it will be ignored!
     * This is undesireable behavior, so catch it and get out with an error message.
     **************************************************************************************/
    $argx = 0;
    $prefix = "-";
    $allArgs = array();
    $argv = $GLOBALS["argv"];
    $argc = count($GLOBALS["argv"]);
    
    # Keep track of all the arguments.
    while (++$argx < $argc) {
        $matches = array();

        if (preg_match('/^-/', $argv[$argx])) {
            preg_match("/-{1,2}(\w*)/", $argv[$argx], $matches);    # Strip the -
            $arg = $matches[1];
            $arg = (strpos($arg, "vv") === 0) ? "v" : $arg; # Simplify -vv and -vvv to -v (so it will match).

            # Keep track of all the arguments and the prefixes.
            # If the next argument is data, rather than another prefix, grab it and save it along with the current prefix (-d master) vs (-v).
            # Increment the counter, so the data is skipped, on the next pass.
            $allArgs[$arg] = array($matches[0] => ((($argx + 1 < $argc) && !preg_match("/^-/", $argv[$argx + 1])) ? $argv[++$argx] : ""));
        }
    }

    foreach ($allArgs as $prefix => $arg) {
        $key = key($arg);

        if (!array_key_exists($prefix, $parsedArgs)) {
            nagios_exit("Illegal argument \"".$key." ".$arg[$key]."\", please check your command line.", STATUS_UNKNOWN);
        }
    }
    
    /**************************************************************************************
     * Handle version request
     **************************************************************************************/
    if (array_key_exists('version', $parsedArgs) || array_key_exists('V', $parsedArgs)) {
        nagios_exit($VERSION.PHP_EOL, STATUS_OK);
    }
    
    /**************************************************************************************
     * NOTE:  Backslashes in the data MUST be escaped, in order to make it here.
     **************************************************************************************/
    $logger->debug("argv [".var_export($GLOBALS["argv"], true)."]", __METHOD__, __LINE__);
    $logger->debug("argc [".$GLOBALS["argc"]."]", __METHOD__, __LINE__);
    $logger->debug("parsedArgs ".var_export($parsedArgs, true), __METHOD__, __LINE__);

    // Handle help request
    if (array_key_exists("help", $parsedArgs) || array_key_exists("h", $parsedArgs)) {
        print_usage();
        nagios_exit('', STATUS_OK);
    }
    
    /**************************************************************************************
     * Setup the $MODES array.
     **************************************************************************************/
    $MODES = (array_key_exists("checktype", $parsedArgs) && $parsedArgs['checktype'] == 'database') ? $DATABASE_MODES : $SERVER_MODES;
    
    /**************************************************************************************
     * Make sure the input variables are sane.
     * Also check to make sure that all flags marked as required are present.
     **************************************************************************************/
    foreach($argSpecs as $argSpec) {
        $lOptions = $argSpec["long"];
        $sOptions = $argSpec["short"];
        
        if (array_key_exists($lOptions, $parsedArgs) && array_key_exists($sOptions, $parsedArgs)) {
            plugin_error("Command line parsing error: Inconsistent use of flag: ".$argSpec["long"]);
        }

        if (array_key_exists($lOptions, $parsedArgs)) {
            $options[$lOptions] = $parsedArgs[$lOptions];

        } elseif (array_key_exists($sOptions, $parsedArgs)) {
            $options[$lOptions] = $parsedArgs[$sOptions];

        } elseif ($argSpec["required"] == true) {
            plugin_error("Command line parsing error: Required variable \"".$argSpec["long"]."\" not present.");
        }

        // Find and verify the mode test.  --custom overrides mode.
        if ($argSpec['long'] == 'mode') {
            if (array_key_exists($options['mode'], $MODES)) {
                $logger->debug("TEST: ".$options['mode'], __METHOD__, __LINE__);
                $requestedTest = $options['mode'];  # The initial test/mode.  Used when running multiple tests.
            } else {
                plugin_error("Command line parsing error: The specified test \"".$options['mode']."\" is invalid, for the required argument \"mode\".  Please provide a valid test from the list.");
            }
        }
    }
    
    /**************************************************************************************
     * Handle verbosity request
     **************************************************************************************/
    if (array_key_exists('verbose', $parsedArgs) || array_key_exists('v', $parsedArgs)) {
        $level = $logger->getLogLevel();
        $verbosity = $logger->getLevelName($level);

        // We support up to 3 verbosity decreases, e.g., -vvv.
        if (array_key_exists('v', $parsedArgs) && is_array($parsedArgs['v'])) {
            $verboseCntr = 0;

            foreach ($parsedArgs['v'] as $v) {
                $verboseCntr++;

                if ($verboseCntr >= 4) {
                    $logger->notice("Lowest verbosity has already been reached! [".$verboseCntr."]", __METHOD__, __LINE__);
                    break;
                }

                $logger->verbose();
            }

        // Just lower verbosity once.
        } else {
            $logger->verbose();
        }

        $logger->debug("Adding verbosity... Original Log Level [".$level."], New Log Level [".$logger->getLogLevel()."]", __METHOD__, __LINE__);
        $logger->notice("Adding verbosity... Original Log Level [".$verbosity."], New Log Level [".$logger->getLevelName($logger->getLogLevel())."]", __METHOD__, __LINE__);
    }
    
    /**************************************************************************************
     * Handle custom test arguments.
     * NO SINGLE QUOTES!
     *
     * --custom '{"counter_name":"Page life expectancy","instance_name":"000",
     *            "unit":"s","modifier":"","ring_buffer_type":"","xpath":""}'
     * --custom '{"counter_name":"System Idle","instance_name":"","unit":"","modifier":"",
     *            "ring_buffer_type":"RING_BUFFER_SCHEDULER_MONITOR",
     *            "xpath:"/Record/SchedulerMonitorEvent/SystemHealth/SystemIdle[1]"}'
     **************************************************************************************/
    if (array_key_exists("custom", $options) && $options['mode'] != 'custom') {
        plugin_error("Illegal argument: \"--custom\", with --mode \"".$options['mode']."\", please check your command line.");

    } else if (!array_key_exists("custom", $options) && $options['mode'] == 'custom') {
        plugin_error("Missing argument: --mode \"".$options['mode']."\" requires argument --custom '(\"counter_name\":...)', please check your command line.");

    } else if (array_key_exists("custom", $options) && $options['mode'] == 'custom') {
        # Convert the () to {}, for json_decode - the wizard has to use (), instead of {}.
        $jsonString = str_replace("(", "{", str_replace(")", "}", $options['custom']));
        $customArgs = json_decode($jsonString, true);
        $logger->info("custom ".var_export($options['custom'], true), __METHOD__, __LINE__);
        $logger->debug("customArgs ".var_export($customArgs, true), __METHOD__, __LINE__);
    }

    /**************************************************************************************
     * Handle complex logging request
     **************************************************************************************/
    if (array_key_exists("complex", $parsedArgs)) {
        $logger->complex();
    }
    
    /**************************************************************************************
     * Handle SQL Server version requirements
     **************************************************************************************/
    if (array_key_exists("perftype", $parsedArgs)) {
        $perfType = $parsedArgs["perftype"];

        // Verify the key...
        if (!array_key_exists($perfType, $TABLES)) {
            plugin_error("The specified perftype is invalid [$perfType].");
        }
    }

    /**************************************************************************************
     * Setup the MEM_QUERY Global variable and update the 'memory' SERVER_MODE.
     **************************************************************************************/
    if (array_key_exists("checktype", $parsedArgs) && $parsedArgs['checktype'] == 'server') {
        $MEM_QUERY = ($perfType == 'azuresqldb') ? $AZURE_MEM_QUERY : $DEFAULT_MEM_QUERY;
        $MODES['memory']['query'] = $MEM_QUERY;
    }

    if (array_key_exists('memory', $MODES)) {
        $logger->debug("MEM_QUERY [".$MEM_QUERY."]", __METHOD__, __LINE__);
        $logger->debug("MODES['memory']\n".var_export($MODES['memory'], true), __METHOD__, __LINE__);
        $logger->info("Options\n".var_export($options, true), __METHOD__, __LINE__);
    }

    return $options;
}

function plugin_error($error_message) {
    global $logger;
    $logger->debug("", __METHOD__, __LINE__);

    print_usage("***ERROR***:\n\n{$error_message}\n\n");
}

function main() {
    global $logger;
    $logger->debug("", __METHOD__, __LINE__);

    // Handle the user's arguments.
    $options = parse_args();
    
    $check = new Check($options);
    $check->run_check();
}

########################################################################
# Argument handling functions
########################################################################

$VERSION = VERSION;  // Make the constant into a variable, so it can be embedded.  Yes this is annoying.
$mode = new Mode();  // Now it can be embedded in the variable definition, like VERSION.

$USAGE_STRING1 = <<<USAGE

check_mssql_database {$VERSION} - Copyright 2021 Nagios Enterprises, LLC.
                             Portions Copyright others (see source code).
                 
USAGE;

$USAGE_STRING2 = <<<USAGE

Usage: {$argv[0]} <options>

usage = "usage: %prog --checktype type -H hostname -U username -P password --mode testname
                      -w warning -c critical [-i instancename] [--perftype type] [--custom 'args']
                      [-v | -vv | -vvv | --verbose --complex]"

Options:
    -h, --help          Print detailed help screen.
    -V, --version       Print version information.
    -v, --verbose       Optional: Increases verbosity.  Verbosity may be increased up to 3 times,
                        using -v, e.g., -vvv
        --complex       Optional: Show Traditional/Complex log entries, for use with verbose.
        --perftype      Optional: Specify which version of SQL Server you are using.

                        Option      Table used                                SQL Server versions
                        ==========  ========================================  ================================
                        deprecated  sys.sysperfinfo                           SQL Server 2000+ - deprecated
                        default     sys.dm_os_performance_counters            SQL Server 2008+
                        azure       sys.dm_pdw_nodes_os_performance_counters  Azure Synapse Analytics (SQL DW)
                        pdw         sys.dm_pdw_nodes_os_performance_counters  Parallel Data Warehouse

        --checktype     Required: Specify 'database' or 'server' checks.
    -H, --hostname      Required: Hostname of the MSSQL server.
    -U, --username      Required: Username to use when logging into the MSSQL server.
    -P, --password      Required: Password to use when logging into the MSSQL server.

        --mode          Required: Must specify one and only one test, from the list, below...
                        (-h or --help for more information)

{$mode->print_list_of_tests()}

    -I, --instance      Optional: MSSQL Instance. (Overrides port)
    -p, --port          Optional: MSSQL server port. (Default is 1433)
        --tdsversion    Optional: Set the TDS version used when connecting to the MSSQL server.
                        Example: --tdsversion 7.0
    -i, --instancename  Optional: specify the instance_name value, from the perf table.
                        (When run in --checktype 'database', default is 'master')

        --custom 'args' Optional: Make sure you use double quotes (") in the brackets.
                        Requires --mode "custom".

                        Examples...
                        '{"counter_name":"Page life expectancy","instance_name":"000",
                          "unit":"s","modifier":"","ring_buffer_type":"","xpath":""}'
                        '("counter_name":"System Idle","instance_name":"","unit":"",
                          "modifier":"","ring_buffer_type":"RING_BUFFER_SCHEDULER_MONITOR",
                          "xpath":"/Record/SchedulerMonitorEvent/SystemHealth/SystemIdle[1]")'


    -w, --warning=<WARNING>     Required: The warning values, see:
    -c, --critical=<CRITICAL>   Required: The critical values, see
                          
Note: Warning and critical threshold values should be formatted via the
Nagios Plugin guidelines. See guidelines here:
https://nagios-plugins.org/doc/guidelines.html#THRESHOLDFORMAT
    
Examples:   10          Alerts if value is > 10
            30:         Alerts if value < 30
            ~:30        Alerts if value > 30
            30:100      Alerts if 30 > value > 100
            @10:200     Alerts if 30 >= value <= 100
            @10         Alerts if value = 10
                          
This plugin checks the status of an MS SQL Server Database.

USAGE;


########################################################################
# Argument handling functions
########################################################################

function print_usage($message=null) {
    global $USAGE_STRING1;
    global $USAGE_STRING2;
    global $logger;
    $logger->debug("", __METHOD__, __LINE__);

    if ($message) echo(PHP_EOL.$message.PHP_EOL);

    nagios_exit($USAGE_STRING1.$USAGE_STRING2, STATUS_UNKNOWN); // Exit status 3 for Nagios plugins is 'unknown'.
}

class Mode {
    function print_list_of_tests() {
        global $MODES;
        global $DATABASE_MODES;
        global $SERVER_MODES;
        global $options;
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        $checkType = "";

        // Trying to be nice to the users...
        if (!empty($options) && array_key_exists('checktype', $options)) {
            if ($options['checktype'] == 'database') {
                $checkType = 'database';
                $localMODES = $DATABASE_MODES;
            } else if ($options['checktype'] == 'server') {
                $checkType = 'server';
                $localMODES = $SERVER_MODES;
            }
        } else {
            $key = array_search('--checktype', $GLOBALS["argv"]);

            if ($key && $key++ < count($GLOBALS['argv'])) {
                $checkType = $GLOBALS['argv'][$key];

                if ($checkType == 'database') {
                    $localMODES = $DATABASE_MODES;
                } else if ($checkType == 'server') {
                    $localMODES = $SERVER_MODES;
                }
            }
        } 

        if (!empty($localMODES)) {
            $outputString = "                        Available Tests".PHP_EOL.PHP_EOL;

            foreach ($localMODES as $name => $test) {
                $logger->debug("print_list_of_tests key [".($name)."] ".var_export($test, true), __METHOD__, __LINE__);
                $outputString .= "                              ".$name.PHP_EOL;
            }
        } else {
            $outputString = "                        Available Database Tests".PHP_EOL.PHP_EOL;

            foreach ($DATABASE_MODES as $name => $test) {
                $logger->debug("print_list_of_tests key [".($name)."] ".var_export($test, true), __METHOD__, __LINE__);
                $outputString .= "                              ".$name.PHP_EOL;
            }

            $outputString .= PHP_EOL."                        Available Server Tests".PHP_EOL.PHP_EOL;

            foreach ($SERVER_MODES as $name => $test) {
                $logger->debug("print_list_of_tests key [".($name)."] ".var_export($test, true), __METHOD__, __LINE__);
                $outputString .= "                              ".$name.PHP_EOL;
            }

        }

        return $outputString;
    }
}

########################################################################
# Argument validation helper functions
########################################################################

#####
## NOTE: Not currently in use.
#####
// Validate the hostname
function validate_hostname($hostname) {
    $logger->debug("", __METHOD__, __LINE__);

    if (isset($hostname)) {
        if (!preg_match("/^([a-zA-Z0-9-]+[\.])+([a-zA-Z0-9]+)$/", $hostname)) {
            nagios_exit("UNKNOWN: Invalid characters in the hostname.\n", STATUS_UNKNOWN);
        }
    } else {
        nagios_exit("UNKNOWN: The required hostname field is missing.\n", STATUS_UNKNOWN);
    }
}

// Validate the port
function validate_port($port) {
    $logger->debug("", __METHOD__, __LINE__);

    if (isset($port)) {
        if (!preg_match("/^([0-9]{4,5})$/", $port)) {
            nagios_exit("UNKNOWN: The port field should be numeric and in the range 1000-65535.\n", STATUS_UNKNOWN);
        }
    }
}

// Validate the username
function validate_username() {
    $logger->debug("", __METHOD__, __LINE__);

    if (isset($username)) {
        if (!preg_match("/^[a-zA-Z0-9-_\\\@]*$/", $username)) {
            nagios_exit("UNKNOWN: Invalid characters in the username.\n", STATUS_UNKNOWN);
        }
    } else {
        nagios_exit("UNKNOWN: You must specify a valid username for this DB connection [".$username."].\n", STATUS_UNKNOWN);
    }
}

// Validate the warning and critical thresholds
function validate_warning_threshold($warning, $critical) {
    $logger->debug("", __METHOD__, __LINE__);

    $threshold_regex = "/^@?(~?(\d?\.?\d+)?\:?(\d?\.?\d+)?)$/";

    // Validate the warning thresholds
    if ($warning != "" && !preg_match($threshold_regex, $warning)) {
        nagios_exit("UNKNOWN: Invalid warning (-w | --warning) threshold.\n", STATUS_UNKNOWN);
    }

    // Validate the critical threshold
    if ($critical != "" && !preg_match($threshold_regex, $critical)) {
        nagios_exit("UNKNOWN: Invalid critical (-c | --critical) threshold.\n", STATUS_UNKNOWN);
    }

    // Is warning greater than critical? Doesn't care about ranges
    if (!empty($warning) && !empty($critical) && $warning > $critical) {
        $outputMsg = "UNKNOWN: warning value should be lower than critical value.\n";
        nagios_exit($outputMsg, STATUS_UNKNOWN);
    }
}

########################################################################
# Database Connection and test setup class and functions
########################################################################

class Check {
    private $options = null;
    private $connection = null;
    private $testCode = null;
    private $testStatusMsg = null;

    public function __construct($options) {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        $this->options = $options;
    }

    public function run_check() {
        global $MODES;
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        $mode = $this->options['mode'];
        $logger->debug("MODES[$mode]".var_export($MODES[$mode], true), __METHOD__, __LINE__);

        $instancename = (array_key_exists('instancename', $this->options)) ? $this->options['instancename'] : "";
        $logger->debug("instancename [".$instancename."]", __METHOD__, __LINE__);

        // Allow the ability to change the TDS version to support different version of MS SQL
        // Neither freetds.conf nor odbc.ini are respected when you pass a SERVER parameter in the PDO connection string
        // So we need to set the environment variable
        if (array_key_exists('tdsversion', $this->options)) {
            $tdsversion = $this->options["tdsversion"];
            $putstring = "TDSVER={$tdsversion}";
            putenv($putstring);
        }

        try {
            // Start time...
            $startTime = microtime(true);


            // If mssql DBLIB is not present...  CentOS 8's version of PHP (7.2+) does not include PDO_DBLIB (deprecated as of PHP 5.3).
            if (!extension_loaded('pdo_dblib')) {
                $db_dsn_host = "Server=".$this->options['hostname'];

                // We need to check this here, because port is required for ODBC.
                if (!empty($this->options['instance'])) {
                    $db_dsn_host .= "\\".$this->options['instance'];
                } else {
                    $db_dsn_host .= ";Port=".(!empty($this->options['port']) ? $this->options['port'] : "1433");
                }
        
                #$db_dsn = "odbc:mssql; Server=".$this->options['hostname'].(!empty($options['port']) ? "; Port=".$options['port'] : "")."; Database=".(!empty($options['database']) ? $options['database'] : "master");
                $db_dsn = "odbc:Driver=FreeTDS;".$db_dsn_host.";dbname=".(!empty($instancename) ? $instancename : "master").";charset=UTF8";
                $logger->info("Connecting to odbc db_dsn [".$db_dsn."]", __METHOD__, __LINE__);

            // MSSQL DBLIB is present...  PDO_DBLIB (deprecated as of PHP 5.3).
            } else {
                // Attempt to connect to the server
                $db_dsn_host = "host=".$this->options['hostname'];
        
                if (!empty($this->options['instance'])) {
                    $db_dsn_host .= "\\".$this->options['instance'];
                } else if (!empty($this->options['port'])) {
                    $db_dsn_host .= ":".$this->options['port'];
                }

                $db_dsn = "dblib:".$db_dsn_host.";dbname=".(!empty($instancename) ? $instancename : "master").";charset=UTF8";
                $logger->info("Connecting to dblib db_dsn [".$db_dsn."]", __METHOD__, __LINE__);
            }

            /**************************************************************************************************
             * Check the password for backspace escape characters, e.g., '\'.  This would be used to handle 
             * characters like '!', that will break the config files.
             **************************************************************************************************/
            if (array_key_exists("password", $this->options)) {
                $logger->info('Original password ['.$this->options['password'].']', __METHOD__, __LINE__);
                $this->options['password'] = stripslashes($this->options['password']);  # This is for the CCM's Run Check and command line testing.
                $logger->info('Cleaned password ['.$this->options['password'].']', __METHOD__, __LINE__);
            }
    
            $this->connection = new PDO($db_dsn, $this->options['username'], $this->options['password']);

            $this->connection->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

            // End time...
            $endTime = microtime(true);

            $this->connection->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        } catch (PDOException $e) {
            $logger->error("CRITICAL: Could not connect to $db_dsn as ".$this->options['username']." (Exception: " . $e->getMessage() . ")");
            $outputMsg = "CRITICAL: Could not connect to $db_dsn as ".$this->options['username']." (Exception: " . $e->getMessage() . ").\n";

            nagios_exit($outputMsg, STATUS_CRITICAL);
        }

        $connectionTime = round(($endTime - $startTime), 6);
        $logger->info("Successful connecting to ".$this->options['hostname']." [".$connectionTime."]", __METHOD__, __LINE__);
        
        // Run all the tests.
        if ($mode == 'test') {
            $logger->debug("*** Run All the plugins", __METHOD__, __LINE__);
            $this->run_plugin_tests();
            
        } elseif ($mode == 'runall') {
            $logger->debug("*** Run All the Tests", __METHOD__, __LINE__);
            $this->run_all_plugins($connectionTime);
            
        } elseif (!$mode || $mode == 'time2connect') {
            $state = "OK";

            $unit = array_key_exists("unit", $MODES[$mode]) ? $MODES[$mode]['unit'] : "";
            $label = $MODES[$mode]['label'];
            $stdout = $MODES[$mode]['stdout'];
            $logger->debug("*** No Mode specified or time2Connect was specified: connectionTime [$connectionTime] state [$state]", __METHOD__, __LINE__);

            $msSqlTest = new BaseQuery(null, $this->options, $stdout, $label, $unit, null, null);
            $exitCode = $msSqlTest->process_results($connectionTime, null, $this->options, $state, null);
                            
            nagios_exit($msSqlTest->get_outputMsg(), $exitCode);

        } else {
            $logger->debug("*** Run Only One Test", __METHOD__, __LINE__);
            $exitCode = $this->execute_query();
            nagios_exit($this->outputMsg, $exitCode);
        }
    }

    private function execute_query() {
        global $MODES;
        global $TABLES;
        global $MEM_QUERY;
        global $perfType;
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        $exitCode = STATUS_OK;

        # If this is a --mode custom test, setup $MODE for the new/custom query.
        if ($this->options['mode'] == 'custom') {
            $this->options['mode'] = $this->custom_query($this->connection);

            $logger->debug("Custom Query mode [".$this->options['mode']."] MODE ".var_export($MODES, true), __METHOD__, __LINE__);
        }

        $mode = $this->options['mode'];
        $logger->info("MODES name [".$this->options['mode']."]", __METHOD__, __LINE__);
        $logger->debug("MODES mode ".var_export($MODES[$mode], true), __METHOD__, __LINE__);

        $query = array_key_exists('query', $MODES[$mode]) ? $MODES[$mode]['query'] : "";
        $queryType = array_key_exists('type', $MODES[$mode]) ? $MODES[$mode]['type'] : "";
        $counterName = array_key_exists("counter_name", $MODES[$mode]) ? $MODES[$mode]['counter_name'] : "";
        $modifier = array_key_exists("modifier", $MODES[$mode]) ? $MODES[$mode]['modifier'] : "";
        $unit = array_key_exists("unit", $MODES[$mode]) ? $MODES[$mode]['unit'] : "";
        $label = $MODES[$mode]['label'];
        $stdout = $MODES[$mode]['stdout'];

        # instance_name is specified in the MODE entry and may be overridden by the command line option --instancename.
        # MODE entry or empty string (default).
        $instance_name = (array_key_exists('instance_name', $MODES[$mode])) ? $MODES[$mode]['instance_name'] : '';

        # command line option overrides
        if (array_key_exists('instancename', $this->options)) {
            $instance_name = $this->options['instancename'];
        }

        $logger->debug("mode [$mode] queryType [$queryType] counterName [$counterName] instance_name [$instance_name] --instancename [".(array_key_exists('instance_name', $this->options) ? $this->options['instancename'] : '')."]", __METHOD__, __LINE__);

        # If this is a memory test, use the "memory" table, otherwise use the "perf"ormance table.
        $mappedTable = ($MODES[$mode]['query'] == $MEM_QUERY) ? $TABLES[$perfType]["memory"] : $TABLES[$perfType]["perf"];

        $sqlQuery = strtr($query, array("@table" => $mappedTable, "@counter_name" => $counterName, "@instance_name" => $instance_name));
        $logger->debug("queryType [$queryType]", __METHOD__, __LINE__);

        if ($queryType == 'delta') {
            $logger->info("delta: sqlQuery [$sqlQuery]", __METHOD__, __LINE__);
            $msSqlTest = new MSSQLDeltaQuery($sqlQuery, $this->options, $stdout, $label, $unit, $modifier, $queryType, $this->connection);
            $testCode = $msSqlTest->test($this->connection);

        } elseif ($queryType == 'ratio') {
            $logger->info("ratio: sqlQuery [$sqlQuery]", __METHOD__, __LINE__);
            $msSqlTest = new MSSQLRatioQuery($sqlQuery, $this->options, $stdout, $label, $unit, $modifier, $queryType);
            $testCode = $msSqlTest->test($this->connection);

        } elseif ($queryType == 'deltaratio') {
            $logger->info("deltaratio: sqlQuery [$sqlQuery]", __METHOD__, __LINE__);
            $msSqlTest = new MSSQLDeltaRatioQuery($sqlQuery, $this->options, $stdout, $label, $unit, $modifier, $queryType, $this->connection);
            $testCode = $msSqlTest->test($this->connection);

        } elseif ($queryType == 'xml') {
            $logger->info("xml: sqlQuery [$sqlQuery]", __METHOD__, __LINE__);
            $msSqlTest = new MSSQLXMLQuery($sqlQuery, $this->options, $stdout, $label, $unit, $modifier, $queryType);
            $testCode = $msSqlTest->test($this->connection);

        } else {
            $logger->info("standard: sqlQuery [$sqlQuery]", __METHOD__, __LINE__);
            $msSqlTest = new MSSQLQuery($sqlQuery, $this->options, $stdout, $label, $unit, $modifier, $queryType);
            $testCode = $msSqlTest->test($this->connection);
        }

        $this->outputMsg = $msSqlTest->get_outputMsg();
        $this->testStatusMsg = $msSqlTest->get_testStatusMsg(); // This tends to be a shorter error message.

        return $testCode;
    }

    /**********************************************************************************
     * This is just a basic "do all the plugins function" test.
     * Use run_plugin_tests() to get output from each plugin.
     *
     * Note:  Not as useful as the Python version, since try/catch is a bit
     *        lacking in the current 5.4 version of PHP.
     */
    function run_plugin_tests() {
        global $MODES;
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        $failed = 0;
        $total  = 0;

        foreach ($MODES as $mode => $test) {
            $logger->debug("mode [$mode]", __METHOD__, __LINE__);

            if ($mode == 'time2connect' || $mode == 'test' || $mode == 'runall' || $mode == 'custom') {
                $logger->debug("SKIPPING $mode", __METHOD__, __LINE__);
                continue;
            }

            $total += 1;

            // Set the next test to run.
            $this->options['mode'] = $mode;
            $statusCode = $this->execute_query();

            if ($statusCode == STATUS_OK ||
                $statusCode == STATUS_WARNING ||
                $statusCode == STATUS_CRITICAL) {

                print(strtr("@testName passed!".PHP_EOL, array("@testName" => $mode)));

            } else {
                $failed += 1;

                print(strtr("@testName failed with: @testStatusMsg".PHP_EOL, array("@testName" => $mode, "@testStatusMsg" => $this->testStatusMsg)));
            }
        }

        print(strtr("@failed/@total tests failed.", array("@failed" => $failed, "@total" => $total)));
    }
        
    /**********************************************************************************
     * Runs all the plugins and displays the output from each plugin/test.
     */
    function run_all_plugins($connectionTime) {
        global $MODES;
        global $logger;
        global $overrideErrors;
        $logger->debug("", __METHOD__, __LINE__);

        $failed = 0;
        $total  = 0;
        $overrideErrors = true;

        foreach ($MODES as $mode => $test) {
            $logger->debug("mode [$mode]", __METHOD__, __LINE__);

            $total += 1;

            // Set the next test to run.
            $this->options['mode'] = $mode;

            if ($mode == 'test' || $mode == 'runall' || $mode == 'custom') {
                $logger->debug("SKIPPING $mode", __METHOD__, __LINE__);
                $total--;

                continue;
            } else if ($mode == 'time2connect') {
                $state = "OK";

                $unit = array_key_exists("unit", $MODES[$mode]) ? $MODES[$mode]['unit'] : "";
                $label = $MODES[$mode]['label'];
                $stdout = $MODES[$mode]['stdout'];
                $logger->debug("*** No Mode specified or time2Connect was specified: connectionTime [$connectionTime] state [$state]", __METHOD__, __LINE__);

                $msSqlTest = new BaseQuery(null, $this->options, $stdout, $label, $unit, null, null);
                $exitCode = $msSqlTest->process_results($connectionTime, null, $this->options, $state, null);

                $this->outputMsg = $msSqlTest->get_outputMsg();

            } else {
                $statusCode = $this->execute_query();
            }

            print($this->outputMsg);
        }

        print(strtr("@total tests.", array("@total" => $total)).PHP_EOL);
    }
        
    /***************************************************************************************************************
     * Create a custom query (--custom) with a counter_name value --countername, and optional --unit and --modifier.
     *
     * Use their database (we need to connect, anyway) to get the cntr_type, so the correct calculation is used.
     * We could save this to a file, but it may not be worth it, as long as we use the same connection, for both
     * queries.
     ***************************************************************************************************************/
    function custom_query($connection) {
        global $BASE_QUERY;
        global $RATIO_QUERY;
        global $RING_QUERY;
        global $CALC_QUERY;
        global $MODES;
        global $TABLES;
        global $perfType;
        global $customArgs;
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        $ringBufferType = $customArgs['ring_buffer_type'];
        $counterName = $customArgs['counter_name'];
        $instanceName = $customArgs['instance_name'];
        $modifier = $customArgs['modifier'];
        $unit = $customArgs['unit'];
        $xpath = $customArgs['xpath'];

        $checkName = $counterName;
        $label = $counterName;
        $logger->debug("checkName [$checkName] label [$label]", __METHOD__, __LINE__);

        $checkName = strtolower(str_replace("%", "pct", str_replace(" ", "", $checkName))); # e.g. Cache Hit Ratio to 'cachehitratio'
        $label     = strtolower(str_replace("%", "pct", str_replace(" ", "_", $label)));    # e.g. Cache Hit Ratio to 'cache_hit_ratio'
        $logger->debug("ringBufferType [$ringBufferType] counterName [$counterName] instanceName [$instanceName] modifier [$modifier] unit [$unit] xpath [$xpath] checkName [$checkName] label [$label]", __METHOD__, __LINE__);

        $counterType = "RING";
        $returnCode;

        # Figure out which type of calculation we need to do.
        if (empty($ringBufferType)) {
            try {
                $query = "select cntr_type from @table where counter_name = '@counter_name'";

                # Map the name of the "perf"ormance table and the counter_name.
                $sqlQuery = strtr($CALC_QUERY, array("@table" => $TABLES[$perfType]['perf'], "@counter_name" => $counterName, "@instance_name" => $instanceName));
                $logger->debug("sqlQuery [$sqlQuery]", __METHOD__, __LINE__);

                $pdoQuery = $this->connection->prepare($sqlQuery);
                $returnCode = $pdoQuery->execute();

                $logger->debug("returnCode [".$returnCode."]", __METHOD__, __LINE__);

                // Bail if the query failed.
                if (!$returnCode) {
                    nagios_exit("CRITICAL: Could not execute custom setup query.", STATUS_CRITICAL);
                }

                $result = $pdoQuery->fetchAll(PDO::FETCH_ASSOC);
                $logger->debug("result [".var_export($result, true)."]", __METHOD__, __LINE__);

                if (!isset($result) || empty($result) || !array_key_exists('cntr_type', $result[0])) {
                    $outputMsg = "CRITICAL: custom setup query failed. Verify your ".$TABLES[$perfType]['perf']." contains the proper entries for this query.";
                    nagios_exit($outputMsg, STATUS_CRITICAL);
                }

                $counterType = $result[0]['cntr_type'];
                $logger->debug("counterType [$counterType]", __METHOD__, __LINE__);

            } catch (Exception $pdoe) {
                nagios_exit($pdoe->getMessage(), STATUS_UNKNOWN);

                // Clean up.
                $pdoQuery->closeCursor(); // this is not even required
                $pdoQuery = null; // doing this is mandatory for connection to get closed
                $connection = null;
            }
        }

        switch ($counterType) {
            # PERF_COUNTER_LARGE_RAWCOUNT - 65792 (Point in Time) – Returns the last observed value for the counter.
            case 65792:
                $MODES = array(
                    $checkName => array(
                        'stdout' => $counterName.' is @result'.$unit,
                        'label' => $label,    # e.g. Cache Hit Ratio to 'cache_hit_ratio'
                        'query' => $BASE_QUERY,
                        'counter_name' => $counterName,
                        'instance_name' => $instanceName,
                        'type' => 'standard',
                        'unit' => $unit,
                        'modifier' => $modifier,
                    )
                );

                $logger->debug("65792: MODES [".var_export($MODES, true)."]", __METHOD__, __LINE__);
                break;

            # PERF_COUNTER_BULK_COUNT - 272696576 (Delta) – Average # of operations completed during each second of the sample interval.
            case 272696576:
                $MODES = array(
                    $checkName => array(
                        'stdout' => $counterName.' is @result'.$unit,
                        'label' => $label,    # e.g. Cache Hit Ratio to 'cache_hit_ratio'
                        'query' => $BASE_QUERY,
                        'counter_name' => $counterName,
                        'instance_name' => $instanceName,
                        'type' => 'delta',
                        'unit' => $unit,
                        'modifier' => $modifier,
                    )
                );

                $logger->debug("272696576: MODES [".var_export($MODES, true)."]", __METHOD__, __LINE__);
                break;

            # PERF_LARGE_RAw_FRACTION - 537003264 (Ratio)
            case 537003264:
                $MODES = array(
                    $checkName => array(
                        'stdout' => $counterName.' is @result'.$unit,
                        'label' => $label,    # e.g. Cache Hit Ratio to 'cache_hit_ratio'
                        'query' => $RATIO_QUERY,
                        'counter_name' => $counterName,
                        'instance_name' => $instanceName,
                        'type' => 'ratio',
                        'unit' => $unit,
                        'modifier' => $modifier,
                    )
                );

                $logger->debug("537003264: MODES [".var_export($MODES, true)."]", __METHOD__, __LINE__);
                break;

            # PERF_AVERAGE_BULK - 1073874176 (Delta Ratio)
            case 1073874176:
                $MODES = array(
                    $checkName => array(
                        'stdout' => $counterName.' is @result'.$unit,
                        'label' => $label,    # e.g. Cache Hit Ratio to 'cache_hit_ratio'
                        'query' => $RATIO_QUERY,
                        'counter_name' => $counterName,
                        'instance_name' => $instanceName,
                        'type' => 'deltaratio',
                        'unit' => $unit,
                        'modifier' => $modifier,
                    )
                );

                $logger->debug("1073874176: MODES [".var_export($MODES, true)."]", __METHOD__, __LINE__);
                break;

            # RING_QUERY
            case "RING":
                $MODES = array(
                    $checkName => array(
                        'stdout' => $counterName.' @result'.$unit,
                        'label' => $label,    # e.g. Cache Hit Ratio to 'cache_hit_ratio'
                        'query' => $RING_QUERY,
                        'counter_name' => $counterName,
                        'instance_name' => $instanceName,
                        'type' => 'xml',
                        'xpath' => $xpath,
                        'ring_buffer_type' => $ringBufferType,
                        'unit' => $unit,
                        'modifier' => $modifier,
                    )
                );

                $logger->debug("RING: MODES [".var_export($MODES, true)."]", __METHOD__, __LINE__);
                break;

            default:
                nagios_exit("NO MATCH!!", STATUS_UNKNOWN);
                break;
        }

        $logger->info("checkName [".$checkName."]", __METHOD__, __LINE__);
        return $checkName;
    }
}

########################################################################
# Database query classes
########################################################################

class BaseQuery {
    protected static $query = null;
    protected static $options = null;
    protected static $stdout = null;
    protected static $label = null;
    protected static $unit = null;
    protected static $modifier = null;
    protected static $queryType = null;
    protected static $result = array();
    protected static $calculatedResult = null;
    protected static $outputMsg = null;
    protected static $testStatusMsg = null;

    public function __construct($query, $options, $stdout, $label, $unit, $modifier, $queryType) {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        self::$query = $query;
        self::$options = $options;
        self::$stdout = $stdout;
        self::$label = $label;
        self::$unit = $unit;
        self::$modifier = (!empty($modifier) ? $modifier : 1);   // if no modifier, default is 1, no change.
        self::$queryType = $queryType;
        $logger->debug("query [$query] options (Array) stdout [$stdout] label [$label] unit [$unit] modifier [$modifier] queryType [$queryType]", __METHOD__, __LINE__);
    }

    public function get_outputMsg() {
        return self::$outputMsg;
    }

    public function get_testStatusMsg() {
        return self::$testStatusMsg;
    }

    // Function to process the results
    // process_results makes $outputMsg - the Status & Performance data available and returns the exit code.
    public static function process_results($queryDuration, $finalResult, $options, $state, $outputMsg) {
        global $logger;
        global $overrideErrors;
        $logger->debug("", __METHOD__, __LINE__);
        $logger->debug("input queryDuration [$queryDuration] options (Array) state [$state] outputMsg [$outputMsg]", __METHOD__, __LINE__);

        // Save the outputMsg
        self::$outputMsg = $outputMsg;

        $state = "OK";
        $exitCode = STATUS_OK;
        $warning = $options['warning'];
        $critical = $options['critical'];
        
        if (!$queryDuration && !$overrideErrors) {
            $logger->error("queryDuration [".$queryDuration."]", __METHOD__, __LINE__);
            nagios_exit("UNKNOWN: Could not perform query", STATUS_UNKNOWN);
        }

        // time2connect
        if ($options['mode'] == 'time2connect') {
            $finalResult = $queryDuration;
        }
        
        if (!empty($warning)) {
            switch (self::check_nagios_threshold($warning, $finalResult)) {
                case STATUS_UNKNOWN:
                    $exitCode = STATUS_UNKNOWN;
                    $state = "UNKNOWN";
                    self::$outputMsg = "ERROR: In range threshold START:END, START must be less than or equal to END";
                case 1:
                    $state = "WARNING";
                    $exitCode = STATUS_WARNING;
            }

            $logger->debug("Check Warning Threshold: exitCode [$exitCode ] state [$state] outputMsg [self::$outputMsg]", __METHOD__, __LINE__);
        }
        
        if (!empty($critical)) {
            switch (self::check_nagios_threshold($critical, $finalResult)) {
                case STATUS_UNKNOWN:
                    $exitCode = STATUS_UNKNOWN;
                    $state = "UNKNOWN";
                    self::$outputMsg = "ERROR: In range threshold START:END, START must be less than or equal to END";
                case 1:
                    $state = "CRITICAL";
                    $exitCode = STATUS_CRITICAL;
            }

            $logger->debug("Check Critical Threshold: exitCode [$exitCode ] state [$state] outputMsg [self::$outputMsg]", __METHOD__, __LINE__);
        }
        
        if ($options['mode'] == 'time2connect') {
            $queryResult = $queryDuration;
        } else {
            #$result = self::$queryResult;
            $queryResult = $finalResult;
        }

        $mappedResult = strtr(self::$stdout, array("@result" => $queryResult));
        $logger->debug("mappedResult: [$mappedResult] stdout [".self::$stdout."] queryResult [".$queryResult."]", __METHOD__, __LINE__);

        // The output consists of Status Data|Performance Data
        // Status & Performance Data Format from the Nagios Plugins Development Guidelines @ https://nagios-plugins.org/doc/guidelines.html#AEN33
        //
        // Status Data Format
        // SERVICE STATUS: Information text
        //
        // Performance Data Format
        // 'label'=value[UOM];[warning];[critical];[min];[max]
        self::$outputMsg = strtr("@state: @mappedResult|@label=@result@unit;@warning;@critical;;".PHP_EOL,
                          array("@state" => $state, "@mappedResult" => $mappedResult, "@label" => self::$label, "@result" => $queryResult,
                           "@unit" => self::$unit, "@warning" => $warning, "@critical" => $critical));
        $logger->info("outputMsg: [".self::$outputMsg."]", __METHOD__, __LINE__);
        $logger->debug("@state [$state] @stdout [".self::$stdout."] @label [".self::$label."] @result [$mappedResult] @unit [".self::$unit."] @warning [$warning] @critical [$critical]", __METHOD__, __LINE__);

        return $exitCode;
    }

    // Seems to return 0 for OK, 1 for warning/critical (depending on threshold), 3 for UNKNOWN.
    private static function check_nagios_threshold($threshold, $value) {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        $logger->debug("threshold [".$threshold."], value [".$value."]", __METHOD__, __LINE__);

        $inside = ((substr($threshold, 0, 1) == '@') ? true : false);
        $range = str_replace('@','', $threshold);
        $parts = explode(':', $range);
        
        if (count($parts) > 1) {
            $start = $parts[0];
            $end = $parts[1];

        } else {
            $start = 0;
            $end = $range;
        }
        
        if (substr($start, 0, 1) == "~") {
            $start = -999999999;
        }

        if ($end == "") {
            $end = 999999999;
        }

        $logger->debug("start [".$start."] > end [".$end."]", __METHOD__, __LINE__);

        if ($start > $end) {
            $logger->debug("STATUS_UNKNOWN [".STATUS_UNKNOWN."] start [".$start."] > end [".$end."]", __METHOD__, __LINE__);

            return STATUS_UNKNOWN;
        }
        
        if ($start <= $value && $value <= $end) {
            $logger->debug("inside", __METHOD__, __LINE__);

            return $inside;
        }

        $logger->debug("!inside", __METHOD__, __LINE__);

        return !$inside;
    }
}

/*****************************************************************************************************************************************
 * MSSQLQuery handles point in time "calculations" for any test that is NOT type "delta", "ratio" or "xml"
 *
 * Counter              (counter_type) (Category)
 *
 * PERF_COUNTER_LARGE_RAWCOUNT (65792) (Point in Time) – Returns the last observed value for the counter.
 *
 * Categories       Calculation
 * =============    ====================================================================================================================
 * Point in Time    None.  Many of these Counters use instance_name, which may be the name of a database, or something else.
 *
 *****************************************************************************************************************************************/
class MSSQLQuery extends BaseQuery {
    protected static $returnCode = null;
    protected static $utcTimestamp = null;

    public function __construct($query, $options, $stdout, $label, $unit, $modifier, $queryType) {
        global $logger;
        parent::__construct($query, $options, $stdout, $label, $unit, $modifier, $queryType);

        $logger->debug("", __METHOD__, __LINE__);
    }

    protected function run_on_connection($connection) {
        global $logger;
        global $overrideErrors;
        $logger->debug("", __METHOD__, __LINE__);

        try {
            $logger->info("query [".self::$query."]", __METHOD__, __LINE__);
            $pdoQuery = $connection->prepare(self::$query);

            $startTime = microtime(true);
            self::$returnCode = $pdoQuery->execute();
            $endTime = microtime(true);
            $logger->debug("returnCode [".self::$returnCode."]", __METHOD__, __LINE__);

            // Bail if the query failed.
            if (!self::$returnCode) {
                if (!$overrideErrors) {
                    $logger->error("Query Failed!! returnCode [".self::$returnCode."] query [".self::$query."]", __METHOD__, __LINE__);
                    $logger->error("PDO::errorInfo():\n".var_export($pdoQuery->errorInfo(), true), __METHOD__, __LINE__);
                }

                $exit_code = STATUS_CRITICAL;
                $outputMsg = "CRITICAL: Could not execute ".self::$options['mode']." query.";

                if ($overrideErrors) {
                    echo($outputMsg.PHP_EOL);
                    return round(($endTime - $startTime), 6);
                }

                nagios_exit($outputMsg, $exit_code);
            }

            self::$result = $pdoQuery->fetchAll(PDO::FETCH_ASSOC);
            $logger->debug("result [".var_export(self::$result, true)."]", __METHOD__, __LINE__);

            if (!isset(self::$result) || empty(self::$result)) {
                if (!$overrideErrors) {
                    $logger->info("Query (".self::$options['mode'].") Failed!! No Results!  returnCode [".self::$returnCode."] query [".self::$query."]", __METHOD__, __LINE__);
                    $logger->debug("PDO::errorInfo():\n".var_export($pdoQuery->errorInfo(), true), __METHOD__, __LINE__);
                }

                $exit_code = STATUS_CRITICAL;
                $outputMsg = "CRITICAL: ".self::$options['mode']." query failed. This could be caused by your sysperfinfo not containing the proper entries for this query, and you may need to delete this service check.";

                if ($overrideErrors) {
                    echo($outputMsg.PHP_EOL);
                    return round(($endTime - $startTime), 6);
                }

                nagios_exit($outputMsg, $exit_code);
            }
        } catch (Exception $pdoe) {
            $logger->error("Query (".self::$options['mode'].") Failed!! No Results! returnCode [".self::$returnCode."] query [".self::$query."]", __METHOD__, __LINE__);
            $logger->error($pdoe->getMessage(), __METHOD__, __LINE__);
            $outputMsg = "CRITICAL: ".self::$options['mode']." query failed. This could be caused by your sysperfinfo not containing the proper entries for this query, and you may need to delete this service check.";
            nagios_exit($outputMsg, (!isset($exit_code) ? STATUS_UNKNOWN : $exit_code));

            # This is for when we are ignoring errors and nagios_exit() returns, instead of exiting (runall).
            return;
        }

        // Make sure we have well formatted results.
        if (!array_key_exists(0, self::$result) || !array_key_exists('value', self::$result[0])) {
            $logger->error("PDO::errorInfo():\n".var_export($pdoQuery->errorInfo(), true), __METHOD__, __LINE__);
            $exit_code = STATUS_CRITICAL;
            $outputMsg = "CRITICAL: Query Failed!! Bad results array!  Could not execute ".self::$queryType." query.\n";
            nagios_exit($outputMsg, $exit_code);
        }

        // Initialize calculatedResult to the value of the first set.
        self::$calculatedResult = self::$result[0]['value'];

        // Clean up.
        $pdoQuery->closeCursor(); // this is not even required
        $pdoQuery = null; // doing this is mandatory for connection to get closed
        $connection = null;

        // Time to complete the query.
        return round(($endTime - $startTime), 6);
    }

    #
    # Point in Time should never need a modifier, so this is a bit overkill/confusion.
    #
    private function calculate_result() {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        $logger->info("calculatedResult [".self::$calculatedResult."] modifier [".self::$modifier."]", __METHOD__, __LINE__);
        $calculate = self::$calculatedResult;

        if (!empty(self::$modifier)) {
            $calculate = floatval($calculate) * floatval(self::$modifier);
            $logger->debug("calculate [".$calculate."]", __METHOD__, __LINE__);
        }

        self::$calculatedResult = number_format($calculate, 1, '.', '');
        $logger->debug("calculatedResult [".self::$calculatedResult."]", __METHOD__, __LINE__);

        return STATUS_OK;
    }

    public function test($connection) {
        global $logger;
        global $requestedTest;
        global $overrideErrors;
        $logger->debug("", __METHOD__, __LINE__);

        // Attempt to execute the query/stored procedure
        $queryDuration = $this->run_on_connection($connection);

        // If we are running all tests, things may have already gone sideways.
        // Make sure we have valid data and should continue...
        if ($overrideErrors && empty(self::$result)) {
            return STATUS_CRITICAL;
        }

        $logger->info("queryDuration [$queryDuration]", __METHOD__, __LINE__);
        $logger->debug("calculatedResult [".self::$calculatedResult."]", __METHOD__, __LINE__);   // Value, before any calculations
        $logger->debug("returnCode [".self::$returnCode."]", __METHOD__, __LINE__);

        $outputMsg = self::$queryType." duration=$queryDuration seconds.";

        # Use the child's method, if it exists, otherwise this one.
        # e.g., MSSQLRatioQuery, MSSQLDeltaQuery, etc.
        $errorCode = static::calculate_result();

        // Exit on error, unless we are running all the tests.
        if ($errorCode != STATUS_OK) {
            # MODES test and runall, execute multiple tests, so keep going...
            if ($requestedTest == 'test' || $requestedTest == "runall") {
                return $errorCode;
            }

            nagios_exit(self::$outputMsg, $errorCode);
        }

        # Round anything with decimal places > 3, to 4 significant figures.
        $finalResult = $this->sigFig(self::$calculatedResult, 4);

        $logger->info("outputMsg [$outputMsg]", __METHOD__, __LINE__);
        $logger->debug("calculatedResult [".self::$calculatedResult."] finalResult [$finalResult]", __METHOD__, __LINE__);

        $exitCode = $this->process_results($queryDuration, $finalResult, self::$options, STATUS_OK, $outputMsg);

        return $exitCode;
    }

    #
    # Round anything with decimal places > 3, to 4 significant figures, otherwise, ignore.
    # Prevent scientific notation in large numbers.
    # 
    function sigFig($value, $significantDigits) {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

#        $value = ".000000000000000000000000000025677";
#        $value = "1234567349812384891.49872345";
        $decimalPart = ".".substr(strrchr($value, "."), 1);
        $decimalCount = (strpos($decimalPart, ".") == 0) ? strlen($decimalPart) - 1 : 0;
        $integerPart = strstr($value, ".", true);
        $integerCount = strlen($integerPart);
        $logger->debug("decimalPart [$decimalPart] decimalCount [$decimalCount] integerPart [$integerPart] integerCount [$integerCount]");

        if ($decimalCount > 3) {
            if ($value == 0) {
                $decimalPlaces = $significantDigits - 1;
            } else if ($value < 1) {
                $decimalPlaces = $significantDigits - floor(log10($value)) - 1;
                #$decimalPlaces = $significantDigits;
            } elseif ($value < 0) {
                #$decimalPlaces = $significantDigits - floor(log10($value * -1)) - 1;
                $decimalPlaces = $significantDigits - floor(log10($value * -1)) - 2;
            } elseif ($value < 1000) {
                #$decimalPlaces = $significantDigits - floor(log10($value * -1)) - 1;
                $decimalPlaces = 3;
            } else {
                #$decimalPlaces = $significantDigits - floor(log10($value)) - 1;
                $decimalPlaces = 1;
            }

            if ($integerCount > 16) {
                # Avoid scientific notation...
                $roundedDecimals = floatval(round($decimalPart, $decimalCount));

                if ($roundedDecimals >= 1) {
                    $integerPart += 1;

                    if ($roundedDecimals == 1) {
                        $roundedDecimals = str_repeat("0", $decimalPlaces);
                    }
                } else {
                    $roundedDecimals = substr(strrchr($roundedDecimals, "."), 1, $decimalPlaces);
                }

                $answer = "$integerPart.$roundedDecimals";
            } else if ($decimalPlaces > 0) {
                $answer = number_format($value, $decimalPlaces, ".", "");
            } else {
                $answer = round($value, $decimalPlaces);
            }
        } else {
            $answer = $value;
        }

        return $answer;
    }
}

class MSSQLXMLQuery extends MSSQLQuery {

    # Example record from SQL Server...
    #
    # <Record id="20462" type="RING_BUFFER_SCHEDULER_MONITOR" time="1232891461">
    #   <SchedulerMonitorEvent>
    #     <SystemHealth>
    #       <ProcessUtilization>0</ProcessUtilization>
    #       <SystemIdle>88</SystemIdle>
    #       <UserModeTime>625000</UserModeTime>
    #       <KernelModeTime>937500</KernelModeTime>
    #       <PageFaults>1010</PageFaults>
    #       <WorkingSetDelta>3502080</WorkingSetDelta>
    #       <MemoryUtilization>61</MemoryUtilization>
    #     </SystemHealth>
    #   </SchedulerMonitorEvent>
    # </Record>
    #
    # Example xpath, to get requested data...
    #
    # 'xpath' => '/Record/SchedulerMonitorEvent/SystemHealth/ProcessUtilization[1]',
    private function process_xml($xmlString) {
        global $MODES;
        global $logger;
        global $overrideErrors;
        $logger->debug("", __METHOD__, __LINE__);

        $mode = self::$options['mode'];
        $xml = simplexml_load_string($xmlString);

        if ($xml === false) {
            $exit_code = STATUS_CRITICAL;
            $outputMsg = "CRITICAL: Failed loading XML! (".var_export($xml, true).")";

            foreach(libxml_get_errors() as $error) {
                $outputMsg .= ", ".$error->message;
            }

            $outputMsg.PHP_EOL;
            nagios_exit($outputMsg, $exit_code);
        }

        $xpath = array_key_exists("xpath", $MODES[$mode]) ? $MODES[$mode]['xpath'] : "";
        $logger->debug("mode [$mode] xpath [$xpath] xmlString [$xmlString] xml [".var_export($xml, true)."]", __METHOD__, __LINE__);

        $valueArray = empty($xml) ? "": $xml->xpath($xpath);

        if (empty($valueArray) && array_key_exists("xpath2", $MODES[$mode])) {
        $logger->debug("Using xpath2: xpath [$xpath]", __METHOD__, __LINE__);
            $xpath = $MODES[$mode]['xpath2'];
            $valueArray = empty($xml) ? "": $xml->xpath($xpath);
        }
 
        // Make sure we have well formatted results.
        if (!array_key_exists(0, $valueArray)) {
            $logger->debug("Xpath Failed!!".PHP_EOL."xpath [$xpath]".PHP_EOL."xmlString [$xmlString]", __METHOD__, __LINE__);
            $logger->debug(PHP_EOL."Array results from xpath [".var_export($valueArray, true)."]", __METHOD__, __LINE__);

            $exit_code = STATUS_CRITICAL;
            $outputMsg = "CRITICAL: Xpath Failed!! Bad results array! (".self::$options['mode'].")".PHP_EOL;
            nagios_exit($outputMsg, $exit_code);

            # This is for when we are ignoring errors (runall).
            if ($overrideErrors) {
                return;
            }
        }

        $value = $valueArray[0];
        $logger->debug(PHP_EOL."value [".$value."]".PHP_EOL."xpath [$xpath]".PHP_EOL."xmlString [$xmlString]", __METHOD__, __LINE__);

        return floatval($value);
    }

    #
    # In this case, self::$calculatedResult is an xml string...
    # 
    protected function calculate_result() {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        $logger->info("calculatedResult [".self::$calculatedResult."] modifier [".self::$modifier."]", __METHOD__, __LINE__);
        $calculate = $this->process_xml(self::$calculatedResult);
        $logger->debug("After process_xml: calculate [".$calculate."]", __METHOD__, __LINE__);

        if (!empty(self::$modifier)) {
            $calculate = floatval($calculate) * floatval(self::$modifier);
            $logger->debug("calculate [".$calculate."]", __METHOD__, __LINE__);
        }

        self::$calculatedResult = number_format($calculate, 1, '.', '');
        $logger->debug("calculatedResult [".self::$calculatedResult."]", __METHOD__, __LINE__);

        return STATUS_OK;
    }
}

/*****************************************************************************************************************************************
 *
 * Counter              (counter_type) (Category)
 *
 * PERF_LARGE_RAw_FRACTION (537003264) (Ratio)
 *      - # of items processed, on average, during an operation.
 *      - These counter types display a ratio of the items processed (such as bytes sent) to the number of operations
 *        completed, and requires a base property with PERF_LARGE_RAW_BASE as the counter type.
 *
 * Categories       Calculation
 * =============    ====================================================================================================================
 * Ratio            The formula is A1/B1 * 100.  Where A1 is type 537003264 and B1 is type 1073939712.
 * 
 *                  Buffer Cache Hit Ratio % = 100 * Buffer Cache Hit Ratio / Buffer Cache Hit Ratio Base
 *                                           = 100 * 2,135 / 3,573
 *                                           = 59.75%
 * 
 *****************************************************************************************************************************************/
class MSSQLRatioQuery extends MSSQLQuery {

    public function __construct($query, $options, $stdout, $label, $unit, $modifier, $queryType) {
        global $logger;
        parent::__construct($query, $options, $stdout, $label, $unit, $modifier, $queryType);

        $logger->debug("", __METHOD__, __LINE__);
    }

    protected function calculate_result() {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        $logger->info("calculatedResult [".self::$calculatedResult."] modifier [".self::$modifier."]", __METHOD__, __LINE__);

        if (!empty(self::$modifier)) {
            try {
                $logger->debug("result[0] [".self::$result[0]['value']."] / result[1] [".self::$result[1]['value']."] * modifier [".self::$modifier."]", __METHOD__, __LINE__);

                // PHP 5.4...
                if (empty(self::$result[1]['value']) || !self::$result[1]['value']) {
                    $logger->debug("THROWING DIVIDE BY ZERO!!!", __METHOD__, __LINE__);
                    throw new Exception('division by zero');
                }

                #
                # Where the first result [0], is A1 and the second result [1], is B1.
                #
                $calculated = floatval(self::$result[0]['value']) / floatval(self::$result[1]['value']) * floatval(self::$modifier);
                $logger->debug("0 [".self::$result[0]['value']."] / 1 [".self::$result[1]['value']."] * modifier [".self::$modifier."]", __METHOD__, __LINE__);

                self::$calculatedResult = number_format($calculated, 10, '.', '');
                #
            # PHP 7
            #} catch (DivisionByZeroError $dbze) {
            } catch (Exception $exception) {
                $logger->debug("CATCHING DIVIDE BY ZERO!!!", __METHOD__, __LINE__);
                self::$testStatusMsg = $exception->getMessage();    // This is for running all the tests.
                self::$outputMsg = "<type 'exceptions.DivisionByZeroError'>".PHP_EOL."Caught unexpected error processing ".self::$options['mode'].". This could be caused by your sysperfinfo not containing the proper entries for this query, and you may need to delete this service check.".PHP_EOL;

                return STATUS_UNKNOWN;
            }
        }

        return STATUS_OK;
    }
}

/*****************************************************************************************************************************************
 * MSSQLDeltaQuery handles delta calculations for any test with type "delta".
 *
 * Counter                 (counter_type)  (Category)
 * PERF_COUNTER_BULK_COUNT (272696576)     (Delta) – Average # of operations completed during each second of the sample interval.
 *
 * Categories       Calculation
 * =============    ====================================================================================================================
 * Delta            The formula for the current metric value is (A2-A1)/(T2-T1) 
 *
 *                  Where A1 and A2 are the values of the monitored PERF_COUNTER_BULK_COUNT counter, taken at sample times T1 and T2
 *                  T1 and T2 are the times when the sample values are taken.
 *
 *                  Page lookups/sec = (854,521 – 852,433)/(621,366,686-621,303,043) 
 *                                   = 2,088 / 63,643 ms 
 *                                   = 2,088/63 sec = 32.1 /sec
 *
 *****************************************************************************************************************************************/
class MSSQLDeltaQuery extends MSSQLQuery {
    private $fileName = "";
    private $lastRun = null;
    protected $connection = null;
    protected $currentRun = null;

    public function __construct($query, $options, $stdout, $label, $unit, $modifier, $queryType, $connection) {
        global $logger;
        parent::__construct($query, $options, $stdout, $label, $unit, $modifier, $queryType);

        $logger->debug("", __METHOD__, __LINE__);

        $this->connection = $connection;
    }

    // Create a temporary file in the temporary files directory.
    protected function set_file_name() {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        $tmpDir = sys_get_temp_dir();
        $thisTest = self::$options['mode'].self::$options['hostname'];

        if (!is_dir($tmpDir) || !is_writable($tmpDir)) {
            $outputMsg = ((!is_dir($tmpir)) ? "The system tmp directory [$tmpDir] does NOT exist!" : "The system tmp directory [$tmpDir] MUST be writeable!");
            $logger->error($outputMsg, __METHOD__, __LINE__);
            nagios_exit($outputMsg, STATUS_UNKNOWN); 
        }

        $uniqueId = md5($thisTest);
        $logger->debug("Generate filename (md5): uniqueId [$uniqueId] thisTest [$thisTest] mode [".self::$options['mode']."] hostname [".self::$options['hostname']."]", __METHOD__, __LINE__);

        // The hash always returns the same result, so had to add the name of the test, so multiple runs don't overwrite each other.
        // e.g., individual files for each test.
        $fullFilePath = strtr("@tmpDirectory/mssql-@uniqueId.tmp", array("@tmpDirectory" => $tmpDir, "@uniqueId" => $uniqueId));
        $logger->debug("fullFilePath [$fullFilePath]", __METHOD__, __LINE__);

        $this->fileName = $fullFilePath;
    }

    protected function save() {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        $serializedRun = serialize($this->currentRun);
        $returnCode = STATUS_OK;

        $logger->debug("currentRun [".var_export($this->currentRun, true)."]", __METHOD__, __LINE__);
        $logger->debug("serializedRun [$serializedRun]", __METHOD__, __LINE__);

        try {
            $logger->debug("fileName [$this->fileName]", __METHOD__, __LINE__);
            $returnCode = file_put_contents($this->fileName, $serializedRun);

        } catch (Exception $exception) {
            $logger->error("Failed to save data from last run! code [$exception->getCode()] message [$exception->getMessage()]", __METHOD__, __LINE__);
            $logger->debug($exception->getTraceAsString(), __METHOD__, __LINE__);
        }

        if ($returnCode === false) {
            $logger->error("FAILED TO CREATE/WRITE FILE", __METHOD__, __LINE__);
        }

        $logger->debug("file_put_contents: returnCode [$returnCode]", __METHOD__, __LINE__);
    }

    protected function getLastRun() {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        $fileHandle = fopen($this->fileName, 'c+');
        $fileContents = file_get_contents($this->fileName);
        fclose($fileHandle);

        if ($fileContents === FALSE) throw new Exception("Run file does not exist!");

        $this->lastRun = unserialize($fileContents);
        if ($this->lastRun === FALSE) throw new Exception("No run data!");

        return unserialize(file_get_contents($this->fileName));
    }

    protected function calculate_result() {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        $this->set_file_name();
        
        $newTime = self::$result[0]['utctimestamp'];
        $newValue  = self::$result[0]['value'];
        $logger->debug("@@@@@newTime [$newTime] newValue [$newValue]", __METHOD__, __LINE__);

        try {
            $this->lastRun = $this->getLastRun();
            $logger->debug("@@@@@@ oldTime [".$this->lastRun['time']."] oldValue [".$this->lastRun['query_result']."]", __METHOD__, __LINE__);

            if (empty($this->lastRun['time']))  {
                throw new Exception('Data missing from file!');
            }

        } catch (Exception $exception) {
            $logger->debug("First Run, Data Missing or File Not Found!", __METHOD__, __LINE__);

            # Make this run the current values.
            $this->lastRun = array('time' => $newTime, 'query_result' => $newValue);

            # Wait a second and get another run...
            sleep(1);

            # Get a new set of data.
            self::run_on_connection($this->connection);

            $newTime = self::$result[0]['utctimestamp'];
            $newValue  = self::$result[0]['value'];
            $logger->debug("###### oldTime [".$this->lastRun['time']."] oldValue [".$this->lastRun['query_result']."]", __METHOD__, __LINE__);
            $logger->debug("###### newTime [$newTime] newValue [$newValue]", __METHOD__, __LINE__);
        }
        
        $oldTime = $this->lastRun['time'];
        $oldValue = $this->lastRun['query_result'];

        try {
            // PHP 5.4...
            if (empty($newTime) || empty($oldTime) || !($newTime - $oldTime)) {
                throw new Exception('division by zero');
            }

            $logger->debug("(floatval($oldValue) - floatval($newValue)) = [".(floatval($oldValue) - floatval($newValue))."]", __METHOD__, __LINE__);
            $logger->debug("(floatval($oldTime)) - floatval($newTime) = [".(floatval($oldTime) - floatval($newTime))."]", __METHOD__, __LINE__);
            $logger->debug("floatval(".self::$modifier.")", __METHOD__, __LINE__);

            # (A2-A1)/(T2-T1) = (new-old)/(newtime-oldtime)
            $calculate = (floatval($newValue) - floatval($oldValue)) / (floatval($newTime) - floatval($oldTime)) * floatval(self::$modifier);

            self::$calculatedResult = number_format($calculate, 10, '.', '');
            $logger->info("calculate [$calculate] calculatedResult [".self::$calculatedResult."]", __METHOD__, __LINE__);

        # PHP 7
        #} catch (DivisionByZeroError $dbze) {
        } catch (Exception $exception) {
            self::$testStatusMsg = $exception->getMessage();    // This is for running all the tests.
            self::$outputMsg = "<type 'exceptions.DivisionByZeroError'>".PHP_EOL."Caught unexpected error. This could be caused by your sysperfinfo not containing the proper entries for this query, and you may need to delete this service check.".PHP_EOL;

            return STATUS_UNKNOWN;
        }

        $logger->debug("calculatedResult [".self::$calculatedResult."] newValue [$newValue] oldValue [$oldValue] newTime [$newTime] oldTime [$oldTime] modifier [".self::$modifier."]", __METHOD__, __LINE__);
        
        $this->currentRun = array('time' => $newTime, 'query_result' => $newValue);
        $logger->info("currentRun Object ".var_export($this->currentRun, true), __METHOD__, __LINE__);
        $this->save();

        return STATUS_OK;
    }
}

/*****************************************************************************************************************************************
 * Counter                (counter_type) (Category)
 *
 * PERF_AVERAGE_BULK      (1073874176)   (Delta Ratio)
 *      - # of items processed, on average, during an operation.
 *      - These counter types display a ratio of the items processed (such as bytes sent) to the number of operations
 *        completed, and requires a base property with PERF_LARGE_RAW_BASE as the counter type.
 *
 * Categories       Calculation
 * =============    ====================================================================================================================
 * Delta Ratio      The formula is (A2 – A1)/(B2 – B1) / Interval - where A1 and A2 are type 1073874176 and B1 & B2 are type 1073939712,
 *                  A1 & B1, A2 & B2 are collected at the same time and Interval is the time between the two.
 *
 *                  Average Wait Time (ms) for the interval between these two measurements is:
 *                      = (A2 – A1)/(B2 – B1) / Interval
 *                      = (53736 ms -52939 ms)/(23-18) = 797 ms / 5 = 159.4 ms
 *
 *****************************************************************************************************************************************/
class MSSQLDeltaRatioQuery extends MSSQLDeltaQuery {

    public function __construct($query, $options, $stdout, $label, $unit, $modifier, $queryType, $connection) {
        global $logger;
        parent::__construct($query, $options, $stdout, $label, $unit, $modifier, $queryType, $connection);

        $logger->debug("", __METHOD__, __LINE__);
    }

    protected function calculate_result() {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        $this->set_file_name();
        
        $newTime = self::$result[0]['utctimestamp'];
        $newValue  = self::$result[0]['value'];
        $newBase  = self::$result[1]['value'];
        $logger->debug("result ".var_export(self::$result, true),__METHOD__, __LINE__);
        $logger->debug("@@@@@newTime [$newTime] newValue [$newValue] newBase [$newBase]", __METHOD__, __LINE__);

        try {
            $this->lastRun = $this->getLastRun();
            $logger->debug("@@@@@@ oldTime [".$this->lastRun['time']."] oldValue [".$this->lastRun['query_value']."] oldBase [".$this->lastRun['query_base']."]", __METHOD__, __LINE__);

            if (empty($this->lastRun['time']))  {
                throw new Exception('Data missing from file!');
            }

        } catch (Exception $exception) {
            $logger->debug("First Run, Data Missing or File Not Found!", __METHOD__, __LINE__);

            # Make this run the current values.
            $this->lastRun = array('time' => $newTime, 'query_value' => $newValue, 'query_base' => $newBase);

            # Wait a second and get another run...
            sleep(1);

            # Get a new set of data.
            self::run_on_connection($this->connection);

            $newTime = self::$result[0]['utctimestamp'];
            $newValue  = self::$result[0]['value'];
            $newBase  = self::$result[1]['value'];
            $logger->debug("###### oldTime [".$this->lastRun['time']."] oldValue [".$this->lastRun['query_value']."] oldBase [".$this->lastRun['query_base']."]", __METHOD__, __LINE__);
            $logger->debug("###### newTime [$newTime] newValue [$newValue] newBase [$newBase]", __METHOD__, __LINE__);
        }
        
        $oldTime = $this->lastRun['time'];
        $oldValue = $this->lastRun['query_value'];
        $oldBase = $this->lastRun['query_base'];

        try {
            // PHP 5.4...
            if (empty($newTime) || empty($oldTime) || !($newTime - $oldTime)) {
                throw new Exception('division by zero');
            }

            $logger->debug("((floatval($newValue) - floatval($oldValue)) = [".(floatval($newValue) - floatval($oldValue))."]", __METHOD__, __LINE__);
            $logger->debug("(floatval($oldBase) - floatval($newBase)) = [".(floatval($oldBase) - floatval($newBase))."]", __METHOD__, __LINE__);
            $logger->debug("(floatval($newTime) - floatval($oldTime)) = [".(floatval($newTime) - floatval($oldTime))."]", __METHOD__, __LINE__);
            $logger->debug("floatval(".self::$modifier.")", __METHOD__, __LINE__);

            # Delta Ratio - The formula is (A2 – A1)/(B2 – B1) / Interval
            $base = floatval($newBase) - floatval($oldBase);
            $time = floatval($newTime) - floatval($oldTime);

            $calculate = ($base != 0) ? (floatval($newValue) - floatval($oldValue)) / $base / $time * floatval(self::$modifier) : 0;
            #$calculate = (floatval($newValue) - floatval($oldValue)) / (floatval($newBase) - floatval($oldBase)) / (floatval($newTime) - floatval($oldTime)) * floatval(self::$modifier);    # 5.3.3 Issues.

            self::$calculatedResult = number_format($calculate, 10, '.', '');
            $logger->info("calculate [$calculate] calculatedResult [".self::$calculatedResult."]", __METHOD__, __LINE__);

        # PHP 7
        #} catch (DivisionByZeroError $dbze) {
        } catch (Exception $exception) {
            self::$testStatusMsg = $exception->getMessage();    // This is for running all the tests.
            self::$outputMsg = "<type 'exceptions.DivisionByZeroError'>".PHP_EOL."Caught unexpected error. This could be caused by your sysperfinfo not containing the proper entries for this query, and you may need to delete this service check.".PHP_EOL;

            return STATUS_UNKNOWN;
        }

        $logger->debug("calculatedResult [".self::$calculatedResult."] newValue [$newValue] oldValue [$oldValue] newTime [$newTime] oldTime [$oldTime] modifier [".self::$modifier."]", __METHOD__, __LINE__);
        
        $this->currentRun = array('time' => $newTime, 'query_value' => $newValue, 'query_base' => $newBase);
        $logger->info("currentRun Object ".var_export($this->currentRun, true), __METHOD__, __LINE__);
        $this->save();

        return STATUS_OK;
    }
}

########################################################################
# Basic three level verbosity logging.
# Overkill, but could be useful, later.
########################################################################

class Logger {
    #public const DEBUG     = 100;
    const DEBUG     = 100;
    const INFO      = 200;
    const NOTICE    = 250;
    const WARNING   = 300;
    const ERROR     = 400;
    const CRITICAL  = 500;
    const ALERT     = 550;
    const EMERGENCY = 600;

    protected static $levels = array(
        self::DEBUG     => 'DEBUG',
        self::INFO      => 'INFO',
        self::NOTICE    => 'NOTICE',
        self::WARNING   => 'WARNING',
        self::ERROR     => 'ERROR',
        self::CRITICAL  => 'CRITICAL',
        self::ALERT     => 'ALERT',
        self::EMERGENCY => 'EMERGENCY',
    );

    /**
     * @var string
     */
    private $logLevel;
    private $complex;
    private $name;

    public function __construct($channel = '') {
        $this->setName($channel);
        $this->setLogLevel(static::WARNING);
    }

    private function setName($name = '') {
        $this->name = $name;
    }

    public function getLogLevel() {
        return $this->logLevel;
    }

    private function setLogLevel($logLevel = '') {
        $this->logLevel = $logLevel;
    }

    public function complex() {
        $this->complex = true;
    }

    public function verbose() {
        $localLevels = static::$levels;
        $level = $this->getLogLevel();

        while(key($localLevels) !== null && key($localLevels) !== $level) {
            next($localLevels);
        }

        prev($localLevels);

        $level = key($localLevels);

        if (!empty($level)) {
            $this->setLogLevel($level);
        }
    }

    /**
     * Adds a log record at the DEBUG level.
     *
     * @param string $message The log message
     * @param string $caller The calling method (__METHOD__, __LINE__)
     * @param array  $context The log context
     */
    #public function debug($message, array $context = []): void {
    public function debug($message, $caller = "", $lineNumber = "", array $context = array()) {
        if (empty($caller)) {
            // Get the calling method.
            #$dbt = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS,2);    # DEBUG_BACKTRACE_IGNORE_ARGS unavailable before 5.3.6
            $dbt = debug_backtrace(true);    # PHP 5.3.6-
            $caller = isset($dbt[1]['function']) ? $dbt[1]['function'] : "";
        }

        $this->addRecord(static::DEBUG, (string) $message, (string) $caller, $context, $lineNumber);
    }

    public function info($message, $caller = "", $lineNumber = "", array $context = array()) {
        if (empty($caller)) {
            // Get the calling method.
            #$dbt = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS,2);    # DEBUG_BACKTRACE_IGNORE_ARGS unavailable before 5.3.6
            $dbt = debug_backtrace(true);    # PHP 5.3.6-
            $caller = isset($dbt[1]['function']) ? $dbt[1]['function'] : "";
        }

        $this->addRecord(static::INFO, (string) $message, (string) $caller, $context, $lineNumber);
    }

    public function notice($message, $caller = "", $lineNumber = "", array $context = array()) {
        if (empty($caller)) {
            // Get the calling method.
            #$dbt = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS,2);    # DEBUG_BACKTRACE_IGNORE_ARGS unavailable before 5.3.6
            $dbt = debug_backtrace(true);    # PHP 5.3.6-
            $caller = isset($dbt[1]['function']) ? $dbt[1]['function'] : "";
        }

        $this->addRecord(static::NOTICE, (string) $message, (string) $caller, $context, $lineNumber);
    }

    public function warning($message, $caller = "", $lineNumber = "", array $context = array()) {
        if (empty($caller)) {
            // Get the calling method.
            #$dbt = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS,2);    # DEBUG_BACKTRACE_IGNORE_ARGS unavailable before 5.3.6
            $dbt = debug_backtrace(true);    # PHP 5.3.6-
            $caller = isset($dbt[1]['function']) ? $dbt[1]['function'] : "";
        }

        $this->addRecord(static::WARNING, (string) $message, (string) $caller, $context, $lineNumber);
    }

    public function error($message, $caller = "", $lineNumber = "", array $context = array()) {
        if (empty($caller)) {
            // Get the calling method.
            #$dbt = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS,2);    # DEBUG_BACKTRACE_IGNORE_ARGS unavailable before 5.3.6
            $dbt = debug_backtrace(true);    # PHP 5.3.6-
            $caller = isset($dbt[1]['function']) ? $dbt[1]['function'] : "";
        }

        $this->addRecord(static::ERROR, (string) $message, (string) $caller, $context, $lineNumber);
    }

    public function critical($message, $caller = "", $lineNumber = "", array $context = array()) {
        if (empty($caller)) {

            // Get the calling method.
            #$dbt = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS,2);    # DEBUG_BACKTRACE_IGNORE_ARGS unavailable before 5.3.6
            $dbt = debug_backtrace(true);    # PHP 5.3.6-
            $caller = isset($dbt[1]['function']) ? $dbt[1]['function'] : "";
        }

        $this->addRecord(static::CRITICAL, (string) $message, (string) $caller, $context, $lineNumber);
    }

    public function alert($message, $caller = "", $lineNumber = "", array $context = array()) {
        if (empty($caller)) {
            // Get the calling method.
            #$dbt = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS,2);    # DEBUG_BACKTRACE_IGNORE_ARGS unavailable before 5.3.6
            $dbt = debug_backtrace(true);    # PHP 5.3.6-
            $caller = isset($dbt[1]['function']) ? $dbt[1]['function'] : "";
        }

        $this->addRecord(static::ALERT, (string) $message, (string) $caller, $context, $lineNumber);
    }

    public function emergency($message, $caller = "", $lineNumber = "", array $context = array()) {
        if (empty($caller)) {
            // Get the calling method.
            #$dbt = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS,2);    # DEBUG_BACKTRACE_IGNORE_ARGS unavailable before 5.3.6
            $dbt = debug_backtrace(true);    # PHP 5.3.6-
            $caller = isset($dbt[1]['function']) ? $dbt[1]['function'] : "";
        }

        $this->addRecord(static::EMERGENCY, (string) $message, (string) $caller, $context, $lineNumber);
    }

    /**
     * Adds a log record.
     *
     * @param  int    $level   The logging level
     * @param  string $message The log message
     * @param  string $caller The calling method (__METHOD__ or backtrace)
     * @param  array  $context The log context
     * @return bool   Whether the record has been processed
     */
    #public function addRecord(int $level, string $message, array $context = array()) bool {
    public function addRecord($level = 0, $message = "", $caller = "", $context = array(), $lineNumber = "") {
        // check if we need to handle this message.
        // Only proceed if the current message's level is equal to or greater than the minimum level to display.
        if ($level < $this->getLogLevel()) {
            return;
        }

        $record = array(
            'message' => $message,
            'context' => $context,
            'level' => $level,
            'level_name' => static::getLevelName($level),
            'channel' => $this->name,
            #'datetime' => new DateTimeImmutable($this->microsecondTimestamps, $this->timezone),
            #'datetime' => new DateTime("now", date_default_timezone_get()),
            'datetime' => microtime(true), #new DateTime("now", $dateTime->getTimezone()),
            'extra' => array(),
            'line' => $lineNumber,
        );

        #echo("[".$record['datetime']."] ".$record['channel'].".".$record['level'].": ".$record['message']." {".$record['context']."} [".$record['extra']."]".PHP_EOL);
        if ($this->complex) {
            echo("[".$this->udate("Ymd H:i:s.u T", $record['datetime'])."] ".(($record['channel']) ? $record['channel']."." : "").$record['level_name'].": ".((!empty($caller)) ? "[".$caller."] " : "").$record['message'].(!empty($record['line']) ? " (".$record['line'].")" : "").((!empty($record['context'][0])) ? " {".$record['context'][0]."}" : " []").((!empty($record['extra'])) ? " [".$record['extra']."]" : " []").PHP_EOL);flush();
    
        // Simple
        } else {
            echo($record['level_name'].": ".((!empty($caller)) ? "[".$caller."] " : "").$record['message'].(!empty($record['line']) ? " (".$record['line'].")" : "").PHP_EOL);flush();
        }

        return true;
    }

    /**
     * Gets the name of the logging level.
     *
     * @throws \Psr\Log\InvalidArgumentException If level is not defined
     */
    #public static function getLevelName(int $level): string {
    public static function getLevelName($level) {
        if (!isset(static::$levels[$level])) {
            throw new InvalidArgumentException('Level "'.$level.'" is not defined, use one of: '.implode(', ', array_keys(static::$levels)));
        }

        return static::$levels[$level];
    }

    private static function udate($format = 'u', $utimestamp = null) {
        if (is_null($utimestamp))
            $utimestamp = microtime(true);

        $timestamp = floor($utimestamp);
        $milliseconds = round(($utimestamp - $timestamp) * 1000000);

        return date(preg_replace('`(?<!\\\\)u`', $milliseconds, $format), $timestamp);
    }
}   // Logger Class

########################################################################
# Misc. helper functions
########################################################################

/** Echo a message and exit with a status. */
function nagios_exit($message, $status=0) {
    global $logger;
    global $overrideErrors;
    $logger->debug("", __METHOD__, __LINE__);

    echo($message);flush();

    # Make sure the $status is a number.
    if (!is_int($status)) {
        $logger->critical("##### EXIT STATUS CODE MUST BE NUMERIC!!!! [$status]", __METHOD__, __LINE__);
    }

    # Make sure there is an EOL.
    if (!strstr($message, PHP_EOL)) {
        echo PHP_EOL;
    }

    # This is for running multiple tests, so all the tests run, even with failures.
    if ($overrideErrors) {
        return;
    }

    exit($status);
}

########################################################################
# main() - Begin...
########################################################################

main();

?>
