#!/usr/bin/env python3
import winrm
import argparse
import getpass
import os
from sys import argv

try:
    # Parse command-line arguments
    parser = argparse.ArgumentParser(description='winrm_check_cpu.py uses WinRM to obtain CPU statistics on a remote, Windows-based system.')
    parser.add_argument('-H', '--host', required=True, type=str, help='IP address or host name of the Windows system.')
    parser.add_argument('-u', '--user', required=True, type=str, help='Username for connecting to the Windows system.')
    # choices=['cert', 'kerberos', 'credssp', 'basic-http', 'basic-https'], all supported auth methods to potentially add bellow*
    parser.add_argument('-a', '--auth', required=True, type=str, choices=['basic-http', 'basic-https'], help='Authentication mechanism for the Windows system. Strongly recommended you avoid basic-http.')
    parser.add_argument('-p', '--password', required=True, type=str, help='Password for connecting to the Windows system.')
    parser.add_argument('-C', '--certvalidation', required=False, action="count", default=0)

    parser.add_argument('-w', '--warning', required=False, type=float, help='Percent usage at which a warning alert with be sent.', default=0)
    parser.add_argument('-c', '--critical', required=False, type=float, help='Percent usage at which a critical alert with be sent', default=0)

    args = parser.parse_args(argv[1:])

except argparse.ArgumentError as e:
    print(f"\nArgument error: {str(e)}")
    exit(1)
except Exception as e:
    print(f"\nAn error occurred during argument parsing: {str(e)}")
    exit(1)

try:
    authentication = None

    if args.auth == 'basic-http':
        # Use basic-http Authentication
        authentication = 'basic'

    elif args.auth == 'basic-https':
        # Use basic-https Authentication
        authentication = 'ssl'

    elif args.auth == 'cert':
        # Use Certificate Authentication (TLS transport with SSL enabled)
        authentication = winrm.transport.TlsTransport(ssl=True)


    # CPU usage powershell script
    ps_cpu = """
    $strComputer = $Host
    Clear

    (WmiObject Win32_Processor).LoadPercentage | ConvertTo-Json -Compress
    """


    # Create a WinRM session with the provided host, user, password, and authentication method
    if args.certvalidation > 0:
        winrmsession = winrm.Session(args.host, auth=(args.user, args.password), transport=authentication)
    else:
        winrmsession = winrm.Session(args.host, auth=(args.user, args.password), transport=authentication, server_cert_validation='ignore')


    # run the ps script and decode the output
    r = winrmsession.run_ps(ps_cpu)
    final_dictionary = eval(r.std_out.decode('utf-8'))

    # calculate the disk stats
    cpu_usage = 0
    total_cpus = len(final_dictionary)
    for cpu in final_dictionary:
        cpu_usage += cpu
    cpu_usage = cpu_usage/total_cpus

    # check if the disk usage is within bounds of the warning/critical variables
    status_dict = {
        0: "OK",
        1: "WARNING",
        2: "CRITICAL",
        3: "UNKNOWN"
    }

    status_code = 0
    if args.warning and cpu_usage >= args.warning:
        status_code = 1
    if args.critical and cpu_usage >= args.critical:
        status_code = 2
    
    # build and return the status message/perf data
    cpu_warning = args.warning if args.warning else ''
    cpu_critical = args.critical if args.critical else ''
    message = f"{str(status_dict[status_code])}: CPU usage is {str(cpu_usage)}% | 'percent_used_cpu'={str(cpu_usage)}%;{str(cpu_warning)};{str(cpu_critical)};"
    
    print(message)
    exit(status_code)

except winrm.exceptions.WinRMTransportError as e:
    print(f"WinRM transport error: {str(e)}")
    exit(4)
except Exception as e:
    print(f"An error occurred during WinRM session setup or command execution: {str(e)}")
    exit(4)