<?php
//
// Linux SNMP Config Wizard
// Copyright (c) 2011-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

linuxsnmp_configwizard_init();

function linuxsnmp_configwizard_init()
{
    $name = "linux_snmp";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.2",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor a Linux workstation or server using SNMP."),
        CONFIGWIZARD_DISPLAYTITLE => _("Linux SNMP"),
        CONFIGWIZARD_FUNCTION => "linuxsnmp_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "snmppenguin.png",
        CONFIGWIZARD_FILTER_GROUPS => array('linux', 'server'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function linuxsnmp_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "linux_snmp";
    $process_count = 10;

    // Initialize return code and output
    $result = 0;
    $output = "";
    $snmptimeout = 10000000; //10 sec.

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $address = grab_array_var($inargs, "ip_address", "");
            $hostname = grab_array_var($inargs, "hostname", "");
            $port = grab_array_var($inargs, "port", "161");
            $snmpcommunity = grab_array_var($inargs, "snmpcommunity", "public");
            $snmpversion = grab_array_var($inargs, "snmpversion", "2c");
            $selectedhostconfig = grab_array_var($inargs, "selectedhostconfig", "");
            $operation = grab_array_var($inargs, "operation", "new");

            $snmpopts = "";
            $snmpopts_serial = grab_array_var($inargs, "snmpopts_serial", "");

            if ($snmpopts_serial != "") {
                $snmpopts = json_decode(base64_decode($snmpopts_serial), true);
            }

            if (!is_array($snmpopts)) {
                $snmpopts_default = array(
                    "v3_security_level" => "authPriv",
                    "v3_username" => "",
                    "v3_auth_password" => "",
                    "v3_privacy_password" => "",
                    "v3_auth_proto" => "md5",
                    "v3_priv_proto" => "des",
                );

                $snmpopts = grab_array_var($inargs, "snmpopts", $snmpopts_default);
            }

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $hostname = grab_array_var($inargs, "hostname", "");
            $port = grab_array_var($inargs, "port", 161);
            $snmpcommunity = grab_array_var($inargs, "snmpcommunity", "");
            $snmpversion = grab_array_var($inargs, "snmpversion", "");

            $address = nagiosccm_replace_user_macros($address);
            $snmpcommunity = nagiosccm_replace_user_macros($snmpcommunity);

            $snmpopts_serial = grab_array_var($inargs, "snmpopts_serial", "");

            if ($snmpopts_serial == "") {
                $snmpopts = grab_array_var($inargs, "snmpopts");
            } else {
                $snmpopts = json_decode(base64_decode($snmpopts_serial), true);
            }

            // Add port to address if we need to
            $runaddress = $address;

            if (!empty($port) && $port != 161) {
                $runaddress = $address.":".$port;
            }

            // Check for errors
            $errors = [];

            if (have_value($address) == false) {
                $errors[] = _("No address specified.");
            } else if (!valid_ip($address)) {
                $errors[] = _("Invalid IP address.");
            }

            if (have_value($snmpcommunity) == false && $snmpversion != "3") {
                $errors[] = _("No SNMP community specified.");
            }

            $test_timeout = 5000000;
            $test_oid = "iso.3.6.1.2.1.1.1.0"; // Simple, generic OID that should be very easy to respond to

            if ($snmpversion == "3") {
                $securitylevel = grab_array_var($snmpopts, "v3_security_level", "");
                $username = grab_array_var($snmpopts, "v3_username", "");
                $authproto = grab_array_var($snmpopts, "v3_auth_proto", "");
                $privproto = grab_array_var($snmpopts, "v3_priv_proto", "");
                $authpassword = grab_array_var($snmpopts, "v3_auth_password", "");
                $privacypassword = grab_array_var($snmpopts, "v3_privacy_password", "");

                // Run through user macros to account for illegal characters
                $username_replaced = nagiosccm_replace_user_macros($username);
                $authpassword_replaced = nagiosccm_replace_user_macros($authpassword);
                $privacypassword_replaced = nagiosccm_replace_user_macros($privacypassword);

                if (!$username ) {
                    $errors[] = "Username required for v3";
                }

                if ($securitylevel != 'noAuthNoPriv' && ! $authpassword) {
                    $errors[] = "Authentication password required for v3 authNoPriv and authPriv ";
                }

                if ($securitylevel == 'authPriv'  && ! $privacypassword) {
                    $errors[] = "Privacy password required for v3 authPriv ";
                }
                $test_reponse = snmp3_real_walk($runaddress, $username_replaced, $securitylevel, $authproto, $authpassword_replaced, $privproto, $privacypassword_replaced, $test_oid, $test_timeout);

            } else if ($snmpversion == "2c") {
                $test_reponse = snmp2_real_walk($runaddress, $snmpcommunity, $test_oid, $test_timeout);

            } else {
                $test_reponse = snmprealwalk($runaddress, $snmpcommunity, $test_oid, $test_timeout);
            }

            if (!$test_reponse) {
                $errors[] = "This server does not respond to SNMP requests (not snmp, or possibly wrong auth).";
            }


            if (count($errors)) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errors;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $port = intval(grab_array_var($inargs, "port", 161));
            $address_replaced = nagiosccm_replace_user_macros($address);
            $selectedhostconfig = grab_array_var($inargs, "selectedhostconfig", "");
            $operation = grab_array_var($inargs, "operation", "new");

            $hostname = grab_array_var($inargs, "hostname", @gethostbyaddr($address_replaced));

            // Catch if user is using DNS hostname as address and fill in field with the address value
            if (empty($hostname)) {
                $hostname = $address;
            }

            $hostname_replaced = nagiosccm_replace_user_macros($hostname);

            // Add port
            if (!empty($port) && $port != 161) {
                $address_replaced = $address_replaced.":".$port;
            }

            $snmpcommunity = grab_array_var($inargs, "snmpcommunity", "");
            $snmpcommunity_replaced = nagiosccm_replace_user_macros($snmpcommunity);
            $snmpversion = grab_array_var($inargs, "snmpversion", "");
            $snmpopts_serial = grab_array_var($inargs, "snmpopts_serial", "");

            if ($snmpopts_serial == "") {
                $snmpopts = grab_array_var($inargs, "snmpopts");
            } else {
                $snmpopts = json_decode(base64_decode($snmpopts_serial), true);
            }

            $ptips = grab_request_var("ptips", 0);
            $walksuccess = 1;
            $process = array();
            $disk = array();
            $process_oid = "HOST-RESOURCES-MIB::hrSWRunName";
            $disk_oid = "HOST-RESOURCES-MIB::hrFSMountPoint";

            ////////////////////////////////////////////////////////////////////////
            // Walk the process and disk OIDs to display on stage 2 select inputs //
            if ($snmpversion == "3") {
                $securitylevel = grab_array_var($snmpopts, "v3_security_level");
                $username = grab_array_var($snmpopts, "v3_username");
                $authproto = grab_array_var($snmpopts, "v3_auth_proto");
                $privproto = grab_array_var($snmpopts, "v3_priv_proto");
                $authpassword = grab_array_var($snmpopts, "v3_auth_password");
                $privacypassword = grab_array_var($snmpopts, "v3_privacy_password");

                // Run through user macros to account for illegal characters
                $username_replaced = nagiosccm_replace_user_macros($username);
                $authpassword_replaced = nagiosccm_replace_user_macros($authpassword);
                $privacypassword_replaced = nagiosccm_replace_user_macros($privacypassword);

                $process_name_table = snmp3_real_walk($address_replaced, $username_replaced, $securitylevel, $authproto, $authpassword_replaced, $privproto, $privacypassword_replaced, $process_oid, $snmptimeout);
                $disk_name_table = snmp3_walk($address_replaced, $username_replaced, $securitylevel, $authproto, $authpassword_replaced, $privproto, $privacypassword_replaced, $disk_oid, $snmptimeout);
            } else {
                $process_name_table = snmp2_real_walk($address_replaced, $snmpcommunity_replaced, $process_oid, $snmptimeout);
                $disk_name_table = snmp2_walk($address_replaced, $snmpcommunity_replaced, $disk_oid, $snmptimeout);
            }

            $process = array();
            $disk = array();

            // If any walks fail, let user know
            if ($disk_name_table == false || $process_name_table == false) {
                $walksuccess = 0;
            }

            // Processes
            if (!empty($process_name_table)) {
                foreach ($process_name_table as $key => $val) {
                    $matchCount = preg_match('/"([^"]+)"/', $val, $name);
                    $name = (($matchCount > 0) ? $name[1] : '');

                    // Create array of names
                    array_push($process, '"' . $name . '"');
                }

                // Trim repeated process names
                $process = array_unique($process);

                // get count
                $process_count = count($process);
                $process = implode(",", $process);
            }

            // Disks
            if (!empty($disk_name_table)) {
                foreach ($disk_name_table as $key => $val) {
                    $matchCount = preg_match('/"([^"]+)"/', $val, $name);
                    $name = (($matchCount > 0) ? $name[1] : '');

                    // Create array of names
                    array_push($disk, '"' . $name . '"');
                }

                // List of disks
                $disk_count = count($disk);
                $disk = implode(",", $disk);
            }
            // End of walk section //
            /////////////////////////

            $services = "";
            $services_serial = grab_array_var($inargs, "services_serial", "");

            if ($services_serial != "") {
                $services = json_decode(base64_decode($services_serial), true);
            }

            if (!is_array($services)) {
                $services_default = array(
                    "ping" => 'on',
                    "cpu" => 'on',
                    "memory" => 'on',
                    "nobuffers" => 'on',
                    "pagefile" => 'on',
                    "disk" => 'on',
                );

                $services = grab_array_var($inargs, "services", $services_default);
            }

            $serviceargs = "";
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            if (!empty($serviceargs_serial)) {
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            }

            if (!is_array($serviceargs)) {
                $serviceargs_default = array(
                    "cpu_warning" => 80,
                    "cpu_critical" => 90,
                    "memory_warning" => 80,
                    "memory_critical" => 90,
                    "pagefile_warning" => 5,
                    "pagefile_critical" => 10,
                    "disk_warning" => array(),
                    "disk_critical" => array(),
                    "disk" => array(),
                    "processstate" => array(),
                );

                for ($x = 0; $x < 5; $x++) {
                    if (!empty($disk)) { // we have snmpwalk data
                        $serviceargs_default["disk_warning"][$x] = 80;
                        $serviceargs_default["disk_critical"][$x] = 95;
                        $serviceargs_default["disk"][$x] = "";
                    } else { // if we dont have walk data return to default
                        $serviceargs_default["disk_warning"][$x] = 80;
                        $serviceargs_default["disk_critical"][$x] = 95;
                        $serviceargs_default["disk"][$x] = ($x == 0) ? "/" : "";
                    }
                }

                for ($x = 0; $x < 5; $x++) {
                    if (!empty($process)) { // we have snmpwalk data
                        $serviceargs_default['processstate'][$x]['process'] = '';
                        $serviceargs_default['processstate'][$x]['name'] = '';
                        $serviceargs_default['processstate'][$x]['warn'] = '';
                        $serviceargs_default['processstate'][$x]['crit'] = '';
                    } else { // if we dont have walk data return to default
                        if ($x == 0) {
                            $serviceargs_default['processstate'][$x]['process'] = 'httpd';
                            $serviceargs_default['processstate'][$x]['name'] = 'Apache';
                        } else if ($x == 1) {
                            $serviceargs_default['processstate'][$x]['process'] = 'mysqld';
                            $serviceargs_default['processstate'][$x]['name'] = 'MySQL';
                        } else if ($x == 2) {
                            $serviceargs_default['processstate'][$x]['process'] = 'sshd';
                            $serviceargs_default['processstate'][$x]['name'] = 'SSH';
                        } else {
                            $serviceargs_default['processstate'][$x]['process'] = '';
                            $serviceargs_default['processstate'][$x]['name'] = '';
                        }

                        $serviceargs_default['processstate'][$x]['warn'] = '';
                        $serviceargs_default['processstate'][$x]['crit'] = '';
                    }
                }

                $serviceargs = grab_array_var($inargs, "serviceargs", $serviceargs_default);
            }

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $port = grab_array_var($inargs, "port", 161);
            $hostname = grab_array_var($inargs, "hostname");
            $legacy_perfdata = grab_array_var($inargs, "legacy_perfdata");
            $snmpcommunity = grab_array_var($inargs, "snmpcommunity", "");
            $snmpversion = grab_array_var($inargs, "snmpversion", "");
            $services = grab_array_var($inargs, "services", []);
            $serviceargs = grab_array_var($inargs, "serviceargs");

            #
            # Data passed back from "next" steps.
            #
            # Required here so validation passes.
            #
            # TODO: WHY IS THIS NOT BEING HANDLED IN monitoringwizards.php?????
            #
            $passbackdata = grab_array_var($inargs, "passbackdata", "");

            if ($passbackdata) {
                $decodedpassbackdata = json_decode(base64_decode($passbackdata), true);

                $address = $decodedpassbackdata['ip_address'];
                $port = $decodedpassbackdata['port'];
                $hostname = $decodedpassbackdata['hostname'];
                $legacy_perfdata = $decodedpassbackdata["legacy_perfdata"];
                $snmpcommunity = $decodedpassbackdata['snmpcommunity'];
                $snmpversion = $decodedpassbackdata['snmpversion'];

                $snmpopts_serial = $decodedpassbackdata['snmpopts_serial'];

                $services_serial = $decodedpassbackdata['services_serial'];
                $serviceargs_serial = $decodedpassbackdata['serviceargs_serial'];

                $services = json_decode(base64_decode($services_serial), true);
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            }

            // check for errors
            $errors = 0;
            $errmsg = array();

            if (is_valid_host_name($hostname) == false) {
                $errmsg[$errors++] = "Invalid host name.";
            }

            $check_plugin_error = configwizard_snmp_verify_wc($services, $serviceargs);

            if (!empty($check_plugin_error)) {
                foreach($check_plugin_error as $error_message) {
                    $errmsg[$errors++] = $error_message;
                }
            }

            if (count($services) == 0) {
                $errmsg[$errors++] = "You have not selected any OIDs to monitor.";
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $port = grab_array_var($inargs, "port", 161);
            $hostname = grab_array_var($inargs, "hostname");
            $legacy_perfdata = grab_array_var($inargs, 'legacy_perfdata');
            $snmpcommunity = grab_array_var($inargs, "snmpcommunity", "");
            $snmpversion = grab_array_var($inargs, "snmpversion", "");
            $services = grab_array_var($inargs, "services");
            $serviceargs = grab_array_var($inargs, "serviceargs");

            $snmpopts_serial = grab_array_var($inargs, "snmpopts_serial", "");

            if ($snmpopts_serial == "") {
                $snmpopts = grab_array_var($inargs, "snmpopts");
            } else {
                $snmpopts = json_decode(base64_decode($snmpopts_serial), true);
            }

            $services = grab_array_var($inargs, "services");
            $serviceargs = grab_array_var($inargs, "serviceargs");

            # BAD!!  The default values set in the previous stages will overwrite the "new" values passed in by the form.
            #$services_serial = grab_array_var($inargs, "services_serial", base64_encode(json_encode($services)));
            #$serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", base64_encode(json_encode($serviceargs)));
            $services_serial = base64_encode(json_encode($services));
            $serviceargs_serial = base64_encode(json_encode($serviceargs));

            #
            # Data passed back from "next" steps.
            #
            # TODO: WHY IS THIS NOT BEING HANDLED IN monitoringwizards.php?????
            #
            $passbackdata = grab_array_var($inargs, "passbackdata", "");

            if ($passbackdata) {
                $decodedpassbackdata = json_decode(base64_decode($passbackdata), true);

                $address = $decodedpassbackdata['ip_address'];
                $port = $decodedpassbackdata['port'];
                $hostname = $decodedpassbackdata['hostname'];
                $legacy_perfdata = $decodedpassbackdata['legacy_perfdata'];
                $snmpcommunity = $decodedpassbackdata['snmpcommunity'];
                $snmpversion = $decodedpassbackdata['snmpversion'];

                $snmpopts_serial = $decodedpassbackdata['snmpopts_serial'];

                $services_serial = $decodedpassbackdata['services_serial'];
                $serviceargs_serial = $decodedpassbackdata['serviceargs_serial'];
            }

            $output = '

        <input type="hidden" name="ip_address" value="' . encode_form_val($address) . '">
        <input type="hidden" name="port" value="' . intval($port) . '">
        <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '">
        <input type="hidden" name="legacy_perfdata" value="' . encode_form_val($legacy_perfdata) . '">
        <input type="hidden" name="snmpcommunity" value="' . encode_form_val($snmpcommunity) . '">
        <input type="hidden" name="snmpversion" value="' . encode_form_val($snmpversion) . '">
        <input type="hidden" name="snmpopts_serial" value="' . base64_encode(json_encode($snmpopts)) . '">
        <input type="hidden" name="services_serial" value="' . $services_serial . '">
        <input type="hidden" name="serviceargs_serial" value="' . $serviceargs_serial . '">
';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $hostname = grab_array_var($inargs, "hostname", "");
            $address = grab_array_var($inargs, "ip_address", "");
            $port = intval(grab_array_var($inargs, "port", 161));
            $legacy_perfdata = grab_array_var($inargs, 'legacy_perfdata', "");
            $snmpcommunity = grab_array_var($inargs, "snmpcommunity", "");
            $snmpversion = grab_array_var($inargs, "snmpversion", "");
            $hostaddress = $address;

            $snmpopts_serial = grab_array_var($inargs, "snmpopts_serial", "");
            $snmpopts = json_decode(base64_decode($snmpopts_serial), true);

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["ip_address"] = $address;
            $meta_arr["snmpcommunity"] = $snmpcommunity;
            $meta_arr["snmpversion"] = $snmpversion;
            $meta_arr["services"] = $services;
            $meta_arr["serivceargs"] = $serviceargs;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = array();

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_linuxsnmp_host",
                    "host_name" => $hostname,
                    "address" => $hostaddress,
                    "icon_image" => "snmppenguin.png",
                    "statusmap_image" => "snmppenguin.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            $nobuffers = 'Memory';
            if (array_key_exists('nobuffers', $services) && $services['nobuffers'] == 'on') {
                $nobuffers = 'Physical';
            }

            // If this wizard was run before the plugin had the option, retain old perfdata order.
            // Otherwise, we want to have the plugin sort the perfdata to guarantee a consistent ordering.
            // --swolf 2024-07-01
            $legacy_perfdata_args = empty($legacy_perfdata) ? " --sort-perfdata " : "";

            // Determine SNMP args
            $snmpargs = "";

            if (!empty($port) && $port != 161) {
                $snmpargs .= " -p $port";
            }

            if (!empty($snmpcommunity) && $snmpversion != "3") {
                $snmpargs .= " -C " . escapeshellarg_nowrap($snmpcommunity);

                if ($snmpversion == "2c") {
                    $snmpargs .= " --v2c";
                }

            } else if ($snmpversion == "3") {

                // Do snmpv3 connecion auth
                $securitylevel = grab_array_var($snmpopts, "v3_security_level");
                $username = grab_array_var($snmpopts, "v3_username");
                $authproto = grab_array_var($snmpopts, "v3_auth_proto");
                $privproto = grab_array_var($snmpopts, "v3_priv_proto");
                $authpassword = grab_array_var($snmpopts, "v3_auth_password");
                $privacypassword = grab_array_var($snmpopts, "v3_privacy_password");

                if ($username != "")
                    $snmpargs .= " --login=" . $username;
                if ($authpassword != "")
                    $snmpargs .= " --passwd=" . escapeshellarg_nowrap($authpassword);
                if ($privacypassword != "")
                    $snmpargs .= " --privpass=" . escapeshellarg_nowrap($privacypassword);

                if ($authproto != "" && $privproto != "") {
                    $snmpargs .= " --protocols=" . $authproto . "," . $privproto;
                } else if ($authproto != "" ) {
                    $snmpargs .= " --protocols=" . $authproto;
                }
            }

            // see which services we should monitor
            foreach ($services as $svc => $svcstate) {

                //echo "PROCESSING: $svc -> $svcstate<BR>\n";

                switch ($svc) {

                    case "ping":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Ping",
                            "check_command" => "check-host-alive",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "cpu":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "CPU Usage",
                            "use" => "xiwizard_linuxsnmp_load",
                            "check_command" => "check_xi_service_snmp_linux_load!" . $snmpargs . " -w " . $serviceargs["cpu_warning"] . " -c " . $serviceargs["cpu_critical"] . " -f",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "memory":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Memory Usage",
                            "use" => "xiwizard_linuxsnmp_storage",
                            "check_command" => "check_xi_service_snmp_linux_storage!" . $snmpargs . " -m " . $nobuffers . " $legacy_perfdata_args -w " . $serviceargs["memory_warning"] . " -c " . $serviceargs["memory_critical"] . " -f",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "pagefile":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Swap Usage",
                            "use" => "xiwizard_linuxsnmp_storage",
                            "check_command" => "check_xi_service_snmp_linux_storage!" . $snmpargs . " -m 'Swap' $legacy_perfdata_args -w " . $serviceargs["pagefile_warning"] . " -c " . $serviceargs["pagefile_critical"] . " -f",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "disk":
                        $donedisks = array();
                        $diskid = 0;
                        foreach ($serviceargs["disk"] as $diskname) {

                            if ($diskname == "")
                                continue;

                            //echo "HANDLING DISK: $diskname<BR>";

                            // we already configured this disk
                            if (in_array($diskname, $donedisks))
                                continue;
                            $donedisks[] = $diskname;

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => $diskname . " Disk Usage",
                                "use" => "xiwizard_linuxsnmp_storage",
                                "check_command" => "check_xi_service_snmp_linux_storage!" . $snmpargs . " -m \"^" . $diskname . "$\" $legacy_perfdata_args -w " . $serviceargs["disk_warning"][$diskid] . " -c " . $serviceargs["disk_critical"][$diskid] . " -f",
                                "_xiwizard" => $wizard_name,
                            );

                            $diskid++;
                        }
                        break;


                    case "processstate":

                        $enabledprocs = $svcstate;
                        foreach ($enabledprocs as $pid => $pstate) {

                            $pname = $serviceargs["processstate"][$pid]["process"];
                            $pdesc = $serviceargs["processstate"][$pid]["name"];
                            $pwarn = $serviceargs["processstate"][$pid]["warn"];
                            $pcrit = $serviceargs["processstate"][$pid]["crit"];

                            if ($pwarn !== "")
                                $pwarn = "-w '" . $pwarn . "'";

                            if ($pcrit !== "")
                                $pcrit = "-c '" . $pcrit . "'";

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => $pdesc,
                                "use" => "xiwizard_linuxsnmp_process",
                                "check_command" => "check_xi_service_snmp_linux_process!" . $snmpargs . " -r -n " . $pname . " " . $pwarn . " " . $pcrit . "",
                                "_xiwizard" => $wizard_name,
                            );
                        }
                        break;


                    default:
                        break;
                }
            }

            // echo "OBJECTS:<BR>";
            // print_r($objs);
            //exit();

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            /* Debugging
            echo "\n<textarea class='dbg' style='width:400px;height:80px'>\$objs:\n". print_r($objs,true) . "</textarea>\n";

            error_log( "\n" . debug_backtrace()[0]['function'] . " - \$objs: ".print_r($objs,true) . "\n");
            */

            break;

        default:
            break;
    }

    return $output;
}

// Restrict illegal plugin warn/crit values so services checks don't fail on syntax errors
function configwizard_snmp_verify_wc($services, $serviceargs)
{
    $output = array();
    $warning = "";
    $critical = "";

    foreach ($services as $k => $value) {
        switch ($k) {

            case "processstate":
                foreach ($services["processstate"] as $key => $val) {
                    unset($warning_min);
                    unset($warning_max);
                    unset($critical_min);
                    unset($critical_max);

                    if ($val == "on") {
                        $warning = explode(",", $serviceargs["processstate"][$key]['warn']);
                        $critical = explode(",", $serviceargs["processstate"][$key]['crit']);

                        // Set warning min/max
                        if (!empty($warning[1])) {
                            $warning_min = $warning[0];
                            $warning_max = $warning[1];
                        } else {
                            $warning_min = $warning[0];
                        }

                        // Set critical min/max
                        if (!empty($critical[1])) {
                            $critical_min = $critical[0];
                            $critical_max = $critical[1];
                        } else {
                            $critical_min = $critical[0];
                        }

                        // Check to make sure max values are positive
                        if ((!empty($warning_max) && $warning_max < 0) || ((!empty($critical_max) && $critical_max < 0))) {
                            array_push($output, "Error: Max Warning and Critical must be > 0 on process service " . $serviceargs["processstate"][$key]['process']);
                        }

                        // Check critical_min < warning_min < warning_max < critical_max
                        if ($warning_min < $critical_min)
                            array_push($output, "Error: Warning minimum must be >= Critical minimum on process service " . $serviceargs["processstate"][$key]['process']);

                        if (!empty($warning_max)) {
                            if ($warning_max <= $warning_min)
                                array_push($output, "Error: Warning minimum must be < Warning maximum on process service " . $serviceargs["processstate"][$key]['process']);
                        } else if (!empty($critical_max) && ($critical_max <= $warning_min)) {
                            array_push($output, "Error: Warning minimum must be < Critical maximum when no maximum Warning defined on process service " . $serviceargs["processstate"][$key]['process']);
                        } else if (!empty($critical_max) && !empty($warning_max) && ($critical_max < $warning_max)) {
                            array_push($output, "Error: Warning maximum must be <= Critical maximum on process service " . $serviceargs["processstate"][$key]['process']);
                        }
                    }
                }

                break;

            case "disk":

                if ($value == "on") {
                    $warning = $serviceargs[$k . "_warning"];
                    $critical = $serviceargs[$k . "_critical"];

                    foreach ($critical as $key => $val) {
                        $disk_row = $key + 1;
                        if (!empty($val)) {
                            if ($critical[$key] < $warning[$key]) {
                                array_push($output, "Error: Warning must be < Critical when creating a disk service (row " . $disk_row . ")");
                            }
                        }
                    }
                }

                break;

            case "cpu":
            case "memory":
            case "pagefile":

                if ($value == "on") {
                    $warning = $serviceargs[$k . "_warning"];
                    $critical = $serviceargs[$k . "_critical"];

                    if (($critical < $warning) && !empty($critical)) {
                        array_push($output, "Error: Warning must be < Critical when creating a " . $k . " service.");
                    }
                }

                break;

            default:
                break;

        }
    }

    return $output;
}
