<?php
//
// Favorites Menu Component
// Copyright (c) 2020-2024 Nagios Enterprises, LLC. All rights reserved.
//

require_once(dirname(__FILE__) . '/../componenthelper.inc.php');
require_once(dirname(__FILE__) . '/../../common.inc.php');

$favorites_component_name = 'favorites';

favorites_component_init();

function favorites_component_init()
{
    global $favorites_component_name;
    $versionok = favorites_component_checkversion();

    $desc = _('This component adds a \'favorite\' button to every page, and creates a link for favorited pages in the left navigation menu');

    if (!$versionok) {
        $desc = sprintf("<b>%s: %s</b>", _('Error'), _('This component requires Nagios XI 2024R1.1 or later.'));
    }

    $args = array(
        COMPONENT_NAME => $favorites_component_name,
        COMPONENT_VERSION => '2.0.2',
        COMPONENT_AUTHOR => 'Nagios Enterprises, LLC',
        COMPONENT_DESCRIPTION => $desc,
        COMPONENT_TITLE => _("Favorites"),
        COMPONENT_REQUIRES_VERSION => 60100,
        COMPONENT_CONFIGFUNCTION => "favorites_config_func"
    );

    register_component($favorites_component_name, $args);

    if ($versionok) {
        register_callback(CALLBACK_PAGE_HEAD, 'favorites_head_include');
        register_callback(CALLBACK_MENUS_INITIALIZED, 'favorites_component_addmenu');
        register_callback(CALLBACK_BODY_START, 'favorites_bodystart_callback');
    }

}

function favorites_head_include()
{
    $enabled = intval(get_option("favorites_enabled", 1));
    if (!$enabled) {
        return;
    }

    echo "<script type='text/javascript' src='" . get_base_url() . "includes/components/favorites/favorites_head.js'></script>";
}

// Requires XI 2024R1.1 or later
function favorites_component_checkversion()
{
    if (!function_exists('get_product_release')) { return false; }
    if (get_product_release() < 60100) { return false; }
    return true;
}

function favorites_get_user_favorites()
{
    global $db_tables;

    $user_id = 0;
    if (isset($_SESSION['user_id'])) {
        $user_id = $_SESSION['user_id'];
    }

    $sql = "SELECT item_id, user_id, title, partial_href
            FROM " . $db_tables[DB_NAGIOSXI]['cmp_favorites'] ."
            WHERE user_id = " . intval($user_id);

    $rows = exec_sql_query(DB_NAGIOSXI, $sql);
    if ($rows) {
        $rows = $rows->GetRows();
    }

    return $rows;
}

function is_user_favorite($page = null, $query_param = null)
{
    global $db_tables;

    if (!isset($page)) {
        return false;
    }

    $user_id = 0;

    if (isset($_SESSION['user_id'])) {
        $user_id = $_SESSION['user_id'];
    }

    $sql = "SELECT COUNT(item_id) as count
            FROM ".$db_tables[DB_NAGIOSXI]['cmp_favorites']."
            WHERE user_id = ".intval($user_id)."
              AND partial_href LIKE '".escape_sql_param($page, DB_NAGIOSXI)."'";
    if (!empty($query_param)) {
        $sql .= "  OR partial_href LIKE '%".escape_sql_param($query_param, DB_NAGIOSXI)."%'";
    }

    $count = 0;

    $rows = exec_sql_query(DB_NAGIOSXI, $sql);

    if (!empty($rows) && isset($rows->fields['count'])) {
        $count = (int) $rows->fields['count'];
    }

    $retval = ($count > 0) ? true : false;

    return $retval;
}

function favorites_bodystart_callback( $cbtype, &$cbargs )
{
    if (install_needed()) {
        return;
    }
    $enabled = intval(get_option("favorites_enabled", 1));

    if (!$enabled) {
        return;
    }

    $export = grab_request_var('export', 0);
    if ($export) { // favorites icon will be suppressed for scheduled reports
        return;
    }

    // List of pages to not show the favorites icon on
    $pages = array('rr.php', 'install.php', 'dashboard.php', 'classic-dashlet-wrapper.php', 'classic-dashlet-config.php');
    $page = basename($_SERVER['PHP_SELF']);
    $query_string = $_SERVER['QUERY_STRING'];
    parse_str($query_string, $query_params);
    
    if ($cbargs['child'] == 1 && !in_array($page, $pages)) {

        $hide = '';
        
        if (array_key_exists('msg', $_SESSION)) {
            $hide = 'style="display: none;"';
        }

        $menu_info = favorites_get_user_favorites();

        // Wizards have a query parameter 'wizard'
        $wizard_name = (array_key_exists('wizard', $query_params) ? $query_params['wizard'] : "");

        $wizard_required_version = "";

        if (empty($wizard_name)) {
            $request = $_REQUEST;

            // After Step 1, wizards do not have query parameters.  Use Request.
            if (array_key_exists('wizard', $request)) {
                $wizard_name = $request['wizard'];
                $hide = 'style="display: none;"';
            }
        }
        
        # Wizards will now use bootstrap5/html5.
        if (!empty($wizard_name)) {
            $wizard = get_configwizard_by_name($wizard_name);
            $wizard_required_version = $wizard[CONFIGWIZARD_REQUIRES_VERSION];
        }

        if ($wizard_required_version >= CONFIGWIZARD_BOOTSTRAP5_REQUIRES_VERSION) {
            $request_uri = $page;
        } else {
            $request_uri = str_replace("/nagiosxi/", "", $_SERVER['REQUEST_URI']);
        }
?>
        <div class="helpsystem_icon favorites_icon tt-bind">
<?php
            $is_favorite = '';
            $is_favorite_string = "Add to Favorites";

            if (is_user_favorite($request_uri, $wizard_name)) {
                $is_favorite = " style=\"color:gold; font-variation-settings:'FILL' 1\"";
                $is_favorite_string = "Remove from Favorites";
            }

            $not_neptune = 'md-button md-action';

            if (is_neptune()) {
                $not_neptune = '';
            }
?>
            <a class="btn btn-outline-primary btn-nagios btn-icon tt-bind" <?= $hide ?> title="<?= _($is_favorite_string) ?>" data-placement="left" data-user-id="<?= $_SESSION['user_id'] ?>">
                <i class="material-symbols-outlined action-icon <?= $not_neptune ?>" <?= $is_favorite ?>>kid_star</i>
            </a>
        </div>
<?php
    }
}

function favorites_component_addmenu() 
{
    global $menus;

    $enabled = intval(get_option("favorites_enabled", 1));
    if (!$enabled) {
        return;
    }

    $menu_info = favorites_get_user_favorites();
    if (empty($menu_info)) {
        return;
    }

    $top_level_menus = array_keys($menus);

    foreach ($top_level_menus as $name) {

        if(is_neptune()) {
            add_menu_item($name, array(
                'type' => "menusection",
                'title' => _("Favorites"),
                'id' => "menu-$name-section-favorites",
                'order' => 50,
                "opts" => array(),
            ));
            $length = count($menu_info);
            for ($i = 0; $i < $length && $i < 48; $i++) {
                $data = $menu_info[$i];
                add_menu_item($name, array(
                    'type' => MENULINK,
                    'title' => encode_form_val($data['title']),
                    'id' => "menu-$name-favorities-link-".$data['item_id'],
                    'order' => 51 + $i,
                    /* href does not need a slash, but it can have one without issues */
                    "opts" => array('href' => get_base_url(false) . preg_replace('/(?<=&nsp=)(.*)(?=&)/', get_nagios_session_protector_id(), $data['partial_href'])),
                ));
            }
        } else {
            add_menu_item($name, array(
                'type' => "html",
                'order' => 50,
                "opts" => array("html" => '
                    <div class="menusection "><div class="menusectiontitle" data-id="menu-'.$name.'-section-favorites" id="mid-menu-'.$name.'-section-favorites"><i class="fa fa-chevron-down" title="'._('Collapse menu section').'"></i> '._('Favorites').' <a class="favorites_manage tt-bind" title="'._('Manage favorites').'" style="float: right; position: relative; top: -2px;"><img src="'.theme_image('cog.png').'"></a> <a class="favorites_cancel hide tt-bind" title="'._('Cancel').'" style="float: right;"><img src="'.theme_image('disable_small.png').'"></a> <a class="favorites_save hide tt-bind" title="'._('Save changes').'" style="float: right; margin-right: 5px;"><img src="'.theme_image('enable_small.gif').'"></a></div><ul class="menusection">'),
            ));
            $hidden_form_html = '<form class="favorites_hidden_form hide">';

        $length = count($menu_info);
        for ($i = 0; $i < $length && $i < 48; $i++) {
            $data = $menu_info[$i];
            add_menu_item($name, array(
                'type' => MENULINK,
                'title' => encode_form_val($data['title']),
                'id' => "menu-$name-link-$i",
                'order' => 51 + $i,
                /* href does not need a slash, but it can have one without issues */
                /* if there is an old nsp in the url, replace it with the current one */
                "opts" => array('href' => get_base_url(false) . preg_replace('/(?<=&nsp=)(.*)(?=&)/', get_nagios_session_protector_id(), $data['partial_href'])),
            ));

                $hidden_form_html .= '<li class="menulink" style="margin-top: 2px;"><input type="text" class="form-colors form-control condensed" name="item['.$data['item_id'].']" data-original-value="'.encode_form_val($data['title']).'" value="'.encode_form_val($data['title']).'" style="width: 145px"><img class="favorites_delete tt-bind" title="'._('Remove').'" data-placement="right" style="cursor: pointer; margin-left: 3px;" src="'.theme_image('delete_new.png').'"></li>';
            }
            $hidden_form_html .= '</form>';

            add_menu_item($name, array(
                'type' => "html",
                'order' => 98,
                'opts' => array('html' => $hidden_form_html),
            ));

            add_menu_item($name, array(
                'type' => MENUSECTIONEND,
                'id' => 'menu-$name-sectionend-favorites',
                'order' => 99,
                'opts' => ''
            ));
        }
    }

}

function favorites_config_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $result = 0;
    $output = '';

    $component_name = "favorites";

    switch ($mode) {
        case COMPONENT_CONFIGMODE_GETSETTINGSHTML:

            $enabled = intval(get_option("favorites_enabled", 1));

            $output .= '
    <h5 class="ul"> ' . _('Favorites Menu Settings') . '</h5>

    <table class="table table-condensed table-no-border table-auto-width">
        <tr>
            <td></td>
            <td class="checkbox">
                <label>
                    <input type="hidden" name="enabled" value="0" />
                    <input type="checkbox" class="checkbox" id="enabled" name="enabled" value="1" ' . is_checked($enabled, 1) . '>' . _('Enable Favorites Menu').'
                </label>
            </td>
        </tr>
    </table>';

            break;

        case COMPONENT_CONFIGMODE_SAVESETTINGS:
            $enabled = intval(grab_array_var($inargs, 'enabled', '0'));

            set_option('favorites_enabled', $enabled);
            break;
        default:
            break;
    }
    return $output;
}