public class NagiosThresholds {
	private NagiosThreshold warning;
	private NagiosThreshold critical;
	private String type;
	private int status;

	public NagiosThresholds(String warningRaw, String criticalRaw, String type) {

		warning = new NagiosThreshold(warningRaw, type);
		critical = new NagiosThreshold(criticalRaw, type);
		status = 3;

		if(!warning.isNull() && !critical.isNull() && (warning.isNumeric() != critical.isNumeric())) {
			throw new IllegalArgumentException("If both thresholds are set, they must both be of the same type (numeric or string-comparison).");
		}
	}

	public int check (Object value) throws IllegalArgumentException {
		if (critical.checkThreshold(value)) {
			status = 2;
			return 2;
		}
		if (warning.checkThreshold(value)) {
			status = 1;
			return 1;
		}
		status = 0;
		return 0;
	}

	public int getStatus() {
		return status;
	}

	public String getRawThreshold(boolean wantsWarning) {
		if (wantsWarning) {
			return warning.getRaw();
		}
		return critical.getRaw();
	}

	class NagiosThreshold {
		private String type; // long | double | string
		private boolean isFlipped;

		private String raw;
		private String theString;
		private double lowDouble;
		private double highDouble;
		private long lowLong;
		private long highLong;

		public NagiosThreshold(String thresholdRaw, String type) {

			this.raw = thresholdRaw;
			if(type.equals("long") || type.equals("string") || type.equals("double")) {
				this.type = type;
			}
			else if (type.equals("rate")) {
				this.type = "double";
			}
			else {
				throw new IllegalArgumentException("Threshold type " + type + " not string/long/double");
			}
			/*
			char unit = thresholdRaw.charAt(thresholdRaw.length);
			int mult = 1;
			if (unit == 'K') {
				mult = 1024;
			}
			if (unit == 'M') {
				mult = 1048576;
			}
			if (unit == 'G') {
				mult = 1073741824;
			}*/

			isFlipped = false;
			lowDouble = 0;
			lowLong = 0;
			highDouble = 0;
			highLong = 0;

			if (thresholdRaw == null || thresholdRaw.equals("")) {
				return;
			}

			if(thresholdRaw.charAt(0) == '@') {
				isFlipped = true;
				thresholdRaw = thresholdRaw.substring(1);
			}
			String[] thresholdSplit = thresholdRaw.split(":");
			if (thresholdSplit.length == 1 && thresholdRaw.charAt(thresholdRaw.length()-1) == ':') {
				// In any reasonable language this is caught under thresholdSplit.length == 2
				try {
					if (type.equals("long")) {
						lowLong = Long.parseLong(thresholdSplit[0]);
						highLong = Long.MAX_VALUE;
					}
					else if (type.equals("double")) {
						lowDouble = Double.parseDouble(thresholdSplit[0]);
						highDouble = Double.POSITIVE_INFINITY;
					}
					else { // type.equals("string")
						theString = thresholdSplit[0];
					}
				}
				catch (NumberFormatException e) {
					if (thresholdSplit[0].equals("~")) {
						lowDouble = Double.NEGATIVE_INFINITY;
						lowLong = Long.MIN_VALUE;
					}
					else if (thresholdSplit[0].equals("")) {
						// lower bounds already set to 0.
					}
					else {
						throw new IllegalArgumentException("Threshold lower bounds must be a valid double, the tilde (~), or the empty string.");
					}
				}
			}
			else if (thresholdSplit.length == 1) {
				try {
					if (type.equals("long")) {
						highLong = Long.parseLong(thresholdSplit[0]);
					}
					else if (type.equals("double")) {
						highDouble = Double.parseDouble(thresholdSplit[0]);
					}
					else { // type.equals("string")
						theString = thresholdSplit[0];
					}
				}
				catch (NumberFormatException e) {
					if (thresholdSplit[0].equals("")) {
						highDouble = Double.POSITIVE_INFINITY; // the "null" threshold
						highLong = Long.MAX_VALUE;
					}
					else {
						throw new IllegalArgumentException("Singleton Thresholds must be valid numbers (for numeric thresholds) or of the form \"[@]sSomeString\" for string-comparison thresholds.");
					}
				}
			}
			else if (thresholdSplit.length == 2) {
				try {
					if (type.equals("long")) {
						lowLong = Long.parseLong(thresholdSplit[0]);
					}
					else {
						lowDouble = Double.parseDouble(thresholdSplit[0]);
					}
				}
				catch (NumberFormatException e) {
					if (thresholdSplit[0].equals("~")) {
						lowDouble = Double.NEGATIVE_INFINITY;
						lowLong = Long.MIN_VALUE;
					}
					else if (thresholdSplit[0].equals("")) {
						// lower bounds already set to 0.
					}
					else {
						throw new IllegalArgumentException("Threshold lower bounds must be a valid double, the tilde (~), or the empty string.");
					}
				}
				try {
					if (type.equals("long")) {
						highLong = Long.parseLong(thresholdSplit[1]);
					}
					else {
						highDouble = Double.parseDouble(thresholdSplit[1]);
					}
				}
				catch (NumberFormatException e) {
					if (thresholdSplit[0].equals("")) {
						highDouble = Double.POSITIVE_INFINITY;
						highLong = Long.MAX_VALUE;
					}
					else {
						throw new IllegalArgumentException("Threshold upper bounds must be a valid doubleing point number or the empty string.");
					}
				}
			}

			if ((!Double.isNaN(highDouble) && !Double.isNaN(lowDouble) && highDouble < lowDouble) || highLong < lowLong) {
				throw new IllegalArgumentException("Numeric Thresholds must be such that the upper bound is greater than or equal to the lower bound.");
			}
		}

		public boolean isNull() {
			return (lowDouble == 0.0) && (highDouble == Double.POSITIVE_INFINITY) && (lowLong == 0) && (highLong == Long.MAX_VALUE);
		}

		public boolean isNumeric() {
			return type.equals("long") || type.equals("double");
		}

		// return true iff the threshold should alert

		public boolean checkThreshold(Object value) {
			try {
				return checkThresholdSub((Double) value);
			}
			catch (IllegalArgumentException | ClassCastException e) {
				assert true;
			}
			try {
				return checkThresholdSub((Long) value);
			}
			catch (IllegalArgumentException | ClassCastException e) {
				assert true;
			}
			try {
				return checkThresholdSub(new Long(((Integer) value).longValue()));
			}
			catch (IllegalArgumentException | ClassCastException e) {
				assert true;
			}
			try {
				return checkThresholdSub((String) value);
			}
			catch (IllegalArgumentException | ClassCastException e) {
				throw new IllegalArgumentException("value " + value.toString() + " is not of a valid type (Double, Long, String).");
			}
		}


		// I don't know exactly how overloading is done so these have a separate name just to be safe
		public boolean checkThresholdSub(double value) {
			if (type.equals("string")) {
				throw new IllegalArgumentException("This check has type " + type + ", but our metric was of type double.");
			}
			if (isFlipped) {
				return (value >= lowDouble && value <= highDouble);
			}
			else {
				return (value < lowDouble || value > highDouble);
			}
		}

		public boolean checkThresholdSub(String value) {
			if (!type.equals("string")) {
				throw new IllegalArgumentException("This check has type " + type + ", but our metric was of type String.");
			}
			if (isFlipped) {
				return !value.equals(theString);
			}
			return value.equals(theString);
		}

		public boolean checkThresholdSub(long value) {
			if (type.equals("string")) {
				throw new IllegalArgumentException("This check has type " + type + ", but our metric was of type long.");
			}
			if (isFlipped) {
				return (value >= ((long) lowLong) && value <= ((long) highLong));
			}
			else {
				return (value < ((long) lowLong) || value > ((long) highLong));
			}
		}

		public String getRaw() {
			return raw;
		}
	}

}
